/***************************************************************************
 File: sxser.c

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.
			  
 You may use this Sample Code any way you please provided you do not resell 
 the code and that this notice (including the above copyright notice) is 
 reproduced on all copies.  THIS SAMPLE CODE IS PROVIDED "AS IS", WITHOUT 
 WARRANTY OF ANY KIND, AND GO CORPORATION EXPRESSLY DISCLAIMS ALL IMPLIED 
 WARRANTIES, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES OF 
 MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. IN NO EVENT WILL GO 
 CORPORATION BE LIABLE TO YOU FOR ANY CONSEQUENTIAL,INCIDENTAL, OR INDIRECT 
 DAMAGES ARISING OUT OF THE USE OR INABILITY TO USE THIS SAMPLE CODE.

 $Revision:   1.0  $
   $Author:   aloomis  $
     $Date:   13 Aug 1992 12:15:16  $

 This file contains the methods for SIO messages. Also contains the
 entrypoint for the subtask.

****************************************************************************/
#ifndef NOTE_INCLUDED
#include <note.h>
#endif

#ifndef SERVMGR_INCLUDED
#include <servmgr.h>
#endif

#ifndef STREAM_INCLUDED
#include <stream.h>
#endif

#ifndef INTL_INCLUDED
#include <intl.h>
#endif

#ifndef SXAPP_INCLUDED
#include "sxapp.h"
#endif
#include "methods.h"

/* Subtask entry point */
void LOCAL SerialReadTask(P_SX_SUBTASK_DATA);

/* Delay before subtask is signaled to try to connect to the stream */
#define SIGNAL_TASK_DELAY	500

/* Tables for transmission error note */
static const TK_TABLE_ENTRY	sxNoteContent[] = {
	{tagSXTxErrorStr, 0, 0, 0, tkLabelStringId},
	{pNull}
};

const TK_TABLE_ENTRY	sxNoteButton[] = {
	{tagSXOKStr, 0, 0, 0, tkLabelStringId},
	{pNull}
};

/*
 * Respond to msgSXOpenSerial.
 *
 * Open the default (the first one available) serial service instance or the
 * one specified by the user.
 * Initialize to default settings and indicate I want to know about
 * transmission errors. Find out the current connection
 * status, launch subtask to read serial input and finally self send
 * msgSXSetSerialMetrics to initialize port to prefered settings.
 */
MsgHandlerWithTypes(SXOpenSerial, P_UNKNOWN, PP_SX_APP_DATA)
{
	P_SX_APP_DATA pInst = *pData;
 	SM_ACCESS saccess;
	SIO_INIT sinit;
	SIO_EVENT_SET ses;
	SM_GET_STATE servstate;
	OBJECT sio;
	STATUS s;

	Dbg(Debugf(U_L("SXOpenSerial >>"));)

	/* Since I grab the port and won't let go, I can use SMAccess etc..
	 * If you don't need exclusive access to the serial port, bind the
	 * server and only open it if you need it. <servmgr.h> shows how to do
	 * that.
	 */
 	saccess.pServiceName = pInst->sxSerPrefs.spSerServ;
 	saccess.caller = self;

	ObjCallWarn(msgSMAccessDefaults, theSerialDevices, &saccess);
 	ObjCallRet(msgSMAccess, theSerialDevices, &saccess, s);

	sio = saccess.service;

	/* Initialize to default state, use small buffers */
	sinit.inputSize = 512;
	sinit.outputSize = 512;
	ObjCallWarn(msgSioInit, sio, &sinit);

	/* I'm only interested in transmission errors (well not really,
	 * but just to show how).
	 */
	ses.eventMask = sioEventRxError;
	ses.client = self;
	ObjCallWarn(msgSioEventSet, sio, &ses);

	/* The Service Manager will keep me updated about the connection
 	 * status, however I want to find out the inital status.
	 */
	servstate.handle = (OBJECT) saccess.handle;
	servstate.connected = false;
	ObjCallWarn(msgSMGetState, theSerialDevices, &servstate);

	/* Update instance data */
	pInst->sxSerConnected = servstate.connected;
	pInst->sxSIOService = saccess.service;
	pInst->sxSIOHandle = saccess.handle;

	/* Duplicate service for subtask */
	pInst->sxSubTaskData->sdSIOService = pInst->sxSIOService;

	/* And create the subtask to read serial input. Note that task will
	 * start out waiting for me to wake it up.
	 */
	StsRet(OSSubTaskCreate((P_OS_SUBTASK_ENTRY)SerialReadTask, 8192,
			false, (U32)pInst->sxSubTaskData, &(pInst->sxSubTaskID)), s);

	/* Give task a name */
	StsWarn(OSTaskNameSet(pInst->sxSubTaskID, (U_L("SXSR"))));

	/* Set serial preferences. The method handler will signal the
	 * subtask to start reading the serial port.
	 */
	ObjCallWarn(msgSXSetSerialMetrics, self, (P_ARGS)SET_SERIAL_ALL);

	Dbg(Debugf(U_L("<< SXOpenSerial"));)

	return stsOK;

	MsgHandlerParametersNoWarning;

} /* SXOpenSerial */
/*
 * Respond to msgSXCloseSerial.
 *
 * Drop DTR and RTS, terminate subtask and close port.
 */
MsgHandlerWithTypes(SXCloseSerial, P_UNKNOWN, PP_SX_APP_DATA)
{
	P_SX_APP_DATA pInst = *pData;
	SM_RELEASE srelease;
	SIO_CONTROL_OUT_SET sco;
	STATUS s;

	Dbg(Debugf(U_L("SXCloseSerial >>"));)

	/* pull dtr and rts low to physically disconnect */
	sco.dtr = false;
	sco.rts = false;
	sco.out1 = false;
	sco.out2 = false;
	ObjCallWarn(msgSioControlOutSet, (*pData)->sxSIOService, &sco);

	/* release the serial port */
 	srelease.caller = self;
 	srelease.service = pInst->sxSIOService;
	srelease.handle = pInst->sxSIOHandle;
 	ObjCallRet(msgSMRelease, theSerialDevices, &srelease, s);

	/* Terminate serial task. */
	OSTaskTerminate((*pData)->sxSubTaskID, 1);
	pInst->sxSubTaskID = 0;

	/* Update instance data */
	pInst->sxSIOService = objNull;
	pInst->sxSIOHandle = objNull;

	Dbg(Debugf(U_L("<< SXCloseSerial"));)

	return stsOK;

	MsgHandlerParametersNoWarning;
} /* SXCloseSerial */

/*
 * Respond to msgSXSetSerialMetrics.
 *
 * Implements the various ways to set serial metrics. Uses enums to indicate
 * which method should be used. Note that I don't re-read the serial metrics
 * at all if for some obscure reason (like the hw doesn't support it) setting
 * the metrics fails. So it is possible that the settings in the instance data
 * and the hardware settings are out of sync.
 */
MsgHandlerWithTypes(SXSetSerialMetrics, P_ARGS, PP_SX_APP_DATA)
{
	P_SX_APP_DATA pInst = *pData;

	Dbg(Debugf(U_L("SXSetSerialMetrics >>"));)

	switch ((U32)pArgs) {

		case SET_SERIAL_ALL:
			{
			SIO_METRICS smetrics;

			/* Initialize serial port to preferences */
			ObjCallWarn(msgSioGetMetrics, pInst->sxSIOService, &smetrics);
			smetrics.baud = pInst->sxSerPrefs.spBaudRate;
			smetrics.line.dataBits = pInst->sxSerPrefs.spDataBits;
			smetrics.line.stopBits = pInst->sxSerPrefs.spStopBits;
			smetrics.line.parity = pInst->sxSerPrefs.spParity;
			smetrics.flowType.flowControl = pInst->sxSerPrefs.spFlowControl;
			ObjCallWarn(msgSioSetMetrics, pInst->sxSIOService, &smetrics);
			}
			break;

		case SET_SERIAL_BAUDRATE:
			ObjCallWarn(msgSioBaudSet, (*pData)->sxSIOService,
				(P_ARGS)(*pData)->sxSerPrefs.spBaudRate);
			break;

		case SET_SERIAL_LINECONTROL:
			{
			SIO_LINE_CONTROL_SET slc;

			slc.parity = (*pData)->sxSerPrefs.spParity;
			slc.dataBits = (*pData)->sxSerPrefs.spDataBits;
			slc.stopBits = (*pData)->sxSerPrefs.spStopBits;
			ObjCallWarn(msgSioLineControlSet, (*pData)->sxSIOService, &slc);
			}
			break;

		case SET_SERIAL_FLOWCONTROL:
			{
			SIO_FLOW_CONTROL_SET flow;

			flow.flowControl = (*pData)->sxSerPrefs.spFlowControl;

			ObjCallWarn(msgSioFlowControlSet, (*pData)->sxSIOService, &flow);
			}
			break;
		default:
			Dbg(Debugf(U_L("Unknown case switch"));)
			break;
	}

	/* Changing the metrics caused the stream that my subtask is reading
	 * to disconnect. Wake up serial read task and tell it it should try
	 * to connect to the stream again.
	 * Delay to prevent the subtask getting the signal while the
	 * connection status hasn't been resolved yet.
	 */
	OSTaskDelay(SIGNAL_TASK_DELAY);
	OSSemaReset((*pData)->sxSubTaskData->sdEventSemaphore);

	Dbg(Debugf(U_L("<< SXSetSerialMetrics"));)

	return stsOK;

	MsgHandlerParametersNoWarning;
} /* SXSetSerialMetrics */

/*
 * Respond to msgSXSendSerial.
 *
 * Sends pArgs buffer to the serial port. Terminates each string with a
 * CRLF and flushes the stream.
 */
MsgHandlerWithTypes(SXSendSerial, P_CHAR, PP_SX_APP_DATA)
{
	P_SX_APP_DATA pInst = *pData;
	STREAM_READ_WRITE_TIMEOUT srw;
 	STATUS s;

	Dbg(Debugf(U_L("SXSendSerial >>"));)

	/* Do nothing if I'm not connected (or open for that matter) */
	if (pInst->sxSerConnected) {

		/* Support unicode... */
		srw.numBytes = Ustrlen(pArgs) * SizeOf(CHAR);

		/* Do nothing if there were no characters entered in the IP */
		if (srw.numBytes > 0) {

			srw.pBuf = pArgs;
			srw.timeOut = 750;

			ObjCallRet(msgStreamWriteTimeOut, 
				pInst->sxSIOService, &srw, s);
			/* Send CRLF */
			srw.pBuf = "\r\n";
			srw.numBytes = 2;
			ObjCallRet(msgStreamWriteTimeOut, 
				pInst->sxSIOService, &srw, s);

			/* Flush stream */
			ObjCallRet(msgStreamFlush, pInst->sxSIOService,
				pNull, s);
		}
	} else {
		Dbg(Debugf(U_L("Not Open/Connected"));)
	}

	Dbg(Debugf(U_L("<< SXSendSerial"));)

	return stsOK;

	MsgHandlerParametersNoWarning;
} /* SXSendSerial */

/*
 * Respond to msgSXSMConnectedChanged.
 *
 * Send by service manager when a change in the connection status has occured.
 * Signal the subtask to start reading serial input if the connection has
 * been made . Update the serial card Status label to reflect current connection
 * status.
 */
MsgHandlerWithTypes(SXSMConnectedChanged, P_SM_CONNECTED_NOTIFY, PP_SX_APP_DATA)
{
	P_SX_APP_DATA pInst = *pData;

	Dbg(Debugf(U_L("SXSMConnectedChanged >>"));)

	Dbg(Debugf(U_L("Connected %ld"), pArgs->connected);)
	pInst->sxSerConnected = pArgs->connected;

	/* Has a connection been made? */
	if ((*pData)->sxSerConnected) {

		/* Update option card status label, if existent */
		ObjCallWarn(msgSXSetConnectStatusId, self, (P_ARGS)tagSXConnectedStr);

		/* Signal subtask */
		Dbg(Debugf(U_L("Signalling SerialRead Task"));)
		OSTaskDelay(SIGNAL_TASK_DELAY);
		OSSemaReset(pInst->sxSubTaskData->sdEventSemaphore);

	} else {
		/* Lost connection. Serial read task will drop into semawait */
		ObjCallWarn(msgSXSetConnectStatusId, self, (P_ARGS)tagSXNotConnectedStr);
	}

	if (pInst->sxOptWin)
		ObjCallWarn(msgSXResizeCard, self, pInst->sxOptWin);

	Dbg(Debugf(U_L("<< SXSMConnectedChanged"));)

	return stsOK;

	MsgHandlerParametersNoWarning;
} /* SXSMConnectedChanged */

/*
 * Respond to msgSioEventHappened.
 *
 * Respond to an event, in this case only EventRxError. Note that more events
 * bits may be set in the mask, even those I didn't express interest in.
 * Don't do much about the apparent transmission error. Could put up a note
 * or something. You can also be informed when the serial input buffer is no
 * longer empty. That is most suited for non-continious serial I/O, since it
 * has some overhead.
 */
MsgHandlerWithTypes(SXSioEventHappened, P_SIO_EVENT_HAPPENED, PP_SX_APP_DATA)
{
	NOTE_NEW nn;
	MESSAGE m;
	STATUS s;

	Dbg(Debugf(U_L("SXSioEventHappened >>"));)

	if (pArgs->eventMask & sioEventRxError) {

		Dbg(Debugf(U_L("sioEventRxError"));)

		/* Show a simple note */
		ObjCallWarn(msgNewDefaults, clsNote, &nn);
		nn.note.metrics.flags = nfDefaultAppFlags | nfAutoDestroy;
		nn.note.pContentEntries	= sxNoteContent;
		nn.note.pCmdBarEntries	= sxNoteButton;
		ObjCallRet(msgNew, clsNote, &nn, s);

		ObjCallWarn(msgNoteShow, nn.object.uid, &m);

	}

	Dbg(Debugf(U_L("<< SXSioEventHappened"));)

	return stsOK;

	MsgHandlerParametersNoWarning;
} /* SXSioEventHappened */

/*
 * Entry point for the subtask which reads serial input and displays it in the
 * textview. This task will never terminate itself, it will be terminated when
 * the serial port is closed. Two loops are used, the first to wait for its
 * parent to signal it to start reading, the second waiting for input from the
 * serial port. Note that when the stream is disconnected you _must not_ retry;
 * since you'll suck up all CPU time busy-waiting in an endless loop of
 * disconnected status return values, not leaving time for the system to
 * remake the stream status if the connection is re-established.
 *
 * The subtask is passed a pointer to a SX_SUBTASK_DATA structure, which holds
 * the semaphore, serial service and the textdata object to which to write
 * the input. The subtask will NOT write that structure.
 */
void LOCAL SerialReadTask(P_SX_SUBTASK_DATA pSubTaskData)
{
	STREAM_READ_WRITE_TIMEOUT srw;
	U8 charBuf, failureCounter;
	STATUS s;

	Dbg(Debugf(U_L("SerialReadTaskID %ld"), OSThisTask());)

	/* Read serial buffer 1 char at a time (U8, serially I won't get
	 * more than 8 bits). In this case that will be smoother than
	 * waiting for a buffer to fill or a timeout.
	 */
	srw.pBuf = &charBuf;
	srw.numBytes = 1;
	srw.timeOut = osInfiniteTime;

	/* Loop forever waiting for something to happen */
	for (;;) {

		/* Wait for parent to signal me to go ahead */
		OSSemaWait(pSubTaskData->sdEventSemaphore, osInfiniteTime);

		failureCounter = 0;

		/* Good morning. Loop forever waiting input to arrive at the
		 * serial port. Grab each char and stuff it in the textview.
		 * Safe since the user can not modify the textview him/herself
		 * (though clsTextView seems to handle dual input just fine).
		 *
		 * Note that if SX is compiled with DEBUG flags on and the debug
		 * version of PenPoint is used, it is possible to receive a
		 * connection-change message (making the parent wake the subtask
		 * up) before the stream is 'reconnected', causing the subtask to
		 * drop right back in the semaphore wait. To work around this,
		 * a short delay is executed before the semaphore is reset in
		 * the main task (see SXSetSerialMetrics and SXSMConnectedChanged).
		 *
		 * I've never seen this with production-compiled code and a non-
		 * debug version of PenPoint.
		 */
		for (;;) {
			s = ObjectCall(msgStreamReadTimeOut, pSubTaskData->sdSIOService, &srw);

			/* If the connection is dropped, serial metrics are changed,
			 * or the port is closed, the stream will be disconnected.
			 * At that point I cannot retry, it will fail forever. Instead
			 * I'll wait for my parent to either signal me it is ok to wait
			 * for serial I/O again or terminate me (shudder).
			 */
			if (s == stsOK) {

				/* Doesn't do any emulation, not even backspacing */
				TextInsertOne(pSubTaskData->sdTextObject, infTEXT_INDEX, (CHAR)charBuf);
			} else if (s == stsStreamDisconnected) {
				Dbg(Debugf(U_L("Stream Disconnected"));)
				break;
			} else {
				/* try again, but not too often */
				Dbg(Debugf(U_L("StreamRead Failure."));)
				Dbg(StsPrint(s);)
				if (failureCounter++ == 10)
					break;	/* drop into wait */
			}
		}
		/* Set semaphore to 1. Parent will reset it if it wants me to
		 * wake up.
		 */
		OSSemaSet(pSubTaskData->sdEventSemaphore);
	}
} /* SerialReadTask */
