/*****************************************************************************/
/*                     ACPA Device Specific MIDI Functions                   */
/*                                                                           */
/*    DevNoteOn()                                                            */
/*    DevNoteOff()                                                           */
/*    DevAllNotesOff()     Make sure all notes are off                       */
/*    DevAfterTouch()                                                        */
/*    DevControlChange                                                       */
/*    DevProgramChange                                                       */
/*    DevChannelPressure()                                                   */
/*    DevPitchBend                                                           */
/*    DevMTC()                                                               */
/*    DevSysEx()                                                             */
/*    SetClock()           Called when Tempo or PPQN changes                 */
/*                                                                           */
/* 5/04/92 BRR???  Creation of this file. Split out of ACPA.C.               */
/*****************************************************************************/

/*****************************************************************************/
/* DEFINES                                                                   */
/*****************************************************************************/

#define  ARMINUS     0x90
#define  ARPLUS      0xa0
#define  CONFIGMASK  15
#define  HIGHBUF     0x1d00
#define  LOWBUF      0x0c00
#define  PSPRET      0xce26
#define  SACL        0x6000
#define  ZAC         0xca00

/*****************************************************************************/
/* Include DEFINEs                                                           */
/*****************************************************************************/

#if IS_OS2
#define  INCL_DOS
#define  INCL_DOSINFOSEG
#include <os2.h>
#endif

/*****************************************************************************/
/* Include Files                                                             */
/*****************************************************************************/

#include <stdio.h>
#include <conio.h>
#include <stdlib.h>
#include <string.h>

#include "audiodd.h"
#include "auddef.h"
#include "audproto.h"
#include "audiodd2.h"
#include "acpadef.h"

#include "gmvoices.c"

/*****************************************************************************/
/* External Function declarations                                            */
/*****************************************************************************/

extern unsigned int acpa_in  (unsigned int addr);
extern void         acpa_out (unsigned int addr, unsigned int data);

/*****************************************************************************/
/* Internal Function declarations                                            */
/*****************************************************************************/

void acpa_noteoff (char gennum);
void acpa_noteon  (char gennum, char pitch, char velocity);
void setvoice     (int inst, int table, int voiceid);

/*****************************************************************************/
/* External Variable declarations                                            */
/*****************************************************************************/

extern char           acpalock;
extern unsigned int   hostbase;
extern int            max_midi_bytes;
extern unsigned short pspdataloc;
extern unsigned int   timing_prescaler;

/* Track Specific External Variable Declarations                             */
#if NUM_TRACKS > 1
   extern unsigned long pos_buffer_count[];
   extern int           ppqn[];
   extern unsigned long rollover_time[];
   extern unsigned int  tempo[];
   extern unsigned long time_per_block[];
   extern struct vscb   xmitio[];
#else
   extern unsigned long pos_buffer_count;
   extern int           ppqn;
   extern unsigned long rollover_time;
   extern unsigned int  tempo;
   extern unsigned long time_per_block;
   extern struct vscb   xmitio;
#endif

/*****************************************************************************/
/* Internal Variable declarations                                            */
/*****************************************************************************/

unsigned char BlockSysex[10]  = { 0xf0,0,0,0x3a,6,5,0,0,106,0xf7 };
unsigned char qcaps_sysex[10] = { 0xf0,0,0,0x3a,5,1,0,0x10,0,0xf7 };
unsigned char qid_sysex[10]   = { 0xf0,0,0,0x3a,5,4,'A','C','P',0xf7 };
unsigned char TimbreSysex[12] = { 0xf0,0,0,0x3a,6,2,0,0,0,0,0,0xf7 };

unsigned int codefixup;
unsigned int datafixup;
unsigned int dlarea;
unsigned int dlload;
unsigned int dlpsiz;
int          do_relink = 0;
unsigned int gates;
char         instconfigs[16] = { 15,15,15,15, 15,15,15,15, 15,15,15,15, 15,15,15,15 };
unsigned int old_dlarea;
int          ProgChgEnable = 1; /* Program changes enabled by default        */


struct midiblk midiblks[16];
struct noteblk noteblks[8];
struct noteblk *offnotes = NULL;
struct noteblk *onnotes = NULL;


/*****************************************************************************/
/*                         DevNoteOn                                         */
/*  This routine will select a sound generator to use for a noteon.          */
/*  Noteblks are kept for each note generator.  These are linked onto        */
/*  2 queues: onnotes & offnotes, indicating whether they are currently      */
/*  sounding (on), or off.  The oldest notes are located at the end of       */
/*  the queue, with new noteblks added to the front.                         */
/*  It will attempt to locate one in the following priority:                 */
/*       1. If MONO, currently on voice with same channel (& voice?)         */
/*       2. If MONO, currently off voice with same channel & voice           */
/*       3. Oldest Off note with same voice, channel, & keynum               */
/*       4. Oldest Off note with same voice                                  */
/*       5. Oldest Off note with same configuration                          */
/*       6. Oldest Off note                                                  */
/*       7. Oldest On note with same voice                                   */
/*       8. Oldest On note with same config                                  */
/*       9. Oldest On note                                                   */
/*  If a new generator is selected while in MONO mode, and a generator is    */
/*  still on for that channel, then it will be turned off.  This can be      */
/*  caused by changing voices while in MONO mode between note-on & off.      */
/*  In MONO mode, the generator is NOT retriggered if the note-on occurs     */
/*  prior to the note-off for the previous note (in concept anyways.  I'm    */
/*  not sure how this should be done).                                       */
/*  General MIDI specifies that the lower MIDI channels have priority. It    */
/*  doesn't really clarify this though.  My assumption is that the lower     */
/*  channels may steal from higher channels, even if they are still on,      */
/*  but but higher channels may not steal from lower channels 'on' notes.    */
/*****************************************************************************/
void DevNoteOn(midichan,key,velocity)
char midichan,key,velocity;
{
   extern struct noteblk *onnotes,*offnotes;
   extern struct midiblk midiblks[];
   struct midiblk *mptr;
   struct noteblk *nptr;
   struct noteblk *got_nptrs[9];          /* Ptrs to blks in priority seq    */
                                          /* as listed above (-1)            */
   int x,found,pitch;
   char config;
   unsigned char voice;

   if(velocity==0){
      DevNoteOff(midichan,key,0);
      return;
   }

   mptr = &midiblks[midichan & 0x0f];         /* Get midiblk for this chan   */
   for(x=0; x<9; x++) got_nptrs[x] = NULL;    /* Zero out ptrs to found blks */
   found = 0;                                 /* Clear found flag            */

   /* Handle dynamic voice selection for drum kit             General MIDI   */
   if(midichan==9){                           /* Is this the drum kit?       */
      voice = (char)(key+128-35);             /* Convert to drum kit         */
      config = voices[voice][5];              /*                             */
   }else{
      voice = mptr->mbvoicenum;               /* Get id of voice             */
      config = mptr->mbconfig;                /*   and config                */
   }

   /* Search offnote queue                                                   */
   if(nptr = offnotes){
      while(nptr){                            /* Loop for all notes (if any) */
         if(nptr->nbvoicenum == (char)voice){ /* Same voice?                 */
            if((mptr->mbflags1 & MONO)
            && (nptr->nbchannel==midichan)){
               got_nptrs[1] = nptr;            /*  Set priority 2 pointer    */
            }else if(nptr->nbchannel==midichan /*    same channel            */
                  && nptr->nbkeynum==key){     /*    and key also?           */
               got_nptrs[2] = nptr;            /*  Set priority 3 pointer    */
            }else{
               got_nptrs[3] = nptr;            /*  Set priority 4 pointer    */
            }

         }else if(nptr->nbconfig==config){    /* Same config?                */
            got_nptrs[4] = nptr;              /*   Set priority 5 pointer    */

         }else{
            got_nptrs[5] = nptr;              /*   Set priority 6 pointer    */
         }
         found=1;
         nptr = nptr->nbnext;
      }
   }
   if((offnotes==NULL) || (mptr->mbflags1 & MONO)){
      /* Search onnote queue   */
      nptr = onnotes;
      while(nptr){                             /* Loop for all notes         */
         if((mptr->mbflags1 & MONO)            /* MONO mode?                 */
         && (nptr->nbchannel==midichan)){      /*   Same channel?            */
            got_nptrs[0] = nptr;               /*   Set priority 0 ptr       */
         }else if(nptr->nbchannel >= midichan){ // Is note on higher chan? General MIDI
            if(nptr->nbvoicenum==(char)voice){ /* Same voice?                */
               got_nptrs[6] = nptr;            /*   Set priority 7 ptr       */
            }else if(nptr->nbconfig==config){  /* Same config?               */
               got_nptrs[7] = nptr;            /*  Set priority 8 ptr        */
            }else{
               got_nptrs[8] = nptr;            /*  Set priority 9 pointer    */
            }
         }
         nptr = nptr->nbnext;
      }
   }
   /* Now select the highest priority block                                  */
   for(x=0; x<9 && got_nptrs[x]==NULL; x++);
   if(x==9) return;              /* Return if we didn't find one             */
   nptr = got_nptrs[x];

  /* If note was on, then issue noteoff                                      */
  if(x>5){
     acpa_noteoff(nptr->nbgennum);
  }

   /* Unqueue block from whereever it is                                     */
   if(nptr->nbprev==NULL){                      /* Is block at front of q?   */
      if(onnotes==nptr){
         onnotes=nptr->nbnext;
         if(onnotes) onnotes->nbprev=NULL;
      }else if(offnotes==nptr){
         offnotes=nptr->nbnext;
         if(offnotes) offnotes->nbprev=NULL;
      }
   }else{
      nptr->nbprev->nbnext = nptr->nbnext;      /* Set prev's next           */
   }
   if(nptr->nbnext){
      nptr->nbnext->nbprev = nptr->nbprev;      /* Set next's prev           */
   }

   /* Queue block to front of on queue                                       */
   nptr->nbprev = NULL;                         /* Set our prev to NULL      */
   if(onnotes) onnotes->nbprev = nptr;          /* Set next's prev to us     */
   nptr->nbnext = onnotes;                      /* Set our next              */
   onnotes = nptr;                              /* and link us in            */

   /* Now turn on the note                                                   */
   nptr->nbkeynum = key;
   nptr->nbchannel = midichan;
   nptr->nbflags1 = 0;                          /* Clear flags               */
   if(nptr->nbvoicenum!=(char)voice){
      nptr->nbvoicenum=(char)voice;
      setvoice(nptr->nbgennum,0,voice);
   }
   if(nptr->nbconfig!=config){
      nptr->nbconfig=config;
      do_relink = 1;
   }
   if(mptr->mbflags1 & MONO){                   /* MONO mode?                */
      acpa_out(pspdataloc+(128*nptr->nbgennum)+9,1);
   }else{
      acpa_out(pspdataloc+(128*nptr->nbgennum)+9,0);
   }

   /* Add transpose value to pitch, and adjust to within voice range */
   pitch = key+mptr->mbtranspose;
   while((pitch >= 12) && (pitch > mptr->mbkeyhi)) pitch-=12;
   while((pitch <= 127-12) && (pitch < mptr->mbkeylo)) pitch+=12;

   acpa_noteon(nptr->nbgennum,(char)pitch,velocity);
}

void acpa_noteon(gennum,pitch,velocity)
char  gennum,pitch,velocity;
{
   acpa_out(pspdataloc+0x4e+(gennum*128),pitch-12);      /* +78 = pitch      */
   acpa_out(pspdataloc+0x6b+(gennum*128),velocity);      /* +107 = onvel     */
   gates = acpa_in(0x1f80) | ( 0x0101 << gennum);
   acpa_out(0x1f80,gates);
}

/*****************************************************************************/
/* NOTEOFF                                                                   */
/*****************************************************************************/
void DevNoteOff(midichan,key,velocity)
char midichan,key,velocity;
{
   extern struct noteblk *onnotes,*offnotes;
   struct noteblk *nptr;

   if(midiblks[midichan].mbflags1 & SUSTAIN){  /* Is sustain pedal pushed?   */
      nptr = onnotes;                        /* Locate noteblk for this note */
      while (nptr
             && !(nptr->nbchannel==midichan
                  && nptr->nbkeynum==key)){
         nptr = nptr->nbnext;
      }
      if(nptr){
         nptr->nbflags1 |= SUSTAIN;
      }
      return;
   }

   /* Locate noteblk for this note    */
   nptr = onnotes;
   while (nptr && !(nptr->nbchannel==midichan && nptr->nbkeynum==key)){
      nptr = nptr->nbnext;
   }

   /* Did we find it?                                                        */
   if(nptr){

      /* Turn off note                                                       */
      acpa_noteoff(nptr->nbgennum);

      /* Move noteblk to offnotes queue                                      */
      /* Unqueue block from whereever it is                                  */
      if(nptr->nbprev==NULL){                    /* Is block at front of q?  */
         if(onnotes==nptr){
            onnotes=nptr->nbnext;
            if(onnotes) onnotes->nbprev=NULL;
         }
      }else{
         nptr->nbprev->nbnext = nptr->nbnext;    /* Set prev's next          */
      }
      if(nptr->nbnext){
         nptr->nbnext->nbprev = nptr->nbprev;    /* Set next's prev          */
      }

      /* Queue block to front of off queue                                   */
      nptr->nbprev = NULL;                       /* Set our prev to NULL     */
      if(offnotes) offnotes->nbprev = nptr;      /* Set next's prev to us    */
      nptr->nbnext = offnotes;                   /* Set our next             */
      offnotes = nptr;                           /* and link us in           */

   }
}

/*****************************************************************************/
/* acpa_noteoff                                                              */
/*****************************************************************************/
void acpa_noteoff(gennum)
char gennum;
{
   gates = acpa_in(0x1f80) & ~( 1 << gennum);    /* Clear on flag            */
   gates |= 0x0100 << gennum;                    /* Set XOR flag             */
   acpa_out(0x1f80,gates);                       /* write it                 */
}

/*****************************************************************************/
/* All Notes Off                                                             */
/*****************************************************************************/
void DevAllNotesOff()
{
   gates = 0;
   acpa_out(0x1f80,0xff00);
}

/*****************************************************************************/
/* ChannelPressure                                                           */
/*****************************************************************************/
void  DevChannelPressure(chan,press)
char  chan,press;
{
   /* not supported */
}

/*****************************************************************************/
/* Control Change                                                            */
/*****************************************************************************/
void  DevControlChange(chan,cnum,cval)
char  chan,cnum,cval;
{
   struct noteblk *nptr;

   switch(cnum){
      case  64:                                /* Sustain Pedal              */
         if(cval > 63){                        /*    on                      */
            midiblks[chan].mbflags1 |= SUSTAIN;/* set sustain flag           */
         }else{
            midiblks[chan].mbflags1 &= ~SUSTAIN;   /* turn sustain flag on   */
            nptr = onnotes;                    /* locate sustained notes     */
            while(nptr){
               if(nptr->nbchannel==chan        /*  for this channel          */
               && (nptr->nbflags1 & SUSTAIN)){
                  DevNoteOff(chan,nptr->nbkeynum,0);
               }
               nptr = nptr->nbnext;
            }
         }
         break;
      case  123:                               /* ALL NOTES OFF              */
         nptr = onnotes;                       /* locate all on notes        */
         while(nptr){
            if(nptr->nbchannel==chan){
               DevNoteOff(chan,nptr->nbkeynum,0);
               nptr = onnotes;
            }else{
               nptr = nptr->nbnext;            /* next block                 */
            }
         }
         break;
      case  124:                               /* OMNI Mode Off              */
         midiblks[chan].mbflags1 &= ~OMNI;
         break;
      case  125:                               /* OMNI Mode On               */
         midiblks[chan].mbflags1 |= OMNI;
         break;
      case  126:                               /* MONO Mode On               */
         midiblks[chan].mbflags1 |= MONO;
         /* need support for v = #notes */
         break;
      case  127:                               /* MONO Mode Off              */
         midiblks[chan].mbflags1 &= ~MONO;
         break;
      default:
         break;
   }/*switch*/
}

/*****************************************************************************/
/* Program Change                                                            */
/*****************************************************************************/
void  DevProgramChange(chan,prog)
char  chan,prog;
{
   struct midiblk *mptr;

   if(ProgChgEnable){
      mptr = &midiblks[chan];
      mptr->mbvoicenum = prog;
      mptr->mbconfig = voices[prog][5];
      mptr->mbkeylo = voices[prog][69];
      mptr->mbkeyhi = voices[prog][70];
   }
}

/*****************************************************************************/
/* AfterTouch                                                                */
/*****************************************************************************/
void  DevAfterTouch(chan,key,press)
char  chan,key,press;
{

}

/*****************************************************************************/
/* PitchBend                                                                 */
/*****************************************************************************/
void  DevPitchBend(chan,lsb,msb)
char  chan,lsb,msb;
{

}

/*****************************************************************************/
/* MTC                                                                       */
/*****************************************************************************/
void  DevMTC(byte1,byte2)
char  byte1,byte2;
{

}

/*****************************************************************************/
/* Device Dependent SysEx                                                    */
/*****************************************************************************/
void DevDepSX(d1,d2,d3)
char d1,d2,d3;
{

   if(d1==1){                               /* ACPA ID?                      */
      switch(d2){
         case  1:                           /* Program Change Enable         */
            ProgChgEnable = d3;
            break;
         default:
            break;
      }/* switch*/
   }
}

/* General MIDI */
/*****************************************************************************/
/* This array maps AVC generic sound #'s to general midi                     */
/* 0-127 are regular voices,                                                 */
/* 128-174 are in the percussion map (128+key-35)                            */
/*****************************************************************************/
unsigned char gm_map[128] = {
   21,109,105,105,32,32,33,36,
   70,112,128+61-35,61,56,82,127,128+37-35,
   8,42,14,91,91,91,128+39-35,71,
   128+75-35,7,7,56,128+56-35,128+49-35,128+52-35,128+43-35,
   128+36-35,128+37-35,128+38-35,114,59,73,60,9,
   128+52-35,24,27,28,26,25,30,22,
   46,6,128+42-35,128+46-35,93,107,105,105,
   128+70-35,12,68,55,18,16,19,75,
   1,4,0,3,72,128+74-35,128+51-35,64,
   104,128+37-35,48,45,62,87,95,80,
   50,128+54-35,128+54-35,47,128+47-35,128+81-35,57,56,
   58,14,105,11,41,40,45,53,
   52,78,112,115,13,106,113,128+69-35,
   128+63-35,128+74-35,0,0,0,0,0,0,
   0,0,0,0,0,0,0,0,
   0,0,0,0,0,0,0,0
};
/*****************************************************************************/
/* Generic Voice Select                                                      */
/*****************************************************************************/
void DevVoiceSel(inst,type,sel)
char inst,type,sel;
{
   struct midiblk *mptr;
   unsigned char  gm_num;

   gm_num = gm_map[sel];
   mptr = &midiblks[inst];
   mptr->mbvoicenum = gm_num;
   mptr->mbconfig = voices[gm_num][5];
   mptr->mbkeylo = voices[gm_num][69];
   mptr->mbkeyhi = voices[gm_num][70];
}

/*****************************************************************************/
/* Query Timbre Parameter                                                    */
/*****************************************************************************/
void DevQTimbreParm(parmid)
long  parmid;
{
   int  voice;
   char id;

   voice = (char)(parmid >> 7);
   id = (char)(parmid & 0x7f);

   TimbreSysex[6] = (char)((voice >> 7) & 0x7f);   /* Set parm # high 7 bits */
   TimbreSysex[7] = (char)(voice & 0x7f);          /* Set parm # mid 7 bits  */
   TimbreSysex[8] = id;                            /* Set parm # low 7 bits  */
   TimbreSysex[9] = voices[voice][id];             /* Set data lsb           */
   TimbreSysex[10] = 0;                            /* Set data msb (0)       */
   queue_data(&TimbreSysex[0],12,0);               /* Set fixed trk = 0      */

   return;
}

/*****************************************************************************/
/* Set Timbre Parameter                                                      */
/*****************************************************************************/
void DevSetTimbreParm(parmid,parmval)
long parmid;
int parmval;
{
   int x,voice;
   char id;

   voice = (int)(parmid >> 7);
   id = (char)(parmid & 0x7f);

   voices[voice][id] = (unsigned char)parmval;  /* Set parameter             */
   for(x=0; x<8; x++){                          /* Force reload of voice data*/
      if(noteblks[x].nbvoicenum==(char)voice){
         noteblks[x].nbvoicenum = -1;
      }
   }
   return;
}

/*****************************************************************************/
/* Request Timbre Block                                                      */
/*****************************************************************************/
void DevReqTimbreBlk(blknum)
int blknum;
{
   BlockSysex[6] = (unsigned char)((blknum>>7) & 0x7f);
   BlockSysex[7] = (unsigned char)(blknum & 0x7f);
   queue_data(BlockSysex,9,0);                       /* trk = 0              */
   queue_data(voices[blknum],106,0);
   queue_data(&BlockSysex[9],1,0);
   return;
}

/*****************************************************************************/
/* Write Timbre Block                                                        */
/*****************************************************************************/
void DevWriteTimbreBlk(blknum,len,blk)
int blknum,len;
unsigned char *blk;
{
   int x;

   for(x=0; x<len; x++){                    /* Write timbre block            */
      voices[blknum][x] = *(blk+x);
   }

   for(x=0; x<8; x++){                      /* Force reload of voice data    */
      if(noteblks[x].nbvoicenum==(char)blknum){
         noteblks[x].nbvoicenum = -1;
      }
   }
   return;
}

/*****************************************************************************/
/* Set Voice                                                                 */
/*****************************************************************************/
void setvoice(inst,type,id)
int   inst,type,id;
{
   unsigned int loc;
   char *data;

   #if TRACE_ON
   trace(0x02);
   #endif
   instconfigs[inst] = voices[id][5];

   loc = pspdataloc + (inst * 128);
   data = &voices[id][0];

   _asm cli;
   ++acpalock;
   _asm {
      cld
      push  si
      mov   dx,hostbase       ; Set address to instruments data area
      add   dl,4
      mov   ax,loc
      out   dx,ax
      sub   dl,4

      mov   si,data
      mov   cx,73
      sub   ah,ah
   sv_l:
      lodsb
      out   dx,ax
      loop  sv_l

      mov   al,32             ; Set pitch = 32
      out   dx,ax
      mov   al,48             ; Set volume = 48
      out   dx,ax
      mov   ax,id             ; Set voice number = voiceid
      out   dx,ax

      sub   ax,ax             ; Zero out next 32 locations
      mov   cx,32
sv_loop2:
      out   dx,ax
      loop  sv_loop2

      pop   si
   }

   if(--acpalock == 0) _asm sti;
   #if TRACE_ON
   trace(0x12);
   #endif
   return;
}

/*****************************************************************************/
/* Relink WFGEN routines                                                     */
/*****************************************************************************/
void relink()
{
   extern int subend[],wfsubtbl[];

   #if TRACE_ON
   trace(0x03);
   #endif
   do_relink = 0;
   _asm cli;
   acpalock++;
   _asm{
      cld
      push  di
      push  si
      mov   dx,hostbase
      add   dl,4
      mov   ax,1feah          ; Determine dynalink area
      out   dx,ax
      sub   dl,4
      in    ax,dx             ;  get current DLAREA location
      mov   old_dlarea,ax
      out   dx,ax             ; Copy it to DLNEW
 jmp $+2
      in    ax,dx             ; Get prefix size
      mov   dlpsiz,ax
      mov   ax,HIGHBUF
      cmp   old_dlarea,ax
      jne   rl_lowbuf
      mov   ax,LOWBUF         ; if high buffer, set to low
   rl_lowbuf:
      mov   dlarea,ax         ; Set new DLAREA location

      add   dl,4              ; Write 1st 3 instructions
      add   ax,dlpsiz         ; cal start of DL data
      mov   di,ax             ; DI = location counter
      out   dx,ax
      sub   dl,4
      mov   ax,ZAC            ; ZAC
      out   dx,ax
      mov   ax,SACL+ARPLUS    ; SACL *+
      out   dx,ax
      mov   ax,SACL+ARMINUS   ; SACL *-
      out   dx,ax
      mov   datafixup,-16
      mov   bx,offset instconfigs
      mov   cx,8              ; load 8 instruments
   rl_config_l1:
      push  cx                ; save inst counter
      push  bx                ; save ptr to instconfig[]
      mov   codefixup,di      ; calc fixups
      add   datafixup,10h     ; datafixup = inst# * 16
      and   datafixup,70h     ; limit to single page = 0-7 * 16
      mov   bl,[bx]           ; get ptr to WFSUB
      and   bx,CONFIGMASK     ; limit to 0-15
      shl   bx,1              ; times 2 for word pointers
      add   bx,offset wfsubtbl
      mov   si,[bx]           ; Get subroutine data address
      mov   cl,[si]           ; get subrouinte length
      inc   si
      and   cx,00ffh
      jz    rl_config_off
      add   di,cx             ; Update location counter
   rlsubl:
      lodsb                   ; Get fixup flags byte
      mov   bl,al
      lodsw                   ; Get instruction word
      test  bl,1              ; data fixup?
      jz    rlsubl1
      add   ax,datafixup      ;  if so, fix it up
   rlsubl1:
      out   dx,ax             ; write it out
      loop  rlsubl            ; continue for entire subroutine
   rl_config_off:
      pop   bx                ; restore ptr to instconfigs[]
      pop   cx                ; restore inst counter
      inc   bx                ; next instconfig[]
      loop  rl_config_l1      ; continue for all 8 insts

      mov   si,offset subend  ; Now copy ending sequence
      lodsw                   ; (dont use outsw since needs 80286)
      out   dx,ax
      lodsw
      out   dx,ax
      lodsw
      out   dx,ax
      add   di,3              ; update location counter

      mov   ax,dlarea         ; add final BANZ address
      add   ax,dlpsiz
      out   dx,ax
      inc   di
      mov   ax,PSPRET         ;  and RET
      out   dx,ax

      inc   di                ; DI == last instruction in loop
      sub   di,dlarea         ; DI == # instructions in loop
      mov   dlload,di         ; save for load balancing

      add   dl,4
      mov   ax,1febh          ; DLNEW
      out   dx,ax
      sub   dl,4
      mov   ax,dlarea
      out   dx,ax
      pop   si
      pop   di
   }
   if(--acpalock == 0) _asm sti
   #if TRACE_ON
   trace(0x13);
   #endif
   return;
}

/*****************************************************************************/
/* MIDI Control Block Initialization                                         */
/*****************************************************************************/
void midi_cb_init()
{
   int   x;

   for(x=0; x<16; x++){                  /* Initialize all MIDI blocks       */
      midiblks[x].mbflags1 = 0;
      midiblks[x].mbvoicenum = 64;       /* default piano voice              */
      midiblks[x].mbconfig = voices[64][5];
      midiblks[x].mbkeylo = voices[64][69];
      midiblks[x].mbkeyhi = voices[64][70];
      midiblks[x].mbvolume = 96;
      midiblks[x].mbtranspose = 0;
   }
   offnotes = &noteblks[0];
   onnotes = NULL;
   noteblks[0].nbprev = NULL;
   for(x=0; x<8; x++){                   /* Initialize all note blocks       */
      if(x>0){
         noteblks[x].nbprev = &noteblks[x-1];
      }
      if(x<7){
         noteblks[x].nbnext = &noteblks[x+1];
      }
      noteblks[x].nbgennum = (char)x;    /* sound gen = inst # (0 - 7)       */
      noteblks[x].nbvoicenum = -1;       /* force reload of voice data       */
      noteblks[x].nbconfig = -1;         /*  and configuration               */
      noteblks[x].nbflags1 = 0;          /* clear flags                      */
   }
   noteblks[7].nbnext = NULL;
}

/***************************************************************************/
/* Set Clock values                                                        */
/*    Calculate the number of interrupts per second needed, and instruct   */
/*    the ACPA to do it.  The M-ACPA will utilize it's 10Mhz on-board      */
/*    clock to generate timing ints.  It will prescale the 10Mhz clock by  */
/*    20, thus effectively providing a 500khz clock.  We need to provide   */
/*    a 500,000 divisor for the required ints/sec rate.                    */
/*    Since Tempo is expressed as beats per minute * 10, we need to divide */
/*    it by 600 to get actual beats per second.  Thus, the calculation is: */
/*       500,000  /  (Tempo*10 * PPQN) / 600                               */
/*    In addition, we need to calculate the number of MIDI bytes that can  */
/*    be processed each interrupt.  MIDI can only physically transmit up   */
/*    to 3000 bytes per second.  In order to keep our interrupt times down */
/*    to an acceptable length (4msec's), we will emulate this physical     */
/*    limitation, and only process the number of bytes that could actually */
/*    be transmitted on a physical MIDI line each interrupt.               */
/* Variables used:                                                         */
/*    position             # of msec's since audio_init = ROLLOVER_TIME +  */
/*                          (POS_BUFFER_COUNT * 600,000 / TIME_PER_BLOCK)  */
/*    TIME_PER_BLOCK       Tempo(*10) * PPQN                               */
/*    POS_BUFFER_COUNT     Total # of interrupts received                  */
/*    ROLLOVER_TIME        When calc gets too big, rollover position here  */
/*                          and reset POS_BUFFER_COUNT                     */
/*    MAX_BUFFER_COUNT     calc is too big when POS_BUFFER_COUNT == this   */
/*    max_midi_bytes       max # bytes to process each interrupt           */
/***************************************************************************/
void SetClock(trk)
int trk;
{
   unsigned long t;

   /* Set timer = 500,000/(TEMPO*PPQN)/600 */
   if (TIME_PER_BLOCK && POS_BUFFER_COUNT) {
      ROLLOVER_TIME += (POS_BUFFER_COUNT * 600000)/TIME_PER_BLOCK;
      POS_BUFFER_COUNT = 0;
      ((IOB)XMITIO.Ptr)->position = 0;
   } /* endif */
   /* We want to limit the number of interrupts to 500/sec                  */
   /* Therefore: 500,000/250 = 2000 is the minimum t value we want          */
   /* Therefore: 300,000,000/2000 = 150,000 is the maximum TIME_PER_BLOCK value*/
   TIME_PER_BLOCK = (unsigned long)((unsigned long)TEMPO * (unsigned long)PPQN);
   /* Calculate the max # of bytes we should process each interrupt  */
   max_midi_bytes = (int)(3125l*600l / TIME_PER_BLOCK);
   if (TIME_PER_BLOCK > 150000l) {
      timing_prescaler = (unsigned int)(TIME_PER_BLOCK / 150000l) + 1;
   } else {
      timing_prescaler = 1;
   } /* endif */

   /* Now calculate divisor = 500,000 / TIME_PER_BLOCK / 600                   */
   t = (300000000l * timing_prescaler) / TIME_PER_BLOCK;
   acpa_out(0x1fef,(unsigned int)t);                          /* Set TIMDIV */
}

