/*****************************************************************************/
/*                  Audio Device Driver Strategy Routine                     */
/*                                                                           */
/*    CloseRequest()                                                         */
/*    DDSpecificIOCTL()                                                      */
/*    DeInstallRequest()                                                     */
/*    FlushInputRequest()                                                    */
/*    FlushOutputRequest()                                                   */
/*    GenericIOCTLRequest()                                                  */
/*    InitRequest()                                                          */
/*    InputOutputRequest()                                                   */
/*    NonDReadRequest()                                                      */
/*    NonDWriteRequest()                                                     */
/*    OpenRequest()                                                          */
/*    OutputStatusRequest()                                                  */
/*    ReadRequest()                                                          */
/*    ReadStatusRequest()                                                    */
/*    WriteRequest()                                                         */
/*                                                                           */
/*    AudioBuffer()                                                          */
/*    AudioChange()                                                          */
/*    AudioContNotify()                                                      */
/*    AudioControl()                                                         */
/*    AudioDiadRead8()                                                       */
/*    AudioDiadRead16()                                                      */
/*    AudioDiadWrite8()                                                      */
/*    AudioDiadWrite16()                                                     */
/*    AudioHPI()                                                             */
/*    AudioInit()                                                            */
/*    AudioLoad()                                                            */
/*    AudioNotify()                                                          */
/*    AudioPause()                                                           */
/*    AudioResume()                                                          */
/*    AudioStart()                                                           */
/*    AudioStatus()                                                          */
/*    AudioStop()                                                            */
/*    AudioUpdate()                                                          */
/*    AudioWait()                                                            */
/*                                                                           */
/* 5/06/92 BRR???  Creation of this file. Split out of ACPA.C.               */
/* 5/18/92 BRR???  Add support for AUDIO_NOTIFY/CONTINUOUS_NOTIFY            */
/*                 Add support iobuf->moreflags and DELAY from iobuf->delay  */
/*****************************************************************************/

/*****************************************************************************/
/* DEFINEs                                                                   */
/*****************************************************************************/

/* none */

/*****************************************************************************/
/* Include DEFINEs                                                           */
/*****************************************************************************/

#if IS_OS2
   #define  INCL_DOSINFOSEG
   #define  INCL_DOS
   #include <os2.h>
#endif

/*****************************************************************************/
/* Include Files                                                             */
/*****************************************************************************/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "audiodd.h"
#include "auddef.h"
#include "audproto.h"
#include "audiodd2.h"

/* OS/2 Specific Include Files                                               */
#if IS_OS2
   #include "mme.h"
   #include "os2medef.h"
   #include "ssm.h"
   #include "shdd.h"
#endif

/*****************************************************************************/
/* External Function declarations                                            */
/*****************************************************************************/

extern void far *       GetLinearAddr         (void far *info_ptr,
                                               unsigned int size,
                                               unsigned int utilgdt,
                                               void far *utilptr,
                                               int trk);
extern int              init_ptrs             (int trk);

#if TRACE_ON
   extern void trace(unsigned char trace_point);
#endif

/* OS/2 Specific External Function Declarations                              */
#if NOT_OS2
   extern int far    direct_call_1 (void);
   #if NUM_TRACKS > 1
      extern int far direct_call_2 (void);
   #endif
#else
   extern void          MMEDestroyStreams     (int SysFileNum) ;
   extern void          MMEInitStreams        (void) ;
#endif

/*****************************************************************************/
/* Internal Function declarations                                            */
/*****************************************************************************/

unsigned int AudioBuffer         (int);
unsigned int AudioChange         (int);
unsigned int AudioControl        (int);
unsigned int AudioDiag8Read      (int);
unsigned int AudioDiag8Write     (int);
unsigned int AudioDiag16Read     (int);
unsigned int AudioDiag16Write    (int);
unsigned int AudioHPI            (int);
unsigned int AudioInit           (int);
unsigned int AudioLoad           (int);
void         AudioPause          (int);
void         AudioResume         (int);
void         AudioStart          (int);
unsigned int AudioStatus         (int);
void         AudioStop           (int);
unsigned int AudioWait           (int);
unsigned int CloseRequest        (int);
unsigned int DDSpecificIOCTL     (int, unsigned char);
unsigned int DeInstallRequest    (int);
unsigned int GenericIOCTLRequest (int);
unsigned int InitRequest         (int);
unsigned int InputOutputRequest  (int);
unsigned int FlushInputRequest   (int);
unsigned int FlushOutputRequest  (int);
unsigned int NonDReadRequest     (int);
unsigned int OpenRequest         (int);
unsigned int OutputStatusRequest (int);
void         PauseTrk            (int);
unsigned int ReadRequest         (int);
unsigned int ReadStatusRequest   (int);
void         StopTrk             (int);
void         Strategy_c          (struct Packet far *,int);
unsigned int WriteRequest        (int);

#if NOT_K12
   unsigned int AudioUpdate      (int);
#endif

#if NOT_DOS_K12
   unsigned int AudioContNotify  (int);
   void         AudioNotify      (int);
#endif

/*****************************************************************************/
/* External Variable Declarations                                            */
/*****************************************************************************/

extern struct mode_data mode_table[];
extern char far         resident_end;

#if NUM_TRACKS > 1
   extern unsigned int             last_xmit_head_segnum[];
#else
   extern unsigned int             last_xmit_head_segnum;
#endif

/* OS/2 Specific External Variable Declarations                              */
#if IS_OS2
   extern ACPAMME      AcpaMME;
   extern unsigned int UtilGDTs[];              /* Utility GDT's for V86.    */
   extern void far    *UtilPtrs[];              /* Utility GDT ptrs for V86. */
   #if NUM_TRACKS > 1
      extern void far    *GDTrecbuf[];
      extern void far    *GDTrecio[];
      extern unsigned int GDTselector1[];
      extern unsigned int GDTselector2[];
      extern unsigned int GDTselector3[];
      extern unsigned int GDTselector4[];
      extern void far    *GDTxmitbuf[];
      extern void far    *GDTxmitio[];
   #else
      extern void far    *GDTrecbuf;
      extern void far    *GDTrecio;
      extern unsigned int GDTselector1;
      extern unsigned int GDTselector2;
      extern unsigned int GDTselector3;
      extern unsigned int GDTselector4;
      extern void far    *GDTxmitbuf;
      extern void far    *GDTxmitio;
   #endif
#else
   #if NUM_TRACKS > 1
      extern int initcount;
   #endif
#endif

#if NOT_DOS_K12
   #if NUM_TRACKS > 1
      extern int  notify_den[];
      extern int  notify_num[];
   #else
      extern int  notify_den;
      extern int  notify_num;
   #endif
#else
#endif


/*****************************************************************************/
/* Internal Variable Declarations                                            */
/*****************************************************************************/

char                              any_dev_info;
char                              any_mode_info;
int                               change_dev_info[5];
int                               change_mode_info[6];
unsigned int                      hostbase;
unsigned char                     hostrnge;
void far                         *GIODptr;     /* Pointer to IOCTL data      */
struct Packet far                *rb;          /* Request Block pointer      */

/* Track Specific Internal Variable Declarations                             */
#if NUM_TRACKS > 1
   unsigned long         app_version[NUM_TRACKS];
   long                  bits_per_sample[NUM_TRACKS];  /* Bits per sample    */
   long                  bsize[NUM_TRACKS];            /* Block size         */
                                                       /* User callback addr */
   void                  (far * callback_func[NUM_TRACKS])(void);
   int                   channels[NUM_TRACKS];         /* #Channels(2=stereo)*/
   unsigned long         flags[NUM_TRACKS] = { 0 };    /* audio flags        */
   int                   hpilock[NUM_TRACKS] = { 0 };  /* Serializes access  */
                                                       /* to *xmitio & *recio*/
   unsigned int          initflags[NUM_TRACKS] = { 0 };/* 1=INITED, 2=WAITING*/
   struct iobuf          iriobuf[NUM_TRACKS];
   char                  irdata[NUM_TRACKS][RBUFFERSIZE];
   struct iobuf          ixiobuf[NUM_TRACKS];
   char                  ixdata[NUM_TRACKS][XBUFFERSIZE];
   unsigned int          midiflags[NUM_TRACKS] = { 0 };
   int                   mode[NUM_TRACKS];             /* Current initialized*/
                                                       /* mode for the track */
   unsigned long         operation[NUM_TRACKS];        /* current initialized*/
                                                       /* operation          */
   int                   position_type[NUM_TRACKS] = { POS_MSECS };
   unsigned long         prev_recio_count[NUM_TRACKS]={0};
   unsigned long         prev_xmitio_count[NUM_TRACKS]={0};
   unsigned char         rec_buffer_array[NUM_TRACKS][MAX_NUM_BUFFERS] = {0};
   unsigned short        rec_internal[NUM_TRACKS];
   unsigned short        rec_max_num_buffers[NUM_TRACKS] = {0};
   struct vscb           recbuf[NUM_TRACKS];
   struct vscb           recio[NUM_TRACKS];
   unsigned long         recmax[NUM_TRACKS] = { 0 };   /* highest reccount   */
                                                       /* since last open    */
   unsigned long         rollover_time[NUM_TRACKS];
   long                  srate[NUM_TRACKS];            /* Sample Rate        */
   char far             *Srhead[NUM_TRACKS];           /* Shadow rx head ptr */
   char far             *Srtail[NUM_TRACKS];           /* Shadow rx tail ptr */
   char far             *Sxhead[NUM_TRACKS];           /* Shadow tx head ptr */
   char far             *Sxtail[NUM_TRACKS];           /* Shadow tx tail ptr */
   unsigned int          trk_array[NUM_TRACKS];
   unsigned char         trk_assigned_at_init[NUM_TRACKS];
   unsigned long         trk_hvdm[NUM_TRACKS];
   unsigned long         VCB_Hndl[NUM_TRACKS];
   char                  V86_mode[NUM_TRACKS];
   unsigned char         xmit_buffer_array[NUM_TRACKS][MAX_NUM_BUFFERS] = {0};
   unsigned short        xmit_internal[NUM_TRACKS];
   unsigned short        xmit_max_num_buffers[NUM_TRACKS] = {0};
   struct vscb           xmitbuf[NUM_TRACKS];
   struct vscb           xmitio[NUM_TRACKS];
   unsigned long         xmitmax[NUM_TRACKS] = { 0 };  /* highest xmitcnt    */
                                                       /* since last open    */
#else
   unsigned long         app_version;
   long                  bits_per_sample;              /* Bits per sample    */
   long                  bsize;                        /* Block size         */
                                                       /* User callback addr */
   void                  (far * callback_func)(void);
   int                   channels;                     /* #Channels(2=stereo)*/
   unsigned long         flags = {0};                  /* audio flags        */
   int                   hpilock = {0};                /* Serializes access  */
                                                       /* to *xmitio & *recio*/
   unsigned int          initflags = {0};              /* 1=INITED, 2=WAITING*/
   struct iobuf          iriobuf;
   char                  irdata[RBUFFERSIZE];
   struct iobuf          ixiobuf;
   char                  ixdata[XBUFFERSIZE];
   unsigned int          midiflags = {0};
   int                   mode;                         /* Current initialized*/
                                                       /* mode for the track */
   unsigned long         operation;                    /* current initialized*/
                                                       /* operation          */
   int                   position_type = {POS_MSECS};
   unsigned long         prev_recio_count = {0};
   unsigned long         prev_xmitio_count = {0};
   unsigned char         rec_buffer_array[MAX_NUM_BUFFERS] = {0};
   unsigned short        rec_internal;
   unsigned short        rec_max_num_buffers = {0};
   struct vscb           recbuf;
   struct vscb           recio;
   unsigned long         recmax = {0};                 /* highest reccount   */
                                                       /* since last open    */
   unsigned long         rollover_time;
   long                  srate;                        /* Sample Rate        */
   char far             *Srhead;                       /* Shadow rx head ptr */
   char far             *Srtail;                       /* Shadow rx tail ptr */
   char far             *Sxhead;                       /* Shadow tx head ptr */
   char far             *Sxtail;                       /* Shadow tx tail ptr */
   unsigned int          trk_array;
   unsigned char         trk_assigned_at_init;
   unsigned long         trk_hvdm;
   unsigned long         VCB_Hndl;
   char                  V86_mode;
   unsigned char         xmit_buffer_array[MAX_NUM_BUFFERS] = {0};
   unsigned short        xmit_internal;
   unsigned short        xmit_max_num_buffers = {0};
   struct vscb           xmitbuf;
   struct vscb           xmitio;
   unsigned long         xmitmax = {0};                /* highest xmitcnt    */
                                                       /* since last open    */
#endif


/* NOT K12 Specific Internal Variable Declarations                           */
#if NOT_K12
   unsigned char            time_event_sysex[] = {0xf0,0,0,0x3a,1,1,0,0xf7};


   #if NUM_TRACKS > 1
      long                  delay[NUM_TRACKS];
      unsigned char far    *prvtimemsg[NUM_TRACKS] = { NULL };/* Ptr to prv  */
                                                              /* time msg    */
   #else
      long                  delay;
      unsigned char far    *prvtimemsg = { NULL };            /* Ptr to prv  */
                                                              /* time msg    */
   #endif
#endif

   /* OS/2 Specific Internal Variable Declarations                              */
#if IS_OS2
      void far     *iobufvirt;
      unsigned char os2_version;
      int           rmswitch             = 0;
      unsigned int  temp_version;
   #if NUM_TRACKS > 1
      HSEM          callback_sem[NUM_TRACKS];
      /* array that holds if app. using this track is in Real(or V86) or Prot Mode */
      unsigned char trk_mode[NUM_TRACKS] = {0};
   #else
      HSEM          callback_sem;
      /* array that hold whether app. using this track is Real(or V86) or Prot Mode */
      unsigned char trk_mode = {0};
   #endif
#endif

/* AVC 1.03 Specific Internal Variable Declarations                          */
#if IS_AVC103
   int              avc103_mode;
#endif

/**************/
/* STRATEGY_C */
/**************/
void Strategy_c(request_block,trk)
struct Packet far *request_block;              /* Request Block pointer      */
int trk;
{
   unsigned int error_flag;

   rb = request_block;                         /* Copy into global variable  */
   if(MIDIFLAGS & ACPA_INOP){                  /* Is ACPA inoperational?     */
      rb->PktHdr.PktStatus = done+error+general_failure; /* If not, error    */
      return;
   }

   if(INITFLAGS & INITED){                 /* Have we been initialized?  */
      if((OPERATION_G == PLAY) ||          /* verify we have a valid operation*/
         (OPERATION_G == RECORD) ||        /* now, so we can assume we do the */
         (OPERATION_G == PLAY_AND_RECORD) || /* rest of the time in strategy  */
         (OPERATION_G == ANALYSIS) ||
         (OPERATION_G == DISTANCE) ||
         (OPERATION_G == MIGRATION) ||
         (MODE_G == SOURCE_MIX) ||
         (MODE_G == CLAIM_HDWR)) {
         /* do nothing */
      } else {
         rb->PktHdr.PktStatus = done+error+general_failure; /* If not, error  */
         return;
      } /* endif */
   }

   error_flag = done;
   if((rb->PktHdr.PktCmd != 0) &&                  /* If this isn't INIT call */
#if IS_WIN
      (rb->PktHdr.PktCmd != 14) &&                 /* If not CLOSE call       */
#endif
      (rb->PktHdr.PktCmd != 20) &&                 /* if not DeInstall        */
      (rb->PktHdr.PktCmd != 13)){                  /* If not OPEN call        */
         setup_shadows(trk);
   }

   switch(rb->PktHdr.PktCmd){

      case 0:                                /* INIT                         */
           error_flag |= InitRequest(trk);
           break;

      case 4:                                /* READ                         */
           error_flag |= ReadRequest(trk);
           break;

      case 5:                                /* NONDESTRUCTIVE READ, NO WAIT */
           error_flag |= NonDReadRequest(trk);
           break;

      case 6:                                /* READ STATUS                  */
           error_flag |= ReadStatusRequest(trk);
           break;

      case 7:                                /* FLUSH INPUT                  */
           error_flag |= FlushInputRequest(trk);
           break;

      case 8:                                /* WRITE                        */
           error_flag |= WriteRequest(trk);
           break;

      case 10:                               /* WRITE STATUS                 */
           error_flag |= OutputStatusRequest(trk);
           break;

      case 11:                               /* FLUSH OUTPUT                 */
           error_flag |= FlushOutputRequest(trk);
           break;

      case 13:                               /* OPEN                         */
           error_flag |= OpenRequest(trk);
           break;

      case 14:                               /* CLOSE                        */
           error_flag |= CloseRequest(trk);
           break;

      case 3:                                /* IOCTL input                  */
      case 12:                               /* IOCTL output                 */
           error_flag |= InputOutputRequest(trk);
           break;

      case 16:                               /* GENERAL IOCtl (OS/2)         */
      case 19:                               /* GENERAL IOCtl (DOS)          */
           error_flag |= GenericIOCTLRequest(trk);
           break;

      case 20:                               /* DEINSTALL                    */
           error_flag |= DeInstallRequest(trk);
           break;

      default:                               /* BAD COMMAND ID               */
           error_flag = done + error + unknown_command;
           break;

   } /* end switch */

   rb->PktHdr.PktStatus = error_flag;        /* Set done status              */

} /* end Strategy_c */

/*****************************************************************************/
/* INIT Request                                                              */
/*****************************************************************************/

unsigned int InitRequest(trk)
int trk;
{

   unsigned int x;
#if IS_OS2
      /* Get the version of OS/2 that we're running on       */
      if (!(DosGetVersion(&temp_version))) {
         os2_version = HIBYTE(temp_version);
      } else {
         os2_version = 0;
      } /* endif */
#endif

      if (init((struct IPacket far *)rb)) {
         return(done+error+general_failure);  /* If so, flag an error */
      } /* endif */
#if NOT_OS2
#if NUM_TRACKS > 1
      if (initcount==1) {
#endif
#endif
         for (x=0; x<NUM_TRACKS; x++) {
            if (init_ptrs(x)) {
               return(done+error+general_failure);  /* If so, flag an error */
            } /* endif */
         } /* endfor */

#if IS_OS2
         if (!trk)     {                   // Allocating stream table
             MMEInitStreams();             // for both tracks. Do only once
         }
#endif

#if NOT_OS2
#if NUM_TRACKS > 1
      } /* endif */
#endif
#endif
      return(0);

} /* end InitRequest */

/*****************************************************************************/
/* Read Request                                                              */
/*****************************************************************************/

unsigned int ReadRequest(trk)
int trk;
{

   unsigned char far *fptr;
   unsigned long x;
#if IS_OS2
      if(((IOB)RECIO.Ptr)->runflags & CHAIN_BUFFERS){ /* If chain buffers  */
         return(done+error+busy);          /* then don't allow reads  */
      }
      if(!(fptr = PhysToVirt(rb->IOpData,rb->IOcount))){
         return(done+error+general_failure);
      }
#else
      fptr = rb->IOpData;
#endif
      if(((IOB)RECIO.Ptr)->count > RECMAX) RECMAX = ((IOB)RECIO.Ptr)->count;
      for(x = rb->IOcount; x; x--){
#if NOT_K12
         if((MODE_G==MIDI)&&(((IOB)RECIO.Ptr)->tail==PRVTIMEMSG)){
            MIDIFLAGS &= 0xfffa;        /* Clear timer event q'd flags */
            PRVTIMEMSG = NULL;
         }
#endif
         while(((IOB)RECIO.Ptr)->count==0){           /* Wait for some data */
            _asm sti;
#if IS_OS2
            MIDIFLAGS |= INPUT_WAITING;             /* Set waiting flag  */
            Block((long)((char far *)&RECMAX),100l,0);   /* Block thread      */
            MIDIFLAGS &= ~INPUT_WAITING;            /* Clear waiting flag*/
            /* Re-issue phystovirt following block                         */
            if(!(fptr = PhysToVirt(rb->IOpData,rb->IOcount))){
               return(done+error+general_failure);
            }
            fptr += rb->IOcount - x;         /* Skip part already written  */
#endif
         }
         *fptr++ = (unsigned char)midiread(trk);
      }
#if IS_OS2
      UnPhysToVirt();
#endif
      return(0);

} /* end ReadRequest */

/*****************************************************************************/
/* Non Destructive Reqd Request                                              */
/*****************************************************************************/

unsigned int NonDReadRequest(trk)
int trk;
{

      if(((IOB)RECIO.Ptr)->count){                  /* Is anything queued?   */
         rb->PktMediaDesc = *SRTAIL;         /* If so, return it             */
      }else{
         rb->PktHdr.PktStatus = done+busy;   /* Else, return busy flag       */
      }
      return(0);

} /* end NonDReadRequest */

/*****************************************************************************/
/* Read Status Request                                                       */
/*****************************************************************************/

unsigned int ReadStatusRequest(trk)
int trk;
{

      if(((IOB)RECIO.Ptr)->count == 0){      /* Is anything queued?          */
         return(done+busy);             /* If not, return busy flag     */
      }
      return(0);

} /* end ReadStatusRequest */

/*****************************************************************************/
/* Flush Input Request                                                       */
/*****************************************************************************/

unsigned int FlushInputRequest(trk)
int trk;
{

      _asm cli;
      ((IOB)RECIO.Ptr)->head = ((IOB)RECIO.Ptr)->tail = RECBUF.Virt; /* Reset */
      SRHEAD = SRTAIL = RECBUF.Ptr;
      ((IOB)RECIO.Ptr)->count = 0;
      _asm sti;
      return(0);

} /* end FlushInputRequest*/

/*****************************************************************************/
/* Write Request                                                             */
/*****************************************************************************/

unsigned int WriteRequest(trk)
int trk;
{

   unsigned char far *fptr;
   unsigned long x;

#if IS_OS2
      if(((IOB)XMITIO.Ptr)->runflags & CHAIN_BUFFERS){   /* If chain buffers*/
         return(done+error+busy);       /* then don't allow write's   */
      }
      if(!(fptr = PhysToVirt(rb->IOpData,rb->IOcount))){ /* Get Virt Addr  */
         return(done+error+general_failure);
      }
#else
      fptr = rb->IOpData;
#endif
      for(x = rb->IOcount; x; x--){
         while(((IOB)XMITIO.Ptr)->count >= ((IOB)XMITIO.Ptr)->size){
            _asm sti;
#if IS_OS2
            MIDIFLAGS |= OUTPUT_WAITING;               /* Set waiting flag*/
            Block((long)((char far *)&MIDIFLAGS),100l,0); /* Block thread*/
            MIDIFLAGS &= ~OUTPUT_WAITING;           /* Clear waiting flag*/
            /* Re-issue phystovirt following block                         */
            if(!(fptr = PhysToVirt(rb->IOpData,rb->IOcount))){
               return(done+error+general_failure);
            }
            fptr += rb->IOcount - x;         /* Skip part already written  */
#endif
         }
         _asm cli;

         *SXHEAD++ = *fptr++;
         /* Need to wrap ? */

         check_wrap((IOB)XMITIO.Ptr,1,trk,HEAD,TX);
         if (LAST_XMIT_HEAD_SEGNUM) {
#if IS_OS2
            /* Now map new GDT */
            AddrToGDTselector(XMITBUF.Phys,
                              (unsigned)XMITBUF.length,
                              GDTSELECTOR3);
#endif

            SXHEAD = XMITBUF.Ptr;    /* Set our (usable) pointer   */
            ((IOB)XMITIO.Ptr)->head = XMITBUF.Virt; /* Set user's ptr */
         } else {
            ((IOB)XMITIO.Ptr)->head++;
         }
         if(++((IOB)XMITIO.Ptr)->count > XMITMAX)
            XMITMAX = ((IOB)XMITIO.Ptr)->count;
         _asm sti;
      }
#if IS_OS2
      UnPhysToVirt();
#endif
#if NOT_K12
      if(!(MIDIFLAGS & OUTSYNC)){
         MIDIFLAGS |= OUTSYNC;
         DELAY = 1;
      }
#endif
      return(0);

} /* end WriteRequest */

/*****************************************************************************/
/* Output Status Request                                                     */
/*****************************************************************************/

unsigned int OutputStatusRequest(trk)
int trk;
{

      if(((IOB)XMITIO.Ptr)->count == ((IOB)XMITIO.Ptr)->size){ /* Buffer full?*/
         return(done+busy);             /* If not, return busy flag   */
      }
      return(0);

} /* end OutputStatusRequest */

/*****************************************************************************/
/* Flush Output Request                                                      */
/*****************************************************************************/

unsigned int FlushOutputRequest(trk)
int trk;
{

      _asm cli;
      SXHEAD = SXTAIL = XMITBUF.Ptr;         /* Set our usable pointers    */
      ((IOB)XMITIO.Ptr)->head = ((IOB)XMITIO.Ptr)->tail = XMITBUF.Virt;
      ((IOB)XMITIO.Ptr)->count = 0;                 /*  and count = 0      */
      MIDIFLAGS &= ~OUTSYNC;                 /* Clear output pending flag  */
      _asm sti;
      return(0);

} /* end FlushOutputRequest */

/*****************************************************************************/
/* Open Request                                                              */
/*****************************************************************************/

unsigned int OpenRequest(trk)
int trk;
{

#if NOT_OS2  //set up the trk_array for DOS since it doesn't have gendd to do
             // it for it
      if (TRK_ARRAY) {
         return(done+error+device_not_ready);
      } else {
         TRK_ARRAY = 0x5A5A + trk;
      } /* endif */
#endif

#if IS_OS2
      SetProcessMode(trk);                    /* Set global TRK_MODE  */
#endif

#if IS_WIN
      DevInit();
      if (init_ptrs(trk)) {
         return(done+error+general_failure);  /* If so, flag error */
      } /* endif */
#endif

      /* Note that we only get this far if TRK is set to 0 or 1 */
#if IS_OS2
      /* Get Handle to the VDD if OS/2 2.0 */
      if (os2_version >= 20)
         TRK_HVDM = Get_HVDM();
#endif
      setup_shadows(trk);

      if(DevOpen(trk)) {
         TRK_ARRAY = 0;
         return(done+error+general_failure);
      }
#if IS_AVC103
      if (avc103_mode) {
         init_queue_vars(trk);
         INITFLAGS = 0;                    /* Clear INIT flag     */
      }
#endif
      return(0);

} /* end OpenRequest */

/*****************************************************************************/
/* Close Request                                                             */
/*****************************************************************************/

unsigned int CloseRequest(trk)
int trk;
{

      /* Note that we won't even get this far unless TRK is 0 or 1 */
      if (DevClose(trk)) {
         return(done+error+general_failure);
      }
      TRK_HVDM = 0;
      return(DeInitialize(trk));

} /* end CloseRequest */

/*****************************************************************************/
/* DeInstall Request                                                         */
/*****************************************************************************/

unsigned int DeInstallRequest(trk)
int trk;
{

      return(done+error+unknown_command);

} /* end DeInstallRequest */

/*****************************************************************************/
/* DD Specific IOCTL                                                         */
/*****************************************************************************/

unsigned int DDSpecificIOCTL(trk,func)
int trk;
unsigned char func;
{
   unsigned int rc;


   switch(func){

      case AUDIO_INIT:                       /* AUDIO_INIT IOCtl           */
           rc = AudioInit(trk);
           break;

      case AUDIO_STATUS:                     /* AUDIO_STATUS IOCtl         */
           rc = AudioStatus(trk);
           break;

      case AUDIO_CONTROL:                    /* AUDIO_CONTROL IOCtl        */
           rc = AudioControl(trk);
           break;

      case AUDIO_BUFFER:                     /* AUDIO_BUFFER IOCtl         */
           rc = AudioBuffer(trk);
           break;

      case AUDIO_LOAD:                       /* AUDIO_LOAD IOCtl           */
           rc = AudioLoad(trk);
           break;

      case AUDIO_WAIT:                       /* AUDIO_WAIT                 */
           rc = AudioWait(trk);
           break;

      case AUDIO_DIAG8_READ:                 /* Diagnostic 8 bit read      */
           rc = AudioDiag8Read(trk);
           break;

      case AUDIO_DIAG8_WRITE:                /* Diagnostic 8 bit write     */
           rc = AudioDiag8Write(trk);
           break;

      case AUDIO_DIAG16_READ:                /* Diagnostic 16 bit read     */
           rc = AudioDiag16Read(trk);
           break;

      case AUDIO_DIAG16_WRITE:               /* Diagnostic 16 bit write    */
           rc = AudioDiag16Write(trk);
           break;

#if NOT_K12
      case AUDIO_UPDATE:                     /* Update buffer array        */
           rc = AudioUpdate(trk);
           break;
#endif

      case AUDIO_HPI:                        /* High Performance Interface */
           rc = AudioHPI(trk);
           break;

      default:                                  /* Invalid request ID         */
           rc = done+error+unknown_command;
           break;

   } /* end switch */
   return(rc);

} /* end DDSpecificIOCTL */

/*****************************************************************************/
/* AUDIO_INIT IOCTL                                                          */
/*****************************************************************************/

unsigned int AudioInit(trk)
int trk;
{

int             i;
unsigned int    rc = 0;
int             sx;
#if NOT_K12
#endif
#if IS_OS2
   unsigned int x;
   PSTREAM      pStream;
#endif

#if IS_OS2
   if (VerifyAccess(GIODptr,sizeof(struct audio_init_v005),1)) {
      return(done+error+general_failure);            /* If so, flag an error */
   } /* endif */
   ((AIP)GIODptr)->reserved = (void far *)TRK_ARRAY;
#endif

   ((AIP)GIODptr)->rc = 0;
   if (((AIP)GIODptr)->mode == IDLE) {
      if (((IOB)XMITIO.Ptr)->runflags & STARTED)
          ((IOB)XMITIO.Ptr)->runflags |= PAUSED;
      if (((IOB)RECIO.Ptr)->runflags & STARTED)
          ((IOB)RECIO.Ptr)->runflags |= PAUSED;
      rc = DeInitialize(trk);
      DevDeInit(trk);
      return(rc);
   } /* endif */

#if IS_OS2
   SetProcessMode(trk);                              /* Set global TRK_MODE  */
#endif

   init_queue_vars(trk);
   INITFLAGS = 0;                                    /* Clear INIT flag      */

   /* determine what version this is by operation field                      */
   if (((AIP)GIODptr)->operation & 0x80000000) {
#if IS_OS2
      if (VerifyAccess(GIODptr,sizeof(struct audio_init),1)) {
         return(done+error+general_failure); /* If so, flag err */
      } /* endif */
#endif
      APP_VERSION = ((AIP)GIODptr)->version_level;
      ((AIP)GIODptr)->version_level = CURRENT_VERSION;
      if (APP_VERSION > CURRENT_VERSION) {
         ((AIP)GIODptr)->rc = DOWNLEVEL_DD;
         return(done+error+device_not_ready);
      } /* endif */

   } else {
      ((AIP)GIODptr)->operation |= 0x80000000;
      APP_VERSION = 0;
   } /* endif */

   INITFLAGS = INITED;                         /*    set INIT'ed flag        */
   i = determine_mode((AIP)GIODptr);

   /* Copy parameters to our global variables (do also in AVC103.C)          */
   SetGlobalParms((AIP)GIODptr,trk);

#if IS_OS2
   /* Get pointer to stream table and update the TrackNum                    */
   pStream = AcpaMME.paStream;
   for (x=0; x<AcpaMME.usMaxNumStreams; x++) {

      if (pStream->usSysFileNum == TRK_ARRAY) {    //find entry in stream table
         pStream->usTrackNum = (unsigned)trk;
         break;
      } else {
         pStream++;
      }

   } /* endfor */
#endif

   ((AIP)GIODptr)->rc = 0;

   /* Let device process it mode selection */
   if((sx=DevIOCTLinit((AIP)GIODptr,i,trk)) != i){

      if(sx == -1){                     /* Error returned?            */

         /* Check to see if rc already set by DevIOCTLinit     */
         if (((AIP)GIODptr)->rc == 0)
            ((AIP)GIODptr)->rc = OVERLOADED;

         return(done+error+device_not_ready);

#if IS_OS2
         /* Reset the track to indicate it is available IF the init */
         /* failed AND it was the AUDIO_INIT and NOT the OPEN that  */
         /* assigned this track to this application.                */
         if (TRK_ASSIGNED_AT_INIT)
            TRK_ARRAY = 0;
#endif
         INITFLAGS &= ~INITED;        /*  clear INIT'ed flag  */

      }else{

         /* Copy parms to our global variables (do also in AVC103.C)*/
         /*  in case any of them have changed.                          */
         SetGlobalParms((AIP)GIODptr,trk);

      }
   }
   return(0);

} /* end AudioInit */

/*****************************************************************************/
/* AUDIO_HPI IOCTL                                                           */
/*****************************************************************************/

unsigned int AudioHPI (trk)
int trk;
{

#if IS_OS2
   int realmode;
#endif
   unsigned int rc=0;

   HPILOCK++;

#if IS_OS2

   _asm{                   /* Determine if we're in Real Mode        */
      smsw  ax             ; get current msw
      rcr   ax,1           ; Protect bit into C
      mov   ax,0           ; Set AX = 0
      jc    is_protmode    ; Skip if protect mode
      mov   ax,1           ; Set AX = 1 if Real Mode
   is_protmode:
      mov   realmode,ax
   }

   if(!realmode){
      if (VerifyAccess(GIODptr,sizeof(struct audio_hpi),1)) {
         HPILOCK--;
         return(done+error+general_failure);  // Flag error.
      } /* endif */
   } /* endif */

   /* Set up direct call address                                             */
   ((HPI)GIODptr)->ep = NULL;               /* No direct call address        */
#else
#if NUM_TRACKS > 1
   if (trk == 0) {
#endif
      ((HPI)GIODptr)->ep = direct_call_1;   /* Return direct call address    */
#if NUM_TRACKS > 1
   } else {
      ((HPI)GIODptr)->ep = direct_call_2;   /* Return direct call address    */
   } /* endif */
#endif
#endif

   /* Check to see if Callback requested by application                      */
#if IS_OS2
   if (TRK_MODE == REALMODE) {

      CALLBACK_FUNC = ((HPI)GIODptr)->cb;       /* Set callback address      */
      CALLBACK_SEM = 0;

      if(os2_version >= 20){
        if(CALLBACK_FUNC == 0){
          VCB_HNDL = 0;
        } else {
          VCB_HNDL = GetVCB_Hndl(CALLBACK_FUNC,TRK_HVDM);
        }
      }

   } else {

      if (((HPI)GIODptr)->cb == 0) {
         CALLBACK_SEM = 0;
      } else {
         CALLBACK_SEM = SemHandle((HSEM)((HPI)GIODptr)->cb,1);
      } /* endif */

      CALLBACK_FUNC = 0;

   } /* endif */
#else
   CALLBACK_FUNC = ((HPI)GIODptr)->cb;       /* Set callback address         */
#endif

   InitIOBuf(((HPI)GIODptr)->newxbuf,(IOB)XMITIO.Ptr,0,trk);
   ((IOB)XMITIO.Ptr)->head = ((IOB)XMITIO.Ptr)->tail = XMITBUF.Virt;

   InitIOBuf(((HPI)GIODptr)->newrbuf,(IOB)RECIO.Ptr,1,trk);
   ((IOB)RECIO.Ptr)->head = ((IOB)RECIO.Ptr)->tail = RECBUF.Virt;

#if NOT_K12
   XMIT_MAX_NUM_BUFFERS = ((IOB)XMITIO.Ptr)->num_buffers;
   rc = InitBufferArray(XMIT_MAX_NUM_BUFFERS,(IOB)XMITIO.Ptr,
                        XMIT_BUFFER_ARRAY,trk);
   REC_MAX_NUM_BUFFERS = ((IOB)RECIO.Ptr)->num_buffers;
   rc = InitBufferArray(REC_MAX_NUM_BUFFERS,(IOB)RECIO.Ptr,
                        REC_BUFFER_ARRAY,trk);
#endif
   HPILOCK--;
   return(rc);

} /* end AudioHPI */

#if NOT_K12
/*****************************************************************************/
/* AUDIO_UPDATE IOCTL                                                        */
/*****************************************************************************/

unsigned int AudioUpdate(trk)
int trk;
{

   unsigned char        *buff_array;
   void far             *iobufptr;
   unsigned short        max_buffs;
   unsigned char         reset_head;
   unsigned char         reset_tail;
   struct addrs_array    temp_buffer_array[16];
   unsigned int          x,y;
#if IS_OS2
   ULONG                 linear_addr;        /* V86 linear address           */
#endif

   ((UPDATE)GIODptr)->rc = 0;
   HPILOCK++;
#if IS_OS2
   if(TRK_MODE != REALMODE){
      if (VerifyAccess(GIODptr,sizeof(struct audio_update),0)) {
         ((UPDATE)GIODptr)->rc = UPDATE_GENERAL_FAILURE;
         HPILOCK--;
         return(done+error+general_failure);  /* If so, flag an error */
      } /* endif */
   } /* endif !realmode */
#endif

   /* determine which iobuf is being updated */
   if (((UPDATE)GIODptr)->iobuf_type == XMIT_IOBUF) {
      if (XMIT_INTERNAL) {
         ((UPDATE)GIODptr)->rc = UPDATE_GENERAL_FAILURE;
         HPILOCK--;
         return(done+error+general_failure);  /* If so, flag an error */
      } /* endif */
      iobufptr = XMITIO.Ptr;
      buff_array = XMIT_BUFFER_ARRAY;
      max_buffs = XMIT_MAX_NUM_BUFFERS;
#if IS_OS2
      iobufvirt = XMITIO.Virt;
#endif
   } else { /* REC_IOBUF */
      if (REC_INTERNAL) {
         ((UPDATE)GIODptr)->rc = UPDATE_GENERAL_FAILURE;
         HPILOCK--;
         return(done+error+general_failure);  /* If so, flag an error */
      } /* endif */
      iobufptr = RECIO.Ptr;
      buff_array = REC_BUFFER_ARRAY;
      max_buffs = REC_MAX_NUM_BUFFERS;
#if IS_OS2
      iobufvirt = RECIO.Virt;
#endif
   } /* endif */

   /* Null out all buffers that have been processed */

   /* If not buffers in array yet, then both head and tail will need */
   /* to be initialized.                                             */
   if (((IOB)iobufptr)->num_buffers) {
      reset_head = 0;
      reset_tail = 0;
   } else {
      reset_head = 1;
      reset_tail = 1;
   } /* endif */

   /* Take all the "processed" buffers out of the buff_array */
   for (x=0; x < ((IOB)iobufptr)->num_buffers; x++) {
      if (buff_array[x] == 0) {
         ((IOB)iobufptr)->size -= ((IOB)iobufptr)->buf[x].length;

         /* check if this is for rec and update count if it is        */
         /* because we make the ASSUMPTION that any buffers taken off */
         /* the array when recording are full of data                 */
         if (((UPDATE)GIODptr)->iobuf_type == REC_IOBUF) {
            ((IOB)iobufptr)->count -= ((IOB)iobufptr)->buf[x].length;
         } /* endif */
         ((IOB)iobufptr)->buf[x].length = 0;
         ((IOB)iobufptr)->buf[x].Phys = 0;

         /* Are we getting rid of the buffer the head pointer is in? */
         if (x == ((IOB)iobufptr)->head_segnum) reset_head = 1;
         if (x == ((IOB)iobufptr)->tail_segnum) reset_tail = 1;
#if IS_OS2
         if (((IOB)iobufptr)->buf[x].lock_handle) {
            if (UnLock_mem(((IOB)iobufptr)->buf[x].lock_handle,trk)) {
               ((UPDATE)GIODptr)->rc = UPDATE_GENERAL_FAILURE;
               HPILOCK--;
               return(done+error+general_failure);  /* If so, flag an error */
               break;
            } /* endif */
            ((IOB)iobufptr)->buf[x].lock_handle = 0;
         } /* endif */
#endif
      } /* endif */
   } /* endfor */

   /* condense all the buffers now */
   for (y=0,x=((IOB)iobufptr)->tail_segnum; x<((IOB)iobufptr)->num_buffers; x++) {
      if (((IOB)iobufptr)->buf[x].length) {
         copyaddrs((struct addrs_array far *)&temp_buffer_array[y],
                   (struct addrs_array far *)&((IOB)iobufptr)->buf[x]);
         y++;
      } /* endif */
   } /* endfor */
   for (x=0; x<((IOB)iobufptr)->tail_segnum; x++) {
      if (((IOB)iobufptr)->buf[x].length) {
         copyaddrs((struct addrs_array far *)&temp_buffer_array[y],
                   (struct addrs_array far *)&((IOB)iobufptr)->buf[x]);
         y++;
      } /* endif */
   } /* endfor */

   /* Now copy all the buffers back to the iobuf */
   for (x=0; x<y; x++) {
      copyaddrs((struct addrs_array far *)&((IOB)iobufptr)->buf[x],
                (struct addrs_array far *)&temp_buffer_array[x]);
      /* update buff_array                                       */
      buff_array[x] = 1;
   } /* endfor */
   for (x=y; x<((IOB)iobufptr)->num_buffers; x++) {
      /* update buff_array                                       */
      buff_array[x] = 0;
   } /* endfor */

   /* And now the current number of buffers is = y               */
   ((IOB)iobufptr)->num_buffers = y;

   /* Check to see if a buffer was passed down */
   if (((UPDATE)GIODptr)->buffer_address) {

      /* Check to see if there is room in the array of buffers */
      if (y < max_buffs) {

         if ((((UPDATE)GIODptr)->buffer_length > 0x00010000) ||
             (((UPDATE)GIODptr)->buffer_length & 1) ||
             (((UPDATE)GIODptr)->buffer_length == 0))
            ((UPDATE)GIODptr)->rc = INVALID_BUFFER_LENGTH;

         ((IOB)iobufptr)->size += ((UPDATE)GIODptr)->buffer_length;
         ((IOB)iobufptr)->buf[y].length = ((UPDATE)GIODptr)->buffer_length;
         ((IOB)iobufptr)->buf[y].Virt = ((UPDATE)GIODptr)->buffer_address;
#if IS_OS2
         if (V86_MODE) {

            /*******************************************************/
            /****   V86 mode                                    ****/
            /*******************************************************/

            /* get linear addr from buffer address and save in   */
            /* .Phys address for future LinToGDTs at int time    */
            linear_addr=v86toLin(((UPDATE)GIODptr)->buffer_address);
            ((IOB)iobufptr)->buf[y].Phys = (char far *)linear_addr;
            if (y != 0) {
               ((IOB)iobufptr)->buf[y].Phys += TRK_HVDM;
            } /* endif */

            if (!(((IOB)iobufptr)->buf[y].lock_handle =
                 LnLock(linear_addr,((UPDATE)GIODptr)->buffer_length))) {
               ((UPDATE)GIODptr)->rc = UPDATE_GENERAL_FAILURE;
               HPILOCK--;
               return(done+error+general_failure);  /* If so, flag an error */
            } /* endif */

            /* map linear address to one we can use - might as   */
            /* well use real GDT....we'll set it up to the first */
            /* buffer in the array later......                   */
            if (AddrToGDTselector((void far *)linear_addr,
                     (unsigned) ((UPDATE)GIODptr)->buffer_length,
                       UtilGDTs[0])) {
               ((UPDATE)GIODptr)->rc = UPDATE_GENERAL_FAILURE;
               HPILOCK--;
               return(done+error+general_failure);  /* If so, flag an error */
            }

            linear_addr = (ULONG)UtilPtrs[0];

            if (VerifyAccess((void far *)linear_addr,
                     (unsigned int)((UPDATE)GIODptr)->buffer_length,1)) {
               ((UPDATE)GIODptr)->rc = UPDATE_GENERAL_FAILURE;
               HPILOCK--;
               return(done+error+general_failure);  /* If so, flag an error */
            } /* endif */

         } else { /* Non-V86 mode */

            linear_addr = (ULONG)((UPDATE)GIODptr)->buffer_address;

            if ((linear_addr = (ULONG)GetLinearAddr((void far *)linear_addr,(unsigned int)((UPDATE)GIODptr)->buffer_length,
                                UtilGDTs[0],UtilPtrs[0],trk)) == 0) {
               ((UPDATE)GIODptr)->rc = UPDATE_GENERAL_FAILURE;
               HPILOCK--;
               return(done+error+general_failure);  /* If so, flag an error */
            } /* endif */

            if (!((((IOB)iobufptr)->buf[y].lock_handle =   // Lock for
                  Lock(((IOB)iobufptr)->buf[y].Virt,1,0)))) { // VirtToPhys
               ((UPDATE)GIODptr)->rc = UPDATE_GENERAL_FAILURE;
               HPILOCK--;
               return(done+error+general_failure);  /* If so, flag an error */
            } /* endif */

            ((IOB)iobufptr)->buf[y].Phys =
               VirtToPhys(((IOB)iobufptr)->buf[y].Virt);

         } /* endif */                                                      /*wb*/
#endif

         /* check if this is for xmit and update count if it is     */
         if (((UPDATE)GIODptr)->iobuf_type == XMIT_IOBUF) {

            ((IOB)iobufptr)->count += ((UPDATE)GIODptr)->buffer_length;

            /* Always assume that if playing and buffer sent down, that*/
            /* it is full of data...and we should update the head_ptr  */
            reset_head = 1;

         } /* endif */

         /* update number of buffers in the array                   */
         ((IOB)iobufptr)->num_buffers++;

         /* update buff_array                                       */
         buff_array[y] = 1;

      } else { /* position is not available */
         ((UPDATE)GIODptr)->rc = MAX_NUM_BUFFERS_REACHED;
      } /* endif */

   } /* endif buffer passed down */

   if (reset_head) {
      ((IOB)iobufptr)->head_segnum = 0;
      ((IOB)iobufptr)->head = ((IOB)iobufptr)->buf[0].Virt;
   } /* endif */

   /* Tail_segnum is always at 0 now */
   ((IOB)iobufptr)->tail_segnum = 0;

   /* Check to see if the tail_ptr needs to be initialized */
   if (reset_tail) {
      ((IOB)iobufptr)->tail = ((IOB)iobufptr)->buf[0].Virt;
   } /* endif */

   /* if reset head pointer and we are receiving, then need to */
   /* re-setup the RECBUF structure for this new buffer        */
   if ((((UPDATE)GIODptr)->iobuf_type == REC_IOBUF) && (reset_head)) {

      RECBUF.length = ((IOB)iobufptr)->buf[0].length;
#if IS_OS2
      RECBUF.Phys = ((IOB)iobufptr)->buf[0].Phys;
      AddrToGDTselector(RECBUF.Phys,
                        (unsigned)RECBUF.length,
                        GDTSELECTOR4);
      RECBUF.Ptr = RECBUF.GDT;
#else
      RECBUF.Virt = RECBUF.Ptr = ((IOB)iobufptr)->buf[0].Virt;
#endif

   } else if ((((UPDATE)GIODptr)->iobuf_type == XMIT_IOBUF)
               && (reset_tail)) {

      XMITBUF.length = ((IOB)iobufptr)->buf[0].length;
#if IS_OS2
      XMITBUF.Phys = ((IOB)iobufptr)->buf[0].Phys;
      AddrToGDTselector(XMITBUF.Phys,
                        (unsigned)XMITBUF.length,
                        GDTSELECTOR3);
      XMITBUF.Ptr = XMITBUF.GDT;
#else
      XMITBUF.Virt = XMITBUF.Ptr = ((IOB)iobufptr)->buf[0].Virt;
#endif

   } /* endif */

   HPILOCK--;
   return(0);

} /* end AudioUpdate */
#endif

/*****************************************************************************/
/* AUDIO_DIAG16_WRITE IOCTL                                                  */
/*****************************************************************************/

unsigned int AudioDiag16Write(trk)
int trk;
{

#if IS_OS2
         if (VerifyAccess(GIODptr,sizeof(struct diag_regs16),0)) {
            return(done+error+general_failure);  /* If so, flag an error */
         } /* endif */
#endif
         DevIOCTLwrite16((DR16)GIODptr);
         return(0);

} /* end AudioDiag16Write */

/*****************************************************************************/
/* AUDIO_DIAG8_WRITE IOCTL                                                   */
/*****************************************************************************/

unsigned int AudioDiag8Write(trk)
int trk;
{

#if IS_OS2
         if (VerifyAccess(GIODptr,sizeof(struct diag_regs8),0)) {
            return(done+error+general_failure);  /* If so, flag an error */
         } /* endif */
#endif
         DevIOCTLwrite8((DR8)GIODptr);
         return(0);

} /* end AudioDiag8Write */


/*****************************************************************************/
/* AUDIO_DIAG16_READ IOCTL                                                   */
/*****************************************************************************/

unsigned int AudioDiag16Read(trk)
int trk;
{

#if IS_OS2
         if (VerifyAccess(GIODptr,sizeof(struct diag_regs16),1)) {
            return(done+error+general_failure);  /* If so, flag an error */
         } /* endif */
#endif
         DevIOCTLread16((DR16)GIODptr);
         return(0);

} /* end AudioDiag16Read */

/*****************************************************************************/
/* AUDIO_DIAG8_READ IOCTL                                                    */
/*****************************************************************************/

unsigned int AudioDiag8Read(trk)
int trk;
{

#if IS_OS2
         if (VerifyAccess(GIODptr,sizeof(struct diag_regs8),1)) {
            return(done+error+general_failure);  /* If so, flag an error */
         } /* endif */
#endif
         DevIOCTLread8((DR8)GIODptr);
         return(0);

} /* end AudioDiag8Read */

/*****************************************************************************/
/* AUDIO_WAIT IOCTL                                                          */
/*****************************************************************************/

unsigned int AudioWait(trk)
int trk;
{

         if( !(INITFLAGS & INITED)           /*   has INIT been issued?    */
         || !(((IOB)XMITIO.Ptr)->runflags & STARTED)){   /* were we started? */
            return(0);
         }
         /* Clear any latent underrun bit that was set - at this point they*/
         /* don't care about the underrun - they just want to know when they*/
         /* are finished playing-which some devices determine by the       */
         /* underrun flag - therefore clear it before you finish playing   */
         /* just in case                                                   */
         ((IOB)XMITIO.Ptr)->runflags &= ~IOB_UNDERRUN;

         /* Wait until buffers are drained                                 */
         while(((IOB)XMITIO.Ptr)->count){
            /* Wait until xmit queue is empty */
            _asm sti
            INITFLAGS |= WAITING;       /* Set waiting flag           */
#if IS_OS2
            Block((long)((char far *)&XMITIO),100l,0); /* Block this thread*/
#endif
            INITFLAGS &= ~WAITING;      /* Clear waiting flag         */
         }
         if (MODE_G != MIDI) {
            DevIOCTLwait(trk);
         } /* endif */
         return(0);

} /* end AudioWait */


/*****************************************************************************/
/* AUDIO_LOAD IOCTL                                                          */
/*****************************************************************************/

unsigned int AudioLoad(trk)
int trk;
{

   unsigned char far *fptr;

#if IS_OS2
         if ((VerifyAccess(GIODptr,sizeof(struct audio_load),0))) {
            return(done+error+general_failure);  /* If so, flag an error */
         } /* endif */

         fptr = ((ALP)GIODptr)->buffer;

         if ((fptr = GetLinearAddr(fptr,(unsigned int)((ALP)GIODptr)->size,
                                UtilGDTs[0],UtilPtrs[0],trk)) == 0) {
            return(done+error+general_failure);
         } /* endif */
#else
         fptr = ((ALP)GIODptr)->buffer;
#endif
         DevIOCTLload(fptr,
                        ((ALP)GIODptr)->size,
                        ((ALP)GIODptr)->flags,
                        trk);
         return(0);

} /* end AudioLoad */

/*****************************************************************************/
/* AUDIO_BUFFER IOCTL                                                        */
/*****************************************************************************/

unsigned int AudioBuffer(trk)
int trk;
{

#if IS_OS2
   if (VerifyAccess(GIODptr,sizeof(struct audio_buffer),1)) {
      return(done+error+general_failure);  /* If so, flag an error */
   } /* endif */
#endif
   ((ABP)GIODptr)->flags = 0;             /* flags                   */

   if (((IOB)RECIO.Ptr)->runflags & IOB_OVERRUN) {
      ((ABP)GIODptr)->flags |= AUDIO_OVERRUN;
      ((IOB)RECIO.Ptr)->runflags &= ~IOB_OVERRUN;
   }

   if (((IOB)XMITIO.Ptr)->runflags & IOB_UNDERRUN) {
      ((ABP)GIODptr)->flags |= AUDIO_UNDERRUN;
      ((IOB)XMITIO.Ptr)->runflags &= ~IOB_UNDERRUN;
   }

   ((ABP)GIODptr)->read_buf_size=((IOB)RECIO.Ptr)->count; /* Read Q cnt */
   ((ABP)GIODptr)->write_buf_size=((IOB)XMITIO.Ptr)->count; /* Wrt Q cnt*/
   ((ABP)GIODptr)->read_buf_max=RECMAX;   /* highest level of read q    */
   ((ABP)GIODptr)->write_buf_max=XMITMAX; /* highest level of write q   */
   if(OPERATION_G != RECORD) {
      ((ABP)GIODptr)->position = ((IOB)XMITIO.Ptr)->position; //Crrnt pos.

      if (POSITION_TYPE_G == POS_MSECS) {
#if NOT_K12
         if (MODE_G != MIDI) {
#endif
            ((ABP)GIODptr)->write_buf_time =
                       (((ABP)GIODptr)->write_buf_size*1000)
                       / (unsigned long)SRATE_G;
#if NOT_K12
         } /* endif */
#endif
      } else {
         ((ABP)GIODptr)->write_buf_time = 0;    /* # msec's in write Q */
      } /* endif */

   } /* endif */

   if (OPERATION_G != PLAY) {
      ((ABP)GIODptr)->position = ((IOB)RECIO.Ptr)->position; // Curr pos.

      if (POSITION_TYPE_G == POS_MSECS) {
#if NOT_K12
         if (MODE_G != MIDI) {
#endif
            ((ABP)GIODptr)->read_buf_time =
                  (((ABP)GIODptr)->read_buf_size*1000) / (unsigned long)SRATE_G;
#if NOT_K12
         } /* endif */
#endif
      } else {
         ((ABP)GIODptr)->read_buf_time = 0;     /* # msec's in read Q   */
      } /* endif */

   } /* endif */
   ((ABP)GIODptr)->position_type = POSITION_TYPE_G;
   ((ABP)GIODptr)->read_buf_cap = ((IOB)RECIO.Ptr)->size;   // Read Q cap.
   ((ABP)GIODptr)->write_buf_cap = ((IOB)XMITIO.Ptr)->size; // Wrt Q cap.
   ((ABP)GIODptr)->request_buf_cap = MAXREQS;   /* max # requests    */
   return(0);

} /* end AudioBuffer */

/*****************************************************************************/
/* AUDIO_CONTROL IOCTL                                                       */
/*****************************************************************************/

unsigned int AudioControl(trk)
int trk;
{

   unsigned int rc;

           #if IS_OS2
            if (VerifyAccess(GIODptr,sizeof(struct audio_control),1)) {
               return(done+error+general_failure);  /* If so, flag an error  */
            } /* endif */
           #endif

           switch(((ACP)GIODptr)->ioctl_request){

              case AUDIO_CHANGE:            /* Change adapter characteristic */
                   rc = AudioChange(trk);
                   break;

              case AUDIO_START:             /* Start new operation           */
                   AudioStart(trk);
                   break;

              case AUDIO_STOP:              /* Stop current operation        */
                   AudioStop(trk);
                   break;

              case AUDIO_PAUSE:             /* Pause current operation       */
                   AudioPause(trk);
                   break;

              case AUDIO_RESUME:            /* Resume a paused operation     */
                   AudioResume(trk);
                   break;
#if NOT_DOS_K12
              case AUDIO_NOTIFY:            /* Notify app at specified time  */
                   AudioNotify(trk);
                   break;

              case AUDIO_CONTINUOUS_NOTIFY: /* Cont. notify app at specified */
                   rc = AudioContNotify(trk);/* time intervals               */
                   break;
#endif
              default:    /* Unknown control               */
                   ((ACP)GIODptr)->return_code=-1;  /* return an error            */
                   break;

           } /* end switch for AUDIO_CONTROL */

           return(rc);

} /* end AudioControl */

/*****************************************************************************/
/* AUDIO_CHANGE IOCTL                                                        */
/*****************************************************************************/

unsigned int AudioChange(trk)
int trk;
{

#if NOT_K12
#else
   short far *sptr;
   int x;
#endif
#if IS_OS2
   unsigned char far *fptr;
   void  far *dev_infoSave;
   void  far *mode_infoSave;
   unsigned int size;
#endif
   struct audio_change far *chg;

   chg = ((ACP)GIODptr)->request_info;    /* ptr to audio_change    */
#if IS_OS2
   if (APP_VERSION >= VERSION_01000005) {
      size = sizeof(struct audio_change);
   } else {
      size = sizeof(struct audio_change_v00);
   } /* endif */

   /* get virtual addr of change structure                          */
   if ((chg = (struct audio_change far *)
               GetLinearAddr((void far *)chg,
                             size,
                             UtilGDTs[0],UtilPtrs[0],trk)) == 0) {
      return(done+error+general_failure);
   } /* endif */
#endif
   if (APP_VERSION >= VERSION_01000005) {

      if (chg->mode_info) {  /* use UtilPtr[1] */
#if IS_OS2
         mode_infoSave=chg->mode_info;      /*  Save V86? pointer */
         fptr = chg->mode_info;
#endif
         if (MODE_G == MIDI) {
#if IS_OS2
            /* get virtual addr of midi mode infor struct           */
            if ((chg->mode_info = (struct midi_mode_info_struct far *)
                GetLinearAddr((void far *)fptr,
                     (unsigned int)sizeof(struct midi_mode_info_struct),
                     UtilGDTs[1],UtilPtrs[1],trk)) == 0) {
               return(done+error+general_failure);
            } /* endif */
#endif
         } else { /* SPV2 Mode */
#if IS_OS2
            /* get virtual addr of change structure                          */
            if ((chg->mode_info = (struct spv2_mode_info_struct far *)
                 GetLinearAddr((void far *)fptr,
                    (unsigned int)sizeof(struct spv2_mode_info_struct),
                    UtilGDTs[1],UtilPtrs[1],trk)) == 0) {
               return(done+error+general_failure);
            } /* endif */
#endif
#if IS_K12
            sptr = (short far *)chg->mode_info;
            for (x=0; x<6; x++) {
               change_mode_info[x] = *sptr++;
            } /* endfor */
            any_mode_info = 1;
#endif
         } /* endif */
#if IS_K12
      } else {
         any_mode_info = 0;
#endif
      } /* endif */

   } /* endif */

   if (chg->dev_info) { /* use UtilPtr[2] */
#if IS_OS2
      dev_infoSave=chg->dev_info;    /*  Save V86? pointer */
      fptr = chg->dev_info;

      /* get virtual addr of change structure                          */
      if ((chg->dev_info = (struct track_info far *)
           GetLinearAddr((void far *)fptr,
                 (unsigned int)sizeof(struct track_info),
                 UtilGDTs[2],UtilPtrs[2],trk)) == 0) {
         return(done+error+general_failure);
      } /* endif */
#endif
#if IS_K12
      sptr = (short far *)chg->dev_info;
      for (x = 0; x<5; x++) {
         change_dev_info[x] = *sptr++;
      } /* endfor */
      any_dev_info = 1;
   } else {
      any_dev_info = 0;
#endif
   } /* endif */

#if NOT_DOS_K12
   if(queue_request(AUDIO_CHANGE,         /* Request type      */
      ((ACP)GIODptr)->position,           /* position #           */
      chg,                                /* struct of changed fields*/
      trk)){
      ((ACP)GIODptr)->return_code = FULL_QUEUE; /* return error   */
      return(0);
   } /* endif */
#if IS_OS2
   if ((MODE_G == SPV2BCPCM) ||
       (MODE_G == SPV2PCM) ||
       (MODE_G == SPV2NONE)) {
#endif
      ((ACP)GIODptr)->return_code=aud_control(trk);
#if IS_OS2
   } /* endif */
#endif
#else
   ((ACP)GIODptr)->return_code = DevChange(
      chg,
      change_dev_info,
      any_dev_info,
      change_mode_info,
      any_mode_info,
      trk);
#endif
#if IS_OS2
   if (os2_version >= 20) {
      chg->dev_info = dev_infoSave;   /* Restore saved V86? ptr. *wb*/
      chg->mode_info = mode_infoSave; /* Restore saved V86? ptr. *wb*/
   } /* endif */
#endif
   return(0);

} /* end AudioChange */

#if NOT_DOS_K12
/*****************************************************************************/
/* AUDIO_NOTIFY IOCTL                                                        */
/*****************************************************************************/

void AudioNotify(trk)
int trk;
{
   if(queue_request(AUDIO_NOTIFY,                    /* Request type         */
      ((ACP)GIODptr)->position,                      /* position #           */
      0,                                             /* NULL changed fields  */
      trk)){
      ((ACP)GIODptr)->return_code = FULL_QUEUE;      /* return error         */
   } /* endif */

} /* end AudioNotify */

/*****************************************************************************/
/* AUDIO_CONTINUOUS_NOTIFY IOCTL                                             */
/*****************************************************************************/

unsigned int AudioContNotify(trk)
int trk;
{
char pos_not_passed;
unsigned long pos;

   struct continuous_info far *cinfo;

   cinfo = ((ACP)GIODptr)->request_info;   /* ptr to cont_info     */
   if (cinfo) {
#if IS_OS2
      /* get virtual addr of continuous_info structure             */
      if ((cinfo = (struct continuous_info far*)
           GetLinearAddr((void far *)cinfo,
                    (unsigned int)sizeof(struct continuous_info),
                    UtilGDTs[0],UtilPtrs[0],trk)) == 0) {
         return(done+error+general_failure);
      } /* endif */
#endif
      if ((cinfo->numerator == 0) || (cinfo->denominator == 0)) {
         return(done+error+general_failure);
      } /* endif */
      NOTIFY_NUM = cinfo->numerator;
      NOTIFY_DEN = cinfo->denominator;
      pos = ((ACP)GIODptr)->position;
      pos_not_passed = 1;
      while (pos_not_passed) {

         pos = ((pos*NOTIFY_DEN/NOTIFY_NUM) + 1) *
                NOTIFY_NUM / NOTIFY_DEN;

         /* Call the device specific code to setup the DSP code to     */
         /* interrupt at "pos" time. DevNotify returns '0' if time  */
         /* is set, '1' if time requested has already passed.          */
         pos_not_passed = DevNotify(pos,trk);

      } /* end while */

      if(queue_request(AUDIO_CONTINUOUS_NOTIFY,    /* Request type     */
         pos,                                      /* position #       */
         0,                                        /* NULL             */
         trk)){
         ((ACP)GIODptr)->return_code = FULL_QUEUE; /* return error     */
         return(0);
      } /* endif */

   } else {

      NOTIFY_NUM = 0;
      NOTIFY_DEN = 0;

   } /* endif */

   return(0);

} /* end AudioContNotify */
#endif

/*****************************************************************************/
/* AUDIO_START IOCTL                                                         */
/*****************************************************************************/

void AudioStart(trk)
int trk;
{

   if( !(INITFLAGS & INITED)){        /*   has INIT been issued?    */
      ((ACP)GIODptr)->return_code=AC_UNINITED;
      return;                         /* if not, return an error    */
   }
   DevStart(trk);                     /* Do device dependent stuff  */
   if(OPERATION_G != RECORD) {
      ((IOB)XMITIO.Ptr)->runflags |= STARTED;  /*  set started flag */
      ((IOB)XMITIO.Ptr)->position = ((ACP)GIODptr)->position;
   }
   if(OPERATION_G != PLAY) {
      ((IOB)RECIO.Ptr)->runflags |= STARTED;   /*   set started flag */
      ((IOB)RECIO.Ptr)->position = ((ACP)GIODptr)->position;
   }
   ROLLOVER_TIME = ((ACP)GIODptr)->position;
   return;

} /* end AudioStart */

/*****************************************************************************/
/* AUDIO_STOP IOCTL                                                          */
/*****************************************************************************/

void AudioStop(trk)
int trk;
{

#if NOT_DOS_K12
            if(queue_request(AUDIO_STOP,     /* Request type               */
                  ((ACP)GIODptr)->position,  /* position #                 */
                  0,trk) < 0){
               ((ACP)GIODptr)->return_code = FULL_QUEUE; /* return error   */
               return;
            }
#if IS_OS2
            if ((MODE_G == SPV2BCPCM) ||
                (MODE_G == SPV2PCM) ||
                (MODE_G == SPV2NONE)) {
#endif
               ((ACP)GIODptr)->return_code=aud_control(trk);
#if IS_OS2
            } /* endif */
#endif
#else
            _asm cli;
            StopTrk(trk);
            _asm sti;
#endif
            DevStop(trk);
            return;

} /* end AudioStop */



/*****************************************************************************/
/* AUDIO_PAUSE IOCTL                                                         */
/*****************************************************************************/

void AudioPause(trk)
int trk;
{

#if NOT_DOS_K12
            if(queue_request(AUDIO_PAUSE,          /* Request type         */
                  ((ACP)GIODptr)->position,        /* position #           */
                  0,trk) < 0){
               ((ACP)GIODptr)->return_code = FULL_QUEUE; /* return error   */
               return;
            }
#if IS_OS2
            if ((MODE_G == SPV2BCPCM) ||
                (MODE_G == SPV2PCM) ||
                (MODE_G == SPV2NONE)) {
#endif
               ((ACP)GIODptr)->return_code=aud_control(trk);
#if IS_OS2
            } /* endif */
#endif
#else
            PauseTrk(trk);
#endif
            DevPause(trk);

} /* end AudioPause */


/*****************************************************************************/
/* AUDIO_RESUME IOCTL                                                        */
/*****************************************************************************/

void AudioResume(trk)
int trk;
{

   DevResume(trk);
   if(OPERATION_G != RECORD) {
      ((IOB)XMITIO.Ptr)->runflags &= ~PAUSED;   /* clear paused flag */
   }
   if(OPERATION_G != PLAY) {
      ((IOB)RECIO.Ptr)->runflags &= ~PAUSED;    /* clear paused flag */
   }

} /* end AudioResume */


/*****************************************************************************/
/* AUDIO_STATUS IOCTL                                                        */
/*****************************************************************************/

unsigned int AudioStatus(trk)
int trk;
{

#if IS_OS2
   if (APP_VERSION >= VERSION_01000005) {

      if (VerifyAccess(GIODptr,sizeof(struct audio_status),1)) {
         return(done+error+general_failure);       /* If so, flag an error   */
      } /* endif */

      if (((struct audio_status far *)GIODptr)->change.mode_info) {

         if (MODE_G == MIDI) {

            if (VerifyAccess(((struct audio_status far *)
                      GIODptr)->change.mode_info,
                       sizeof(struct midi_mode_info_struct),1)) {
               return(done+error+general_failure);  /* If so, flag an error  */
            } /* endif */

         } else if ((MODE_G == SPV2NONE) ||
            (MODE_G == SPV2PCM) ||
            (MODE_G == SPV2BCPCM)) {

            if (VerifyAccess(((struct audio_status far *)GIODptr)->
               change.mode_info,sizeof(struct spv2_mode_info_struct),1)) {
               return(done+error+general_failure);  /* If so, flag an error  */
            } /* endif */

         } /* endif */

      } /* endif */

   } else {

      if (VerifyAccess(GIODptr,sizeof(struct audio_status_v00),1)) {
         return(done+error+general_failure);  /* If so, flag an error        */
      } /* endif */

   } /* endif */

   if (((struct audio_status far *)GIODptr)->change.dev_info) {

      if (VerifyAccess(((struct audio_status far *)GIODptr)->
         change.dev_info,sizeof(struct track_info),1)) {
         return(done+error+general_failure);  /* If so, flag an error        */
      } /* endif */

   } /* endif */
#endif
   DevIOCTLstatus((ASP)GIODptr,trk);           /*    go get it               */
   return(0);

} /* end AudioStatus */


/*****************************************************************************/
/* Generic IOCTL Request                                                     */
/*****************************************************************************/

unsigned int GenericIOCTLRequest(trk)
int trk;
{

   unsigned char func;

   if(((GIO)rb)->GIOcat != 0x80){              /* Is this our category?      */
      return(done+error+unknown_command);
   }
   GIODptr = ((GIO)rb)->GIOData;               /* Get pointer to IOCTL data  */
   func = (unsigned char)(((GIO)rb)->GIOcode - 0x40); /* 0x40-0x5f -> 0-0x1f */
   return(DDSpecificIOCTL(trk,func));

} /* end GenericIOCTLRequest */

/*****************************************************************************/
/* Input/Output IOCTL Request                                                */
/*****************************************************************************/

unsigned int InputOutputRequest(trk)
int trk;
{

unsigned char func;
   unsigned char far *fptr;

#if IS_OS2
   if(!(fptr = PhysToVirt(rb->IOpData,rb->IOcount))){ /* Get Virt Addr  */
      return(done+error+general_failure);  /* If so, flag an error */
   }
#else
   fptr = rb->IOpData;
#endif
   func = *fptr++;
   GIODptr = fptr;
   return(DDSpecificIOCTL(trk,func));

} /* end InputOutputRequest */

