/********************************************************************/
/*
 * File name: mnatis.c 
 * Synopsis:  This shows the use of the MIL native mode programmer's
 *            kit. This example mixes MIL code and IMAGE SERIES code 
 *            to create a pseudo-MIL function. It creates a
 *            function that adds a constant to a grabbed image,
 *            shifts it right by 1 bit, and writes the result into a
 *            destination buffer in a single processing operation.
 */

/* General header file. */
#include <stdio.h>

/* MIL header file. */
#include <mil.h>

/* IMAGE-Series header files. */
#include <imseries.h>  /* Define for IMAGE system                  */
#include <i_head.h>    /* General IMAGE function call parameters   */
#include <proto.h>     /* C-prototypes for imaging function calls  */

/* Source MIL image file specifications. */ 
#define CONSTANT_TO_ADD        128L
#define EXECUTION_ERROR        (M_NATIVE_ERROR + 1L)

/* Pseudo-MIL function prototype */
void MyGrabAddConstantAndShift(MIL_ID MilCamera,
                               MIL_ID MilImage,
                               long Constant);

void main(void)
{
  MIL_ID MilApplication,    /* Application Identifier.      */
         MilSystem,         /* System Identifier.           */
         MilDisplay,        /* Display Identifier.          */
         MilCamera,         /* Camera Identifier.           */
         MilImage;          /* Image buffer Identifier.     */

  /* Allocate default system, camera and image. */
   MappAllocDefault(M_SETUP, &MilApplication, &MilSystem,
                             &MilDisplay, &MilCamera, &MilImage);

  /* Pause to show the original image. */
   printf("This program will add a constant to a grabbed image\n");
   printf("and right shift it by 1 in a single processing\n");
   printf("operation using the MIL native mode on IMAGE SERIES.\n");
   printf("Press <Enter> to continue.\n");
   getchar();

  /* Do processing on IM-SERIES with native mode */
   MyGrabAddConstantAndShift(MilCamera,MilImage,CONSTANT_TO_ADD);

  /* Print results. */
   printf("\n");
   printf("A constant was added to a grabbed image and\n");
   printf("was shiffed right by 1.\n");
   printf("Press <Enter> to end.");
   getchar();

  /* Free defaults */
   MappFreeDefault(MilApplication, MilSystem, MilDisplay,
                   MilCamera, MilImage);
}


/* Pseudo-MIL function definition */
void MyGrabAddConstantAndShift(MIL_ID MilCamera,
                               MIL_ID MilImage,
                               long Constant)

{
   MIL_ID FuncId;          /* allocated function ID                 */
   MIL_ID SystemId;        /* system owner of image buffer          */
   long   ErrorStatus = 0; /* current error status                  */
   long   Mask;            /* mask variable                         */
   long   SizeInBits;      /* depth of image buffer                 */
   long   Sign;            /* sign of image buffer                  */
   long   BufFormat;       /* specific image buffer location format */
   long   CamFormat;       /* specific camera format                */
   short  CamType;         /* specific type of camera               */

   short  CamSizeX,        /* Camera width variable.                */
          CamSizeY,        /* Camera height variable.               */
          CamSizeBit;      /* Camera bit depth variable.            */


   /* Prepare start of the function and register parameters */
   FuncId = MfuncAlloc("MyGrabAddConstantAndShift",3);
   if ( FuncId && MfuncParamRegister(FuncId))
      {
      MfuncParamId(FuncId,1,MilCamera,M_DIGITIZER,M_OUT);
      MfuncParamId(FuncId,2,MilImage,M_IMAGE,M_IN);
      MfuncParamLong(FuncId,3,Constant);
      }

   /* Mark the start of the function */
   if (MfuncStart(FuncId))
      {
      /* Reads camera X, Y and depth dimensions. */
      CamSizeX   = (short)MdigInquire(MilCamera,M_SIZE_X,  M_NULL);
      CamSizeY   = (short)MdigInquire(MilCamera,M_SIZE_Y,  M_NULL);
      CamSizeBit = (short)MdigInquire(MilCamera,M_SIZE_BIT,M_NULL);

      /* Find camera data format and type */
      if (CamSizeBit == 8L)
         {
         CamFormat  = I_UNSIGNED_8;
         CamType    = I_DIG_8;
         }
      else
         {
         CamFormat  = I_UNSIGNED_16;
         CamType    = I_DIG_16;
         }

      /* Inquire buffer caracteritic to find data format and owner system */
      SizeInBits = MbufInquire(MilImage,M_SIZE_BIT,    M_NULL);
      Sign       = MbufInquire(MilImage,M_SIGN,        M_NULL);
      SystemId   = MbufInquire(MilImage,M_OWNER_SYSTEM,M_NULL);

      /* Find buffer data format */
      if (SizeInBits == 8L  && Sign == M_UNSIGNED)
         BufFormat = I_UNSIGNED_8;
      else
      if (SizeInBits == 8L  && Sign == M_SIGNED)
         BufFormat = I_SIGNED_8;
      else
      if (SizeInBits == 16L && Sign == M_UNSIGNED)
         BufFormat = I_UNSIGNED_16;
      else 
         BufFormat = I_SIGNED_16;

      /* Find mask for processing */
      Mask = (1L << SizeInBits)-1;

      /* Enter IM-SERIES native programming mode */
      ErrorStatus = MnatEnterNativeMode(SystemId,M_DEFAULT);

      /* Report error (if any) through MIL error logging mechanism */
      if (ErrorStatus == M_NULL)
         {
         /* Select frame buffer organisation */
         pcstfborg((short)MnatGetLocOrg(MilImage));

         /* Set ROI size and position */
         pcstroisiz(CamSizeX,CamSizeY);
         pcstroipos(I_ROI0,0,0);
         pcstroipos(I_ROI1,(short)MnatGetLocPosX(MilImage),
                           (short)MnatGetLocPosY(MilImage));


         /* Set data format */
         pcstformat(CamFormat,BufFormat,BufFormat);

         /* Set data mask */
         pcstsrcmsk(I_SRC1, (short)Mask);
         pcstsrcmsk(I_SRC2, (short)Mask);
         pcstdstmsk(Mask);

         /* Set synchronization and number of fields to process */
         pcslsync(I_PROC_CAMERA);
         pcstcond(0,I_FOR,2);

         /* Add constant to the image and right shift by 1 bit
            in a single pass */
         ptdoalumap(CamType + I_ROI0, 
                    I_CONSTANT,                              
                    (short)MnatGetLocSurf(MilImage) + I_ROI1, 
                    I_ADD,                                   
                    1,                                       
                    (short)Constant);
                    
         }
      else
         MfuncErrorReport(FuncId, EXECUTION_ERROR,
                         "MyAddConstantAndShift",
                         "Function execution error.\n",
                         "Native mode function fail.\n",
                         M_NULL
                         );

      /* Leave native programming mode */
      MnatLeaveNativeMode(SystemId, M_DEFAULT);
      }

   /* Mark the end of the function */
   MfuncFreeAndEnd(FuncId);
}
