package org.apache.log4j.appender;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.helpers.FileHelper;

import junit.framework.Assert;
import junit.framework.TestCase;

public class TestLogFileCompressor extends TestCase {

  private static final String[] BACKUP_FILES;

  static {
    List names = new ArrayList();
    for (int i = 1; i <= 128; i++) {
      names.add("mylog.log.10." + i);
    }
    BACKUP_FILES = (String[]) names.toArray(new String[] {});
  }

  private final FileTestHelper fileTestHelper = new FileTestHelper();

  protected void setUp() throws Exception {
    this.fileTestHelper.setUp();
    for (int i = 0; i < BACKUP_FILES.length; i++) {
      this.fileTestHelper.writeFile(BACKUP_FILES[i], 10);
    }
  }

  protected void tearDown() throws Exception {
    this.fileTestHelper.tearDown();
  }

  // TODO This test is non-deterministic - has timing issues
  public void testCompressZipNoBlocking() {
    TimeAndSizeRollingAppender appender = new TimeAndSizeRollingAppender();
    AppenderRollingProperties properties = new AppenderRollingProperties();
    properties.setCompressionAlgorithm("ZIP");
    properties.setCompressionBlocking(false);
    LogFileCompressor compressor = new LogFileCompressor(appender, properties);
    compressor.begin();
    try {
      for (int i = 0; i < BACKUP_FILES.length; i++) {
        compressor.compress(this.fileTestHelper.fileForTest(BACKUP_FILES[i]));
      }
    } finally {
      try {
        /*
         * Expecting all files that fit into the queue to be compressed in this
         * time, others to have been skipped
         */
        Thread.sleep(10000L);
      } catch (InterruptedException e) {
        Thread.currentThread().interrupt();
      }
      if (compressor != null) {
        compressor.end();
      }
    }
    File dir = new FileHelper().parentDirOf(this.fileTestHelper
        .fileForTest(BACKUP_FILES[0]));
    Assert.assertNotNull(dir);
    Assert.assertTrue(dir.isDirectory());
    Assert.assertEquals(BACKUP_FILES.length, dir.listFiles().length);
    File[] fileList = dir.listFiles();
    /*
     * Expecting at least one file to be in the process of being compressed
     * whilst the remaining files are added to the compressor's queue, i.e. one
     * plus the queue limit. Therefore, start the count at -1 to account for the
     * file being compressed whilst the queue is filled up.
     */
    int zipCount = -1;
    for (int i = 0; i < fileList.length; i++) {
      if (fileList[i].getName().endsWith(".zip")) {
        Assert.assertTrue(new FileHelper().isZip(fileList[i]));
        zipCount++;
      }
    }
    Assert.assertEquals(compressor.getQueueLimit(), zipCount);
  }

  public void testCompressZipWithBlocking() {
    TimeAndSizeRollingAppender appender = new TimeAndSizeRollingAppender();
    AppenderRollingProperties properties = new AppenderRollingProperties();
    properties.setCompressionAlgorithm("ZIP");
    properties.setCompressionBlocking(true);
    LogFileCompressor compressor = new LogFileCompressor(appender, properties);
    compressor.begin();
    try {
      for (int i = 0; i < BACKUP_FILES.length; i++) {
        compressor.compress(this.fileTestHelper.fileForTest(BACKUP_FILES[i]));
      }
    } finally {
      try {
        /*
         * Expecting all files that fit into the queue to be compressed in this
         * time
         */
        Thread.sleep(10000L);
      } catch (InterruptedException e) {
        Thread.currentThread().interrupt();
      }
      if (compressor != null) {
        compressor.end();
      }
    }
    File dir = new FileHelper().parentDirOf(this.fileTestHelper
        .fileForTest(BACKUP_FILES[0]));
    Assert.assertNotNull(dir);
    Assert.assertTrue(dir.isDirectory());
    Assert.assertEquals(BACKUP_FILES.length, dir.listFiles().length);
    File[] fileList = dir.listFiles();
    /*
     * Expecting all backup files to have been compressed.
     */
    for (int i = 0; i < fileList.length; i++) {
      Assert.assertTrue(new FileHelper().isZip(fileList[i]));
    }
  }

  public void testCompressZipWithBlockingMinQueueSize() {
    TimeAndSizeRollingAppender appender = new TimeAndSizeRollingAppender();
    AppenderRollingProperties properties = new AppenderRollingProperties();
    properties.setCompressionAlgorithm("ZIP");
    properties.setCompressionBlocking(true);
    properties.setCompressionMinQueueSize(20);
    LogFileCompressor compressor = new LogFileCompressor(appender, properties);
    compressor.begin();
    try {
      for (int i = 0; i < BACKUP_FILES.length; i++) {
        compressor.compress(this.fileTestHelper.fileForTest(BACKUP_FILES[i]));
      }
    } finally {
      try {
        /*
         * Expecting all files that fit into the queue to be compressed in this
         * time
         */
        Thread.sleep(10000L);
      } catch (InterruptedException e) {
        Thread.currentThread().interrupt();
      }
      if (compressor != null) {
        compressor.end();
      }
    }
    File dir = new FileHelper().parentDirOf(this.fileTestHelper
        .fileForTest(BACKUP_FILES[0]));
    Assert.assertNotNull(dir);
    Assert.assertTrue(dir.isDirectory());
    Assert.assertEquals(BACKUP_FILES.length, dir.listFiles().length);
    File[] fileList = dir.listFiles();
    /*
     * Expecting 19 backup files to be uncompressed.
     */
    int compressedCount = 0;
    for (int i = 0; i < fileList.length; i++) {
      if (new FileHelper().isZip(fileList[i])) {
        compressedCount++;
      }
    }
    Assert.assertEquals((BACKUP_FILES.length - 19), compressedCount);
  }
}
