/* $Id: dsmcheck.c,v 1.7 2001/07/29 18:00:17 richdawe Exp $ */

/*
 *  dsmcheck.c - DSM file format checker
 *  Copyright (C) 1999-2001 by Richard Dawe
 *      
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "common.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

/* libzippo includes */
#include <libzippo/dsm.h>
#include <libzippo/package.h>
#include <libzippo/util.h>
#include <libzippo/version.h>

#include "version.h"

/* Compile time information */
#include "comptime.h"

/* Copyright info */
#define COPYRIGHT_NAME        "dsmcheck"
#define COPYRIGHT_DESCRIPTION "DJGPP Software Manifest syntax checker"
#define COPYRIGHT_AUTHOR      "Richard Dawe"
#define COPYRIGHT_EMAIL       "rich@phekda.freeserve.co.uk"
#define COPYRIGHT_YEAR        "1999-2001"

#include <zlib.h>
#ifndef ZLIB_COPYRIGHT
#define ZLIB_COPYRIGHT	\
"Copyright (C) 1995-1998 Jean-loup Gailly and Mark Adler"
#endif

extern char unz_copyright[];

/* Pwetty messages */
static void show_copyright (int license_hint);
static void show_help      (void);
static void show_license   (void);
static void show_version   (void);

/* --------
 * - main -
 * -------- */

int
main (int argc, char *argv[])
{
  PACKAGE_INFO  package;
  DSM_ERROR    *dsm_err  = NULL;	/* List of errors in DSM parsing */
  DSM_ERROR    *nde      = NULL;
  char         *filename = NULL;
  int           tret     = DSM_OK;
  int           ret, i;
	
  /* We need an argument. If none found, display help. */
  if (argc < 2) {
    show_copyright(1);
    show_help();		
    return(EXIT_SUCCESS);
  }

  /* Check whether we have a help or version option. */
  for (i = 1; i < argc; i++) {
    /* If we encounter '--', stop checking args. */
    if (strcmp(argv[i], "--") == 0) break;

    /* Help options */
    if (   (strcmp(argv[i], "-h") == 0)
	|| (strcmp(argv[i], "--help") == 0) ) {
      show_copyright(1);
      show_help();		
      return(EXIT_SUCCESS);
    }

    /* Version options (NB: '-v' is reserved for verbose mode). */
    if (   (strcmp(argv[i], "-V") == 0)
	|| (strcmp(argv[i], "--version") == 0) ) {
      show_copyright(1);
      show_version();

      return(EXIT_SUCCESS);
    }

    /* License option */
    if (   (strcmp(argv[i], "-L") == 0)
	|| (strcmp(argv[i], "--license") == 0) ) {
      show_copyright(0);
      show_license();		
      return(EXIT_SUCCESS);
    }
  }
	
  /* Loop over all specified files */
  for (i = 1; i < argc; i++) {
    filename = argv[i];

    if (access(filename, R_OK) != 0) {
      fprintf(stderr, "Unable to read %s file!\n", filename);
      continue;
    }

    ret = dsm_get_and_parse(filename, &package, NULL, &dsm_err);

    if (ret != DSM_OK) {
      /* Pass through failure code. */
      tret = DSM_FAILED;

      /* If there were any parsing errors, display them. */
      for (nde = dsm_err; nde != NULL; nde = (DSM_ERROR *) nde->q_forw) {
	dsm_perror(filename, nde);
      }
		
      /* Tidy up */
      dsm_free_error_list(dsm_err);
    } else {
      printf("DSM '%s' OK\n", filename);
    }
  }
	
  return((tret == DSM_OK) ? EXIT_SUCCESS : EXIT_FAILURE);
}

/* ------------------
 * - show_copyright -
 * ------------------ */

static void
show_copyright (int license_hint)
{
  printf("%s %d.%d.%d - %s\nCopyright (C) %s by %s <%s>\n",	       
	 COPYRIGHT_NAME,
	 ZIPPO_VERSION_MAJOR, ZIPPO_VERSION_MINOR, ZIPPO_VERSION_SUBMINOR,
	 COPYRIGHT_DESCRIPTION,
	 COPYRIGHT_YEAR, COPYRIGHT_AUTHOR, COPYRIGHT_EMAIL);

  if (license_hint) {
    printf("Distributed under terms of the GNU GPL "
	   "('%s -L' to view)\n",
	   COPYRIGHT_NAME);
  }

  printf("\n");
}

/* -------------
 * - show_help -
 * ------------- */

static void
show_help (void)
{
  printf("Syntax: %s [<Switches>] <DSM file> [<DSM file> ... <DSM file>]\n\n"
	 "Switches:\n"
	 "    -h, --help    - Display this help\n"
	 "    -V, --version - Display version information\n"
	 "    -L, --license - Display licensing information\n\n",
	 COPYRIGHT_NAME);
}

/* ----------------
 * - show_license -
 * ---------------- */

static void
show_license (void)
{
  printf(
"   This program is free software; you can redistribute it and/or modify\n"
"   it under the terms of the GNU General Public License as published by\n"
"   the Free Software Foundation; either version 2 of the License, or\n"
"   (at your option) any later version.\n"
"\n"
"   This program is distributed in the hope that it will be useful,\n"
"   but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
"   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
"   GNU General Public License for more details.\n"
"\n"
"   You should have received a copy of the GNU General Public License\n"
"   along with this program; if not, write to the Free Software\n"
"   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.\n"
"\n");
}

/* ----------------
 * - show_version -
 * ---------------- */

static void
show_version (void)
{
  printf("Compilation information:\n");

  printf(". Built on '%s' by %s", zippo_build_host, zippo_build_user);
  if ((zippo_build_user_mail != NULL) && (zippo_build_user_mail[0] != '\0'))
    printf(" <%s>", zippo_build_user_mail);
  printf("\n. Built on %s %s\n", zippo_build_date, zippo_build_time);

  printf(". libzippo %d.%d.%d\n", LIBZIPPO_VERSION_MAJOR,
	 LIBZIPPO_VERSION_MINOR, LIBZIPPO_VERSION_SUBMINOR);

#ifdef __DJGPP__
  printf(". DJGPP %d.0%d\n", DJGPP, DJGPP_MINOR);
#else
  printf(". %s\n", zippo_build_uname);
#endif

  printf(". %s\n. %s\n. zlib %s %s\n.%s\n", zippo_build_gcc,
	 zippo_build_binutils, ZLIB_VERSION, ZLIB_COPYRIGHT, unz_copyright);

  printf("\nDSM built-ins:\n");

  printf(". DSM version: %d.%d.%d\n",
	 DSM_VERSION_MAJOR, DSM_VERSION_MINOR, DSM_VERSION_SUBMINOR);

  printf("\n");
}
