/* ppmchange.c - change a given color to another
**
** Copyright (C) 1991 by Wilson H. Bent, Jr.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
**
** Modified by Alberto Accomazzi (alberto@cfa.harvard.edu).
**     28 Jan 94 -  Added support for multiple color substitution.
*/

#include "ppm.h"
#define TCOLS 256
#define SQRT3 1.73205080756887729352
    /* The square root of 3 */

struct cmdline_info {
    /* All the information the user supplied in the command line,
       in a form easy for the program to use.
    */
    char *input_filespec;  /* Filespecs of input files */
    int ncolors;      /* Number of valid entries in color0[], color1[] */
    pixel color0[TCOLS];  /* colors user wants replaced */
    pixel color1[TCOLS];  /* colors with which he wants them replaced */
    int closeness;    /* -closeness option value */
};



static void
parse_command_line(int argc, char ** argv,
                   struct cmdline_info *cmdlineP) {
/*----------------------------------------------------------------------------
   Note that the file spec array we return is stored in the storage that
   was passed to us as the argv array.
-----------------------------------------------------------------------------*/
    optStruct *option_def = malloc(100*sizeof(optStruct));
        /* Instructions to OptParseOptions2 on how to parse our options.
         */
    optStruct2 opt;

    unsigned int option_def_index;

    option_def_index = 0;   /* incremented by OPTENTRY */
    OPTENTRY(0,   "closeness",     OPT_INT,   &cmdlineP->closeness,     0);

    /* Set the defaults */
    cmdlineP->closeness = 0;

    opt.opt_table = option_def;
    opt.short_allowed = FALSE;  /* We have no short (old-fashioned) options */
    opt.allowNegNum = FALSE;  /* We may have parms that are negative numbers */

    pm_optParseOptions2(&argc, argv, opt, 0);
        /* Uses and sets argc, argv, and some of *cmdlineP and others. */

    if ((argc-1) % 2 == 0) 
        cmdlineP->input_filespec = "-";
    else
        cmdlineP->input_filespec = argv[argc-1];

    {
        int argn;
        cmdlineP->ncolors = 0;  /* initial value */
        for (argn = 1; 
             argn+1 < argc && cmdlineP->ncolors < TCOLS; 
             argn += 2) {
            cmdlineP->color0[cmdlineP->ncolors] = 
                ppm_parsecolor(argv[argn], PPM_MAXMAXVAL);
            cmdlineP->color1[cmdlineP->ncolors] = 
                ppm_parsecolor(argv[argn+1], PPM_MAXMAXVAL);
            cmdlineP->ncolors++;
            argn += 2;
        }
    }
}



static int
sqri(const int N) {
    return N*N;
}


static double
sqrf(const float F) {
    return F*F;
}



static int 
colormatch(const pixel comparand, const pixel comparator, 
           const float closeness) {
/*----------------------------------------------------------------------------
   Return true iff 'comparand' matches 'comparator' in color within the
   fuzz factor 'closeness'.
-----------------------------------------------------------------------------*/
    /* Fast path for usual case */
    if (closeness == 0)
        return PPM_EQUAL(comparand, comparator);

    return (sqri(PPM_GETR(comparand) - PPM_GETR(comparator)) +
            sqri(PPM_GETG(comparand) - PPM_GETG(comparator)) +
            sqri(PPM_GETB(comparand) - PPM_GETB(comparator)) 
            <= sqrf(closeness)
        );
}



int
main(int argc, char *argv[]) {
    struct cmdline_info cmdline;
    FILE* ifp;
    int format;
    int rows, cols;
    pixval maxval;
    float closeness;
    int row;
    pixel* inrow;
    pixel* outrow;

    ppm_init( &argc, argv );

    parse_command_line(argc, argv, &cmdline);

    ifp = pm_openr(cmdline.input_filespec);

    ppm_readppminit(ifp, &cols, &rows, &maxval, &format);

    closeness = SQRT3 * maxval * cmdline.closeness/100;

    ppm_writeppminit( stdout, cols, rows, maxval, 0 );
    inrow = ppm_allocrow(cols);
    outrow = ppm_allocrow(cols);

    /* Scan for the desired color */
    for (row = 0; row < rows; row++) {
        int col;
        ppm_readppmrow(ifp, inrow, cols, maxval, format);
        for ( col = 0; col < cols; ++col ) {
            int i;
            i = 0;  /* Start with first color in list */
            while (i < cmdline.ncolors && 
                   ! colormatch(inrow[col], cmdline.color0[i], closeness))
                i++;

            if (i < cmdline.ncolors)
                outrow[col] = cmdline.color1[i];
            else
                outrow[col] = inrow[col];
        }
        ppm_writeppmrow(stdout, outrow, cols, maxval, 0);
    }

    pm_close(ifp);

    exit(0);
}
