/* Copyright (C) 1996-1998 Robert Hhne, see COPYING.RH for details */
/* This file is part of RHIDE. */
/* This program can be used to translate the strings in a more friendly
   way. It knows about the special char '~' in keywords, so there is
   no necessary at all all, to retranslate the string, if the hotkey
   in the key has changed. It can read also any previous translated
   files to get possible translations from those files.
   It uses the conion functions for highlighting text on the screen and
   it may work (bether the color highlight may work) only on graphic cards
   wich can show correct the textcolor(7) and textbackground(7) colors.

   The program uses the great 'fstrcmp()' function from the gettext lib,
   which does a 'fuzzy' seach for finding a similar string.

   The usage is:

   msgcat [-l] [-s] [--only-similar] [-cat cat-file ] [-ocat cat-file]
          oldest-file any-newer-file potfile output-file

   where old-file(s) are the files, with existing
   translations in descending order by time, because if a file contains
   a translation, which was already found, the new one is taken.

   The file cat-file can be used to read from and to store in all the
   translations. This file can contain also translations, which are
   currently not used by the potfile.

   If -ocat is given, the cat-file specified there is used as output
   cat file instead of (if given) the cat-file from the -cat option.

   if -s was given, the program does not prompt for translating, it
   translates silently all exact msgid's and the others not.

   if -l was given, all the comments (which include for instance
   the line numbers of the input-files) are written in the output.

   if --only-similar is given, the program prompst only for strings,
   which are found to be similar.

*/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#ifdef __DJGPP__
#include <conio.h>

static void NORM()
{
  textcolor(7);
  textbackground(0);
}

static void LIGHT()
{
  textcolor(0);
  textbackground(7);
}

#define CRLF "\r\n"
#define CR "\r"

#else

#include <curses.h>

static void NORM()
{
#if 0
  attr_set(A_NORMAL);
#endif
}

static void LIGHT()
{
#if 0
  attr_set(A_REVERSE);
#endif
}

#define CRLF "\n"
#define CR "\n"

#define cscanf scanw
#define cprintf printw

#include <unistd.h>

static int __file_exists(const char *fname)
{
  return (access(fname,R_OK) == 0);
}

#endif

#include <unistd.h>

double fstrcmp(const char *,const char *);

struct msg
{
  char *id;
  char *id_;
  char *msg;
};

struct msg *msgs = NULL;
int msg_count = 0;
static int silent = 0;
static int only_similar = 0;
static int write_all = 0;

/* This function returns the string without any '~', because this is a
   special case in TV programs, so if I change the hotkey for a string
   there is no need to retranslate the string.
*/
char *similar(char *msg)
{
  static char buffer[4000];
  char *t1,*t2;
  strcpy(buffer,msg);
  t1 = strchr(buffer,'~');
  while (t1)
  {
    t2 = strchr(t1+1,'~');
    if (!t2) break;
    strcpy(t1,t1+1);
    strcpy(t2-1,t2);
    t1 = strchr(buffer,'~');
  }
  return strdup(buffer);
}

static int
cmp_msg(const void *m1, const void *m2)
{
  return strcmp(((struct msg *)m1)->id, ((struct msg *)m2)->id);
}

static void
sort_msgs()
{
  qsort(msgs, msg_count, sizeof(struct msg), cmp_msg);
}

static struct msg *find_msg(char *id)
{
  int i;
  for (i=0;i<msg_count;i++)
  {
    if (strcmp(msgs[i].id,id) == 0) return &msgs[i];
  }
  return NULL;
}

void add_msg(char *id,char *msg)
{
  struct msg *_msg = find_msg(id);
  if (_msg)
  {
    free(_msg->msg);
    _msg->msg = strdup(msg);
    return;
  }
  msgs = (struct msg *)realloc(msgs,sizeof(struct msg)*(msg_count+1));
  msgs[msg_count].id = strdup(id);
  msgs[msg_count].msg = strdup(msg);
  msgs[msg_count].id_ = similar(id);
  msg_count++;
}

static char MSG[4000];
static char ID[4000];
static FILE *fi,*fo;
static char LINE[4000];
static char line[4000];
static int line_count = 0;
static char *read_line()
{
  line_count ++;
  return fgets(LINE,4000,fi);
}

int read_id(int write_it)
{
  int started = 0;
  char *start,*end;
  ID[0] = 0;
  while (read_line())
  {
    if (!started && strncmp(LINE,"msgid ",6) != 0)
    {
      if (write_it && write_all) fprintf(fo,"%s",LINE);
      continue;
    }
    if (started && LINE[0] != '\"') return 1;
    if (!started && write_it)
      fprintf(fo, "msgid  ");
    start = strchr(LINE,'\"');
    if (!start) return 0;
    start++;
    end = strrchr(LINE,'\"');
    if (!end) return 0;
    *end = 0;
    if (write_it)
      fprintf(fo, "\"%s\"\n", start);
    strcat(ID,start);
    started = 1;
  }
  return 0;
}

int read_msg()
{
  int started = 0;
  char *start,*end;
  char *xline;

  MSG[0] = 0;
  do
  {
    xline=LINE;
    /* Cut spaces on the beginning */
    while (*xline==' ' || *xline=='\t')
      xline++;

/* fprintf(stderr,"%s\n",LINE); */
    if (!started && (strncmp(xline,"msgstr ",7) != 0))
    {
      fprintf(stderr,"Error: No 'msgstr' keyword found. (line %d)\n",line_count);
    }
    if (started && xline[0] != '\"') return 1;
    started = 1;
    start = strchr(xline,'\"')+1;
    end = strrchr(xline,'\"');
    *end = 0;
    strcat(MSG,start);
  } while (read_line());
  return 0;
}

static
void put_breakline(FILE *f,int start_len,int max_len,char *s)
{
  int len;
  while (1)
  {
    char *tmp;
    len = strlen(s);
    if (len+start_len <= max_len)
    {
      fprintf(f,"\"%s\"\n",s);
      return;
    }
    tmp = s + max_len-start_len;
    while (tmp > s && *tmp != ' ') tmp--;
    if (tmp == s)
    {
      fprintf(f,"\"%s\"\n",s);
      return;
    }
    *tmp = 0;
    fprintf(f,"\"%s \"\n",s);
    *tmp = ' ';
    start_len = 0;
    s = tmp+1;
  }
}

void write_msg(char *msg)
{
  char *tmp;
  char *start = msg;
  int started = 7;
  while ((tmp = strstr(start,"\\n")) != NULL)
  {
    char c = tmp[2];
    tmp += 2;
    *tmp = 0;
    if (started) fprintf(fo,"msgstr ");
    put_breakline(fo, started, 75, start);
    started = 0;
    *tmp = c;
    start = tmp;
  }
  if (started) fprintf(fo,"msgstr ");
  if (started || *start) put_breakline(fo, started, 75, start);
  fprintf(fo,"\n");
}

static void convert(char *s)
{
  char *b;
  while ((b = strchr(s,'\b')) != NULL)
  {
    strcpy(b-1,b+1);
  }
}

int fuzzy_search(char *id,char *idd)
{
  int i;
  char c;
  double best_weight;
  int best = -1;
  double weight;
  best_weight = 0.6;
  for (i=0;i<msg_count;i++)
  {
    weight = fstrcmp(id,msgs[i].id_);
    if (weight > best_weight)
    {
      best_weight = weight;
      best = i;
    }
  }
  while (best == -1)
  {
    if (only_similar) return -1;
    NORM();
    cprintf("Please translate:"CRLF"  ");
    LIGHT();
    cprintf("%s",idd);
    NORM();
    cprintf(CRLF"Enter the string: ");
    LIGHT();
    line[0] = 0;
    cscanf("%[^"CR"]",line);
    convert(line);
    NORM();
    cprintf("\nAccept ? (y)es/(n)o/(c)ancel ");
    c = tolower(getch());
    do
    {
      c = tolower(getch());
    } while (c != 'y' && c != 'n' && c != 'c');
    NORM();
    cprintf("%c"CRLF,c);
    if (c == 'y')
    {
      add_msg(idd,line);
      return msg_count-1;
    }
    if (c == 'c') return -1;
  }
  NORM();
  cprintf("translate:"CRLF"  ");
  LIGHT();
  cprintf("%s",idd);
  NORM();
  cprintf(CRLF"from"CRLF"  ");
  LIGHT();
  cprintf("%s",msgs[best].id);
  NORM();
  cprintf(CRLF"to"CRLF"             ");
  LIGHT();
  cprintf("%s",msgs[best].msg);
  NORM();
  cprintf(CRLF"??? (y)es/(n)o/(c)hange ");
  do
  {
    c = tolower(getch());
  } while (c != 'y' && c != 'n' && c != 'c');
  NORM();
  cprintf("%c"CRLF,c);
  if (c == 'y')
  {
    strcpy(MSG,msgs[best].msg);
    strcpy(MSG+strlen(MSG),"\\0# FUZZY");
    add_msg(idd,MSG);
    return msg_count-1;
/*  return best; */
  }
  while (c == 'c')
  {
    NORM();
    cprintf("Translation: ");
    LIGHT();
    line[0] = 0;
    cscanf("%[^"CR"]",line);
    convert(line);
    NORM();
    c = tolower(getch());
    cprintf("\nAccept ? (y)es/(n)o/(c)hange ");
    do
    {
      c = tolower(getch());
    } while (c != 'y' && c != 'n' && c != 'c');
    NORM();
    cprintf("%c"CRLF,c);
  }
  if (c == 'n') return -1;
  add_msg(idd,line);
  return msg_count-1;
}

void translate()
{
  int i;
  char *id_;
/*  if (ID[0] == 0)
    {
      write_msg("");
      return;
    } */
  for (i=0;i<msg_count;i++)
  {
    if (strcmp(msgs[i].id,ID) == 0)
    {
      write_msg(msgs[i].msg);
      return;
    }
  }
  if (silent)
  {
    write_msg("");
    return;
  }
  id_ = similar(ID);
  for (i=0;i<msg_count;i++)
  {
    if (strcmp(msgs[i].id_,id_) == 0) // do it not silently
    {
      char c;
      NORM();
      cprintf("translate:"CRLF"  ");
      LIGHT();
      cprintf("%s",ID);
      NORM();
      cprintf(CRLF"from"CRLF"  ");
      LIGHT();
      cprintf("%s",msgs[i].id);
      NORM();
      cprintf(CRLF"to"CRLF"             ");
      LIGHT();
      cprintf("%s",msgs[i].msg);
      NORM();
      cprintf(CRLF"??? (y)es/(n)o/(c)hange ");
      do
      {
        c = tolower(getch());
      } while (c != 'y' && c != 'n' && c != 'c');
      NORM();
      cprintf("%c"CRLF,c);
      if (c == 'y')
      {
        add_msg(ID,msgs[i].msg);
        write_msg(msgs[i].msg);
        return;
      }
      while (c == 'c')
      {
        NORM();
        cprintf("Translation: ");
        LIGHT();
        line[0] = 0;
        cscanf("%[^"CR"]",line);
        convert(line);
        NORM();
        c = tolower(getch());
        cprintf("\nAccept ? (y)es/(n)o/(c)hange ");
        do
        {
          c = tolower(getch());
        } while (c != 'y' && c != 'n' && c != 'c');
        NORM();
        cprintf("%c"CRLF,c);
      }
      if (c != 'n')
      {
        add_msg(ID,line);
        write_msg(line);
        return;
      }
    }
  }
  i = fuzzy_search(id_,ID);
  if (i != -1)
    write_msg(msgs[i].msg);
  else
    write_msg("");
}

char **potable;
int PotableItems;
int main(int argc,char *argv[])
{
  char *catfile;
  char *ocatfile;
  char *outputfile;
  char *potfile;

  int i;
#ifdef __linux__
  initscr();
#endif

  catfile = NULL;
  ocatfile = NULL;
  outputfile=NULL;
  potfile = NULL;


  potable=malloc(sizeof(char *));
  PotableItems=0;
  *potable=NULL;

/* Parse Arguments */
  while (argc > 1)
  {
    if (strcmp(argv[1],"-l") == 0)
    {
      write_all = 1;
      argc--;
      argv++;
      continue;
    }

    if (strcmp(argv[1],"--help") == 0 || strcmp(argv[1],"-?") == 0 )
    {
      printf(
 "\n>>>>msgcat<<<\n"
 "This program can be used to translate the strings in a more friendly\n"
 "way. It knows about the special char '~' in keywords, so there is\n"
 "no necessary at all all, to retranslate the string, if the hotkey\n"
 "in the key has changed. It can read also any previous translated\n"
 "files to get possible translations from those files.\n"
 "\n   The usage is:\n"
 " msgcat [-l] [-s] [--only-similar] [-cat cat-file ] [-ocat cat-file]\n"
 "oldest-file any-newer-file potfile output-file\n"
 "\n"
 "where old-file(s) are the files, with existing translations in descending\n"
 "   order by time, because if a file contains a translation,\n"
 "   which was already found, the new one is taken.\n"
 "The file cat-file can be used to read from and to store in all the\n"
 "   translations. This file can contain also translations, which are\n"
 "   currently not used by the potfile.");
      return(0);
    }

    if (strcmp(argv[1],"--only-similar") == 0)
    {
      only_similar = 1;
      argc--;
      argv++;
      continue;
    }

    if (strcmp(argv[1],"-s")==0 || strcmp(argv[1],"--silent") == 0)
    {
      silent = 1;
      argc--;
      argv++;
      continue;
    }

    if (argc > 1 && strcmp(argv[1],"-cat") == 0)
    {
      catfile = argv[2];
      ocatfile = argv[2];
      argc -= 2;
      argv += 2;
      continue;
    }

    if (argc > 1 && strcmp(argv[1],"-ocat") == 0)
    {
      ocatfile = argv[2];
      argc -= 2;
      argv += 2;
      continue;
    }

    if (strcmp(argv[1],"-o") == 0)
    {
      outputfile = argv[2];
      argc -= 2;
      argv += 2;
      continue;
    }

    if (strcmp(argv[1],"-po") == 0)
    {
      potfile = argv[2];
      argc -= 2;
      argv += 2;
      continue;
    }

    if (argv[1])
    {
      potable=realloc(potable,(PotableItems+1) * sizeof(char *));
      potable[PotableItems]=argv[1];
      PotableItems++;
    }
    argc--;
    argv++;
  }
/* End of reading arguments */


  if (PotableItems<2 && outputfile==NULL)
  {
    puts("Error: The output file is unknown!");
    return(-1);
  }
  if (outputfile==NULL)
  {
    PotableItems--;
    outputfile=potable[PotableItems];
  }

  if (PotableItems<2 && potfile==NULL)
  {
    puts("Error: The pot file is unknown!");
    return(-2);
  }
  if (potfile==NULL)
  {
    PotableItems--;
    potfile=potable[PotableItems];
  }

/* Read the .po files */
  for (i=0;i<PotableItems;i++)
  {
    if ((fi = fopen(potable[i],"r"))==NULL)
    {
      printf("Error: Cannot open a file %s\n!",potable[i]);
      continue;
    }

    line_count = 0;
    while (read_id(0))
    {
      read_msg();
      if (strlen(MSG) > 0) add_msg(ID,MSG);
    }
    fclose(fi);
  }

  if (catfile && __file_exists(catfile))
  {
    fi = fopen(catfile,"r");
    line_count = 0;
    while (read_id(0))
    {
      read_msg();
      if (strlen(MSG) > 0) add_msg(ID,MSG);
    }
    fclose(fi);
  }

/* Write output po file to the disk */
  if ((fi = fopen(potfile,"r"))==NULL)
  {
    printf("Error: Cannot open pot file!");
    return(-3);
  }
  if ((fo = fopen(outputfile,"w"))==NULL)
  {
    printf("Error: Cannot open output file!");
    return(-3);
  }
  line_count = 0;
  while (read_id(1))
  {
    read_msg();
    translate();
  }
  fclose(fi);
  fclose(fo);

/* Write output cat file to the disk */
  if (ocatfile)
  {
    sort_msgs();
    fo = fopen(ocatfile,"w+");
    for (i=0;i<msg_count;i++)
    {
      fprintf(fo,"msgid \"%s\"\nmsgstr \"%s\"\n\n",msgs[i].id,msgs[i].msg);
    }
    fclose(fo);
  }

#ifdef __linux__
  endwin();
#endif
  return 0;
}

