#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <pc.h>
#include <keys.h>
#include <string.h>
#include <dpmi.h>
#include "serio.h"

/*
This program is based on the serial io functions from BCSERIO.ZIP
A function has been added for checking if output-buffer has been
sent completly.
*/

// Functions for command-line processing as getting filenames and parameters
// in different formats.

int num_arguments; // set to argc of main
char**arguments;   // set to argv of main

// get parameter preceeded by 'name'
// return NULL if not found
char *get_char_param(const char *name)
{
  unsigned namelen=strlen(name);
  int a;
  for (a=1; a<num_arguments; a++)
	if (!strncmp(arguments[a],name,namelen))
	  return arguments[a]+namelen;
  return NULL;
}

// get parameter preceeded with 'name' as integer, return 0 if not given
int get_int_param(const char *name)
{
  char *param=get_char_param(name);
  if (param)
	return atoi(param);
  return 0;
}

void help()
{
  puts("USAGE: term <OPTIONS>");
  puts("OPTIONS:");
  puts("  -comX:B,P,D,S  ... select IO-port (default: com1:9600,n,8,2)");
  puts("     X ... port-number (1-4)");
  puts("     B ... baud-rate (300-115200)");
  puts("     P ... parity (n/o/e/m/s)");
  puts("     D ... data-bits (5-8)");
  puts("     S ... stopbits (1/2)");
  puts("  -com:P,I:B,P,D,S ... selectio IO-port (default com:3f8,4:9600,n,8,2)");
  puts("     B,P,D,S ... as above");
  puts("     P ... base port");
  puts("     I ... irq number (0-15)");
  puts("  -oNAME ... write input to file 'NAME'");
  puts("  -iNAME ... send file 'NAME'");
  puts("  -waitX ... wait X seconds for response after sending file");
  puts("  -echo  ... local echo on (screen only!)");
  puts("  -help  ... print this screen");
  puts("Keyboard-COMMANDS in program:");
  puts("  ALT-X  ... QUIT");
  puts("  ALT-E  ... toggle local echo");
  exit(1);
}

int main(int argc, char *argv[])
{
  int comnr=1, baud=9600, bits=8, stop=2;
  char parity='n';
  int raw=0;
  int base=0x3f8, irq=4;

  int done=0,echo=0,waitsent=0,c;
  clock_t wait=0;

  enum sioWordSize siobits;
  enum sioParity sioparity;
  enum sioStopBits siostop;
  int divisor;

  FILE *in=NULL;
  FILE *out=NULL;

  SioPort *port=NULL;

  printf("simple terminal-program (c) Roland Exler, %s\n",__DATE__);
  num_arguments=argc; arguments=argv;
  if (get_char_param("-h")) help();

  if (get_char_param("-com"))
  {
	char *com=get_char_param("-com");
	if (com[0]==':')
	{
	  raw=1;
	  if (sscanf(com,":%d,%d:%d,%c,%d,%d",&base,&irq,&baud,&parity,&bits,&stop)!=6)
	  {
	  	char dummy;
		if (sscanf(com,":0%[xX]%x,%d:%d,%c,%d,%d",&dummy,&base,&irq,&baud,&parity,&bits,&stop)!=7)
		{
		  fprintf(stderr,"error processing string '-com%s'",get_char_param("-com"));
		  abort();
		}
	  }
	}
	else
	{
	  raw=0;
	  if (sscanf(com,"%d:%d,%c,%d,%d",&comnr,&baud,&parity,&bits,&stop)!=5)
	  {
		fprintf(stderr,"error processing string '-com%s'",get_char_param("-com"));
		abort();
	  }
	}
  }

// check baudrate
  if ((baud>115200) || (baud<300) || (115200%baud))
  {
	fprintf(stderr,"illegal baudrate selected\n");
	abort();
  }

// check databits
  switch(bits)
  {
	case 5: siobits=sio5Bits; break;
	case 6: siobits=sio6Bits; break;
	case 7: siobits=sio7Bits; break;
	case 8: siobits=sio8Bits; break;
	default:
	  fprintf(stderr,"%d databits invalid!\n",bits);
	  abort();
  }

// check parity
  switch(parity)
  {
	case 'n': sioparity=sioNoParity; break;
	case 'o': sioparity=sioOddParity; break;
	case 'e': sioparity=sioEvenParity; break;
	case 'm': sioparity=sioMarkParity; break;
	case 's': sioparity=sioSpaceParity; break;
	default:
	  fprintf(stderr,"parity %c invalid!\n",parity);
	  abort();
  }

// check stopbits
  switch(stop)
  {
	case 1:	siostop=sio1StopBit; break;
	case 2: siostop=sio2StopBits; break;
	default:
	  fprintf(stderr,"%d stopbits invalid!\n",stop);
	  abort();
  }

// init sio and interrupt
  if (raw)
  {
	if (irq<0 || irq>15)
	{
		fprintf(stderr,"invalid irq nubmer %d\n",irq);
		abort();
	}
    port=sio_openport(base,irq);
  }
  else
  {
	switch(comnr)
	{
	  case 1: port=sio_openport(0x3f8,4); break;
	  case 2: port=sio_openport(0x2f8,3); break;
	  case 3: port=sio_openport(0x3e8,4); break;
	  case 4: port=sio_openport(0x2e8,3); break;
	  default:
		fprintf(stderr,"don't know how to handle COM%d\n",comnr);
		abort();
	}
  }

  sio_setparms(port,siobits,sioparity,siostop);

  divisor=115200/baud;
  sio_setspeed(port,divisor);

// open input-file (binary!)
  if (get_char_param("-i"))
	in=fopen(get_char_param("-i"),"rb");

// open output-file (binary!)
  if (get_char_param("-o"))
	out=fopen(get_char_param("-o"),"wb");

  while (!done)
  {
// characters from keyboard
	if (kbhit())
	{
	  c=getkey();
	  if (c>0x100)
	  {
		switch (c)
		{
		  case K_Alt_E: echo=!echo; break;
		  case K_Alt_X: done=1; break;
		}
	  }
	  else
	  {
		while (sio_put(port,c)) {};
		if (echo)
		{
		  putchar(c); if (c=='\r') putchar('\n');
		  fflush(stdout);
		}
	  }
	}
// characters from input-file
	if (in)
	{
	  c=fgetc(in);
	  if (feof(in))
	  {
		fclose(in); in=NULL;
		waitsent=1;
	  }
	  else
	  {
		while (sio_put(port,c)) {};
		if (echo)
		{
		  putchar(c); if (c=='\r') putchar('\n');
		  fflush(stdout);
		}
	  }
	}
// input from rs232
	if (sio_charready(port))
	{
	  c=sio_get(port);
	  putchar(c);
	  if (out) fputc(c,out);
	  // if (c=='\r') putchar('\n');
	  fflush(stdout);
	}
// wait for output-buffer to be sent
	if (waitsent && !sio_senddone(port))
	{
	  waitsent=0;
	  wait=get_int_param("-wait");
	  if (!wait) done=1; // don't wait for response
	  wait*=CLOCKS_PER_SEC;
	}
// wait additional time for response
	if (wait && (clock()>=wait)) done=1;
  }
  if (out) fclose(out);
  if (in) fclose(in);
  sio_closeport(port);
  return 0;
}

extern char sltext[] asm ("sltext");
extern char eltext[] asm ("eltext");
extern char sldata[] asm ("sldata");
extern char eldata[] asm ("eldata");

static void __attribute__((contructor))lock_sections(void)
{
	_go32_dpmi_lock_code(sltext,eltext-sltext);
	_go32_dpmi_lock_data(sldata,eldata-sldata);
}

/*
TODO:
moved to todo.txt

*/
