/****************************************************************************/
/* SCREEN                                                                   */
/*--------------------------------------------------------------------------*/
/* Fonctions de manipulation de l'cran Texte (VGA)                         */
/*--------------------------------------------------------------------------*/
/* Auteur     : DELPRAT Jean-Pierre                                         */
/* Cr le    : 10/01/95                                                    */
/****************************************************************************/

#include <conio.h>
#include <dir.h>
#include <dos.h>
#include <process.h>
#include <stdlib.h>
#include <stdio.h>

#include "JPAppli.h"

#include "Compat.h"
#include "Const.h"
#include "Vocab.h"
#include "Types.h"

#include "Cursor.h"
#include "Errors.h"
#include "Files.h"
#include "JPData.h"
#include "Mouse.h"
#include "SpChars.h"
#include "Strings.h"

#include "Screen.h"


/*ͻ*/
/*                                 CONSTANTS                              */
/*ͼ*/

// Video interrupt

#define INT_VIDEO 0x10


/*ͻ*/
/*                             GLOBAL VARIABLES                           */
/*ͼ*/

// A VGA card is present : is only significant when the screen has
// been initialized

boolean   s_vga_card_present=TRUE;


/*ͻ*/
/*                             STATIC VARIABLES                           */
/*ͼ*/

// Text mode

static TTextMode s_text_mode=TEXTMODE_80x25_9;

// Function to call in order to set the chosen text mode

static void (*s_set_mode_function)()=NULL;

// Screen size in characters

static int s_screen_width=80;
static int s_screen_height=25;

// Font size in dots

static int s_font_height=16;
static int s_font_width=9;

// L'cran a-t-il t initialis

static boolean s_screen_initialized=FALSE;

// The screen is hidden

static boolean s_screen_hidden=FALSE;

// The cursor was visible before hiding the screen

static boolean s_cursor_was_visible=FALSE;


#ifdef __TCPLUSPLUS__

static int s_critical_error_handler(int /*errval*/,int /*ax*/,int /*bp*/,int /*si*/)
{
  _hardresume(_HARDERR_FAIL);
  return(0);
}

#endif


/*ͻ*/
/*                                 FONCTIONS                              */
/*ͼ*/

bool TEXT_redefined_symbol(int ch, const char* text)
{
  for (size_t i = 0; text[i] != 0; i++) if 
    ((unsigned char)text[i] == ch) return true;
  return false;
}

bool redefined_symbol(int ch)
{
  bool redefined = false;
  switch (ch) {
    case GMOUSE_1ST_CHAR: case GMOUSE_1ST_CHAR+1:
    case GMOUSE_1ST_CHAR+2:case GMOUSE_1ST_CHAR+3:
    case (unsigned char)CHAR_UP_LEFT_CORNER:
    case (unsigned char)CHAR_UP_HLINE:
    case (unsigned char)CHAR_UP_RIGHT_CORNER:
    case (unsigned char)CHAR_MIDDLE_RIGHT_CORNER:
    case (unsigned char)CHAR_LEFT_VLINE:
    case (unsigned char)CHAR_RIGHT_VLINE:
    case (unsigned char)CHAR_BOTTOM_LEFT_CORNER:
    case (unsigned char)CHAR_BOTTOM_HLINE:
    case (unsigned char)CHAR_BOTTOM_RIGHT_CORNER:
    case (unsigned char)CHAR_TICK:
    case (unsigned char)CHAR_UP_BOTTOM_HLINES:
    case (unsigned char)CHAR_ALT_UP_LEFT_CORNER:
    case (unsigned char)ADDON_CHAR_UP_LEFT_CORNER:
    case (unsigned char)ADDON_CHAR_UP_HLINE:
    case (unsigned char)ADDON_CHAR_BOTTOM_LEFT_CORNER:
    case (unsigned char)ADDON_CHAR_MIDDLE_UP_DOUBLE_LINE:
    case (unsigned char)ADDON_CHAR_MIDDLE_UP_LINE:
    case 16: // '' redefined too
      redefined = true; break;
    default: 
      if (ch != ' ') {
        if (redefined=TEXT_redefined_symbol(ch, TEXT_ARROW_UP)) break;
        if (redefined=TEXT_redefined_symbol(ch, TEXT_ARROW_DOWN)) break;
        if (redefined=TEXT_redefined_symbol(ch, TEXT_ARROW_LEFT)) break;
        if (redefined=TEXT_redefined_symbol(ch, TEXT_ARROW_RIGHT)) break;
        if (redefined=TEXT_redefined_symbol(ch, TEXT_CLOSE_BUTTON)) break;
        if (redefined=TEXT_redefined_symbol(ch, TEXT_CHECKED_RADIO_BUTTON)) break;
        if (redefined=TEXT_redefined_symbol(ch, TEXT_UNCHECKED_RADIO_BUTTON)) break;
        if (redefined=TEXT_redefined_symbol(ch, TEXT_CHECKED_CHECK_BOX)) break;
        if (redefined=TEXT_redefined_symbol(ch, TEXT_UNCHECKED_CHECK_BOX)) break;
      }
  }
  return redefined;
}

int ToScreenChar(int ch)
// Fix: converts all redefined font symbols to '.'
{
  return redefined_symbol(ch) ? '.' : ch;
}

/****************************************************************************/
/* WaitRetrace                                                              */
/*--------------------------------------------------------------------------*/
/* Attend le retour vertical du rayon lectronique de l'cran               */
/****************************************************************************/

void WaitRetrace()
{
//  Fix: too slow redrawing
//  while (inportb(0x3da)&0x08);
//  while ((inportb(0x3da)&0x08)==0);
}

/****************************************************************************/
/* GetSetCharDefinition                                                     */
/*--------------------------------------------------------------------------*/
/* Lit/Modifie la dfinition de caractres texte                            */
/****************************************************************************/

void GetSetCharDefinition(boolean set,int nb_chars,char first_char,char *char_definition)
{
  int font_height;

  if (   (!s_screen_initialized)
      || (!s_vga_card_present))
    return;

  if (nb_chars<1)
    return;

  // Retrace

  WaitRetrace();

  // Accs linaire  la mmoire VGA

  disable();
  outport(0x3c4,0x0402);
  outport(0x3c4,0x0704);
  outport(0x3ce,0x0204);
  outport(0x3ce,0x0005);
  outport(0x3ce,0x0406);

  #ifdef __TCPLUSPLUS__

  asm {
	    mov    ax,s_font_height
	    mov    font_height,ax

	    mov    bx,32
	    sub    bx,ax
	    mov    dx,nb_chars
	    push   ds

	    mov    ax,set
	    cmp    ax,0
	    jz     unset
      }

  set:

  asm {
	lds    si,char_definition
	mov    ax,0xA000
	mov    es,ax
	mov    ax,0
	mov    al,first_char
	shl    ax,5
	mov    di,ax
      }

  loop_set:

  asm {
	mov    cx,font_height
	shr    cx,1
    rep movsw
	add    di,bx
	dec    dx
	jnz    loop_set
	jmp    end_asm
      }

  unset:

  asm {
	les    di,char_definition
	mov    ax,0xA000
	mov    ds,ax
	mov    ax,0
	mov    al,first_char
	shl    ax,5
	mov    si,ax
      }

  loop_get:

  asm {
	mov    cx,font_height
	shr    cx,1
    rep movsw
	add    si,bx
	dec    dx
	jnz    loop_get
      }

  end_asm:

  asm {
	pop    ds
      }


  #else

  asm ("
	    movl   _s_font_height,%%eax
	    movl   %%eax,%0   # font_height

	    movl   $32,%%ebx
	    subl   %%eax,%%ebx
	    movl   %5,%%edx  # nb_chars

	    movl   %4,%%eax  # set
	    cmpl   $0,%%eax
	    jz     2f        # unset:

	 # set:

	    pushw  %%es
	    movl   %1,%%esi  # char_definition
	    movw   %2,%%es   # _dos_ds
	    movl   $0xA0000,%%edi
	    movl   $0,%%eax
	    movb   %3,%%al   # first_char
	    shll   $5,%%eax
	    addl   %%eax,%%edi

      0: # loop_set:

	    movl   %0,%%ecx  # font_height
	    shrl   $1,%%ecx
	rep;movsw
	    addl   %%ebx,%%edi
	    decl   %%edx
	    jnz    0b
	    popw   %%es
	    jmp    3f        #end_asm:

      2: # unset:

	    pushw  %%ds
	    movl   %1,%%edi
	    movw   %2,%%ds
	    movl   $0xA0000,%%esi
	    movl   $0,%%eax
	    movb   %3,%%al
	    shll   $5,%%eax
	    addl   %%eax,%%esi

      1: # loop_unset:

	    movl   %0,%%ecx
	    shrl   $1,%%ecx
	rep;movsw
	    addl   %%ebx,%%esi
	    decl   %%edx
	    jnz    1b
	    popw   %%ds

      3: # end_asm

	   "
	   :"=m" (font_height)       // 0
	   : "m" (char_definition),  // 1
	    "m" (_dos_ds),           // 2
	    "m" (first_char),        // 3
	    "m" (set),		     // 4
	    "m" (nb_chars)           // 5
	   :"%esi","%edi","%eax","%ebx","%ecx","%edx");

  #endif


  // Accs par plans  la mmoire VGA

  outport(0x3c4,0x0302);
  outport(0x3c4,0x0304);
  outport(0x3ce,0x0004);
  outport(0x3ce,0x1005);
  outport(0x3ce,0x0e06);
  enable();
}


/****************************************************************************/
/* SetTextMode/SetUserTextMode                                              */
/*--------------------------------------------------------------------------*/
/* Sets the text mode of the screen					    */
/****************************************************************************/

static void SetMode(TTextMode text_mode,void (*set_mode_function)())
{
  boolean init;

  init=s_screen_initialized;
  if (init)
    HideScreen();

  s_text_mode=text_mode;

  s_set_mode_function=set_mode_function;

  if (init)
    UnhideScreen();
}

void SetTextMode(TTextMode text_mode)
{
  if (text_mode==TEXTMODE_USER)
    text_mode=TEXTMODE_80x25_9;

  SetMode(text_mode,NULL);
}

void SetUserTextMode(void (*set_mode_function)())
{
  SetMode(TEXTMODE_USER,set_mode_function);
}


/****************************************************************************/
/* GetTextMode                                                              */
/*--------------------------------------------------------------------------*/
/* Returns the text mode of the screen					    */
/****************************************************************************/

TTextMode GetTextMode()
{
  return(s_text_mode);
}

/****************************************************************************/
/* GetScreenWidth/GetScreenHeight                                           */
/*--------------------------------------------------------------------------*/
/* Returns the size of the screen in characters 			    */
/****************************************************************************/

int     GetScreenWidth()
{
  return(s_screen_width);
}

int     GetScreenHeight()
{
  return(s_screen_height);
}

/****************************************************************************/
/* SetFont/GetFont                                            		    */
/*--------------------------------------------------------------------------*/
/* Sets/Returns the font used by JPTUI					    */
/****************************************************************************/

void SetFont(TFont font)
{
  boolean init=s_screen_initialized;

  if ((init) &&
      (!s_vga_card_present) &&
      (font==FONT_JPTUI))
    return;

  if (init)
    HideScreen();

  JPFont=font;
  JPRedrawWindows();

  if (init)
    UnhideScreen();
}

TFont GetFont()
{
  return(JPFont);
}


/****************************************************************************/
/* GetFontHeight/GetFontWidth                                  		    */
/*--------------------------------------------------------------------------*/
/* Returns the size of the font   		                 	    */
/****************************************************************************/

int     GetFontHeight()
{
  return(s_font_height);
}

int     GetFontWidth()
{
  return(s_font_width);
}

/****************************************************************************/
/* InitScreen                                                               */
/*--------------------------------------------------------------------------*/
/* Initialise la souris, l'cran (Fond intense),le curseur, la police de    */
/* caracteres    							    */
/* Montre la souris.                                                        */
/* Quitte le programme en cas de problme				    */
/****************************************************************************/

void SetChosenMode()
{
  int width;
  __dpmi_regs regs;
  u_char tmp;

  if (s_text_mode==TEXTMODE_USER)
    {
      if (s_set_mode_function!=NULL)
	s_set_mode_function();
    }
  else
    {
      // 80x25

      regs.x.ax=0x0003;
      __dpmi_int(INT_VIDEO, &regs);

      // Nombre de lignes

      regs.x.ax=0x1114;
      switch (s_text_mode)
	{
	  case TEXTMODE_80x25_8 :
	  case TEXTMODE_80x25_9 : regs.h.al=0x14;break;
	  case TEXTMODE_80x28_8 :
	  case TEXTMODE_80x28_9 : regs.h.al=0x11;break;
	  case TEXTMODE_80x50_8 :
	  case TEXTMODE_80x50_9 : regs.h.al=0x12;break;
	  default               : break;
	}

      regs.h.bl=0;
      __dpmi_int(INT_VIDEO, &regs);

      // 8 ou 9 pixels de large

      if (   (s_text_mode==TEXTMODE_80x25_8)
	  || (s_text_mode==TEXTMODE_80x28_8)
	  || (s_text_mode==TEXTMODE_80x50_8))
	{
	  width=8;
	  regs.x.bx=0x0001;
	}
      else
	{
	  width=9;
	  regs.x.bx=0x0800;
	}

      tmp=inportb(0x3CC) & (255-12);
      if (width==9)
	tmp|=4;
      outportb(0x3C2,tmp);

      disable();
      outport(0x3C4,0x0100);
      outport(0x3C4,0x01+(regs.h.bl<<8));
      outport(0x3C4,0x0300);
      enable();

      regs.x.ax=0x1000;
      regs.h.bl=0x13;
      __dpmi_int(INT_VIDEO, &regs);

    }
}

// Attention : peut retablir le curseur par defaut

void SetChosenFont()
{
  char font_file[12];
  char *char_definition;
  __dpmi_regs regs;
  long length;
  char *extension;

  // Police de caractres JPTUI

  if (JPFont==FONT_JPTUI)
  {
    // Chargement de la police de caractres

    sprintf(font_file,"JP%1dx%d",s_font_width,s_font_height);

    switch (GetLanguage())
      {
	case GREEK  : extension=".GRK";
		      break;

	default     : extension=".FNT";
      }

    strcat(font_file,extension);

    LoadDataFile(font_file,char_definition,length);
    if ((char_definition==NULL) || (length!=(256*s_font_height)))
      {
	if (char_definition!=NULL)
	  delete []char_definition;
	FatalError(GetInvalidDataFileMessage());
      }

    // Modification de la police de caractres
    // Fix: redefine only needed symbols
    // DefineChars(256,0,char_definition);
    for (int i=0; i<256; i++)
      if (redefined_symbol(i)) DefineChars(1, i, char_definition+i*s_font_height);
    delete []char_definition;
  }

  // Police de caractres standard

  else
  {
    switch (s_font_height)
    {
      case 8 : regs.x.ax = 0x1112;
	       break;
      case 14: regs.x.ax = 0x1111;
	       break;
      default: regs.x.ax = 0x1114;
	       break;
    }

    regs.h.bl=0;
    __dpmi_int(INT_VIDEO, &regs);

    // Attention : retablit le curseur par defaut
  }
}


void  InitScreen()
{
  __dpmi_regs regs;

  // Y a-t-il une carte VGA ?

  regs.x.ax=0x1A00;
  __dpmi_int(INT_VIDEO, &regs);

  if (regs.h.al!=0x1A)
    {
      s_vga_card_present = FALSE;

      // Les caracteres speciaux sont interdits

      SetFont(FONT_STANDARD);
      SetMousePointerAspect(MPA_MOUSE_DRIVER);
    }
  else
    {
      s_vga_card_present = TRUE;
    }

  // On peut maintenant fermer l'cran

  s_screen_initialized=TRUE;

  // Passage en mode texte

  SetChosenMode();

  // Rcupration des paramtres du nouveau mode vido

  s_screen_width=_peekw((0x40<<4)+0x4a);
  s_screen_height=((int)_peekb((0x40<<4)+0x84)+1);
  s_font_height=_peekw((0x40<<4)+0x85);

  outportb(0x3c4,0x1);
  s_font_width=((inportb(0x3c5) & 1)==0)?9:8;

  // Police de caractres

  SetChosenFont();

  // Dissimulation du curseur texte

  HideTextCursor();

  // Fonds intenses

  regs.x.ax=0x1003;
  regs.h.bl=0;
  __dpmi_int(INT_VIDEO, &regs);

  // Initialisation et affichage de la souris

  InitMouse();
  ShowMouse();

  // Dsactivation de l'erreur critique

  #ifdef __TCPLUSPLUS__
  harderr(s_critical_error_handler);
  #endif
}

/****************************************************************************/
/* CloseScreen                                                              */
/*--------------------------------------------------------------------------*/
/* Remet l'cran dans sa configuration normale afin de quitter le programme */
/****************************************************************************/

void CloseScreen()
{
  __dpmi_regs regs;

  // Si l'cran n'a pas t initialis, rien  faire

  if (!s_screen_initialized)
    return;

  // On repasse ventuellement en mode texte

  textmode(C80);

  // Effacement de l'cran

  ClearScreen((BLACK<<4)+(unsigned)LIGHTGRAY,' ');

  // Disparition du curseur souris
  // Affichage du curseur texte

  HideMouse();

  _setcursortype(_NORMALCURSOR);
  gotoxy(1,1);

  // Fin de la souris

  CloseMouse();

  // Fonds normaux (non intenses)

  regs.x.ax=0x1003;
  regs.h.bl=1;
  __dpmi_int(INT_VIDEO, &regs);

  // On restaure la police d'origine

  regs.x.ax = 0x1114;
  regs.h.bl=0;
  __dpmi_int(INT_VIDEO, &regs);

  // L'cran est ferm

  s_screen_initialized=FALSE;

}

/****************************************************************************/
/* ScreenInitialized                                                        */
/*--------------------------------------------------------------------------*/
/* L'cran a-t-il t initialis ?                                          */
/****************************************************************************/

boolean ScreenInitialized()
{
  return (s_screen_initialized);
}

/****************************************************************************/
/* HideScreen                                                               */
/*--------------------------------------------------------------------------*/
/* Cache l'cran (le sauve et restaure l'cran standard)                    */
/****************************************************************************/

void HideScreen()
{
  if (s_screen_hidden)
    return;

  s_cursor_was_visible=TextCursorIsVisible();
  CloseScreen();
  s_screen_hidden=TRUE;
}

/****************************************************************************/
/* UnhideScreen                                                             */
/*--------------------------------------------------------------------------*/
/* Remontre l'cran cach par HideScreen                                    */
/****************************************************************************/

void UnhideScreen()
{
  if (!s_screen_hidden)
    return;

  InitScreen();
  JPRefreshAll();
  if (s_cursor_was_visible)
    ShowTextCursor();
  s_screen_hidden=FALSE;
}

/****************************************************************************/
/* ClearScreen                                                              */
/*--------------------------------------------------------------------------*/
/* Effacement de l'cran avec le caractre et la couleur indiqus           */
/****************************************************************************/

void ClearScreen(char attribute,char character)
{
  int screen_size=s_screen_width*s_screen_height;

  HideMouse();

  #ifdef __TCPLUSPLUS__

  asm {
	mov ax,0xB800
	mov es,ax
	mov di,0
	mov ah,attribute
	mov al,character
	mov cx,screen_size
    rep stosw
      }

  #else

  asm("
	pushw %%es
	movw %0,%%es
	movl $0xB8000,%%edi
	movb %1,%%bh
	movb %2,%%bl
	movw %%bx,%%ax
	shll $16,%%eax
	movw %%bx,%%ax
	movl %3,%%ecx
	shrl $1,%%ecx
    rep;stosl
	popw %%es
      "
      :
      :"m" (_dos_ds),
       "m" (attribute),
       "m" (character),
       "m" (screen_size)
      :"%edi","%bx","%eax","%ecx");

  #endif

  ShowMouse();

}

/****************************************************************************/
/* ClearPartOfScreen                                                        */
/*--------------------------------------------------------------------------*/
/* Effacement d'1 partie de l'cran avec le caractre et la couleur indiqus*/
/****************************************************************************/

void ClearPartOfScreen(int x1,int y1,int x2,int y2,char attribute,char character)
{
  int height,width,start_offset;


  if ((x1>s_screen_width) || (y1>s_screen_height) || (x2<1) || (y2<1)
      || (x2<x1) || (y2<y1))
    return;

  if (x2>s_screen_width)
    x2=s_screen_width;
  if (x1<1)
    x1=1;

  if (y2>s_screen_height)
    y2=s_screen_height;
  if (y1<1)
    y1=1;

  width=x2-x1+1;
  height=y2-y1+1;
  start_offset=(((y1-1)*s_screen_width)+(x1-1))<<1;

  HideMouse();

  #ifdef __TCPLUSPLUS__

  asm {
	mov ax,0xB800
	mov es,ax
	mov di,start_offset
	mov ah,attribute
	mov al,character
	mov bx,s_screen_width
	sub bx,width
	shl bx,1
	mov dx,height
      }

  loop:

  asm {
	mov cx,width
    rep stosw
	add di,bx
	dec dx
	jnz loop
      }

  #else

  asm("
	pushw %%es
	movw %0,%%es
	movl $0xB8000,%%edi
	addl %1,%%edi
	movb %2,%%ah
	movb %3,%%al
	movl %6,%%ebx
	subl %4,%%ebx
	shll $1,%%ebx
	movl %5,%%edx
0:
	movl %4,%%ecx
    rep;stosw
	addl %%ebx,%%edi
	decl %%edx
	jnz  0b
	popw %%es
      "
      :
      :"m" (_dos_ds),        // 0
       "m" (start_offset),   // 1
       "m" (attribute),      // 2
       "m" (character),      // 3
       "m" (width),          // 4
       "m" (height),         // 5
       "m" (s_screen_width)    // 6
      :"%ax","%ebx","%ecx","%edx","%edi");

  #endif

  ShowMouse();
}

/****************************************************************************/
/* PartOfBufferToScreen                                                     */
/*--------------------------------------------------------------------------*/
/* Affiche une partie d'un buffer contenant du texte  l'cran              */
/*--------------------------------------------------------------------------*/
/* x,y             : Coordonnes d'affichage du morceau de buffer (1..)     */
/* buffer          : Tableau de word (attribut+caractre)                   */
/* x1_part,y1_part : Coordonnes dans le buffer, du morceau  afficher (0..)*/
/* part_width,                                                              */
/* part_height     : Dimensions du morceau  afficher                       */
/* buffer_width    : Largeur du buffer                                      */
/****************************************************************************/

void PartOfBufferToScreen(int x,int y,
			 word *buffer,
			 int x1_part,int y1_part,
			 int part_width,int part_height,
			 int buffer_width)

{
  word *buffer_ptr;
  int  screen_offset;

  if ((x>s_screen_width) || (y>s_screen_height))
    return;

  if (x+part_width>(s_screen_width+1))
    part_width=s_screen_width+1-x;

  if (y+part_height>(s_screen_height+1))
    part_height=s_screen_height+1-y;

  if ((part_width<=0) || (part_height<=0))
    return;

  buffer_ptr=buffer+(y1_part*buffer_width)+x1_part;
  screen_offset=(((y-1)*s_screen_width)+(x-1))<<1;

  HideMouse();

  #ifdef __TCPLUSPLUS__

  asm {
	mov    ax,0xB800
	mov    es,ax
	mov    di,screen_offset
	mov    ax,buffer_width
	sub    ax,part_width
	shl    ax,1
	mov    bx,s_screen_width
	sub    bx,part_width
	shl    bx,1
	mov    dx,part_height

	push   ds
	lds    si,buffer_ptr

      }

  loop:

  asm {
	mov    cx,part_width
    rep movsw
	add    si,ax
	add    di,bx
	dec    dx
	jnz    loop
	pop    ds
      }

  #else

  asm("
	movl   %3,%%eax
	subl   %4,%%eax
	shll   $1,%%eax
	movl   %5,%%ebx
	subl   %4,%%ebx
	shll   $1,%%ebx
	movl   %6,%%edx

	pushw  %%es
	movl   %0,%%esi
	movw   %1,%%es
	movl   $0xB8000,%%edi
	addl   %2,%%edi

  0:
	movl   %4,%%ecx
    rep;movsw
	addl   %%eax,%%esi
	addl   %%ebx,%%edi
	decl   %%edx
	jnz    0b
	popw   %%es
      "
      :
      :"m" (buffer_ptr),        // 0
       "m" (_dos_ds),           // 1
       "m" (screen_offset),     // 2
       "m" (buffer_width),      // 3
       "m" (part_width),        // 4
       "m" (s_screen_width),      // 5
       "m" (part_height)        // 6
      :"%esi","%edi","%eax","%ebx","%ecx","%edx");

  #endif

  ShowMouse();
}

/*************************************************/
/* HorizontalShadow : Fonce le morceau de la     */
/* ----------------   ligne horizontale dsigne */
/*************************************************/

void HorizontalShadow(int x1,int x2,int y)
{
  int width;
  int start_offset;

  if (   (y<1)
      || (y>s_screen_height)
      || (x2<x1)
      || (x1>s_screen_width))
    return;

  if (x2>s_screen_width)
    x2=s_screen_width;

  width=x2-x1+1;
  start_offset=((s_screen_width*(y-1))+(x1-1))<<1;

  HideMouse();

  #ifdef __TCPLUSPLUS__

  asm {
	push es
	mov  ax,0xB800
	mov  es,ax
	mov  di,start_offset

	mov  dx,width

      }

  loop:

  asm {
	mov  ax,es:[di]

	mov  bx,ax         // Couleur de fond claire
	and  bx,0x8000
	jz   fondfonce
	and  ax,0x77FF     // -> couleur de fond et de texte assombries
	jmp  suite
      }

  fondfonce:

  asm {                    // Couleur de fond claire
	and  ax,0x07FF     // -> couleur de fond noire et texte assombri
      }

  suite:

  asm {
	mov  bx,ax         // bx=(caractre & 0xF000) >>4 =couleur fond
	and  bx,0xF000
	shr  bx,4

	mov  cx,ax
	and  cx,0x0F00     // cx=(caractre & 0x0F00)=couleur texte

	cmp  bx,cx         // Si couleur fond=couleur texte,
	jnz  writechar

	mov  cx,ax         // la couleur de texte devient la complmentaire
	not  ax            // de celle de fond et le texte est remplac par
	shr  ax,4          // un espace (pour le curseur graphique souris
	and  ax,0x0F00     // soit visible)

	and  cx,0xF000
	or   ax,cx
	or   ax,0x020

      }
  writechar:

  asm {
	mov  es:[di],ax
	inc  di
	inc  di
	dec  dx
	jnz  loop
	pop  es
      }

  #else

  asm ("
	pushw %%es
	movw  %0,%%es       # _dos_ds

	movl  $0xB8000,%%edi
	addl  %1,%%edi      # start_offset

	movl  %2,%%edx      # width

  0:

	movw  %%es:(%%edi),%%ax

	movw  %%ax,%%bx     # Couleur de fond claire
	andw  $0x8000,%%bx
	jz    1f
	andw  $0x77FF,%%ax  # -> couleur de fond et de texte assombries
	jmp   2f

  1:

	andw  $0x07FF,%%ax  # -> couleur de fond noire et texte assombri

  2:

	movw  %%ax,%%bx     # bx=(caractre & 0xF000) >>4 =couleur fond
	andw  $0xF000,%%bx
	shrw  $4,%%bx

	movw  %%ax,%%cx
	andw  $0x0F00,%%cx  # cx=(caractre & 0x0F00)=couleur texte

	cmpw  %%bx,%%cx     # Si couleur fond=couleur texte,
	jnz   3f

	movw  %%ax,%%cx     # la couleur de texte devient la complmentaire
	notw  %%ax          # de celle de fond et le texte est remplac par
	shrw  $4,%%ax       # un espace (pour le curseur graphique souris
	andw  $0x0F00,%%ax  # soit visible)
	andw  $0xF000,%%cx
	orw   %%cx,%%ax
	orw   $0x020,%%ax

  3:

	movw  %%ax,%%es:(%%edi)
	incl  %%edi
	incl  %%edi
	decw  %%dx
	jnz   0b
	popw  %%es
      "
      :
      : "m" (_dos_ds),      // 0
	"m" (start_offset), // 1
	"m" (width)         // 2
      : "%edi","%eax","%ebx","%ecx","%edx");

  #endif

  ShowMouse();
}

/*********************************************/
/* VerticalShadow : Fonce le morceau de la   */
/* --------------   ligne verticale dsigne */
/*********************************************/

void VerticalShadow(int x,int y1,int y2)
{
  int height,start_offset;
  int increment;

  if (   (x<1)
      || (x>s_screen_width)
      || (y2<y1)
      || (y1>s_screen_height))
    return;

  if (y2>s_screen_height)
    y2=s_screen_height;

  height=y2-y1+1;
  start_offset=((s_screen_width*(y1-1))+(x-1))<<1;
  increment=s_screen_width<<1;

  HideMouse();

  #ifdef __TCPLUSPLUS__

  asm {
	push es
	mov  ax,0xB800
	mov  es,ax
	mov  di,start_offset

	mov  dx,height

      }

  loop:

  asm {
	mov  ax,es:[di]

	mov  bx,ax         // Couleur de fond claire
	and  bx,0x8000
	jz   fondfonce
	and  ax,0x77FF     // -> couleur de fond et de texte assombries
	jmp  suite
      }

  fondfonce:

  asm {                    // Couleur de fond claire
	and  ax,0x07FF     // -> couleur de fond noire et texte assombri
      }

  suite:

  asm {
	mov  bx,ax         // bx=(caractre & 0xF000) >>4 =couleur fond
	and  bx,0xF000
	shr  bx,4

	mov  cx,ax
	and  cx,0x0F00     // cx=(caractre & 0x0F00)=couleur texte

	cmp  bx,cx         // Si couleur fond=couleur texte,
	jnz  writechar

	mov  cx,ax         // la couleur de texte devient la complmentaire
	not  ax            // de celle de fond et le texte est remplac par
	shr  ax,4          // un espace (pour le curseur graphique souris
	and  ax,0x0F00     // soit visible)

	and  cx,0xF000
	or   ax,cx
	or   ax,0x020

      }
  writechar:

  asm {
	mov  es:[di],ax
	add  di,increment
	dec  dx
	jnz  loop
	pop  es
      }


  #else

  asm ("
	pushw %%es
	movw  %0,%%es       # _dos_ds

	movl  $0xB8000,%%edi
	addl  %1,%%edi      # start_offset

	movl  %2,%%edx      # height

  0:

	movw  %%es:(%%edi),%%ax

	movw  %%ax,%%bx     # Couleur de fond claire
	andw  $0x8000,%%bx
	jz    1f
	andw  $0x77FF,%%ax  # -> couleur de fond et de texte assombries
	jmp   2f

  1:

	andw  $0x07FF,%%ax  # -> couleur de fond noire et texte assombri

  2:

	movw  %%ax,%%bx     # bx=(caractre & 0xF000) >>4 =couleur fond
	andw  $0xF000,%%bx
	shrw  $4,%%bx

	movw  %%ax,%%cx
	andw  $0x0F00,%%cx  # cx=(caractre & 0x0F00)=couleur texte

	cmpw  %%bx,%%cx     # Si couleur fond=couleur texte,
	jnz   3f

	movw  %%ax,%%cx     # la couleur de texte devient la complmentaire
	notw  %%ax          # de celle de fond et le texte est remplac par
	shrw  $4,%%ax       # un espace (pour le curseur graphique souris
	andw  $0x0F00,%%ax  # soit visible)
	andw  $0xF000,%%cx
	orw   %%cx,%%ax
	orw   $0x020,%%ax

  3:

	movw  %%ax,%%es:(%%edi)
	addl  %3,%%edi
	decw  %%dx
	jnz   0b
	popw  %%es
      "
      :
      : "m" (_dos_ds),       // 0
	"m" (start_offset),  // 1
	"m" (height),        // 2
	"m" (increment)      // 3
      : "%edi","%eax","%ebx","%ecx","%edx");

  #endif

  ShowMouse();
}

