/*
 * Copyright (C) 1996, jack
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

static char rcsid[]= "$Id: genmap.c,v 1.3 1996/10/25 13:56:18 jack Exp $";

#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define TABLE_SIZE (1<<16)
#define ERROR_SJIS (0xffffU)
#define ERROR_JIS (ERROR_SJIS)
#define ERROR_UNICODE (0xffffU)

/* mapping tables */
int hjis_table[TABLE_SIZE];
int sjis_table[TABLE_SIZE];
int jis_table[TABLE_SIZE];
int unicode_table[TABLE_SIZE];
int uni0208_table[94][94];
int uni0212_table[94][94];

/* my name */
char *progname = NULL;



/* safety functions */

void fatal (char *format, ...);
void *xmalloc (size_t size);
void *xrealloc (void *p, size_t size);
FILE *xfopen (char *fname, char *mode);
char *fgets_line (FILE *fp);

void
fatal (char *format, ...)
{
  va_list ap;

  va_start (ap, format);
  fprintf (stderr, "%s: ", progname);
  vfprintf (stderr, format, ap);
  va_end (ap);
  fprintf (stderr, "\n");
  exit (EXIT_FAILURE);
}

void *
xmalloc (size_t size)
{
  void *p;

  p = malloc (size);
  if (p == NULL)
    fatal ("Memory allocate error");

  return p;
}

void *
xrealloc (void *p, size_t size)
{
  p = realloc (p, size);
  if (p == NULL)
    fatal ("Memory allocate error");

  return p;
}

FILE *
xfopen (char *fname, char *mode)
{
  FILE *fp;

  fp = fopen (fname, mode);
  if (fp == NULL)
    fatal ("Cannot open %s", fname);

  return fp;
}

#define TMPBUF_SIZE 256

char *
fgets_line (FILE *fp)
{
  char *dst, *p;
  int len, n;

  dst = p = xmalloc (TMPBUF_SIZE);
  dst[0] = '\0';
  len = 0;

  while (fgets (p, TMPBUF_SIZE, fp) != NULL)
    {
      n = strlen (p);
      len += n;
      if (n + 1 < TMPBUF_SIZE || dst[len - 1] == '\n')
	return xrealloc (dst, len + 1);
      else
	dst = xrealloc (dst, len + TMPBUF_SIZE);
      p = dst + len;
    }

  if (feof (fp) && len != 0)
    {
      dst = xrealloc (dst, len + 2);
      dst[len] = '\n';
      dst[len + 1] = '\0';
      return dst;
    }

  free (dst);
  return NULL;
}


void
fill_table (void)
{
  int i, j;

  for (i = 0; i < TABLE_SIZE; i++)
    {
      hjis_table[i] = ERROR_SJIS;
      sjis_table[i] = ERROR_SJIS;
      jis_table[i] = ERROR_JIS;
      unicode_table[i] = ERROR_UNICODE;
    }

  for (i = 0; i < 94; i++)
    for (j = 0; j < 94; j++)
      {
	uni0208_table[i][j] = ERROR_UNICODE;
	uni0212_table[i][j] = ERROR_UNICODE;
      }
}

void
dump_table (void)
{
#if defined (SEPARATE_JIS)
  FILE *fp_hjis, *fp_sjis, *fp_jis;
  char *hjis_mapname = "hjismap.c";
  char *sjis_mapname = "sjismap.c";
  char *jis_mapname = "jismap.c";
#else
  FILE *fp_mixjis;
  char *mixjis_mapname = "mjismap.c";
#endif
  FILE *fp_unicode;
  char *unicode_mapname = "unimap.c";
  FILE *fp_unisepa;
  int j;
  char *unisepa_mapname = "unisepa.c";
  char *note;
  int i;

  printf ("Dumping\n");

#if defined (SEPARATE_JIS)
  fp_hjis = xfopen (hjis_mapname, "wt");
  fp_sjis = xfopen (sjis_mapname, "wt");
  fp_jis = xfopen (jis_mapname, "wt");
#else
  fp_mixjis = xfopen (mixjis_mapname, "wt");
#endif
  fp_unicode = xfopen (unicode_mapname, "wt");
  fp_unisepa = xfopen (unisepa_mapname, "wt");

  note = "/* Generated automatically by 'genmap'. */\n\n";
#if defined (SEPARATE_JIS)
  fputs (note, fp_hjis);
  fputs ("unsigned short __unicode_to_hjis_mapping[] =\n{\n", fp_hjis);
  fputs (note, fp_sjis);
  fputs ("unsigned short __unicode_to_sjis_mapping[] =\n{\n", fp_sjis);
  fputs (note, fp_jis);
  fputs ("unsigned short __unicode_to_jis_mapping[] =\n{\n", fp_jis);
#else
  fputs (note, fp_mixjis);
  fputs ("unsigned short __unicode_to_mixjis_mapping[] =\n{\n", fp_mixjis);
#endif
  fputs (note, fp_unicode);
  fputs ("unsigned short __mixjis_to_unicode_mapping[] =\n{\n", fp_unicode);

#if 1
  fputs (note, fp_unisepa);

  fputs ("unsigned short __jis0201_to_unicode_mapping[] =\n{\n", fp_unisepa);
  for (i = 0; i < 256; i++)
    {
      if ((i % 8) == 0)
	fputs ("  ", fp_unisepa);
      fprintf (fp_unisepa, "0x%4.4x,", unicode_table[i]);
      if ((i % 8) == 7)
	fputc ('\n', fp_unisepa);
      else
	fputc (' ', fp_unisepa);
    }
  fputs ("};\n\n", fp_unisepa);

  fputs ("unsigned short __jis0208_to_unicode_mapping[] =\n{\n", fp_unisepa);
  for (i = 0; i < 94; i++)
    for (j = 0; j < 94; j++)
      {
	if ((j % 8) == 0)
	  fputs ("  ", fp_unisepa);
	fprintf (fp_unisepa, "0x%4.4x,", uni0208_table[i][j]);
	if ((j % 8) == 7 || j == 93)
	  fputc ('\n', fp_unisepa);
	else
	  fputc (' ', fp_unisepa);
      }
  fputs ("};\n\n", fp_unisepa);

  fputs ("unsigned short __jis0212_to_unicode_mapping[] =\n{\n", fp_unisepa);
  for (i = 0; i < 94; i++)
    for (j = 0; j < 94; j++)
      {
	if ((j % 8) == 0)
	  fputs ("  ", fp_unisepa);
	fprintf (fp_unisepa, "0x%4.4x,", uni0212_table[i][j]);
	if ((j % 8) == 7 || j == 93)
	  fputc ('\n', fp_unisepa);
	else
	  fputc (' ', fp_unisepa);
      }
  fputs ("};\n", fp_unisepa);

  fclose (fp_unisepa);
#endif

  for (i = 0; i < TABLE_SIZE; i++)
    {
      printf ("%d\r", i);
      fflush (stdout);

      if ((i % 8) == 0)
	{
#if defined (SEPARATE_JIS)
	  if (i < 256)
	    fputs ("  ", fp_hjis);
	  fputs ("  ", fp_sjis);
	  fputs ("  ", fp_jis);
#else
	  fputs ("  ", fp_mixjis);
#endif
	  fputs ("  ", fp_unicode);
	}

#if defined (SEPARATE_JIS)
      if (i < 256)
	fprintf (fp_hjis, "0x%4.4x,", hjis_table[i]);
      fprintf (fp_sjis, "0x%4.4x,", sjis_table[i]);
      fprintf (fp_jis, "0x%4.4x,", jis_table[i]);
#else
      if ((i < 256) && (hjis_table[i] != ERROR_SJIS))
	{
	  if ((sjis_table[i] != ERROR_SJIS) || (jis_table[i] != ERROR_JIS))
	    printf ("Duplicated %x\n", i);
	  fprintf (fp_mixjis, "0x%4.4x,", hjis_table[i]);
	}
      else if (sjis_table[i] != ERROR_SJIS)
	{
	  if (jis_table[i] != ERROR_JIS)
	    printf ("Duplicated %x\n", i);
	  fprintf (fp_mixjis, "0x%4.4x,", sjis_table[i]);
	}
      else if (jis_table[i] != ERROR_JIS)
	{
	  fprintf (fp_mixjis, "0x%4.4x,", jis_table[i]);
	}
      else
	{
	  fprintf (fp_mixjis, "0x%4.4x,", ERROR_SJIS);
	}
#endif
      fprintf (fp_unicode, "0x%4.4x,", unicode_table[i]);

      if ((i % 8) == 7)
	{
#if defined (SEPARATE_JIS)
	  if (i < 256)
	    fputc ('\n', fp_hjis);
	  fputc ('\n', fp_sjis);
	  fputc ('\n', fp_jis);
#else
	  fputc ('\n', fp_mixjis);
#endif
	  fputc ('\n', fp_unicode);
	}
      else
	{
#if defined (SEPARATE_JIS)
	  if (i < 256)
	    fputc (' ', fp_hjis);
	  fputc (' ', fp_sjis);
	  fputc (' ', fp_jis);
#else
	  fputc (' ', fp_mixjis);
#endif
	  fputc (' ', fp_unicode);
	}
    }
  printf ("\n");

#if defined (SEPARATE_JIS)
  fputs ("};\n", fp_hjis);
  fputs ("};\n", fp_sjis);
  fputs ("};\n", fp_jis);
#else
  fputs ("};\n", fp_mixjis);
#endif
  fputs ("};\n", fp_unicode);

  fclose (fp_unicode);
#if defined (SEPARATE_JIS)
  fclose (fp_hjis);
  fclose (fp_sjis);
  fclose (fp_jis);
#else
  fclose (fp_mixjis);
#endif
}

void
set_c0 (void)
{
  int i;

  /* 00-1f: control code */
  for (i = 0; i < 0x20; i++)
    {
      hjis_table[i] = i;
      unicode_table[i] = i;
    }
  /* 7f: delete */
  hjis_table[0x7f] = 0x7f;
  unicode_table[0x7f] = 0x7f;
}

void
scan_jis0201 (char *source)
{
  FILE *fp;
  char *lp;
  int lines = 0;
  char jis_string[11];
  char unicode_string[11];
  int jis_code, unicode;

  printf ("Scanning %s\n", source);

  fp = xfopen (source, "rt");
  while ((lp = fgets_line (fp)) != NULL)
    {
      printf ("%d\r", lines);
      fflush (stdout);
      if (sscanf (lp, "%10[^#\t]\t%10[^#\t]%*[^\n]\n", jis_string, unicode_string) == 2)
	{
	  jis_code = (int) strtol (jis_string, NULL, 0);
	  unicode = (int) strtol (unicode_string, NULL, 0);

	  hjis_table[unicode] = jis_code;
	  unicode_table[jis_code] = unicode;
	}

      free (lp);
      lines++;
    }
  printf ("\n");
  fclose (fp);
}

void
scan_jis0208 (char *source)
{
  FILE *fp;
  char *lp;
  int lines = 0;
  char sjis_string[11];
  char jis_string[11];
  char unicode_string[11];
  int sjis_code, jis_code, unicode;

  printf ("Scanning %s\n", source);

  fp = xfopen (source, "rt");
  while ((lp = fgets_line (fp)) != NULL)
    {
      printf ("%d\r", lines);
      fflush (stdout);
      if (sscanf (lp, "%10[^#\t]\t%10[^#\t]\t%10[^#\t]%*[^\n]\n", sjis_string, jis_string,
		  unicode_string) == 3)
	{
	  sjis_code = (int) strtol (sjis_string, NULL, 0);
	  jis_code = (int) strtol (jis_string, NULL, 0);
	  unicode = (int) strtol (unicode_string, NULL, 0);

	  sjis_table[unicode] = sjis_code;
	  unicode_table[sjis_code] = unicode;
	  uni0208_table[((jis_code >> 8) & 0xff) - 0x21][(jis_code & 0xff) - 0x21] = unicode;
	}

      free (lp);
      lines++;
    }
  printf ("\n");
  fclose (fp);
}

void
scan_jis0212 (char *source)
{
  FILE *fp;
  char *lp;
  int lines = 0;
  char jis_string[11];
  char unicode_string[11];
  int jis_code, unicode;

  printf ("Scanning %s\n", source);

  fp = xfopen (source, "rt");
  while ((lp = fgets_line (fp)) != NULL)
    {
      printf ("%d\r", lines);
      fflush (stdout);
      if (sscanf (lp, "%10[^#\t]\t%10[^#\t]%*[^\n]\n", jis_string, unicode_string) == 2)
	{
	  jis_code = (int) strtol (jis_string, NULL, 0);
	  unicode = (int) strtol (unicode_string, NULL, 0);

	  jis_table[unicode] = jis_code;
	  unicode_table[jis_code] = unicode;
	  uni0212_table[((jis_code >> 8) & 0xff) - 0x21][(jis_code & 0xff) - 0x21] = unicode;
	}

      free (lp);
      lines++;
    }
  printf ("\n");
  fclose (fp);
}

int
main (int argc, char *argv[])
{
  char *jis0201_source = "JIS0201";
  char *jis0208_source = "JIS0208";
  char *jis0212_source = "JIS0212";

  progname = argv[0];
  if (argc >= 2)
    jis0201_source = argv[1];
  if (argc >= 3)
    jis0208_source = argv[2];
  if (argc >= 4)
    jis0212_source = argv[3];

  fill_table ();
  set_c0 ();
  scan_jis0201 (jis0201_source);
  scan_jis0208 (jis0208_source);
  scan_jis0212 (jis0212_source);
  dump_table ();

  return 0;
}
