/* SAMEFILE.C - function to test whether two files are actually the same file
 ******************************************************************************
 *
 *	int same_dos_file(char *src, struct stat *src_sb, char *dst,
 *							  struct stat *dst_sb)
 *
 ******************************************************************************
 * edit history is at the end of the file
 ******************************************************************************
 * Copyright 1995 by the Summer Institute of Linguistics, Inc.
 *
 * This file is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; see the file COPYING.LIB.  If
 * not, write to the Free Software Foundation, Inc., 675 Mass Ave,
 * Cambridge, MA 02139, USA.
 */
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>

extern char *_fullpath(char *buffer, const char *pathname, size_t maxlen);

/*****************************************************************************
 * NAME
 *    same_dos_file
 * ARGUMENTS
 *    src    - source file pathname
 *    src_sb - pointer to source file stat buffer
 *    dst    - destination file pathname
 *    dst_sb - pointer to destination file stat buffer
 * DESCRIPTION
 *    Test whether src and dst are actually the same file.
 * RETURN VALUE
 *    0 if files different, 1 if files the same, 2 if files probably the same
 */
int same_dos_file(char *src, struct stat *src_sb, char *dst,
							  struct stat *dst_sb)
{
char *sfile, *dfile, *p;
/*
 *  try the trivial tests first
 */
if ((src == dst) || (strcmp(src, dst) == 0))
    return 1;		/* signal same file for sure */
/*
 *  force full pathnames and compare those
 */
sfile = _fullpath((char *)NULL, src, 0);
if (sfile != (char *)NULL)
    {
    strlwr(sfile);
    dfile = _fullpath((char *)NULL, dst, 0);
    if (dfile != (char *)NULL)
	{
	strlwr(dfile);
	if (strcmp(sfile, dfile) == 0)
	    {
	    free(sfile);
	    free(dfile);
	    return 1;		/* signal same file for sure */
	    }
	free(dfile);
	}
    free(sfile);
    }
/*
 *  SUBST can fool _fullpath(), so compare basenames and valid stat fields
 */
sfile = src;
if (isalpha(sfile[0]) && (sfile[1] == ':'))
    sfile += 2;
p = strrchr(sfile, '/');
if (p != (char *)NULL)
    sfile = p + 1;
dfile = dst;
if (isalpha(dfile[0]) && (dfile[1] == ':'))
    dfile += 2;
p = strrchr(dfile, '/');
if (p != (char *)NULL)
    dfile = p + 1;
if (	(strcmp(sfile, dfile) == 0) &&
	(src_sb->st_mode  == dst_sb->st_mode) &&
	(src_sb->st_size  == dst_sb->st_size) &&
	(src_sb->st_atime == dst_sb->st_atime) &&
	(src_sb->st_ctime == dst_sb->st_ctime) &&
	(src_sb->st_mtime == dst_sb->st_mtime) )
    return 2;			/* signal probably the same file */

return 0;		/* signal that the two files are different */
}

/******************************************************************************
 * EDIT HISTORY
 ******************************************************************************
 * 16-Mar-95	SRMc - write same_dos_file() for use by mv.c and cp.c
 */
