/* File-name wildcard pattern matching for GNU. Part 1/2
   Copyright (C) 1985, 1988, 1989, 1990 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 1, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

/* To whomever it may concern: I have never seen the code which most
   Unix programs use to perform this function.  I wrote this from scratch
   based on specifications for the pattern matching.  --RMS.  */

#include <sys/types.h>

#if	defined(STDC_HEADERS) || defined(__GNU_LIBRARY__)
#include <stdlib.h>
#include <string.h>
#define bcopy(s, d, n) memcpy ((d), (s), (n))
#define index strchr
#define rindex strrchr
#else

#if defined(USG) || defined(MSDOS)
#include <string.h>
#include <memory.h>
#define bcopy(s, d, n) memcpy ((d), (s), (n))
#define index strchr
#define rindex strrchr
#else /* not USG */
#include <strings.h>
extern void bcopy ();
#endif /* not USG */

#ifdef IGNCASE
#include <ctype.h>
#endif

#ifdef MSDOS
#include <malloc.h>
#else
extern char *malloc ();
extern char *realloc ();
extern void free ();
#endif

#ifndef NULL
#define NULL 0
#endif
#endif	/* Not STDC_HEADERS or __GNU_LIBRARY__.  */

#ifdef __GNUC__
#define alloca __builtin_alloca
#else /* Not GCC.  */
#ifdef sparc
#include <alloca.h>
#else /* Not sparc.  */
#ifndef MSDOS
extern char *alloca ();
#endif
#endif /* sparc.  */
#endif /* GCC.  */

/* Nonzero if '*' and '?' do not match an initial '.' for glob_filename.  */
int noglob_dot_filenames = 1;

static int glob_match_after_star (char *pattern, char *text);
int glob_pattern_p (char *pattern);
int glob_match (char *pattern, char *text, int dot_special);

/* Return nonzero if PATTERN has any special globbing chars in it.  */

int glob_pattern_p (pattern)
     char *pattern;
{
  register char *p = pattern;
  register char c;

  while ((c = *p++) != '\0')
    switch (c)
      {
      case '?':
      case '[':
      case '*':
	return 1;

      case '\\':
	if (*p++ == '\0')
	  return 0;
      }

  return 0;
}


/* Match the pattern PATTERN against the string TEXT;
   return 1 if it matches, 0 otherwise.

   A match means the entire string TEXT is used up in matching.

   In the pattern string, `*' matches any sequence of characters,
   `?' matches any character, [SET] matches any character in the specified set,
   [!SET] matches any character not in the specified set.

   A set is composed of characters or ranges; a range looks like
   character hyphen character (as in 0-9 or A-Z).
   [0-9a-zA-Z_] is the set of characters allowed in C identifiers.
   Any other character in the pattern must be matched exactly.

   To suppress the special syntactic significance of any of `[]*?!-\',
   and match the character exactly, precede it with a `\'.

   If DOT_SPECIAL is nonzero,
   `*' and `?' do not match `.' at the beginning of TEXT.  */

int glob_match (pattern, text, dot_special)
     char *pattern, *text;
     int dot_special;
{
  register char *p = pattern, *t = text;
  register char c, ct;

  while ((c = *p++) != '\0')
    switch (c)
      {
      case '?':
	if (*t == '\0' || (dot_special && t == text && *t == '.'))
	  return 0;
	else
	  ++t;
	break;

      case '\\':
	if (*p++ != *t++)
	  return 0;
	break;

      case '*':
	if (dot_special && t == text && *t == '.')
	  return 0;
	return glob_match_after_star (p, t);

      case '[':
	{
	  register char c1 = *t++;
	  int invert;

	  if (c1 == '\0')
	    return 0;

	  invert = (*p == '!');

	  if (invert)
	    p++;

	  c = *p++;
	  while (1)
	    {
	      register char cstart = c, cend = c;

	      if (c == '\\')
		{
		  cstart = *p++;
		  cend = cstart;
		}

	      if (cstart == '\0')
		return 0;	/* Missing ']'. */

	      c = *p++;

	      if (c == '-')
		{
		  cend = *p++;
		  if (cend == '\\')
		    cend = *p++;
		  if (cend == '\0')
		    return 0;
		  c = *p++;
		}
	      if (c1 >= cstart && c1 <= cend)
		goto match;
	      if (c == ']')
		break;
	    }
	  if (!invert)
	    return 0;
	  break;

	match:
	  /* Skip the rest of the [...] construct that already matched.  */
	  while (c != ']')
	    {
	      if (c == '\0')
		return 0;
	      c = *p++;
	      if (c == '\0')
		return 0;
	      if (c == '\\')
		p++;
	    }
	  if (invert)
	    return 0;
	  break;
	}

      default:
#ifdef IGNCASE
        ct = *t++;
	if (tolower(c) != tolower(ct))
#else
	if (c != *t++)
#endif
	  return 0;
      }

  return *t == '\0';
}

/* Like glob_match, but match PATTERN against any final segment of TEXT.  */

static int glob_match_after_star (pattern, text)
     char *pattern, *text;
{
  register char *p = pattern, *t = text;
  register char c, c1;

  while ((c = *p++) == '?' || c == '*')
    if (c == '?' && *t++ == '\0')
      return 0;

  if (c == '\0')
    return 1;

  if (c == '\\')
    c1 = *p;
  else
    c1 = c;

  --p;
  while (1)
    {
      if ((c == '[' || *t == c1) && glob_match (p, t, 0))
	return 1;
      if (*t++ == '\0')
	return 0;
    }
}
