-----------------------------------------------------------------------
--
--  File:        conio.ads
--  Description: DJGPP console I/O
--  Rev:         0.2
--  Date:        03/01/97
--  Author:      Jerry van Dijk
--  Mail:        jerry@jvdsys.nextjk.stuyts.nl
--
--  Copyright (c) Jerry van Dijk, 1996, 1997
--  Forelstraat 211
--  2037 KV  HAARLEM
--  THE NETHERLANDS
--  tel int + 31 23 540 1052
--
--  Permission granted to use for any purpose, provided this copyright
--  remains attached and unmodified.
--
--  THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
--  IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
--  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
--
-----------------------------------------------------------------------

--  ************************************************************
--  **                                                        **
--  **                   PORTABILITY WARNING                  **
--  **                   -------------------                  **
--  **                                                        **
--  **  The console package is DJGPP specific and is not      **
--  **  portable to other enviroments. Use at your own risk!  **
--  **                                                        **
--  ************************************************************

---------------------------------------------------------------------
--                     PACKAGE CONTAINS                            --
---------------------------------------------------------------------
--                                                                 --
-- Types:                                                          --
--                                                                 --
--    Background_Color - Defines background text colors            --
--    Cursor_Size      - Defines possible cursor sizes             --
--    Display_Cell     - Defines character + attribute on screen   --
--    Foreground_Color - Defines foreground text colors            --
--    Num_Lines        - Defines possible line sizes               --
--    Screen_Buffer    - Hold an screen area                       --
--    Text_Attribute   - Defines a text attribute                  --
--    Text_Info        - Specifies text mode info                  --
--    Text_Mode        - Defines possible text modes               --
--    X_Pos            - Specifies a horizontal location           --
--    Y_Pos            - Specifies a vertical position             --
--                                                                 --
-- Functions:                                                      --
--                                                                 --
--    Getch            - Read an single character from the console --
--    Getche           - Read an single character from the console --
--    Kbit             - Determines if character is waiting        --
--    Puttext          - Writes a screen buffer on the screen      --
--    Wherex           - Returns horizontal position of cursor     --
--    Wherey           - Returns vertical position of cursor       --
--                                                                 --
-- Procedures:                                                     --
--                                                                 --
--    Cgets            - Reads a string from the console           --
--    Clreol           - Clear to end of line                      --
--    Clrscr           - Clear the screen                          --
--    Conio_Init       - Re-initializes conio                      --
--    Cputs            - Writes a string on the console            --
--    Delline          - The line the cursor is on is deleted      --
--    Gettext          - Read screen area into a Screen_Buffer     --
--    Gettextinfo      - Fills a Text_Info record                  --
--    Gotoxy           - Move cursor to x, y                       --
--    Highvideo        - All subsequent writes are bright          --
--    Insline          - Insert blank line at cursor location      --
--    Lowvideo         - All subsequent writes are dim             --
--    Movetext         - Copy an area on screen to new position    --
--    Normvideo        - Reset the video attribute                 --
--    Putch            - Print character at cursor position        --
--    Setcursortype    - Set size of cursor                        --
--    Set_Screen_Lines - Sets number of screen lines               --
--    Textattr         - Set new text attribute                    --
--    Textbackground   - Set new background color                  --
--    Textcolor        - Set new foreground color                  --
--    Textmode         - Set a new video mode                      --
--    Ungetch          - Puts a character back in the stream       --
--    Window           - Set new output window coordinates         --
--    Wscroll          - Set window scrolling                      --
--                                                                 --
-- Unimplemented:                                                  --
--                                                                 --
--    Cscanf           - Type unsafe, variable number of arguments --
--    Printf           - Type unsafe, variable number of arguments --
--                                                                 --
---------------------------------------------------------------------

package Conio is

   ----------------------------------------------
   -- NAME:    X_Pos                           --
   -- PURPOSE: Specifies a horizontal position --
   ----------------------------------------------
   subtype X_Pos is Positive range 1 .. 80;

   --------------------------------------------
   -- NAME:    Y_Pos                         --
   -- PURPOSE: Specifies a vertical position --
   --------------------------------------------
   subtype Y_Pos is Positive range 1 .. 50;

   ---------------------------------------------
   -- NAME:    Background_Color               --
   -- PURPOSE: Defines background text colors --
   ---------------------------------------------
   type Background_Color is (Black, Blue, Green, Cyan, Red,
                             Magenta, Brown, Light_Gray);

   --------------------------------------------
   -- NAME:    Cursor_Size                   --
   -- PURPOSE: Defines possible cursor sizes --
   --------------------------------------------
   type Cursor_Size is (None, Solid, Normal);

   ---------------------------------------------
   -- NAME:    Foreground_Color               --
   -- PURPOSE: Defines foreground text colors --
   ---------------------------------------------
   type Foreground_Color is (Black, Blue, Green, Cyan, Red, Magenta, Brown,
                             Light_Gray, Dark_Gray, Light_Blue, Light_Green,
                             Light_Cyan, Light_Red, Light_Magenta, Yellow,
                             White);

   -------------------------------------------------
   -- NAME:    Num_Lines                          --
   -- PURPOSE: Defines possible screen line sizes --
   -------------------------------------------------
   type Num_Lines is (L25, L28, L35, L40, L43, L50);
   for Num_Lines use (L25 => 25, L28 => 28, L35 => 35,
                      L40 => 40, L43 => 43, L50 => 50);

   ---------------------------------------------
   -- NAME:    Text_Attribute                 --
   -- PURPOSE: Defines a video text attribute --
   ---------------------------------------------
   type Text_Attribute is
      record
         Blink      : Boolean;
         Background : Background_Color;
         Foreground : Foreground_Color;
      end record;

   for Text_Attribute use
      record
         Blink      at 0 range 7 .. 7;
         Background at 0 range 4 .. 6;
         Foreground at 0 range 0 .. 3;
      end record;

   for Text_Attribute'Size use 8;

   --------------------------------------------------------------
   -- NAME:    Display_Cell                                    --
   -- PURPOSE: Defines a character and attribute on the screen --
   --------------------------------------------------------------
   type Display_Cell is
      record
         Attr  : Text_Attribute;
         Value : Character;
      end record;

   for Display_Cell use
      record
         Attr  at 1 range 0 .. 7;
         Value at 0 range 0 .. 7;
      end record;

   for Display_Cell'Size use 16;

   ----------------------------------
   -- NAME:    Screen_Buffer       --
   -- PURPOSE: Holds a screen area --
   ----------------------------------
   type Screen_Buffer is
     array (X_Pos range <>, Y_Pos range <>) of Display_Cell;

   ---------------------------------------
   -- NAME:    Text_Info                --
   -- PURPOSE: Specifies text mode info --
   ---------------------------------------
   type Text_Info is
      record
         Curr_X        : X_Pos;
         Curr_Y        : Y_Pos;
         Window_Top    : Y_Pos;
         Window_Left   : X_Pos;
         Window_Right  : X_Pos;
         Screen_Width  : X_Pos;
         Screen_Height : Y_Pos;
         Window_Bottom : Y_Pos;
         Curr_Attr     : Text_Attribute;
         Norm_Attr     : Text_Attribute;
      end record;

   ---------------------------------------
   -- NAME:    Text_Mode                --
   -- PURPOSE: Defines video text modes --
   ---------------------------------------
   type Text_Mode is (Last_Mode, BW40, C40, BW80, C80, Mono, C4350);
   for Text_Mode use (Last_Mode => -1, BW40 => 0, C40 => 1,
                      BW80 => 2,       C80 => 3,  Mono => 7,
                      C4350 => 64);

   ------------------------------------------------------------
   -- NAME:    Cgets                                         --
   -- PURPOSE: Get a string from the console                 --
   -- OUTPUT:  String read, number of chars read             --
   -- NOTES:   1. Returns when S'Length characters are typed --
   --          2. String is right-padded with spaces         --
   ------------------------------------------------------------
   procedure Cgets (S : out String; N : out Natural);

   -----------------------------------
   -- NAME:    Clreol               --
   -- PURPOSE: Clear to end of line --
   -----------------------------------
   procedure Clreol;

   --------------------------------------
   -- NAME:    Clrscr                  --
   -- PURPOSE: Clear the entire screen --
   --------------------------------------
   procedure Clrscr;

   ---------------------------------------------------------
   -- NAME:    Cputs                                      --
   -- PURPOSE: Puts the string onto the console           --
   -- INPUT:   String to be printed                       --
   -- NOTES:   1. Returns immediatly if String'Length = 0 --
   --          2. The cursor position is updated          --
   ---------------------------------------------------------
   procedure Cputs (S : in String);

   -------------------------------------
   -- NAME:    Conio_Init             --
   -- PURPOSE: re-init conio          --
   -- NOTES:   call after mode change --
   -------------------------------------
   procedure Conio_Init;

   ---------------------------------------------------
   -- NAME:    Delline                              --
   -- PURPOSE: The line the cursor is on is deleted --
   -- NOTES:   Lines below it scroll up             --
   ---------------------------------------------------
   procedure Delline;

   ----------------------------------------------------------
   -- NAME:    Getch                                       --
   -- PURPOSE: Read an single character from the console   --
   -- RETURNS: The character pressed                       --
   -- NOTES:   1. Read is un-buffered                      --
   --          2. Characters 'ungetted' are returned first --
   --          3. Character is not echoed to the screen    --
   --          4. If character is NUL then next call       --
   --             retrieves function key                   --
   ----------------------------------------------------------
   function Getch return Character;

   ----------------------------------------------------------
   -- NAME:    Getche                                      --
   -- PURPOSE: Read an single character from the console   --
   -- RETURNS: The character pressed                       --
   -- NOTES:   1. Read is un-buffered                      --
   --          2. Characters 'ungetted' are returned first --
   --          3. Character is echoed to the screen        --
   --          4. If character is NUL then next call       --
   --             retrieves function key                   --
   ----------------------------------------------------------
   function Getche return Character;

   -----------------------------------------------------
   -- NAME:    Gettext                                --
   -- PURPOSE: Read screen area into a Screen_Buffer  --
   -- INPUTS:  Location of screen area, screen buffer --
   -- RETURNS: Buffer filled if Result is True        --
   -----------------------------------------------------
   procedure Gettext (Left   : in     X_Pos; Top    : in Y_Pos;
                      Right  : in     X_Pos; Bottom : in Y_Pos;
                      Buffer :    out Screen_Buffer;
                      Result :    out Boolean);

   ---------------------------------------
   -- NAME:    Gettextinfo              --
   -- PURPOSE: Fills a Text_Info record --
   ---------------------------------------
   procedure Gettextinfo (Info : out Text_Info);

   ---------------------------------------------
   -- NAME:    Gotoxy                         --
   -- PURPOSE: Moves cursor to x, y position  --
   -- INPUTS:  x and y values                 --
   -- NOTES:   1. Home position is upper-left --
   --          2. Values are 1 based          --
   ---------------------------------------------
   procedure Gotoxy (X : in X_Pos; Y : in Y_Pos);

   -----------------------------------------------
   -- NAME:    Highvideo                        --
   -- PURPOSE: All subsequent writes are bright --
   -----------------------------------------------
   procedure Highvideo;

   ------------------------------------------------------
   -- NAME:    Insline                                 --
   -- PURPOSE: Insert blank line at cursor position    --
   -- NOTES:   The previous lines below it scroll down --
   ------------------------------------------------------
   procedure Insline;

   -----------------------------------------------------------------
   -- NAME:    Kbhit                                              --
   -- PURPOSE: Determines if character is waiting at the keyboard --
   -- RETURNS: True if character waiting                          --
   -- NOTES:   Also True if an ungetch character is waiting       --
   -----------------------------------------------------------------
   function Kbhit return Boolean;

   --------------------------------------------
   -- NAME:    Lowvideo                      --
   -- PURPOSE: All subsequent writes are dim --
   --------------------------------------------
   procedure Lowvideo;

   --------------------------------------------------------------
   -- NAME:    Movetext                                        --
   -- PURPOSE: Copy an area on the screen to a new location    --
   -- INPUTS:  Location of area to copy, top-left new location --
   -- RETURNS: False if call failed                            --
   --------------------------------------------------------------
   function Movetext (Left     : in X_Pos; Top     : in Y_Pos;
                      Right    : in X_Pos; Bottom  : in Y_Pos;
                      New_Left : in X_Pos; New_Top : in Y_Pos)
                      return Boolean;

   -----------------------------------------------
   -- NAME:    Normvideo                        --
   -- PURPOSE: Resets video attribute           --
   -- NOTES:   Resets to program start-up value --
   -----------------------------------------------
   procedure Normvideo;

   -------------------------------------------------------------
   -- NAME:    Putch                                          --
   -- PURPOSE: Print character at cursor position             --
   -- INPUT:   Character to print                             --
   -- NOTES:   1. Cursor position is updated                  --
   --          2. Handles: return, linefeed, bell, backspace, --
   --             line wrap and scrolling                     --
   -------------------------------------------------------------
   procedure Putch (C : in Character);

   ---------------------------------------------------
   -- NAME:    Puttext                              --
   -- PURPOSE: Writes a screen buffer on the screen --
   -- INPUTS:  Screen area, screen buffer           --
   -- RETURNS: True if write successfull            --
   ---------------------------------------------------
   function Puttext (Left   : in X_Pos; Top    : in Y_Pos;
                     Right  : in X_Pos; Bottom : in Y_Pos;
                     Buffer : in Screen_Buffer) return Boolean;

   ------------------------------
   -- NAME:    Setcursortype   --
   -- PURPOSE: Set cursor size --
   -- INPUTS:  Cursor size     --
   ------------------------------
   procedure Setcursortype (Size : in Cursor_Size);

   ---------------------------------------------------
   -- NAME:    Set_Screen_Lines                     --
   -- PURPOSE: Set the number of screen lines       --
   -- INPUT:   Number of screen lines               --
   -- NOTES:   1. CGA only supports 25 lines        --
   --          2. EGA only supports 25, 35 and 43   --
   --          3. Use gettextinfo to check linesize --
   --          4. All line size are 80 columns wide --
   --          5. Clears the screen!                --
   ---------------------------------------------------
   procedure Set_Screen_Lines (N : in Num_Lines);

   -------------------------------------
   -- NAME:    Textattr               --
   -- PURPOSE: Set new text attribute --
   -- INPUTS:  New attribute          --
   -------------------------------------
   procedure Textattr (Attr : in Text_Attribute);

   ---------------------------------------
   -- NAME:    Textbackground           --
   -- PURPOSE: Set new background color --
   -- INPUTS:  New color                --
   ---------------------------------------
   procedure Textbackground (Color : in Background_Color);

   ---------------------------------------
   -- NAME:    Textcolor                --
   -- PURPOSE: Set new foreground color --
   -- INPUTS:  New color                --
   ---------------------------------------
   procedure Textcolor (Color : in Foreground_Color);

   -----------------------------------------------------------
   -- NAME:    Textmode                                     --
   -- PURPOSE: Set a new text mode                          --
   -- INPUT:   New text mode                                --
   -- NOTES:   1. C4350 is 43 lines on EGA, 50 lines on VGA --
   --          2. LASTMODE reverts to previous mode         --
   -----------------------------------------------------------
   procedure Textmode (Mode : in Text_Mode);

   -------------------------------------------------------
   -- NAME:    Ungetch                                  --
   -- PURPOSE: Put a character back in the input stream --
   -- INPUT:   Character to put back                    --
   -- NOTES:   Can only push back ASCII characters      --
   -------------------------------------------------------
   procedure Ungetch (SC : in Character);

   ----------------------------------------------------
   -- NAME:    Wherex                                --
   -- PURPOSE: Returns horizontal position of cursor --
   -- RETURNS: Cursor position                       --
   -- NOTES:   1. Cursor position is 1 based         --
   --          2. Home position is upper-left        --
   ----------------------------------------------------
   function Wherex return X_Pos;

   --------------------------------------------------
   -- NAME:    Wherey                              --
   -- PURPOSE: Returns vertical position of cursor --
   -- RETURNS: Cursor position                     --
   -- NOTES:   1. Cursor position is 1 based       --
   --          2. Home position is upper-left      --
   --------------------------------------------------
   function Wherey return Y_Pos;

   ----------------------------------------------------
   -- NAME:    Window                                --
   -- PURPOSE: Set new output window                 --
   -- INPUTS:  New window coordinates                --
   -- NOTES:   1. Coordinates relative to Upper-Left --
   --          2. Coordinates are 1 based            --
   ----------------------------------------------------
   procedure Window (Left  : in X_Pos; Top    : in Y_Pos;
                     Right : in X_Pos; Bottom : in Y_Pos);

   ----------------------------------------
   -- NAME:    Wscroll                   --
   -- PURPOSE: Set window scrolling      --
   -- INPUT:   True if window can scroll --
   ----------------------------------------
   procedure Wscroll (Scroll : in Boolean);

private

   ----------------------------
   -- Import DJGPP functions --
   ----------------------------
   pragma Import (C, Getch, "getch");
   pragma Import (C, Wherex, "wherex");
   pragma Import (C, Wherey, "wherey");
   pragma Import (C, Window, "window");
   pragma Import (C, Clreol, "clreol");
   pragma Import (C, Clrscr, "clrscr");
   pragma Import (C, Getche, "getche");
   pragma Import (C, Gotoxy, "gotoxy");
   pragma Import (C, Delline, "delline");
   pragma Import (C, Insline, "insline");
   pragma Import (C, Ungetch, "ungetch");
   pragma Import (C, Lowvideo, "lowvideo");
   pragma Import (C, Highvideo, "highvideo");
   pragma Import (C, Normvideo, "normvideo");
   pragma Import (C, Conio_Init, "gppconio_init");

   --------------------------------
   -- Inline interface functions --
   --------------------------------
   pragma Inline (Cputs);
   pragma Inline (Cgets);
   pragma Inline (Kbhit);
   pragma Inline (Putch);
   pragma Inline (Wscroll);
   pragma Inline (Gettext);
   pragma Inline (Puttext);
   pragma Inline (Textmode);
   pragma Inline (Textattr);
   pragma Inline (Movetext);
   pragma Inline (Textcolor);
   pragma Inline (Gettextinfo);
   pragma Inline (Setcursortype);
   pragma Inline (Set_Screen_Lines);
   pragma Inline (Textbackground);

end Conio;
