-----------------------------------------------------------------------
--
--  File:        sounds.adb
--  Description: HANGMAN sounds
--  Rev:         0.2
--  Date:        03/01/97
--  Author:      Jerry van Dijk
--  Mail:        jerry@jvdsys.nextjk.stuyts.nl
--
--  Copyright (c) Jerry van Dijk, 1997
--  Forelstraat 211
--  2037 KV  HAARLEM
--  THE NETHERLANDS
--  tel int + 31 23 540 1052
--
--  Permission granted to use for any purpose, provided this copyright
--  remains attached and unmodified.
--
--  THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
--  IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
--  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
--
-----------------------------------------------------------------------
With Interfaces;
Use  Interfaces;

package body Sound is

   procedure Bios_Wait_Ticks (Number_Of_Ticks : in Integer) is
      type Dpmi_Regs is
         record
            Di     : Unsigned_16;
            Di_Hi  : Unsigned_16;
            Si     : Unsigned_16;
            Si_Hi  : Unsigned_16;
            Bp     : Unsigned_16;
            Bp_Hi  : Unsigned_16;
            Res    : Unsigned_16;
            Res_Hi : Unsigned_16;
            Bl     : Unsigned_8;
            Bh     : Unsigned_8;
            Bl_Hi  : Unsigned_8;
            Bh_Hi  : Unsigned_8;
            Dl     : Unsigned_8;
            Dh     : Unsigned_8;
            Dl_Hi  : Unsigned_8;
            Dh_Hi  : Unsigned_8;
            Cl     : Unsigned_8;
            Ch     : Unsigned_8;
            Cl_Hi  : Unsigned_8;
            Ch_Hi  : Unsigned_8;
            Al     : Unsigned_8;
            Ah     : Unsigned_8;
            Al_Hi  : Unsigned_8;
            Ah_Hi  : Unsigned_8;
            Flags  : Unsigned_16;
            Es     : Unsigned_16;
            Ds     : Unsigned_16;
            Fs     : Unsigned_16;
            Gs     : Unsigned_16;
            Ip     : Unsigned_16;
            Cs     : Unsigned_16;
            Sp     : Unsigned_16;
            Ss     : Unsigned_16;
         end record;
      pragma Convention(C, Dpmi_Regs);
      procedure Dpmi_Int(Vector : in Unsigned_16; Regs : in out Dpmi_Regs);
      pragma Import(C, Dpmi_Int, "__dpmi_int");
      Regs : Dpmi_Regs;
      Current_Tick, Wait_For_Tick : Integer;
      function Make_Integer(Hi_Hi : in Unsigned_8;
                            Hi_Lo : in Unsigned_8;
                            Lo_Hi : in Unsigned_8;
                            Lo_Lo : in Unsigned_8)
                            return Integer is
      begin
         return Integer(Lo_Lo)
           + 2**8  * Integer(Lo_Hi)
           + 2**16 * Integer(Hi_Lo)
           + 2**24 * Integer(Hi_Hi);
      end Make_Integer;
      pragma Inline(Make_Integer);
      Bios_Get_Clock       : constant Unsigned_8  := 16#00#;
      Bios_Clock_Interrupt : constant Unsigned_16 := 16#001a#;
   begin
      -- Get current and future clock ticks
      Regs.Ah := Bios_Get_Clock;
      Dpmi_Int(Bios_Clock_Interrupt, Regs);
      Current_Tick  := Make_Integer(Regs.Ch, Regs.Cl, Regs.Dh, Regs.Dl);
      Wait_For_Tick := Current_Tick + Number_Of_Ticks;
      -- Wait for ticks reached
      while Current_Tick /= Wait_For_Tick loop
         Regs.Ah := Bios_Get_Clock;
         Dpmi_Int(Bios_Clock_Interrupt, Regs);
         Current_Tick := Make_Integer(Regs.Ch, Regs.Cl, Regs.Dh, Regs.Dl);
      end loop;
   end Bios_Wait_Ticks;

   procedure Speaker (Frequency : in Integer; Ticks : in Integer) is
      Old_B        : Unsigned_8;
      Freq_Hi      : Unsigned_8 := Unsigned_8 (Frequency / 256);
      Freq_Low     : Unsigned_8 := Unsigned_8
        (Frequency - 256 * Integer (Freq_Hi));
      Speaker_On   : constant Unsigned_8  := 16#03#;
      Pit_Divider  : constant Unsigned_16 := 16#0042#;
      Pit_Mode     : constant Unsigned_16 := 16#0043#;
      Port_B_Out   : constant Unsigned_16 := 16#0061#;
      Set_Pit_Mode : constant Unsigned_8  := 2#10110110#;
      procedure Outportb (Port : in Unsigned_16; Data : in Unsigned_8);
      pragma Import(C, Outportb, "outportb");
      function Inportb (Port : in Unsigned_16) return Unsigned_8;
      pragma Import(C, Inportb, "inportb");
   begin
      -- Set speaker frequency
      Outportb (Pit_Mode, Set_Pit_Mode);
      Outportb (Pit_Divider, Freq_Low);
      Outportb (Pit_Divider, Freq_Hi);
      -- Enable speaker
      Old_B := Inportb(Port_B_Out);
      Outportb(Port_B_Out, Old_B or Speaker_On);
      -- Wait...
      Bios_Wait_Ticks(Ticks);
      -- Disable speaker
      Outportb(Port_B_Out, Old_B);
   end Speaker;

   procedure Dentist_Sound is
   begin
      Speaker (257, 1);
   end Dentist_Sound;

   procedure Sad_Sound is
   begin
      Speaker (2000, 8);
   end Sad_Sound;

   procedure Vague_Sound is
   begin
      Speaker (2000, 4);
      Speaker (2200, 4);
      Speaker (2400, 4);
      Speaker (2600, 4);
      Speaker (2800, 4);
   end Vague_Sound;

end Sound;
