WITH Ada.Text_IO;
WITH Screen;
PROCEDURE Lock_Screen IS
------------------------------------------------------------------
--| Shows tasks writing into their respective columns on the
--| screen. To ensure that an entire "message" is written, even
--| if a task loses the CPU momentarily, we use a locking task.
--| Author: Michael B. Feldman, The George Washington University 
--| Last Modified: December 1995                                     
------------------------------------------------------------------

  TASK TYPE LockType IS
  -- This is a typical task used to lock a resource.
  -- In this case each task will lock the screen for its own use,
  -- so that no other task can write to the screen until the
  -- lock is released.

    ENTRY Seize;
    ENTRY Release;

  END LockType;
 
  TASK BODY LockType IS
  -- The lock task continues to run, alternately accepting
  -- Seize and Release calls. The SELECT and TERMINATE ensure
  -- that when the other tasks are finished, the Lock task
  -- will also terminate. Otherwise it might "hang", waiting for
  -- a call that would never come.
  BEGIN -- LockType
    
    LOOP
      SELECT
        ACCEPT Seize;
        ACCEPT Release;
      OR
        TERMINATE;
      END SELECT;
    END LOOP;
 
  END LockType;
 
  Lock: LockType;

  TASK TYPE SimpleTask (Message: Character; 
                        HowMany: Screen.Depth;
                        Column:  Screen.Width) IS

    -- This specification provides a "start button" entry.
    ENTRY StartRunning;

  END SimpleTask;

  TASK BODY SimpleTask IS

  BEGIN -- SimpleTask
    
    -- Each task will write its message in its own column
    -- Now the task locks the screen before moving the cursor,
    -- unlocking it when writing is completed.

    ACCEPT StartRunning;

    FOR Count IN 1..HowMany LOOP
      Lock.Seize;           -- grab screen for myself
      Screen.MoveCursor(Row => Count, Column => Column);
      Ada.Text_IO.Put(Item => "Hello from Task " & Message);
      Lock.Release;         -- let another task have the screen

      DELAY 0.5;            -- lets another task have the CPU
    END LOOP;

  END SimpleTask;

  -- Now we declare three variables of the type
  Task_A: SimpleTask(Message => 'A', HowMany => 5, Column => 1);
  Task_B: SimpleTask(Message => 'B', HowMany => 7, Column => 26);
  Task_C: SimpleTask(Message => 'C', HowMany => 4, Column => 51);

BEGIN -- Lock_Screen
  
  Screen.ClearScreen;
  Task_B.StartRunning;
  Task_A.StartRunning;
  Task_C.StartRunning;

END Lock_Screen;
