PACKAGE VStrings IS
------------------------------------------------------------------
--| Specification for ADT to handle strings of variable length.
--| Maximum length must be at least 1.
--| Author: Michael B. Feldman, The George Washington University 
--| Last Modified: September 1995                                     
------------------------------------------------------------------
 
  TYPE    VString(MaxLength: Positive) IS PRIVATE;
 
  -- exceptions

  StringOverflow   : EXCEPTION;
  EmptyString      : EXCEPTION;
  InvalidArguments : EXCEPTION;

  -- operators

  -- constructors

  FUNCTION MakeVString(S : String; MaxLength: Positive) RETURN VString;
  -- Pre:    S and MaxLength are defined
  -- Post:   returns a VString with S as the Value part,
  --   MaxLength as the MaxLength part and S'Length as the Length part
  -- Raises: StringOverflow if S is longer than MaxLength characters

  FUNCTION MakeVString(C : Character; MaxLength: Positive) RETURN VString;
  -- Pre:    C and MaxLength are defined
  -- Post:   returns a VString with C as the Value part, Length = 1

  FUNCTION EmptyVString(MaxLength: Positive) RETURN VString;
  -- Pre:    MaxLength is defined
  -- Post:   returns a empty VString with the given MaxLength

  -- selectors

  FUNCTION Length(S : VString)    RETURN Natural;
  FUNCTION MaxLength(S : VString) RETURN Positive;
  FUNCTION Value(S : VString)     RETURN String;
  -- Pre:    S is defined
  -- Post:   returns the Length and Value of S, respectively
 
  FUNCTION Head(S : VString) RETURN Character;
  -- Pre:    S is defined
  -- Post:   returns the first character of S
  -- Raises: EmptyString if S is empty

  -- inquiry

  FUNCTION IsEmpty(S : VString) RETURN Boolean;
  -- Pre: S is defined
  -- Post: returns True if S is empty, False otherwise

  -- concatenation

  FUNCTION "&" (S1, S2 : VString) RETURN VString;
  FUNCTION "&" (S1 : VString; C : Character) RETURN VString;
  FUNCTION "&" (C : Character; S1 : VString) RETURN VString;
  FUNCTION "&" (S1 : VString; S : String) RETURN VString;
  FUNCTION "&" (S : string; S1 : VString) RETURN VString;
  -- Pre:    parameters are defined
  -- Post:   each operator returns the concatenation of its arguments;
  --   the maximum length of the result is the larger of the two
  --   maximum lengths.
  -- Raises: StringOverflow if the result would be longer than
  --   the longer of the two arguments
 
  -- lexical comparison

  FUNCTION "<"  (S1, S2 : VString) RETURN Boolean;
  FUNCTION "<=" (S1, S2 : VString) RETURN Boolean;
  FUNCTION ">"  (S1, S2 : VString) RETURN Boolean;
  FUNCTION ">=" (S1, S2 : VString) RETURN Boolean;
  -- Pre: S1 and S2 are defined
  -- Post: carries out the desired comparison, returning True or False
 
  -- search

  FUNCTION Locate(Sub : VString; Within : VString) RETURN Natural;
  FUNCTION Locate(Sub : String;  Within : VString) RETURN Natural;
  FUNCTION Locate(C : Character; Within : VString) RETURN Natural;
  -- Pre:   Sub, Within. and C are defined
  -- Post:  returns the index of the first character of Sub in Within;
  --   returns 0 if Sub is not present in Within
 
  FUNCTION Tail(S : VString) RETURN VString;
  -- Pre:    S is defined
  -- Post:   returns a string like S but with the first character removed;
  -- Raises: EmptyString if S is empty

  FUNCTION Slice(S : VString; Start, Finish : Positive) RETURN VString;
  -- Pre:    parameters are defined
  -- Post:   returns a VString whose value is
  --   the substring slice starting at position Start in S.
  --   This behaves consistently with Ada's predefined slice.
  -- Raises: InvalidParameters if Start or Finish > Length(S).
 
PRIVATE

  TYPE VString(MaxLength: Positive) IS RECORD
    CurrentLength : Natural := 0;
    StringPart : String(1 .. MaxLength) := (OTHERS => ASCII.NUL);
  END RECORD;

END VStrings;
