;--------------------------------
;Term.Asm  from 'Advanced Assembly Language'

IDEAL
DOSSEG
        MODEL  small
        STACK  1024
JUMPS

TimeOut     equ     364                 ;20 seconds (18.2 * 20)
NormWait    equ     18                  ;1 second
BuffSize    equ     2048                ;Size of incoming buffer (in bytes)
ComInt      equ     0Bh                 ;COM1/COM3=0Ch, COM2/COM4=0Bh
ComAddr     equ     2F8h                ;COM1=3F8h, COM2=2F8h, COM3=3E8h,
                                        ;COM4=2E8h
BRDMSB      equ     0                   ;See book for common BRD values
BRDLSB      equ     06h
DF_N81      equ     00000011b           ;8 data bits, 1 stop bit, no parity
DF_E71      equ     00011010b           ;7 data bits, 1 stop bit, even parity

DATASEG
VectorSeg   dw      0000
VectorOff   dw      0000
XmitCount   dw      0000
CTMax       dw      0000
SetUpMsg    db      'Setting up modem information ...',13,10,0
MemMsg      db      'Could not allocate buffer space',13,10,0
RemindMsg   db      'Press ESC to exit program',13,10,13,10,0

CODESEG
PROC    Term
        mov     bx,200h
        mov     ah,4ah
        int     21h
        jc      @@AllDone
        mov     ax,@data
        mov     ds,ax
        mov     es,ax
        mov     ah,48h
        mov     bx,Buffsize/16
        int     21h
        jc      @@AllDone
@@MemOK:mov     [cs:BuffSeg],ax     ;Store segment address
        call    HookIn              ;Go hook interrupt information
        call    Cls
        mov     si,OFFSET RemindMsg ;Reminder about how to exit
        call    PrtString
@@InputLoop:
        mov     ah,1                ;Check keyboard status
        int     16h
        jz      @@NoKey             ;No key there
        mov     ah,0                ;Get key
        int     16h
        cmp     ah,1                ;Was the key escape? (scan code=1)
        je      @@AllDone           ;Yes, so exit
        call    XmitChar            ;Send the character in AL to modem
        jnc     @@NoKey             ;No error, so continue
        mov     dl,'!'              ;Indication there was a timeout
        mov     ah,02h              ;Output a character
        int     21h
@@NoKey:call    BuffChar            ;Go see if anything in buffer to display
        jmp     @@InputLoop
@@AllDone:
        mov     es,[cs:BuffSeg]
        mov     ah,49h              ;Release memory block at ES
        int     21h
        call    HookOut
        mov     ah,4ch
        int     21h
ENDP Term

PROC HookIn
        mov     si,OFFSET SetUpMsg
        call    PrtString
        push    es
        mov     al,ComInt
        mov     ah,35h              ;Get interrupt vector
        int     21h
        mov     [VectorOff],bx      ;Store original offset
        mov     [VectorSeg],es      ;Store original segment
        pop     es
        mov     al,ComInt
        push    ds
        push    cs
        pop     ds
        mov     dx,OFFSET IntHandler
        mov     ah,25h              ;Set interrupt vector
        int     21h
        pop     ds
        cli
        mov     dx,ComAddr+3        ;Point to line control register
        mov     al,DF_N81
        out     dx,al               ;Send byte
        inc     dx                  ;Point to modem control register
        mov     al,00001011b
        out     dx,al               ;Set it for interrupts
        in      al,21h              ;Get current contents of interrupt
                                    ; request mask
        and     al,11100111b        ;Make sure that both COMs are enabled
        out     21h,al
        mov     dx,ComAddr+1        ;Point to interrupt enable register
        mov     al,00000001b        ;Turn on bit 0
        out     dx,al               ;Set it
        sti
        call    SetBRD              ;Go set for desired baud rate
        ret
ENDP HookIn

PROC HookOut
        mov     dx,ComAddr+1        ;Point to interrupt enable register
        mov     al,0                ;Disable the interrupts
        out     dx,al               ;Set it
        mov     dx,ComAddr+1        ;Point to interrupt enable register
        mov     al,00000000b        ;Turn off all bits
        out     dx,al               ;Set it
        in      al,21h              ;Get current contents of interrupt
                                    ; request mask
        or      al,00011000b        ;Make sure that both COMs are disabled
        out     21h,al
        push    ds                  ;Hold current data segment
        mov     ds,[VectorSeg]      ;Get original segment
        mov     dx,[VectorOff]
        mov     al,ComInt
        mov     ah,25h              ;Set interrupt vector
        int     21h
        pop     ds                  ;Get the data segment back
        ret
ENDP HookOut

PROC BuffChar
        mov     si,[cs:Tail]
        cmp     si,[cs:Head]        ;Any characters there?
        je      @@NoInput           ;Nope, so continue
        mov     es,[cs:BuffSeg]     ;Point to buffer area
        cld
@@BCLoop:
        mov     dl,[es:si]          ;Get character
        inc     si                  ;Point to next character
        mov     ah,02h              ;Output character in DL
        int     21h
        cmp     si,BuffSize         ;End of buffer?
        jne     @@NotAtEnd          ;No, continue
        mov     si,0                ;Point to beginning
@@NotAtEnd:
        cmp     si,[cs:Head]        ;At end of buffered data yet?
        jne     @@BCLoop            ;No, grab another
        mov     [cs:Tail],si        ;Update pointer
@@NoInput:
        ret
ENDP BuffChar

PROC XmitChar
        push    ax                  ;Save character for a moment
        mov     ah,0
        int     1ah                 ;Get clock ticks
        mov     [XmitCount],dx      ;Store ticks for later
@@XC1:  mov     dx,ComAddr+5        ;Point to line status register
        in      al,dx               ;Get status byte
        test    al,00100000b        ;Test if ready to transmit character
        jnz     @@XC2               ;Yes, so continue
        mov     bx,[XmitCount]
        mov     ax,NormWait         ;Wait 1 second
        call    CheckTime           ;Go see if time expired
        jnc     @@XC1               ;Time not up, wait in loop
        pop     ax                  ;Get back character
        jmp     @@XCBad             ;Time up, exit with error
@@XC2:  pop     ax                  ;Get character back
        mov     dx,ComAddr          ;Point to port
        out     dx,al
        clc
        jnc     @@XCExit
@@XCBad:stc
@@XCExit:
        ret
ENDP XmitChar

PROC CheckTime
        mov     [CTMax],ax          ;Store maximum ticks
        mov     ah,0
        int     1ah                 ;Check for timeout
        cmp     bx,dx               ;Check for wrap around
        jg      @@CT1               ;Yup, it was there
        sub     dx,bx               ;Now we have elapsed ticks in DX
        jmp     @@CT2
@@CT1:  mov     ax,0FFFFh
        sub     ax,bx
        add     dx,ax
@@CT2:  cmp     dx,[CTMax]          ;Is our time up?
        ja      @@TimeUp            ;Yes, so exit accordingly
        clc                         ;No, no error
        jnc     @@CTExit
@@TimeUp:
        stc                         ;Set for time up
@@CTExit:
        ret
ENDP CheckTime

PROC SetBRD
        mov     dx,ComAddr+3        ;Point to line control register
        in      al,dx               ;Get what is there
        or      al,10000000b        ;Going to set bit 7 high
        out     dx,al               ;Send the byte
        dec     dx                  ;Point to MSB of baud rate divisor
        dec     dx
        mov     al,BRDMSB
        out     dx,al
        dec     dx                  ;Point to LSB of baud rate divisor
        mov     al,BRDLSB
        out     dx,al
        add     dx,3                ;Point to line control register
        in      al,dx               ;Get what is there
        and     al,01111111b        ;Going to set bit 7 low
        out     dx,al               ;Send the byte
        ret
ENDP SetBRD

PROC Cls
        mov     ah,6                ;Scroll window up
        mov     al,0                ;Scroll full screen
        mov     bh,7                ;Normal white on black
        mov     cx,0                ;Upper left corner of screen
        mov     dh,24               ;Bottom right
        mov     dl,79
        int     10h
        mov     dx,0                ;Upper left corner of screen
        mov     bh,0                ;Assume page 0
        mov     ah,2                ;Set cursor position
        int     10h
        ret
ENDP Cls

PROC PrtString
@@PS1:  mov     dl,[si]             ;Get character
        inc     si                  ;Point to next one
        cmp     dl,0                ;End of string?
        je      @@PS2               ;Yes, so exit
        mov     ah,02h              ;Output a character
        int     21h
        jmp     @@PS1               ;Keep doing it
@@PS2:  ret
ENDP PrtString

PROC IntHandler      FAR
        push    ax
        push    dx
        mov     dx,ComAddr+4        ;Point to modem control register
        in      al,dx
        and     al,11111101b        ;Sets bit 1 to 0 (turn off RTS)
        out     dx,al               ;Set it
        push    ds
        push    es
        push    di
        mov     ax,cs
        mov     ds,ax
        mov     es,[cs:BuffSeg]
        mov     di,[cs:Head]
        cld
Receive:mov     dx,ComAddr          ;Get COM base address
        in      al,dx               ;Get the character
        stosb                       ;Store character in buffer
        cmp     di,BuffSize         ;At end?
        jne     NoWrap              ;No, so continue
        mov     di,0                ;Point to start
NoWrap: mov     [cs:Head],di
        mov     dx,ComAddr+2        ;Point to interrupt identification register
        in      al,dx               ;Get the value that is there
        test    al,1                ;Is another request pending?
        jz      Receive             ;Yes, so go handle again
        mov     al,20h              ;Send end-of-interrupt code
        out     20h,al
        pop     di
        pop     es
        pop     ds
        mov     dx,ComAddr+4        ;Point to modem control register
        in      al,dx
        or      al,00000010b        ;Sets bit 1 to 1
        out     dx,al               ;Set it
        pop     dx
        pop     ax
        iret

BuffSeg     dw      0000
Head        dw      0000
Tail        dw      0000

ENDP IntHandler

END Term

