/****************************************************************

	as_test.c       test astronomical (as) subsystem

			Copyright (c) 1991, Ted A. Campbell

			Bywater Software
			P. O. Box 4023 
			Duke Station 
			Durham, NC  27706

			email: tcamp@hercules.acpub.duke.edu

	Copyright and Permissions Information:

	All U.S. and international copyrights are claimed by the
	author. The author grants permission to use this code
	and software based on it under the following conditions:
	(a) in general, the code and software based upon it may be 
	used by individuals and by non-profit organizations; (b) it
	may also be utilized by governmental agencies in any country,
	with the exception of military agencies; (c) the code and/or
	software based upon it may not be sold for a profit without
	an explicit and specific permission from the author, except
	that a minimal fee may be charged for media on which it is
	copied, and for copying and handling; (d) the code must be 
	distributed in the form in which it has been released by the
	author; and (e) the code and software based upon it may not 
	be used for illegal activities. 

****************************************************************/

#include "stdio.h"

#include "math.h"
#include "time.h"
#include "bw.h"
#include "as.h"

#ifndef  __STDC__
#define time_t long
#endif

struct as_focus test_focus;
struct as_orbit test_orbit;

char bw_ebuf[ BW_EBUFSIZE ];

/***************************************************************

   main()      Main function for stand-alone TEST

***************************************************************/

main()
   {
   static char c;
   int keep_on;

   keep_on = 1;
   while( keep_on == 1 )
      {

      vpt_init();
      vpt_level = 1;

      ast_rollup();
      printf( "Test \"as\" (Astronomical) Functions: \n" );
      printf( "----------------------------------  \n" );
      printf( "\n" );
      printf( "     f - Test Focal Datafile (fd) Read Function \n" );
      printf( "     o - Test Orbital Functions   \n" );
      printf( "     r - Test Angular Radius Function \n" );
      printf( "     s - Test Spherical Projection \n" );
      printf( "\n" );
      printf( "     x - Exit                     \n" );
      printf( "\n" );
      printf( "     Select:  " );
      c = getchar();
      fflush( stdin );

      switch( c )
	 {
	 case 'F':
	 case 'f':
	    ast_readfd();
	    break;
	 case 'O':
	 case 'o':
	    ast_orbital();
	    break;
         case 'R':
         case 'r':
            ast_angrad();
            break;
	 case 'S':
	 case 's':
	    ast_spj();
	    break;
	 case 'X':
	 case 'x':
	    ast_rollup();
	    keep_on = 0;
	    break;
	 default:
	    break;
	 }
      }
   }

ast_rollup()
   {
   register int c;

   for ( c = 0; c < 25; ++c )
	{
	printf( "\n" );
	}
   }

/***************************************************************

   ast_orbital()      test orbital functions

***************************************************************/

ast_orbital()
   {
   static long t, r, n;
   static double lat, lon, ap, aa;

   test_orbit.focus = &test_focus;

   ast_rollup();
   printf( "Test Orbital Functions \n" );
   printf( "---------------------- \n\n" );

   printf( "Radius of the focus (km):               " );
   scanf( "%lf", &test_orbit.focus->radius );

   printf( "Sidereal period of the focus:           " );
   scanf( "%lf", &test_orbit.focus->sid_day );

   printf( "Mass of the focus (grams):              " );
   scanf( "%lf", &test_orbit.focus->mass );

   printf( "Altitude of perigee (km):               " );
   scanf( "%lf", &ap );

   printf( "Altitude of apogee (km):                " );
   scanf( "%lf", &aa );

   printf( "Inclination (degrees):                  " );
   scanf( "%lf", &test_orbit.inclination );

   printf( "Longitude of ascending node (degrees):  " );
   scanf( "%lf", &test_orbit.lon_an );

   printf( "Argument of the perigee (degrees):      " );
   scanf( "%lf", &test_orbit.arg_per );

   fflush( stdin );

   or_init( &test_orbit, ap, aa );

   printf( "\n" );
   printf( "Focus:               Radius         = %lf\n", test_orbit.focus->radius );
   printf( "                     Sidereal Day   = %lf\n", test_orbit.focus->sid_day );
   printf( "                     Mass           = %lf\n", test_orbit.focus->mass );
   printf( "Orbital Parameters:  Semimajor axis = %lf\n", test_orbit.semimajor );
   printf( "                     Semiminor axis = %lf\n", test_orbit.semiminor );
   printf( "                     Period (T)     = %lf\n", test_orbit.period );
   printf( "                     Eccentricity   = %lf\n", test_orbit.eccentricity );
   printf( "                     Lon. As. Node  = %lf\n", test_orbit.lon_an );
   printf( "                     Inclination    = %lf\n", test_orbit.inclination  );
   printf( "                     Arg. Perigee   = %lf\n", test_orbit.arg_per );
   printf( "                     Long. inc.     = %lf\n", test_orbit.lif );
   printf( "\n" );
   ast_wait();

   for ( t = 0; t < ( test_orbit.period * 3); t += ( test_orbit.period / 12 ))
      {
      or_ssp( &test_orbit, t, &lat, &lon, &r, &n );
      printf( "Orbit %ld, time %ld: \t\nDistance:  %ld km \tlat: %lf deg \tlon: %lf deg\n", 
	 n, t, (long) (r - test_orbit.focus->radius ), lat,
	 lon );
      }
   }

/***************************************************************

   ast_angrad()      test angular radius function

***************************************************************/

ast_angrad()
   {
   static double distance, radius;
   double ang_rad;

   ast_rollup();
   printf( "Test Angular Radius Function \n" );
   printf( "-------------------------- \n\n" );

   printf( "Radius of the focus (km):           " );
   scanf( "%lf", &radius );

   printf( "Distance to the focus (km):         " );
   scanf( "%lf", &distance );

   ang_rad = spj_angrad( distance, radius );
   printf( "\nThe angular radius is %0.9lf degrees. \n", ang_rad );
   ast_wait();
   
   }

ast_wait()
   {
   printf( "\nPress any key: " );
   getchar();
   }

/***************************************************************

   ast_spj()      test spherical projection

***************************************************************/

ast_spj()
   {
   static double x, y, side;
   static double radius, distance;
   static double vlat, vlon;
   static double plat, plon;

   ast_rollup();
   printf( "Test Spherical Projection Point Calculation Function \n" );
   printf( "---------------------------------------------------- \n\n" );

   printf( "Radius of the point (km):             " );
   scanf( "%lf", &radius );
   printf( "Latitude of the point:                " );
   scanf( "%lf", &plat );
   printf( "Longitude of the point:               " );
   scanf( "%lf", &plon );
   printf( "Distance from viewer (km):            " );
   scanf( "%lf", &distance );
   printf( "Latitude of viewer:                   " );
   scanf( "%lf", &vlat );
   printf( "Longitude of viewer:                  " );
   scanf( "%lf", &vlon );

   spj_point( plat, plon, radius, vlat, vlon, distance, 0.0,
      SPJ_ALLSIDES, &x, &y, &side );
   printf( "\n\nPoint is on the " );
   if ( side == SPJ_NEARSIDE )
      {
      printf( "near " );
      }
   else
      {
      printf( "far " );
      }
   printf( "side of the sphere:\n" );
   printf( "x axis: %lf degrees of arc (left or right)\n", x );
   printf( "y axis: %lf degrees of arc (up or down)\n", y );
   ast_wait();

   }

/***************************************************************

   ast_readfd()      test read focal data file function

***************************************************************/

ast_readfd()
   {
   static char fdfile[ 128 ];

   ast_rollup();
   printf( "Test Focal Datafile Read Function \n" );
   printf( "--------------------------------- \n\n" );

   printf( "Enter name of a focal data file:    " );
   gets( fdfile );
   fflush( stdin );

   if ( as_readfd( fdfile, &test_focus ) == BW_ERROR )
      {
      return;
      }

   printf( "\n\n" );
   printf( "Name of orbital focus:             %s\n", test_focus.name );
   printf( "Adjective for orbital focus:       %s\n", test_focus.adjective );
   printf( "Radius of orbital focus:           %.0lf km\n", test_focus.radius );
   printf( "Mass of orbital focus:             %.2le kg\n", test_focus.mass );
   printf( "Sidereal period of orbital focus:  %.0lf seconds\n", test_focus.sid_day );

   ast_wait();

   }

bw_message( m )
   char *m;
   {
   fprintf( stderr, "%s\n", m );
   }

bw_error( m )
   char *m;
   {
   fprintf( stderr, "ERROR: %s\n", m );
   getchar();
   fflush( stdin );
   }

#ifdef  DEBUG
bw_debug( m )
   char *m;
   {
   fprintf( stderr, "DEBUG: %s\n", m );
   getchar();
   fflush( stdin );
   }
#endif

