
/*
 *  Activate.c (v1.01)
 *
 *  Utility for setting the Active partition on a HD.
 *
 *  Author: Ronan Mullally October 1992.
 *          < H235_003@ccvax.ucd.ie >
 *          < ronan@bermuda.ucd.ie >
 *          < ronan.mullally@f151.n263.z2.fidonet.org >
 *
 *  Usage: Activate [-d] [-l] [partition #]
 *
 *         -d: Deactivate given partition #
 *         -l: List partition Table(s)
 *         # : Number of partition to (De)Activate
 *
 *         if no command line params are given -l is assumed
 *
 *
 *  Notes: I make *no* guarentees that this program is 100% safe - All
 *         I'll say is that it hasn't damaged my own Hard Disk (yet!)
 *
 *         Most of the code should be fairly portable - the only exceptions
 *         are the functions contained in the DOS-ACT.c and NDOS-ACT.c files.
 *         which contain the routines for reading and writing the 1st sector
 *         of the hard drive.
 *
 */

#include "activate.h"
#include "config.h"               /* Set any system dependant stuff here */


static systypes SystemTypes[NUMBER_OF_SYSTEM_TYPES] =
{
  { 0x00, "Empty" },                              /* Most of these were  */
  { 0x01, "DOS 12-bit FAT" },                     /* 'plagerised' from   */
  { 0x02, "XENIX root" },                         /* the fdisk program   */
  { 0x03, "XENIX user" },                         /* provided with Linux */
  { 0x04, "DOS 16-bit FAT, < 32MB" },
  { 0x05, "DOS Extended" },
  { 0x06, "DOS 16-bit FAT, >= 32MB" },
  { 0x07, "OS/2 HPFS" },
  { 0x08, "AIX" },
  { 0x09, "AIX bootable / Coherent" },
  { 0x0A, "OPUS" },
  { 0x40, "Venix" },
  { 0x51, "Ontrack extended partition" },
  { 0x52, "Microport" },
  { 0x63, "GNU HURD" },
  { 0x64, "Novell" },
  { 0x75, "PC/IX" },
  { 0x80, "Old MINIX" },
  { 0x81, "Linux / MINIX" },
  { 0x82, "Linux swap" },
  { 0x83, "Linux extended fs" },
  { 0x93, "Amiga" },
  { 0x94, "Amiga BBT" },
  { 0xB7, "BSDI" },
  { 0xB8, "Syrinx" },
  { 0xDB, "CP/M" },
  { 0xE1, "DOS access" },
  { 0xE3, "DOS R/O" },
  { 0xF2, "DOS secondary" },
  { 0xFF, "BBT" },
  { 0x00, "Unknown" }                  /* Displayed if system ID unknown */
};

int display = 0, activate = 0, deactivate = 0;
byte buffer[512];                       /* Store Master Boot Sector here */


/*
 * Calculates the Start and End boundaries for the given partn table entry.
 * The 2 high-order bits of the Cylinder are stored in the 2 high order bits
 * of start_sector / end_sector, this these sector values are only 6 bits
 *
 */

void get_partition_boundaries(entry, Start, End)
partition_table_entry entry;
location *Start;
location *End;
{
  Start->Side = entry.start_sector_head;
  Start->Sector = entry.start_sector & 0x3F;   /* 2 high bits are for cyl */
  Start->Cylinder = ((entry.start_sector & 0xC0) << 2)+(entry.start_cylinder);

  End->Side = entry.end_sector_head;
  End->Sector = entry.end_sector & 0x3F;
  End->Cylinder = ((entry.end_sector & 0xC0) << 2) + (entry.end_cylinder);
}


/*
 * Returns index into the SystemTypes array where the Description of the
 * given ID byte can be found.  If no match is found the index of the last
 * entry is given - this corresponds to the description "Unknown".
 *
 */

int get_system_type(system_ID)
byte system_ID;
{
  int x;

  for(x = 0; x < NUMBER_OF_SYSTEM_TYPES; x++)    /* Search for known ID */
    if(SystemTypes[x].ID == system_ID)           /* if a match is found */
      return(x);                                 /* return desc's index */

  return(NUMBER_OF_SYSTEM_TYPES);                /* return unknown index */
}


/*
 *  Displays Partition Table data
 *
 */

void show_partition_table(partn, drive)
partition_table partn;
int drive;
{
  int x;
  location Start, End;

#ifdef __MSDOS__
  printf("\nPartition Table for Hard Drive %d:", drive);
#else
  printf("\nPartition Table for Hard Drive %s:", drive_name[drive]);
#endif

  printf("\n\t\t\t\t\t\t    Starting       Ending");
  printf("\nActive\t\tSystem Type\t    Size (KB)    Side Cyl Sect  Side Cyl Sect\n");
  for(x = 0; x < 4; x++)
  {
    get_partition_boundaries(partn.entry[x], &Start, &End);
    printf("  %-3s", partn.entry[x].boot_flag ? "Yes" : "No");
    printf("\t%-30s", SystemTypes[get_system_type(partn.entry[x].ID)].Desc);
    printf("%6ld", partn.entry[x].sectors_in_partn / 2);
    printf("%8d %4d %3d", Start.Side, Start.Cylinder, Start.Sector);
    printf("%6d %4d %3d\n", End.Side, End.Cylinder, End.Sector);
  }
}


/*
 * Makes the given partition the booting partition - done simply by setting
 * the boot_flag = 0x80.
 *
 */

void activate_partition(partition)
partition_table_entry *partition;
{
   partition->boot_flag = 0x80;
}


/*
 * Resets the given partition's boot_flag, thus making it inactive
 *
 */

void deactivate_partition(partition)
partition_table_entry *partition;
{
  partition->boot_flag = 0x00;
}


/*
 *  Displays an error message and gives Usage information
 *
 */

void Usage(message, path, errorlevel)
char *message;
char *path;
int errorlevel;
{
  printf("\nActivate v1.01 Hard Disk Partition selector (Ronan Mullally, 1992)\n");
  if(message != "") printf("\n%s", message);

  printf("\n\n%cUsage:  %s [-d] [-l] [partition #]\n", 7, path);
  printf("\t-d\t   : Deactivate given partition\n");
  printf("\t-l\t   : Display Partition Table(s)\n");
  printf("\tpartition #: Number of Partition to activate\n\n");
  printf(" Eg: %s 1     Activates 1st partition on Drive 0\n", path);
  printf("     %s -d 6  Deactivates 2nd partition on Drive 1\n\n", path);

  exit(errorlevel);
}


/*
 * Checks the command line for invalid parameters, processes valid options
 * and extracts a partition number (if given).
 *
 * Returns partition number given on command line, or aborts if invalid
 * options were given
 *
 */

int check_command_line(argc, argv)
int argc;
char *argv[];
{
  int x, y = 0, partition_number = 0, error = 0;
  char *buffer[40];

  for(x = 1; x < argc && !error; x++)
  {
    for(y = 0; !error && argv[x][y] != '\0'; y++)
      if(!isdigit(argv[x][y]))
        error = 1;

    if(!error)
    {
      partition_number = atoi(argv[x]);
      activate = !deactivate;
    }
    else if(!strcmp(argv[x], "-l"))
    {
      display = 1;
      error = 0;
      continue;
    }
    else if(!strcmp(argv[x], "-d"))
    {
      deactivate = 1;
      activate = 0;
      error = 0;
      continue;
    }
    else
    {
      error = 1;
      break;
    }
  }

  if(error)
  {
    sprintf((char *) buffer, "Invalid command line argument: %s", argv[x]);
    Usage((char*) buffer, argv[0], BAD_COMMAND_LINE);
  }
  return(partition_number);
}





int main(argc, argv)
int argc;
char *argv[];
{
#ifdef __MSDOS__
  int drive = 0x80;          /* for DOS drives are referenced as numbers */
#else
  char *drive;                            /* for *nix they are filenames */
#endif

  int max_partition, partition_to_activate = 0, x, y, drive_count;
  partition_table partn;

  if(argc > 4)
    Usage("Too many parameters on command line", argv[0], BAD_COMMAND_LINE);

  else if(argc == 1) display = 1;         /* no options given, just list */

  else partition_to_activate = check_command_line(argc, argv);

  drive_count = determine_number_of_drives();
  max_partition = drive_count * 4;                 /* 4 partns per drive */

  if(display)
    printf("\n\n%d Hard Drive%s found", drive_count, drive_count > 1 ? "s" : "");

  if((!partition_to_activate && !display) || (partition_to_activate > max_partition))
    Usage("Invalid partition number", argv[0], BAD_PARTITION_NUMBER);

  for(x = 0; x < drive_count; x++)
  {
#ifdef __MSDOS__
    drive += x;
#else
    drive = drive_name[x];
#endif

    read_partition_table(&partn, drive);

    if((partition_to_activate-1) / 4 == x)  /* partition on this drive ? */
    {
      if(activate)                           /* Activate given partition */
      {
        for(y = 0; y < 4; y++)
          deactivate_partition(&partn.entry[y]);   /* Deactive all first */
        activate_partition(&partn.entry[partition_to_activate - (4*x)-1 % 4]);
        write_partition_table(partn, drive);
      }
      
      else if(deactivate)                  /* Deactivate given partition */
      {
        deactivate_partition(&partn.entry[partition_to_activate-(4*x)-1 % 4]);
        write_partition_table(partn, drive);
      }    
    }
    if(display) show_partition_table(partn, x);
  }

  return(0);
}


