/*
Ŀ
 Title   : jzinstr							     
 Purpose : Provide a sophisticated means of reading input from the keyboard.
	    This routine allows editing via the arrow keys and delete, insert
	    etc are supported as a subset of the wordstar commands.	     
									     
 Parms:								     
   fstr : string to return input into					     
   flen : maximum length of string. Input will be terminated when string is 
	   longer than this value					     
   frow : row position of input					     
   fcol : starting column position.					     
   fattr: color attribute of field. See tech reference manual for details   
	   or calculate as (Background*16|foreground)&0x7f;		     
   ftime: Input is timed. If seconds exceed this value, input is terminated 
	   as if the user hit the <ENTER> key.				     
   fkeys: string of characters whose ordinal values equal the scan codes of 
	   any special characters which will terminate the input. i.e. If you
	   wanted the user to be able to quit the read via the ESC key, you  
	   would pass "\001" which is the scan code for the ESC key. Note    
	   the routine passes back the keystroke that terminated the read    
	   and that multiple keys can be put into this field.		     
   Notes:								     
	   The keypad characters are mapped to their wordstar equivalents as 
	   so that UPARROW is mapped to CTRL_E, etc so keep this in mind     
	   when checking return values. 				     
									     
	Written by Jack Zucker - 75766,1336    301-794-5950  on 	     

*/

jzinstr(fstr,flen,frow,fcol,fattr,ftime,fkeys)
char *fstr;
int flen,frow,fcol,fattr;
long ftime;
char *fkeys;
{


  static int WINSERT = 0;	/* insert mode flag	 */
  int WDONE = 0;		/* init loop condition	 */
  int w1,w2,wstart,wend;	/* work variables for cursor */
  char wtemp[256];		/* temporary work string */
  int WNULLFOUND;		/* flag variable	 */
  int wcol;			/* work column variable  */
  long wtime,wtics;		/* time accumulator	 */
  int wch,wscan;		/* char and scan code	 */

  wcol	= fcol; 		/* work column indicator */

  if (MEMB(0x40,0x49) == 3) {	/* color mode */
    wstart = 6;
    wend   = 7;
  }
  else {
    wstart = 12;
    wend   = CTRL_M;
  }

  jzsetcur(WINSERT ? wstart-4 : wstart , wend); /* set cursor size */

  sprintf(wtemp,"%s%s",fstr,jzpad(flen-strlen(fstr),' '));
  jzscrprn(wtemp,frow,fcol,fattr);

  while (! WDONE) {		     /* endless loop */

    jzloccur(frow,wcol);	/* position cursor correctly */

    wtime = time(&wtics);	  /* initialize elapsed time counter */

    if (wcol - fcol == flen) {
      wch = CTRL_M;
      break;
    }

    while ( ! jzkeyprs()) {
      if (time(&wtics) - wtime >= ftime) {
	wch = CTRL_M;
	break;
      }
    }

    if (WDONE) break;	/* must have timed out */

    /* at this point there is a key sitting in the buffer for us */

    wch = jzinkey(&wscan);    /* read char in buffer */

    /* look for a command key and exit if hit */
    if (strchr(fkeys,wscan)) {
      wch = wscan;
      WDONE = 1;
    }
    else if (wscan == PLUS) {  /* fake like it's a special key */
      jzredscr(wtemp,frow,fcol,flen);
      strcpy(fstr,jzrgtjst(wtemp,flen));
      jzscrprn(fstr,frow,fcol,fattr);
      WDONE = 1;
      wch = CTRL_M;
    }

    if (! WDONE)
      do {
	WNULLFOUND = 0; 	   /* init boolean	    */
	switch (wch) {
	  case 0 :		   /* some kind of command key */
	    WNULLFOUND = 1;	   /* got a command key        */
	    switch(wscan) {	   /* translate arrow keys to wordstar */
	      case HOME :
		wch = CTRL_A;
		break;
	      case END	:
		wch = CTRL_F;
		break;
	      case UARR :
		wch = CTRL_E;
		break;
	      case DARR :
		wch = CTRL_X;
		break;
	      case LARR :
		wch = CTRL_S;
		break;
	      case RARR :
		wch = CTRL_D;
		break;
	      case DELETE :
		wch = CTRL_G;
		break;
	      case INSERT :
		wch = CTRL_V;
		break;
	      case PGUP   :
		wch = CTRL_R;
		break;
	      case PGDN   :
		wch = CTRL_C;
		break;
	      case BACKTAB :
		wch = BACKTAB;
		break;

	      default :
		WNULLFOUND = 0; 	   /* reset to false */
		break;
	    }	   /* end to inner case statement */
	    break;
	  case CTRL_E :
	  case CTRL_X :
	  case CTRL_C :
	  case CTRL_R :
	    WDONE = 1;
	    break;
	  case CTRL_G :
	    if (flen-(wcol-fcol)-1)
	      jzredscr(wtemp,frow,wcol+1,flen-(wcol-fcol)-1);
	    else
	      wtemp[0] = 0;
	    strcat(wtemp," ");
	    jzscrprn(wtemp,frow,wcol,fattr);
	    break;
	  case BACKSPACE :
	    if (wcol > fcol) {
	      jzredscr(wtemp,frow,wcol,flen-(wcol-fcol));
	      strcat(wtemp," ");
	      wcol = max(fcol,wcol-1);
	      jzscrprn(wtemp,frow,wcol,fattr);
	    }
	    break;
	  case CTRL_A :
	    wcol = fcol;	  /* back to column 1 of field */
	    break;
	  case CTRL_F :
	    wcol = fcol + flen - 1;   /* end of field */
	    break;
	  case CTRL_S :
	    wcol = max(fcol,wcol-1);
	    break;
	  case CTRL_D :
	    wcol = min(fcol+flen,wcol+1);
	    break;
	  case CTRL_V :
	    WINSERT = ! WINSERT;
	    if (WINSERT) jzsetcur(wstart-4,wend);
	    else jzsetcur(wstart,wend);
	    break;
	  case CTRL_M :
	    WDONE = 1;
	    break;
	  case TAB :
	    wcol = min(fcol+flen-1,wcol+8);
	    break;
	  case BACKTAB :
	    wcol = max(fcol,wcol-8);
	    break;
	  default :
	    if (wch >= ' ' && wch <= '~')
	      if (! WINSERT) {
		jzwrtchr(wch,fattr,1);
		wcol ++;
	      }
	      else if (flen-(wcol-fcol)-1) {	  /* insert mode */
		jzredscr(wtemp,frow,wcol,flen-(wcol-fcol)-1);
		jzwrtchr(wch,fattr,1);
		jzscrprn(wtemp,frow,++wcol,fattr);
	      }
	      else {
		jzwrtchr(wch,fattr,1);
		++wcol;
	      }
	    break;
	}
      } while (WNULLFOUND);
  }

  jzsetcur(wstart,wend);     /* restore cursor size */
  jzredscr(fstr,frow,fcol,flen);

  -- flen;
  if (fstr[flen] == ' ') {  /* get rid of trailing spaces */
    do
      flen --;
    while (fstr[flen] == ' ');
    fstr[flen + 1] = 0;
  }
  return(wch);		     /* maximum length */
}
