/* Copyright (C) Stephen Chung, 1991-1993.  All rights reserved. */

#include "jwp.h"
#include <math.h>


MEASUREINFO Measurements[] = {
    { "inches",         "\"",       2,  1.00 },
    { "centimeters",    " cm",      1,  2.536 },
    { "millimeters",    " mm",      0,  25.36 },
    { "points",         " pts",     1,  72.00 },
    { "pixels",         " dots",    0,  0.00 },
    { NULL,             NULL,       0,  0.00 }
};

static int PrintFrom, PrintTo;
static HWND printdlghwnd;
static BOOL UserAbort, PrintError;


#define ESCAPECHAR      '&'

static struct {
    char escape;
    int summary;
} HeaderStrings[] = {
    { 'L', 0 },
    { 'S', 1 },
    { 'A', 2 },
    { 'K', 3 },
    { 'C', 4 },
    { 'T', -1 },
    { 'D', -2 },
    { 'P', -3 },
    { 'F', -4 },
    { '\0', 0 }
};



MEASUREMENT FindMeasurement (char *name, MEASUREMENT measure)
{
    int i;

    if (measure >= 0) {
        strcpy(name, Measurements[measure].name);
        return (measure);
    }

    for (i = 0; Measurements[i].name != NULL; i++) {
        if (!stricmp(name, Measurements[i].name)) return ((MEASUREMENT) i);
    }
    return (-1);
}



HDC GetPrinterDC (BOOL IcOnly, char *name)
{
    char buffer[MAXLINELEN];
	char *device, *driver, *output;

    /* Opens the printer device */

    GetProfileString ("windows", "device", ",,,", buffer, MAXLINELEN);

	if ((device = strtok(buffer, ",")) &&
		(driver = strtok(NULL, ", ")) &&
        (output = strtok(NULL, ", "))) {
            if (name != NULL) sprintf(name, "%s on %s", device, output);

            if (IcOnly) {
                return (CreateIC(driver, device, output, NULL));
            } else {
                return (CreateDC(driver, device, output, NULL));
            }
    }

    if (name != NULL) strcpy(name, "(None)");
    return (NULL);
}



BOOL SetupPrinter (BOOL ReloadDriver)
{
    HDC hdc;
	POINT printdots;
	double temp;
    double nr_points[2], dpp[2];
    char buffer[MAXLINELEN];

	if (ReloadDriver) {
        hdc = GetPrinterDC(TRUE, buffer);
        if (hdc == NULL) {
            ErrorMessage(global.hwnd, "There are no printer currently connected.\n\n");
            hdc = CreateDC("DISPLAY", NULL, NULL, NULL);
            global.printer = "DISPLAY";
        } else if (!(GetDeviceCaps(hdc, RASTERCAPS) & RC_BITBLT)) {
            ErrorMessage(global.hwnd, "The current printer is incapable of printing bitmaps.\n\n"
                                         "You will not be able to print.");
            DeleteDC(hdc);
            hdc = CreateDC("DISPLAY", NULL, NULL, NULL);
            global.printer = "DISPLAY";
        } else {
            global.printer = DupString(buffer);
        }

		global.resolution.x = GetDeviceCaps(hdc, LOGPIXELSX);
		global.resolution.y = GetDeviceCaps(hdc, LOGPIXELSY);

        global.printable.x = GetDeviceCaps(hdc, HORZRES);
        global.printable.y = GetDeviceCaps(hdc, VERTRES);

        Measurements[M_PIXELS].conversion = global.resolution.y;

        global.paper[0] = (double) GetDeviceCaps(hdc, HORZSIZE) / Measurements[M_MM].conversion;
        global.paper[1] = (double) GetDeviceCaps(hdc, VERTSIZE) / Measurements[M_MM].conversion;
	}

	dpp[0] = (double) global.resolution.x / 72.0;
	dpp[1] = (double) global.resolution.y / 72.0;


    /* We compensate for the printer's aspect ratio.  If necessary, */
    /* stretching or compressing the horizontal axis.               */

	printdots.y = PRINTFONT->height * global.printscale;
	temp = (double) PRINTFONT->width * (double) global.printscale;
	temp = temp * (double) global.resolution.x / (double) global.resolution.y;
	printdots.x = temp;

	/* This is the number of points per printing font */

	nr_points[0] = (double) printdots.x / dpp[0];
	nr_points[1] = (double) printdots.y / dpp[1];

	global.dispscale = (double) BASEFONT->height / (double) PRINTFONT->height;

    if (ReloadDriver) DeleteDC(hdc);
    return (TRUE);
}



void InitPrinting (void)
{
    SetupPrinter(TRUE);
}


void ChangePrinterOrFonts (void)
{
    FILEOPTIONS *f;
    HCURSOR hcursor;
    HFONT hfont;
    HDC hdc;
    HWND FocusHwnd;

    hcursor = SetCursor(LoadCursor(NULL, IDC_WAIT));
    ShowCursor(TRUE);

    FocusHwnd = SetFocus(NULL);

    SetupPrinter(TRUE);

    hdc = GetPrinterDC(TRUE, NULL);
    if (hdc == NULL) hdc = CreateIC("DISPLAY", NULL, NULL, NULL);

    hfont = SelectAsciiFont(hdc, DefAsciiFont.facename, DefAsciiFont.size, NULL);

    if (DefAsciiFont.hfont != NULL) DeleteObject(DefAsciiFont.hfont);

    DefAsciiFont.hfont =
        FindMatchingScreenFont(hfont, global.dispscale / global.printscale,
                               &(DefAsciiFont.textmetric));
    DeleteObject(hfont);
    DeleteDC(hdc);

    FontCharWidth (0, -1);      /* Reload width tables */

    global.leading = (int) ((double) PRINTFONT->leading * global.dispscale);
    if (global.leading % 2 != 0) global.leading++;

    global.spacing = (int) ((double) PRINTFONT->spacing * global.dispscale);
    if (global.spacing % 2 != 0) global.spacing++;

    for (f = fileoptions; f != NULL; f = f->next) {
        f->leading = global.leading;
        f->spacing = global.spacing;
        f->basefont = BASEFONT;
        f->linelen = CalcLineLength(f);
        ReformatFile(f);

        if (!FindCaret(f, TRUE)) MoveIntoWindow(f);
        InvalidateRect(f->hwnd, NULL, TRUE);
        if(RULER(f).hwnd != NULL) InvalidateRect(RULER(f).hwnd, NULL, TRUE);
    }

    ShowCursor(FALSE);
    SetCursor(hcursor);

    if (FocusHwnd != NULL) SetFocus(FocusHwnd);
}



static void FormatTimeDate (char *timep, char *datep)
{
    time_t t;
    struct tm *tmp;
    static int idate, itime;
    static char sdate[3], stime[3], sam[6], spm[6];
    static BOOL Filled = FALSE;

    if (!Filled || (timep == NULL && datep == NULL)) {
		idate = GetProfileInt("intl", "iDate", 0);
        itime = GetProfileInt("intl", "iTime", 0);

        GetProfileString("intl", "sDate", "/", sdate, 2);
        GetProfileString("intl", "sTime", ":", stime, 2);
        GetProfileString("intl", "s1159", "AM", sam, 5);
        GetProfileString("intl", "s2359", "PM", spm, 5);
        Filled = TRUE;
    }

    time(&t);
	tmp = localtime(&t);

	if (datep != NULL) {
        sprintf(datep, "%d%s%02d%s%02d",
                idate == 1 ? tmp->tm_mday : idate == 2 ? (tmp->tm_year % 100) : (tmp->tm_mon + 1), sdate,
                idate == 1 ? (tmp->tm_mon + 1) : idate == 2 ? (tmp->tm_mon + 1) : tmp->tm_mday, sdate,
                idate == 1 ? (tmp->tm_year % 100) : idate == 2 ? tmp->tm_mday : (tmp->tm_year % 100));
    }
    if (timep != NULL) {
        if (itime == 1) {
            sprintf(timep, "%02d%s%02d", tmp->tm_hour, stime, tmp->tm_min);
        } else {
            sprintf(timep, "%d%s%02d %s",
                    (tmp->tm_hour % 12) ? (tmp->tm_hour % 12) : 12, stime,
                    tmp->tm_min, (tmp->tm_hour / 12) ? spm : sam);
		}
    }
}



static void BuildHeaderString (FILEOPTIONS *f, KANJI far *kp, int PageNum, KANJI far *out)
{
    int i, j, k;
    char cch;
    KANJI kch;
    KANJI far *kp1;

    j = 0;

    for (i = 0; kp[i]; i++) {
        kch = kp[i];
		if (ISKANJI(kch)) cch = LOBYTE(TranslateJAscii(kch, TRUE));
		else cch = LOBYTE(kch);
		if ('a' <= cch && cch <= 'z') cch -= 32;

		if (cch == ESCAPECHAR) {
			i++;
			if (ISKANJI(kp[i])) cch = LOBYTE(TranslateJAscii(kp[i], TRUE));
			else cch = LOBYTE(kp[i]);
			if ('a' <= cch && cch <= 'z') cch -= 32;

			for (k = 0; HeaderStrings[k].escape; k++) {
				if (HeaderStrings[k].escape == cch) break;
			}

			if (!HeaderStrings[k].escape) {
				out[j++] = kch; out[j++] = kp[i]; break;
			} else if (HeaderStrings[k].summary >= 0) {
				kp1 = f->summary[HeaderStrings[k].summary];
				if (kp1 != NULL) for (k = 0; kp1[k]; k++) out[j++] = kp1[k];
            } else {
                char buffer[50];

                switch (HeaderStrings[k].summary) {
                    case -1:    /* Time */
                        FormatTimeDate(buffer, NULL);
                        break;

                    case -2:    /* Date */
                        FormatTimeDate(NULL, buffer);
                        break;

                    case -3:    /* Page */
                        sprintf(buffer, "%d", PageNum);
                        break;

                    case -4:    /* File name */
                        for (k = strlen(f->filename) - 1; k >= 0; k--) {
							if (strchr(":/\\", f->filename[k]) != NULL) break;
                        }
                        strcpy(buffer, f->filename + k + 1);
                        break;
                }

				for (k = 0; buffer[k]; k++) out[j++] = buffer[k];
            }
            continue;
        }

		out[j++] = kch;
    }

	out[j] = 0;
}



static BOOL MoveToNextLine (PARAGRAPH far **pp, ONELINE far **lp, int Direction)
{
    if (Direction > 0) {
        *lp = (*lp)->next;
        if (*lp == NULL) {
            *pp = (*pp)->next;
            if (*pp == NULL) return (FALSE);
            *lp = (*pp)->lines;
        }
        return (TRUE);
    } else if (Direction < 0) {
        *lp = (*lp)->prev;
        if (*lp == NULL) {
            *pp = (*pp)->prev;
            if (*pp == NULL) return (FALSE);
            *lp = (*pp)->lastline;
        }
        return (TRUE);
    } else {
        return (TRUE);
    }
}



// PageNum <> 0: Find the page
// PageNum == 0 and dPage <> 0: Jump pages, starting from pos
// PageNum == 0 and dPage == 0: Find the page containing pos

static int FindPage (FILEOPTIONS *f, int PageLength, int PageNum, int dPage, POSITION *pos)
{
    int i, y;
    int LineHeight;
	PARAGRAPH far *pp;
    ONELINE far *lp;


    LineHeight = (int) floor((double) (PRINTFONT->height + PRINTFONT->spacing) * global.printscale + 0.5);
    POSOF(*pos) = 0;

    /* Find absolute page */

    if (PageNum > 0) {
        i = 1;
        y = 0;

        for (pp = f->paragraph, lp = pp->lines; ; ) {
            if (y + LineHeight >= PageLength) {
                i++;
                y = 0;
            }
            if (i >= PageNum) {
                PARAOF(*pos) = pp;
                LINEOF(*pos) = lp;
                return (i);
            }
            y += LineHeight;
            if (!MoveToNextLine (&pp, &lp, +1)) break;
        }
        return (0);
    }

    /* Skip pages */

    if (dPage != 0) {
        i = y = 0;
        if (dPage > 0) {
            for (pp = PARAOF(*pos), lp = LINEOF(*pos); ; ) {
                if (y + LineHeight >= PageLength) {
                    i++;
                    y = 0;
                }
                if (i >= dPage) {
                    PARAOF(*pos) = pp;
                    LINEOF(*pos) = lp;
                    return (i);
                }
                y += LineHeight;
                if (!MoveToNextLine (&pp, &lp, +1)) break;
            }
        } else {
            for (pp = PARAOF(*pos), lp = LINEOF(*pos); ; ) {
                if (y + LineHeight >= PageLength) {
                    i--;
                    y = 0;
                }
                if (i <= dPage) {
                    PARAOF(*pos) = pp;
                    LINEOF(*pos) = lp;
                    return (i);
                }
                y += LineHeight;
                if (!MoveToNextLine (&pp, &lp, -1)) break;
            }
        }
        return (0);
    }

    /* Find page number */

    i = 1;
    y = 0;

    for (pp = f->paragraph, lp = pp->lines; ; ) {
        if (y + LineHeight >= PageLength) {
            i++;
            y = 0;
        }
        if (pp == PARAOF(*pos) && lp == LINEOF(*pos)) return (i);
        y += LineHeight;
        if (!MoveToNextLine (&pp, &lp, +1)) break;
    }

    return (0);
}



BOOL FAR PASCAL PrintAbortProc (HDC hdc, short nCode)
{
	MSG msg;

    while (!UserAbort && PeekMessage(&msg, NULL, 0, 0, PM_REMOVE)) {
        if (printdlghwnd == NULL || !IsDialogMessage(printdlghwnd, &msg)) {
            TranslateMessage(&msg);
            DispatchMessage(&msg);
        }
    }
    return (!UserAbort);
}



BOOL FAR PASCAL PrintDlgProc (HWND hwnd, WORD message, WORD wParam, LONG lParam)
{
    switch (message) {
        case WM_INITDIALOG:
            CenterDialogBox(hwnd);
            return (TRUE);

		case WM_COMMAND:
            switch (wParam) {
                case IDCANCEL:
                    UserAbort = TRUE;
                    return (TRUE);
            }
            break;
    }

    return (FALSE);
}



static void PrintErrorMessage (int error, char *action)
{
    switch (error) {
        case SP_ERROR:      ErrorMessage(global.hwnd, "Cannot %s!", action); break;
        case SP_APPABORT:
        case SP_USERABORT:  break;
        case SP_OUTOFDISK:  ErrorMessage(global.hwnd, "Out of Disk Space!  Cannot continue printing."); break;
        case SP_OUTOFMEMORY:ErrorMessage(global.hwnd, "Out of Memory!  Cannot continue printing."); break;
    }
}



#define LEFTMG           (MarginDots[0])
#define RIGHTMG          (MarginDots[1])
#define TOPMG            (MarginDots[2])
#define BOTTOMMG         (MarginDots[3])

#define CHECKABORT()     if (!abortprinting(hdc, 0)) break;

void PrintFile (FILEOPTIONS *f)
{
    int i, j, k, h, n, r;
    int x, y, num, temp, size, ScreenLen;
    int PrintWidth, PrintHeight;
    int PrintLeading, PrintSpacing;
    int PrintCell, PageNum, NrPages;
    int WhiteSpaces[2], MarginDots[4];
    int errorcode;
    BOOL ChangeView = FALSE;
    JUSTIFICATION Jmode;
	double ratio;
    UNIT far *up;
    HDC hdc, hdcmem;
    FARPROC abortprinting;
    HFONT hfont;
    HCURSOR hcursor;
    RECT rect;
    PARAGRAPH far *pp;
    ONELINE far *lp;
    POSITION StartPos, NextPos;
    HBITMAP hbitmap;
    TEXTMETRIC tm;
    BYTE far *cbufp;
    char buffer[MAXLINELEN];
    int Widths[256];


    hdc = GetPrinterDC(FALSE, buffer);
    if (hdc == NULL) {
        ErrorMessage(global.hwnd, "Cannot get printer driver!");
        return;
    }
    if (!(GetDeviceCaps(hdc, RASTERCAPS) & RC_BITBLT)) {
        DeleteDC(hdc);
        ErrorMessage(global.hwnd, "Cannot print to the current printer!");
        return;
    }


    /* Initialize stuff */

    SetBkMode(hdc, TRANSPARENT);

	ratio = (double) global.resolution.x / (double) global.resolution.y;
    PrintHeight = floor((double) PRINTFONT->height * global.printscale + 0.5);
    PrintWidth = floor((double) PrintHeight * ratio + 0.5);
	PrintLeading = floor((double) PRINTFONT->leading * global.printscale * ratio + 0.5);
    PrintSpacing = floor((double) PRINTFONT->spacing * global.printscale + 0.5);
    PrintCell = PrintWidth + PrintLeading;

	hbitmap = CreateBitmap(PRINTFONT->width, PRINTFONT->height, 1, 1, NULL);
    hdcmem = CreateCompatibleDC(hdc);
	SetMapMode(hdcmem, GetMapMode(hdc));

    /* Load the ASCII font */

    hfont = SelectAsciiFont(hdc, DefAsciiFont.facename, DefAsciiFont.size, &tm);
    SelectObject(hdc, hfont);
    GetCharWidth(hdc, 0, 255, Widths);

    WhiteSpaces[0] = ((global.paper[0] * global.resolution.x) - global.printable.x) / 2;
	WhiteSpaces[1] = ((global.paper[1] * global.resolution.y) - global.printable.y) / 2;

	MarginDots[0] = (f->pagesetup.margins[0] * global.resolution.x) - WhiteSpaces[0];
	MarginDots[1] = (f->pagesetup.margins[1] * global.resolution.x) - WhiteSpaces[0];
	MarginDots[2] = (f->pagesetup.margins[2] * global.resolution.y) - WhiteSpaces[1];
	MarginDots[3] = (f->pagesetup.margins[3] * global.resolution.y) - WhiteSpaces[1];


    /* Set the abort procedures */

    abortprinting = MakeProcInstance(PrintAbortProc, hInstance);

    EnableWindow(global.hwnd, FALSE);
    EnableWindow(global.clienthwnd, FALSE);

    UserAbort = PrintError = FALSE;
    printdlghwnd = CreateDialog(hInstance, "Printing", f->parent, PrintDlgProc);

    hcursor = SetCursor(LoadCursor(NULL, IDC_WAIT));
    ShowCursor(TRUE);


    /* Reformat? */

    if (global.draftview) {
        SetDlgItemText(printdlghwnd, 4201, "Reformatting File...");
        abortprinting(hdc, 0);
        global.draftview = FALSE;
        ReformatFile(f);
        ChangeView = TRUE;
    }


    /* Pagenate */

    SetDlgItemText(printdlghwnd, 4201, "Repagenating File...");
    abortprinting(hdc, 0);

    if (PrintFrom < 0 && PrintTo < 0) {     /* Current Page */
        StartPos = f->current;
        PageNum = FindPage(f, global.printable.y - TOPMG - BOTTOMMG, 0, 0, &StartPos);
        i = FindPage(f, global.printable.y - TOPMG - BOTTOMMG, PageNum, 0, &StartPos);
        if (i == 0) {
            ErrorMessage(global.hwnd, "There %s less than %d page%s in this document!",
                            (PrintFrom > 1) ? "are" : "is", PrintFrom, (PrintFrom > 1) ? "s" : "");
            goto CleanUp;
        }
        NrPages = 1;
    } else {
        if (PrintFrom <= 0) {
            PARAOF(StartPos) = f->paragraph;
            LINEOF(StartPos) = f->paragraph->lines;
			PageNum = 1;
        } else {
            i = FindPage(f, global.printable.y - TOPMG - BOTTOMMG, PrintFrom, 0, &StartPos);
			if (i == 0) {
                ErrorMessage(global.hwnd, "There %s less than %d page%s in this document!",
                                (PrintFrom > 1) ? "are" : "is", PrintFrom, (PrintFrom > 1) ? "s" : "");
                goto CleanUp;
            }
            PageNum = PrintFrom;
        }

        if (PrintTo <= 0) {
			NrPages = -1;
        } else if (PrintFrom <= 0) {
            NrPages = PrintTo;
		} else {
            NrPages = PrintTo - PrintFrom + 1;
            if (NrPages <= 0) NrPages = 1;
        }
    }

    NextPos = StartPos;
    i = FindPage(f, global.printable.y - TOPMG - BOTTOMMG, 0, +1, &NextPos);
    if (i == 0) PARAOF(NextPos) = NULL;


    if (!(f->filename[0])) {
        strcpy(buffer, "Printing ");
        GetWindowText(f->parent, buffer + strlen(buffer), MAXLINELEN);
        sprintf(buffer + strlen(buffer), "\n\n%s", global.printer);
    } else {
        sprintf(buffer, "Printing %s\n\n%s", f->filename, global.printer);
    }

    SetDlgItemText(printdlghwnd, 4201, buffer);

    SetDlgItemText(printdlghwnd, 4202, "Initializing printer...");
    abortprinting(hdc, 0);

    sprintf(buffer, "%s ", PROGNAME);
    GetWindowText(f->parent, buffer + strlen(buffer), MAXLINELEN);

    errorcode = Escape(hdc, STARTDOC, strlen(buffer), buffer, NULL);
    if (errorcode <= 0) {
        PrintErrorMessage(errorcode, "initialize printer");
        goto CleanUp;
    }

    Escape(hdc, SETABORTPROC, 0, (LPSTR) abortprinting, NULL);


    /* Do the true printing */

Loop:

    sprintf(buffer, "Currently Printing Page %d...", PageNum);
    SetDlgItemText(printdlghwnd, 4202, buffer);
    abortprinting(hdc, 0);

	/* Process each band */

    errorcode = Escape(hdc, NEXTBAND, 0, NULL, (LPSTR) &rect);
    if (errorcode < 0) {
        PrintError = TRUE;
        PrintErrorMessage(errorcode, "continue printing");
    }

    SelectObject(hdc, hfont);
    SetBkMode(hdc, TRANSPARENT);

    while (!PrintError && !UserAbort && !IsRectEmpty(&rect)) {
        /* Header / Footer ? */

        CHECKABORT();

		for (h = 0; h < 2; h++) {
			KANJI kbuffer[MAXLINELEN];

            if (f->nofirstpage && PageNum <= 1) break;

			switch (h) {
                default: continue;

                case 0: y = TOPMG - PrintHeight - PrintSpacing;
                        if (rect.top >= y) continue;
                        y -= PrintHeight;
                        if (f->lrheader) {
                            num = (PageNum % 2 == 0) ? 1 : 0;
                        } else {
                            num = 0;
						}
                        break;

                case 1: y = global.printable.y - BOTTOMMG + PrintHeight + PrintSpacing;
                        if (rect.bottom <= y) continue;
                        if (f->lrheader) {
                            num = (PageNum % 2 == 0) ? 2 : 3;
                        } else {
                            num = 2;
                        }
                        break;
            }

			for (i = 0; i < NRHEADERS; i++) {
				if (f->header[num][i] == NULL) continue;
                CHECKABORT();

				BuildHeaderString (f, f->header[num][i], PageNum, kbuffer);

				n = kanjilen(kbuffer);
				if (n <= 0) continue;

				for (j = k = 0; kbuffer[j]; j++) {
                    CHECKABORT();
					if (ISKANJI(kbuffer[j])) {
						k += PrintCell;
                    } else {
						k += Widths[kbuffer[j]];
                        if (buffer[j+1] && ISKANJI(kbuffer[j+1])) k += PrintLeading;
                    }
                }

                switch (i) {
                    default:
                    case 0: x = LEFTMG; break;
                    case 1: x = (f->linelen * PrintCell - PrintLeading - k) / 2 + LEFTMG; break;
                    case 2: x = f->linelen * PrintCell - PrintLeading - k + LEFTMG; break;
                }

				for (j = 0; kbuffer[j]; j++) {
                    CHECKABORT();
					if (ISKANJI(kbuffer[j])) {
						r = Jis2Index(kbuffer[j], PRINTFONT->holes);
                        if (r < 0) r = Jis2Index(BADKANJI, PRINTFONT->holes);
                        r = GetKanjiBitmap(PRINTFONT, r, &cbufp);
						SetBitmapBits(hbitmap, (DWORD) r, cbufp);
                        SelectObject(hdcmem, hbitmap);
                        StretchBlt(hdc, x, y, PrintWidth, PrintHeight, hdcmem, 0, 0, PRINTFONT->width, PRINTFONT->height, SRCCOPY);
                        x += PrintCell;
                    } else {
                        TextOut(hdc, x, y + PrintHeight - tm.tmHeight, kbuffer + j, 1);
						x += Widths[kbuffer[j]];
						if (kbuffer[j+1] && ISKANJI(kbuffer[j+1])) x += PrintLeading;
					}
				}
            }
        }

        if (UserAbort) break;

        pp = PARAOF(StartPos);
        lp = LINEOF(StartPos);
        y = TOPMG;

		for (;;) {
            if (pp == PARAOF(NextPos) && lp == LINEOF(NextPos)) break;

            if (y > rect.bottom) break;

            CHECKABORT();

            if (y + PrintHeight + PrintSpacing < rect.top) {
				if (!MoveToNextLine(&pp, &lp, +1)) break;
                y += PrintHeight + PrintSpacing;
                continue;
            }

            x = ((lp == pp->lines) ? pp->firstindent : pp->leftindent) * PrintCell + LEFTMG;
			ScreenLen = ((lp == pp->lines) ? pp->firstindent : pp->leftindent) * BASEWIDTH(f);

            up = pp->text + lp->position;

			for (i = 0; i < lp->length; i++) {
                if (x > rect.right) break;

                CHECKABORT();

                if (ISKANJI(up->kanji) && x + PrintCell < rect.left) {
                    x += PrintCell;
					ScreenLen += BASEWIDTH(f);
                    up++;
                    continue;
                }

                if (up->kanji == 0x2121 /* Blank */) {
                    x += PrintCell;
                    ScreenLen += BASEWIDTH(f);
                    up++;
                } else if (ISKANJI(up->kanji)) {
                    r = Jis2Index(up->kanji, PRINTFONT->holes);
                    if (r < 0) r = Jis2Index(BADKANJI, PRINTFONT->holes);
                    r = GetKanjiBitmap(PRINTFONT, r, &cbufp);
					SetBitmapBits(hbitmap, (DWORD) r, cbufp);
					SelectObject(hdcmem, hbitmap);
					StretchBlt(hdc, x, y, PrintWidth, PrintHeight, hdcmem, 0, 0, PRINTFONT->width, PRINTFONT->height, SRCCOPY);
                    x += PrintCell;
					ScreenLen += BASEWIDTH(f);
                    up++;
                } else {
					/* Get the text string */

                    SetTextAlign(hdc, TA_LEFT | TA_BOTTOM | TA_NOUPDATECP);
                    SetTextJustification(hdc, 0, 0);

                    ratio = (double) global.resolution.y / (double) global.resolution.x;
                    ratio *= global.dispscale / global.printscale;

                    for (j = k = 0; i < lp->length; i++, j++, up++) {
						if (ISKANJI(up->kanji)) break;
						if (up->kanji == '\t') break;

						buffer[j] = LOBYTE(up->kanji);

                        //temp = FontCharWidth(up->kanji, 1) - FontDisplayError(up->kanji, pp, lp);
                        temp = FontCharWidth(up->kanji, 1) - FontDisplayError(up->kanji, i);

                        ScreenLen += temp;
                        k += Widths[up->kanji];
					}
					buffer[j] = 0;

                    Jmode = global.shortjust;

					if (i >= lp->length || i <= j) {    /* Open ended */
                        Jmode = global.longjust;
                        if (Jmode == J_RIGHT || Jmode == J_CENTER) Jmode = J_JUSTIFY;
                        if (lp->next == NULL && Jmode == J_JUSTIFY) Jmode = J_LEFT;

                        if (buffer[0] && i >= lp->length) {
                            for (temp = strlen(buffer) - 1; temp >= 0 && buffer[temp] == ' '; temp--);
                            buffer[++temp] = '\0';
                            k -= (j - temp) * Widths[' '];
                        }
					} else if (up->kanji != '\t') {     /* In between */
                        Jmode = J_LEFT;
					}

                    k = LOWORD(GetTextExtent(hdc, buffer, strlen(buffer)));

					/* Now, get the amount of tab */

                    if (Jmode != J_LEFT) {
						if (i >= lp->length) {
							ScreenLen = (f->linelen - pp->rightindent) * BASEWIDTH(f) - global.leading;
                            r = (f->linelen - pp->rightindent) * PrintCell - PrintLeading - x - k + LEFTMG;
						} else {
							ScreenLen += BASEWIDTH(f) - (ScreenLen % BASEWIDTH(f));
                            r = (ScreenLen / BASEWIDTH(f)) * PrintCell - PrintLeading - x - k + LEFTMG;
						}
                    } else if (up->kanji == '\t') {
                        ScreenLen += BASEWIDTH(f) - (ScreenLen % BASEWIDTH(f));
                        r = (ScreenLen / BASEWIDTH(f)) * PrintCell - PrintLeading - x - k + LEFTMG;
					} else {
						r = 0;
					}

                    if (r < 0) Jmode = J_JUSTIFY;

					switch (Jmode) {
						LeftText:
                        case J_LEFT:
                            CHECKABORT();
                            TextOut(hdc, x, y + PrintHeight, buffer, strlen(buffer));
                            x += k + r;
							break;

						RightText:
                        case J_RIGHT:
                            CHECKABORT();
                            TextOut(hdc, x + r, y + PrintHeight, buffer, strlen(buffer));
                            x += k + r;
							break;

						CenterText:
                        case J_CENTER:
                            TextOut(hdc, x + r/2, y + PrintHeight, buffer, strlen(buffer));
                            x += k + r;
							break;

                        case J_JUSTIFY:
							/* Count the nuber of spaces */
                            for (temp = n = 0; buffer[temp]; temp++)
                                if (buffer[temp] == ' ') n++;

							if (n > 0) {
                                if (r > Widths[' ']) {
                                    if (buffer[0] != ' ' && i > j) {
                                        x += Widths[' ']/2;
                                        r -= Widths[' ']/2;
                                    }
                                    if (i < lp->length) {
                                        SetTextJustification(hdc, r - Widths[' ']/2, n);
                                    } else {
                                        SetTextJustification(hdc, r, n);
                                    }
                                } else {
                                    SetTextJustification(hdc, r, n);
                                }
                                CHECKABORT();
                                TextOut(hdc, x, y + PrintHeight, buffer, strlen(buffer));
                                x += k + r;
                            } else if (i >= lp->length) {
                                if (i <= j) goto LeftText; else goto RightText;
							} else {
                                if (i <= j) goto LeftText; else goto CenterText;
							}
							break;
                    }

					if (i >= lp->length) break;
					if (ISKANJI(up->kanji)) {
						i--;
						x += PrintLeading;
						ScreenLen += global.leading;
					}
					if (up->kanji == '\t') {
                        x += PrintLeading;
						up++;
					}
				}
			}

			y += PrintHeight + PrintSpacing;

            CHECKABORT();

			if (!MoveToNextLine(&pp, &lp, +1)) break;
		}

        errorcode = Escape(hdc, NEXTBAND, 0, NULL, (LPSTR) &rect);
        if (errorcode < 0) {
            PrintError = TRUE;
            PrintErrorMessage(errorcode, "continue printing");
            break;
        }
	}

    /* Another page? */

    if (!UserAbort && !PrintError && (NrPages < 0 || NrPages > 1) && PARAOF(NextPos) != NULL) {
        if (NrPages > 0) NrPages--;
        StartPos = NextPos;
        i = FindPage(f, global.printable.y - TOPMG - BOTTOMMG, 0, +1, &NextPos);
        if (i != 0) PageNum++; else PARAOF(NextPos) = NULL;
        goto Loop;
    }

	/* Finishes the accounting */

    if (!PrintError) {
        if (UserAbort) {
            SetDlgItemText(printdlghwnd, 4202, "Printing Aborted!");
            abortprinting(hdc, 0);
            Escape(hdc, ABORTDOC, 0, NULL, NULL);
        } else {
            SetDlgItemText(printdlghwnd, 4202, "Printing Completed.");
            abortprinting(hdc, 0);
            Escape(hdc, ENDDOC, 0, NULL, NULL);
        }
    }


CleanUp:

    /* Clean up... */

    if (ChangeView) {
        SetDlgItemText(printdlghwnd, 4201, "Reformatting File...");
        abortprinting(hdc, 0);
        global.draftview = TRUE;
        ReformatFile(f);
    }

    EnableWindow(global.hwnd, TRUE);
    EnableWindow(global.clienthwnd, TRUE);

    DestroyWindow(printdlghwnd);
    printdlghwnd = NULL;

    FreeProcInstance(abortprinting);

    ShowCursor(FALSE);
    SetCursor(hcursor);

    if (hdcmem != NULL) DeleteDC(hdcmem);
    if (hdc != NULL) DeleteDC(hdc);
    if (hbitmap != NULL) DeleteObject(hbitmap);
    if (hfont != NULL) DeleteObject(hfont);
}



BOOL FAR PASCAL PrinterSetupProc (HWND hwnd, WORD message, WORD wParam, LONG lParam)
{
    static char *Devices = NULL;
    static char *CurDevice = NULL;
    static HANDLE library = NULL;

    switch (message) {
        case WM_INITDIALOG: {
            int i, j;
			char printer[MAXLINELEN];
            char text[MAXLINELEN];
            char *driver, *output;

            if (Devices != NULL) FreeMem(Devices);
            Devices = MemAlloc(MAXLINELEN);
            CurDevice = NULL;

            EnableWindow(GetDlgItem(hwnd, 4201), FALSE);

            GetProfileString("devices", NULL, "", Devices, MAXLINELEN);
            for (i = 0, j = 0;; j++) {
                if (!Devices[i]) break;
				GetProfileString("devices", Devices+ i, "", printer, MAXLINELEN);
                driver = strtok(printer, ", ");
                output = strtok(NULL, ", ");
                sprintf(text, "%s on %s", Devices + i, output);
                SendDlgItemMessage(hwnd, 4211, LB_ADDSTRING, 0, (LONG) ((char far *) text));
                if (!stricmp(text, global.printer)) {
                    CurDevice = Devices + i;
                    SendDlgItemMessage(hwnd, 4211, LB_SETCURSEL, j, 0L);
                    EnableWindow(GetDlgItem(hwnd, 4201), TRUE);
                }
                i += strlen(Devices + i) + 1;
            }

			CenterDialogBox(hwnd);

            SetFocus(GetDlgItem(hwnd, 4211));

            return (TRUE);
        }

        case WM_COMMAND:
            switch (wParam) {
                case 4211:      /* List box */
					switch (HIWORD(lParam)) {
                        case LBN_SELCHANGE: {
                            int i, j;

                            i = SendDlgItemMessage(hwnd, 4211, LB_GETCURSEL, 0, 0L);
                            if (i == LB_ERR) {
                                MessageBeep(0);
                                return (TRUE);
                            }

                            /* Find the device */

                            CurDevice = Devices;
                            for (j = 0; j < i; j++) {
                                if (!CurDevice[0]) {
                                    CurDevice = Devices;
                                    MessageBeep(0);
                                    return (TRUE);
                                }
                                CurDevice += strlen(CurDevice) + 1;
                            }
                            EnableWindow(GetDlgItem(hwnd, 4201), TRUE);
                            return (TRUE);
                        }

                        case LBN_DBLCLK:
                            SendMessage(hwnd, WM_COMMAND, IDOK, 0L);
                            return (TRUE);
                    }
                    return (TRUE);

                case 4201: {    /* Options */
                    char driver[MAXLINELEN];
                    char driverfile[MAXLINELEN];
                    char *output;
                    FARPROC devicemode;

                    GetProfileString("devices", CurDevice, "", driver, MAXLINELEN);
                    (void) strtok(driver, ", ");
                    output = strtok(NULL, ", ");
                    sprintf(driverfile, "%s.DRV", driver);

                    if (library >= 32) FreeLibrary(library);
                    library = LoadLibrary(driverfile);
                    if (library >= 32) {
                        devicemode = GetProcAddress(library, "DEVICEMODE");
                        devicemode(hwnd, library, (LPSTR) driver, (LPSTR) output);
                    }
					return (TRUE);
				}

                case IDOK: {
                    char buffer[MAXLINELEN];

                    if (CurDevice != NULL) {
                        sprintf(buffer, "%s,", CurDevice);
                        GetProfileString("devices", CurDevice, "", buffer + strlen(buffer), MAXLINELEN);
                        WriteProfileString("windows", "device", buffer);
                        FreeMem(CurDevice);
                        ChangePrinterOrFonts();
                    }
                    CurDevice = NULL;
                    EndDialog(hwnd, TRUE);
                    return (TRUE);
                }

                case IDCANCEL:
                    if (CurDevice != NULL) FreeMem(CurDevice);
                    CurDevice = NULL;
                    EndDialog(hwnd, FALSE);
                    return (TRUE);
            }
            break;

        case WM_DESTROY:
            if (library >= 32) FreeLibrary(library);
            library = NULL;
            return (TRUE);
    }

    return (FALSE);
}



BOOL FAR PASCAL PageSetupProc (HWND hwnd, WORD message, WORD wParam, LONG lParam)
{
	static MEASUREMENT OldMeasure;

	switch (message) {
		case WM_INITDIALOG: {
            int i;
			MEASUREINFO *mp;
			char buffer[MAXLINELEN];
			char format[30];

			SetDlgItemText(hwnd, 4201, global.printer);

            /* Page orientation */

			SendDlgItemMessage(hwnd, 4211, CB_ADDSTRING, 0, (LONG) "Portrait");
			SendDlgItemMessage(hwnd, 4211, CB_ADDSTRING, 0, (LONG) "Landscape");
            SendDlgItemMessage(hwnd, 4211, CB_SETCURSEL, global.pagesetup.landscape ? 1 : 0, 0L);

            /* Measurements */

            for (i = 0; Measurements[i].name != NULL; i++) {
				SendDlgItemMessage(hwnd, 4212, CB_ADDSTRING, 0, (LONG) Measurements[i].name);
            }
            mp = Measurements + global.measure;
            SendDlgItemMessage(hwnd, 4212, CB_SETCURSEL, global.measure, 0L);

            /* Line length */

			if (global.active != NULL) {
                SetDlgItemInt(hwnd, 4213, global.active->linelen, TRUE);
            } else {
                SetDlgItemInt(hwnd, 4213, CalcLineLength(global.active), TRUE);
			}
            /* Margins */

            sprintf(format, "%%%d.%dlf%%s", mp->decimal + 2, mp->decimal);

            if (global.active != NULL) {
				for (i = 0; i < 4; i++) {
                    sprintf(buffer, format, global.active->pagesetup.margins[i] * mp->conversion, "");
                    SetDlgItemText(hwnd, 4221 + i, buffer);
				}
            } else {
                for (i = 0; i < 4; i++) {
                    sprintf(buffer, format, global.pagesetup.margins[i] * mp->conversion, "");
					SetDlgItemText(hwnd, 4221 + i, buffer);
                }
            }

            for (i = 0; i < 4; i++) {
                sprintf(buffer, format, global.pagesetup.margins[i] * mp->conversion, mp->symbol);
				SetDlgItemText(hwnd, 4231 + i, buffer);
			}

			OldMeasure = global.measure;

            CenterDialogBox(hwnd);
            return (TRUE);
        }

        case WM_COMMAND: {
            int i, j;
            double num, temp, minmargins[4];
            int NeedReformat;   /* 0 = no, 1 = file, 2 = all */
			MEASUREINFO *mp, *mp1;
            FILEOPTIONS *f;
			char buffer[MAXLINELEN];
			char format[30];

            minmargins[0] = minmargins[1] = ((global.paper[0] * global.resolution.x) - global.printable.x) / 2;
            minmargins[2] = minmargins[3] = ((global.paper[1] * global.resolution.y) - global.printable.y) / 2;

            switch (wParam) {
				case 4241:      /* Make default */
                    if (YesNo(hwnd, "Make these margin settings the default?") == IDYES) {
						mp = Measurements + OldMeasure;
                        sprintf(format, "%%%d.%dlf%%s", mp->decimal + 2, mp->decimal);

						for (i = 0; i < 4; i++) {
							GetDlgItemText(hwnd, 4221 + i, buffer, MAXLINELEN);
							num = atof(buffer);

                            switch (i) {
                                case 0:
                                case 1: temp = (double) minmargins[i] / (double) global.resolution.x;
                                        break;
                                case 2:
                                case 3: temp = (double) minmargins[i] / (double) global.resolution.x;
                                        break;
                            }

                            if (num < temp) num = temp;

							sprintf(buffer, format, num, mp->symbol);
							SetDlgItemText(hwnd, 4231 + i, buffer);
						}
					}
					return (TRUE);

				case 4212:      /* Measurement */
					j = SendDlgItemMessage(hwnd, 4212, CB_GETCURSEL, 0, 0L);
					if (j == CB_ERR) return (TRUE);
                    if (j == OldMeasure) return (TRUE);

					mp = Measurements + OldMeasure;
					mp1 = Measurements + j;
					sprintf(format, "%%%d.%dlf%%s", mp1->decimal + 2, mp1->decimal);

					for (i = 0; i < 4; i++) {
						GetDlgItemText(hwnd, 4221 + i, buffer, MAXLINELEN);
						num = atof(buffer);

                        switch (i) {
                            case 0:
                            case 1: temp = (double) minmargins[i] / (double) global.resolution.x;
                                    break;
                            case 2:
                            case 3: temp = (double) minmargins[i] / (double) global.resolution.x;
                                    break;
						}

                        if (num < temp) num = temp;

                        sprintf(buffer, format, num * mp1->conversion / mp->conversion, "");
                        SetDlgItemText(hwnd, 4221 + i, buffer);
					}
					for (i = 0; i < 4; i++) {
                        GetDlgItemText(hwnd, 4231 + i, buffer, MAXLINELEN);
						num = atof(buffer);

                        switch (i) {
                            case 0:
							case 1: temp = (double) minmargins[i] / (double) global.resolution.x;
                                    break;
                            case 2:
                            case 3: temp = (double) minmargins[i] / (double) global.resolution.x;
                                    break;
                        }

                        if (num < temp) num = temp;

                        sprintf(buffer, format, num * mp1->conversion / mp->conversion, mp1->symbol);
                        SetDlgItemText(hwnd, 4231 + i, buffer);
					}

					OldMeasure = j;

					num = (double) (PRINTFONT->width + PRINTFONT->leading) * global.printscale;
					num *= (double) global.resolution.x / (double) global.resolution.y;
					SetDlgItemInt(hwnd, 4213, (int) floor((double) global.printable.x / num), TRUE);
					return (TRUE);

                case IDOK:
                    NeedReformat = 0;

                    /* Orientation */

					i = SendDlgItemMessage(hwnd, 4211, CB_GETCURSEL, 0, 0L);
                    if (i != CB_ERR && i != global.active->pagesetup.landscape) {
						OptionsChanged = TRUE;
                        global.active->pagesetup.landscape = i;
                        NeedReformat = 2;
                    }

                    /* Measurement */

                    i = SendDlgItemMessage(hwnd, 4212, CB_GETCURSEL, 0, 0L);
                    if (i != CB_ERR && i != global.measure) {
                        OptionsChanged = TRUE;
                        global.measure = i;
                    }

                    /* Margins */

                    if (global.active != NULL) {
                        for (i = 0; i < 4; i++) {
                            GetDlgItemText(hwnd, 4221 + i, buffer, MAXLINELEN);
                            num = atof(buffer);

                            switch (i) {
                                case 0:
                                case 1: temp = (double) minmargins[i] / (double) global.resolution.x;
                                        break;
                                case 2:
                                case 3: temp = (double) minmargins[i] / (double) global.resolution.x;
                                        break;
                            }

                            if (num < temp) num = temp;
                            num /= Measurements[global.measure].conversion;

                            if (fabs(num - global.active->pagesetup.margins[i]) > 0.01) {
                                global.active->pagesetup.margins[i] = num;
                                NeedReformat = 1;
                                OptionsChanged = TRUE;
                            }
                        }
                    }

                    for (i = 0; i < 4; i++) {
                        GetDlgItemText(hwnd, 4231 + i, buffer, MAXLINELEN);
                        num = atof(buffer);

                        switch (i) {
                            case 0:
                            case 1: temp = (double) minmargins[i] / (double) global.resolution.x;
                                    break;
                            case 2:
                            case 3: temp = (double) minmargins[i] / (double) global.resolution.x;
                                    break;
                        }

                        if (num < temp) num = temp;
                        num /= Measurements[global.measure].conversion;

                        if (fabs(num - global.pagesetup.margins[i]) > 0.01) {
                            global.pagesetup.margins[i] = num;
                            OptionsChanged = TRUE;
                        }
                    }

                    switch (NeedReformat) {
                        case 0: break;

                        case 1: if (global.active != NULL) {
                                    global.active->linelen = CalcLineLength(global.active);
                                    ReformatFile(global.active);
                                    InvalidateRect(global.active->hwnd, NULL, TRUE);
                                }
                                break;

                        case 2: for (f = fileoptions; f != NULL; f = f->next) {
                                    f->linelen = CalcLineLength(f);
                                    ReformatFile(f);
                                    InvalidateRect(f->hwnd, NULL, TRUE);
                                }
                                break;
                    }

                    EndDialog(hwnd, NeedReformat);
                    return (TRUE);

                case IDCANCEL:
                    EndDialog(hwnd, 0);
                    return (TRUE);
            }
            break;
		}
		break;
    }
    return (FALSE);
}



BOOL FAR PASCAL PrintProc (HWND hwnd, WORD message, WORD wParam, LONG lParam)
{
    switch (message) {
        case WM_INITDIALOG: {
            char buffer[MAXLINELEN];
            char format[30];
            MEASUREINFO *mp;

            SetDlgItemText(hwnd, 4211, global.printer);
            SetDlgItemText(hwnd, 4212, global.active->pagesetup.landscape ? "Landscape" : "Portrait");

            mp = Measurements + global.measure;
            sprintf(format, "%%%d.%dlf%s x %%%d.%dlf%s",
                        mp->decimal + 2, mp->decimal, mp->symbol,
                        mp->decimal + 2, mp->decimal, mp->symbol);
            sprintf(buffer, format, global.paper[0] * mp->conversion, global.paper[1] * mp->conversion);
            SetDlgItemText(hwnd, 4213, buffer);

            SendDlgItemMessage(hwnd, 4221, BM_SETCHECK, TRUE, 0L);
            SendDlgItemMessage(hwnd, 4222, BM_SETCHECK, FALSE, 0L);
            SendDlgItemMessage(hwnd, 4223, BM_SETCHECK, FALSE, 0L);

            EnableWindow(GetDlgItem(hwnd, 4231), FALSE);
            EnableWindow(GetDlgItem(hwnd, 4232), FALSE);

            SendDlgItemMessage(hwnd, 4241, CB_ADDSTRING, 0, (LONG) "Left");
            SendDlgItemMessage(hwnd, 4241, CB_ADDSTRING, 0, (LONG) "Right");
            SendDlgItemMessage(hwnd, 4241, CB_ADDSTRING, 0, (LONG) "Center");
            SendDlgItemMessage(hwnd, 4241, CB_ADDSTRING, 0, (LONG) "Justify");
            switch (global.shortjust) {
                case J_LEFT:    SendDlgItemMessage(hwnd, 4241, CB_SETCURSEL, 0, 0L); break;
                case J_RIGHT:   SendDlgItemMessage(hwnd, 4241, CB_SETCURSEL, 1, 0L); break;
                default:
                case J_CENTER:  SendDlgItemMessage(hwnd, 4241, CB_SETCURSEL, 2, 0L); break;
                case J_JUSTIFY: SendDlgItemMessage(hwnd, 4241, CB_SETCURSEL, 3, 0L); break;
            }

            SendDlgItemMessage(hwnd, 4242, CB_ADDSTRING, 0, (LONG) "Left");
            SendDlgItemMessage(hwnd, 4242, CB_ADDSTRING, 0, (LONG) "Justify");
            switch (global.longjust) {
                case J_LEFT:    SendDlgItemMessage(hwnd, 4242, CB_SETCURSEL, 0, 0L); break;
                default:
                case J_JUSTIFY: SendDlgItemMessage(hwnd, 4242, CB_SETCURSEL, 1, 0L); break;
            }

            CenterDialogBox(hwnd);
            return (TRUE);
        }

        case WM_COMMAND:
            switch (wParam) {
                case 4221:      /* All */
                    SendDlgItemMessage(hwnd, 4221, BM_SETCHECK, TRUE, 0L);
                    SendDlgItemMessage(hwnd, 4222, BM_SETCHECK, FALSE, 0L);
                    SendDlgItemMessage(hwnd, 4223, BM_SETCHECK, FALSE, 0L);
                    EnableWindow(GetDlgItem(hwnd, 4231), FALSE);
                    EnableWindow(GetDlgItem(hwnd, 4232), FALSE);
                    return (TRUE);

                case 4222:      /* Current */
                    SendDlgItemMessage(hwnd, 4221, BM_SETCHECK, FALSE, 0L);
                    SendDlgItemMessage(hwnd, 4222, BM_SETCHECK, TRUE, 0L);
                    SendDlgItemMessage(hwnd, 4223, BM_SETCHECK, FALSE, 0L);
                    EnableWindow(GetDlgItem(hwnd, 4231), FALSE);
                    EnableWindow(GetDlgItem(hwnd, 4232), FALSE);
                    return (TRUE);

                case 4223:      /* Selected */
                    SendDlgItemMessage(hwnd, 4221, BM_SETCHECK, FALSE, 0L);
                    SendDlgItemMessage(hwnd, 4222, BM_SETCHECK, FALSE, 0L);
                    SendDlgItemMessage(hwnd, 4223, BM_SETCHECK, TRUE, 0L);
                    EnableWindow(GetDlgItem(hwnd, 4231), TRUE);
                    EnableWindow(GetDlgItem(hwnd, 4232), TRUE);
                    return (TRUE);

                case 4201:      /* Page setup */
                    DialogBox(hInstance, "PageSetup", hwnd, PageSetupProc);
                    return (TRUE);

                case 4202:      /* Printer setup */
                    DialogBox(hInstance, "PrinterSetup", hwnd, PrinterSetupProc);
                    return (TRUE);

                case IDOK: {
                    int i;
                    char buffer[MAXLINELEN];

                    i = SendDlgItemMessage(hwnd, 4241, CB_GETCURSEL, 0, 0L);
                    if (i != CB_ERR) {
                        switch (i) {
                            case 0: global.shortjust = J_LEFT; break;
                            case 1: global.shortjust = J_RIGHT; break;
                            case 2: global.shortjust = J_CENTER; break;
                            case 3: global.shortjust = J_JUSTIFY; break;
                        }
                    }
                    i = SendDlgItemMessage(hwnd, 4242, CB_GETCURSEL, 0, 0L);
                    if (i != CB_ERR) {
                        switch (i) {
                            case 0: global.longjust = J_LEFT; break;
                            case 1: global.longjust = J_JUSTIFY; break;
                        }
                    }

                    if (IsDlgButtonChecked(hwnd, 4221)) {
                        PrintFrom = PrintTo = 0;
                    } else if (IsDlgButtonChecked(hwnd, 4222)) {
                        PrintFrom = PrintTo = -1;
                    } else {
                        GetDlgItemText(hwnd, 4231, buffer, MAXLINELEN);
                        PrintFrom = atoi(buffer);
                        if (PrintFrom < 0) PrintFrom = 0;
                        GetDlgItemText(hwnd, 4232, buffer, MAXLINELEN);
                        PrintTo = atoi(buffer);
                        if (PrintTo < PrintFrom) PrintTo = PrintFrom;
                    }
                    EndDialog(hwnd, TRUE);
                    return (TRUE);
                }

                case IDCANCEL:
                    EndDialog(hwnd, FALSE);
                    return (TRUE);
            }
            break;
    }
    return (FALSE);
}
