#include "pt.h"
#include "string.h"

extern struct window *windows;
extern struct window *windowList;

void pascal
/* XTAG:initWindows */
initWindows()
{
	extern struct window *hiddenList;
	extern unsigned char charTable[];
	extern int maxFiles;

	register int i;

	for(i = 0; i < maxFiles; i++)
		windows[i].state = 0;
	windowList = NULL;
	hiddenList = NULL;
	charTable['\n'] = 1;
	charTable[0] = 2;
	charTable['\t'] = 3;
	charTable['\r'] = 4;
#ifdef ANASAZI
	/* the anasazi display mode escape character */
	charTable[0xFE] = (unsigned char)(anasazi ? 5 : 0);
#endif
}

struct window * pascal
/* XTAG:createWindow */
createWindow(fileName, row1, col1, row2, col2, topOrBottom, windowType)
	unsigned char *fileName;
	int row1, col1, row2, col2, topOrBottom, windowType;
{
	extern struct window *activeWindow;
	extern struct window *selWindow;
	extern long selBegin, selEnd;
	extern unsigned char msgBuffer[];
	extern unsigned char textColor, selColor;
	extern unsigned char bannerColor, borderColor, elevColor;
	extern int scrRows, scrCols;
	extern struct openFile *files;
	extern unsigned char *userMessages[];
	extern int maxFiles;
	extern int debug;

	register struct window *w;
	struct window *w1;
	register int i;

	/* find an unused window structure */
	w = NULL;
	for(i = 0; i < maxFiles; i++)
		if( windows[i].state == 0) {
			w = &windows[i];
			break;
		}
	if( w == NULL ) {
		msg(userMessages[OUTOFWINDOWS], 3);
		return NULL;
	}

	/* open file to be in the window */
	w->fileId = getFileId(fileName);
	if( w->fileId == -1 )	/* file not found, no not open the window */
		return NULL;
	w->nameOffset = getBaseName(files[w->fileId].origName);

	/* insert w into the doubly linked windowList */
	if( windowList == NULL ) {
		w->prevWindow = NULL;
		w->nextWindow = NULL;
		windowList = w;
	} else {
		if( topOrBottom == CRTOP ) {
			windowList->prevWindow = w;
			w->prevWindow = NULL;
			w->nextWindow = windowList;
			windowList = w;
		} else {	/* insert on the bottom */
			w1 = windowList;
			while( w1->nextWindow != NULL )
				w1 = w1->nextWindow;
			w1->nextWindow = w;
			w->prevWindow = w1;
			w->nextWindow = NULL;
		}
	}

	/* set the passed window parameters */
	w->saveRow1 = w->row1 = row1;
	w->saveRow2 = w->row2 = row2;
	w->saveCol1 = w->col1 = col1;
	w->saveCol2 = w->col2 = col2;

	/* set the default window parameters */
	w->posTopline = 0;
	w->numTopline = 1;
	w->indent = 0;

	if( windowType == 0 )
		/* set the state of unixMode */
		w->state = getUnixState(w->fileId);
	else {
		/* 0x1 => used */
		/* 0xC => no banner and temporary (windowType==1) */
		/* 0x8 => no banner but permanent (windowType==2) */
		w->state = 0x1 + (windowType==1 ? 0xC : 0x8);
	}
	
	/* set up the last row cache */
	w->posCurLast = 0;
	w->lastPosTop = 0;
	w->rowCurLast = 0;
	
	/* set up the window colors */
	w->textCycle = w->borderCycle = 1;
	setWindowColors(w);

	/* if there is a file in the window, and the window is */
	/* on top or there is not current selection, */
	/* then move the selection to the new window */
	if( w->fileId != -1 && selWindow == NULL ) {
		/* put the selection at the first char in this window */
		eraseSelection();
		selWindow = w;
		selBegin = 0;
		selEnd = 0;
		if( readChar(w->fileId, 0L) == '\r' ) {
			if( readChar(w->fileId, 1L) == '\n' )
				selEnd = 1;
		}
	}

	redrawWindow(w);

	/* make the new window the active window */
	activeWindow = w;
	redoBorders(0);
	updateScreen(0, scrRows-1);

	return w;
}

void pascal
/* XTAG:setWindowColors */
setWindowColors(w)
	register struct window *w;
{
	extern struct colorCycle colorCycles[];

	register int cycle;

	cycle = w->textCycle;
	w->textColor = colorCycles[cycle].textColor;
	w->selColor = colorCycles[cycle].selColor;
	cycle = w->borderCycle;
	w->bannerColor = colorCycles[cycle].bannerColor;
	w->borderColor = colorCycles[cycle].borderColor;
	w->elevColor = colorCycles[cycle].elevColor;
}

int pascal
/* XTAG:getBaseName */
getBaseName(s)
	unsigned char *s;
{
	register int n;
	register unsigned char ch;

	n = strlen(s) - 1;
	while( 1 ) {
		ch = s[n];
		if( ch == ':' || ch == '\\' )
			break;
		if( --n < 0 )
			break;
	}
	return n+1;
}

int pascal
/* XTAG:closeWindow */
closeWindow(w, ask, redraw)
	register struct window *w;
	int ask, redraw;
{
	extern struct window *selWindow;
	extern struct window *activeWindow;
	extern long selBegin, selEnd;
	extern int scrRows, scrCols;
	extern int menuLine;

	if( closeFile(w->fileId, ask) == -1 )
		return -1;
	w->state = 0;	/* free the window structure */

	/* unlink the window from the list of active windows */
	if( w->prevWindow != NULL )
		w->prevWindow->nextWindow = w->nextWindow;
	else	/* must be first on the list -- the top window */
		windowList = w->nextWindow;
	if( w->nextWindow != NULL )
		w->nextWindow->prevWindow = w->prevWindow;

	if( redraw ) {
		redrawBox( w->row1, w->col1, w->row2, w->col2);
		updateScreen(w->row1, w->row2);
	}

	/* is the selection in this window? */
	if( w == selWindow ) {
		/* move the selection to the top window */
		selWindow = windowList;
		if( selWindow != NULL ) {
			selBegin = 0;
			selEnd = 0;
			if( readChar(selWindow->fileId, 0L) == '\r' ) {
				if( readChar(selWindow->fileId, 1L) == '\n' )
					selEnd = 1;
			}
			if( redraw )
				redrawWindow(selWindow);
		}
	}
	/* is this the active window? */
	if( w == activeWindow ) {
		activeWindow = windowList;
		redoBorders(0);
		if( redraw )
			updateScreen((menuLine > 0 ? 1 : 0), scrRows-1);
	}
	return 0;
}

void pascal
/* XTAG:topWindow */
topWindow(w)
	register struct window *w;
{
	extern struct window *activeWindow;

	if( w != windowList ) {
		w->prevWindow->nextWindow = w->nextWindow;
		if( w->nextWindow != NULL )
			w->nextWindow->prevWindow = w->prevWindow;
		w->prevWindow = NULL;
		windowList->prevWindow = w;
		w->nextWindow = windowList;
		windowList = w;
	}
	activeWindow = w;
}

void pascal
/* XTAG:redrawWindow */
redrawWindow(w)
	register struct window *w;
{
	/* set up the screen map and draw the window */
	setMap(w->row1, w->col1, w->row2, w->col2, 1, 0x07);
	maskTop(w);
	drawWindow(w);
}

int pascal
/* XTAG:moveWindow */
moveWindow(w, row1, col1, row2, col2)
	register struct window *w;
	int row1, col1, row2, col2;
{
	extern int scrRows, scrCols;
	extern struct window *activeWindow;
	extern unsigned char msgBuffer[];
	extern int topOnFind;

	int r1, c1, r2, c2;
	int redoAll;

	/* if we have to erase the old active window */
	/* just redo the whole screen*/
	redoAll = (w != activeWindow);

	/* fix impossible corners */
	if( row1 < 0 )
		row1 = 0;
	if( row2 > (scrRows-1) )
		row2 = (scrRows-1);
	if( col1 < 0 )
		col1 = 0;
	if( col2 > (scrCols-1) )
		col2 = (scrCols-1);

	/* do not allow very small windows */
	if( row2 < row1+2 || col2 < col1+10 )
		return 1;

	/* find the bound of the redrawing  */
	r1 = min(row1, w->row1);
	c1 = min(col1, w->col1);
	r2 = max(row2, w->row2);
	c2 = max(col2, w->col2);
	
	/* only these have to change in the window structure */
	w->row1 = row1;
	w->col1 = col1;
	w->row2 = row2;
	w->col2 = col2;

	if( redoAll ) {
		r1 = 0;
		c1 = 0;
		r2 = scrRows - 1;
		c2 = scrCols - 1;
	}

	/* redraw all the windows that might have changed */
	redrawBox(r1, c1, r2, c2);
	updateScreen(r1, r2);

	return 0;
}

void pascal
/* XTAG:redrawBox */
redrawBox(row1, col1, row2, col2)
	int row1, col1, row2, col2;
{
	extern struct window *windowList;
	extern unsigned char textColor;
	extern int scrRows, scrCols;
	extern int menuLine;
	extern int menuShowing;
	extern unsigned char toplineVector[];
	extern int debug;

	register struct window *w1;
	int lowRow, highRow, n;

	/* set up the screen map and clear the screen area */
	setMap(row1, col1, row2, col2, 2, 0x7);
	
	/* restore the menu line if necessary */
	if( (menuLine > 0 && row1 == 0)
	 || (menuLine < 0 && row2 == (scrRows-1)) ) {
	 	/* find the first top line menu (to be the default) */
	 	for(n = 0; n < 64; n++) {
	 		if( (lowRow = toplineVector[n]) != 0 )
	 			break;;
	 	}
	 	switch( lowRow ) {
	 		default:
	 		case 50: lowRow = 1; break;
	 		case 51: lowRow = 2; break;
	 		case 52: lowRow = 3; break;
	 		case 60: lowRow = 4; break;
	 		case 65: lowRow = 5; break;
	 		case 67: lowRow = 6; break;
	 	}
		pulldown(lowRow);
		menuShowing = lowRow;
	}

	/* redraw all the windows that might have changed */
	w1 = windowList;
	lowRow = scrRows;
	highRow = -1;
	while( w1 != NULL ) {
		/* have these rows already been written? */
		if( w1->row1 >= lowRow && w1->row2 <= highRow && debug==0)
			goto nextOne;
		/* if the window is full width, remember the rows */
		if( w1->col1 == 0 && w1->col2 == (scrCols-1) ) {
			/* will this cover some rows below row1? */
			if( w1->row1 < lowRow )
				lowRow = w1->row1;
			/* will this cover some rows above row2? */
			if( w1->row2 > highRow )
				highRow = w1->row2;
		}
		drawWindow(w1);
	nextOne:
		w1 = w1->nextWindow;
	}
}

void pascal
/* XTAG:redoBorders */
redoBorders(bottomLineOnly)
	int bottomLineOnly;
{
	extern struct window *windowList;
	extern struct window *activeWindow;
	extern unsigned char *screenMap;
	extern struct openFile *files;
	extern int scrRows, scrCols;
	extern unsigned char border1[], border2[], border3[], border4[];
	extern int menuLine;
	extern int debug;

	register struct window *w1;
	unsigned char *border;
	int i, n, row1, row2, col1, col2;

	/* set up the screen map */
	n = (bottomLineOnly ? scrRows-1 : 0);

	/* if there are not windows, then clear the bottom line */
	/* of the screen by sending a "2" instead of a "1" */
	i = (windowList==NULL ? 2 : 1);
	setMap(n, 0, scrRows-1, scrCols-1, i, 0x7);

	/* redraw all the window borders */
	w1 = windowList;
	while( w1 != NULL ) {
		if( !bottomLineOnly ) {
			banner(w1);
		}
		row1 = w1->row1;
		col1 = w1->col1;
		row2 = w1->row2;
		col2 = w1->col2;
		if( !bottomLineOnly || row2 == scrRows - 1 ) {
			/* redraw the window border */
			if( w1 != activeWindow )
				border = &border1[0];
			else
				border = &border2[0];
			drawBorder(border, row1, col1, row2, col2,
				w1->borderColor, bottomLineOnly);

			/* return after we draw the bottom line */
			if( bottomLineOnly && row2 == scrRows - 1 &&
				col1 == 0 && col2 == scrCols -1 )
				return;
			if( !bottomLineOnly ) {
				/* mask off the window area */
				n = col2 - col1 - 1;
				for(i = row1+1; i < row2; i++)
					memset(screenMap+scrCols*i+col1+1,
						0, n);
			}
		}

		/* go to next window in the list */
		w1 = w1->nextWindow;
	}
}
