/*******************  start of original comments  ********************/
/*
 * This file contains the utilities to read in a ".tdecfg" file.
 *
 * Most of this stuff is duplicated from the cfgfile.c functions.  In
 *  Linux, this utility searches the CWD first then it searches the
 *  HOME directory for the ".tdecfg" file.
 *
 * Many thanks to <intruder@link.hacktic.nl> for the idea and sample code
 *  for this function.
 *
 *********************  start of Jason's comments  *********************
 *
 * DJGPP also requires the configuration file ("tde.cfg"), so to provide
 *  the same functionality across versions, DOS gets it as well.
 *  I also added a custom help file (".tdehlp" and "tde.hlp").
 *
 * Load the file from the HOME or executable directory first, and then
 *  from the current directory. This allows global and local settings.
 *  The help file is loaded from the current directory, or the HOME /
 *  executable directory.
 *
 *********************   end of Jason's comments   *********************
 *
 * New editor name:  TDE, the Thomson-Davis Editor.
 * Author:           Frank Davis
 * Date:             June 5, 1991, version 1.0
 * Date:             July 29, 1991, version 1.1
 * Date:             October 5, 1991, version 1.2
 * Date:             January 20, 1992, version 1.3
 * Date:             February 17, 1992, version 1.4
 * Date:             April 1, 1992, version 1.5
 * Date:             June 5, 1992, version 2.0
 * Date:             October 31, 1992, version 2.1
 * Date:             April 1, 1993, version 2.2
 * Date:             June 5, 1993, version 3.0
 * Date:             August 29, 1993, version 3.1
 * Date:             November 13, 1993, version 3.2
 * Date:             June 5, 1994, version 4.0
 * Date:             December 5, 1998, version 5.0 (jmh)
 *
 * This code is released into the public domain, Frank Davis.
 * You may distribute it freely.
 */


#include "tdestr.h"             /* tde types */
#include "common.h"
#include "define.h"
#include "tdefunc.h"
#include "config.h"


char *line_in;                  /* line buffer */
char *line_out;                 /* line buffer */
unsigned int line_no;           /* global variable for line count */
int  process_macro;             /* multi-line macro definition */
int  stroke_count;              /* global variable for macro strokes */
long macro_key;
MACRO *mac;
TREE *branch;
TREE *cfg_key_tree;
int  existing_macro;

int need_a_redraw;              /* if we redefined colors, then redraw screen */
int need_mode_line;             /* if we redefined modes,  then redraw line */
int need_rulers;                /* if we redefined rulers, then redraw ruler */

#if defined( __UNIX__ )
/*
 * UNIX stuff:  let's make us an array of colors used in curses.
 */
int curse_col[8] = { COLOR_BLACK, COLOR_RED,     COLOR_GREEN, COLOR_YELLOW,
                     COLOR_BLUE,  COLOR_MAGENTA, COLOR_CYAN,  COLOR_WHITE };
#endif


/*
 * Name:    tdecfgfile
 * Date:    June 5, 1994
 * Notes:   read in a configuration file at any time.
 *          read in a configuration file when we first fire up TDE in
 *           a linux (unix) and djgpp environment.
 *          August 20, 1997, Jason Hood - read in a help file as well.
 * jmh 980721: replaces load_strokes.
 * jmh 981127: only bring up the directory list if the macro name contains
 *              a wildcard pattern.
 */
int  tdecfgfile( TDE_WIN *window )
{
FILE *config;
int  rc = OK;
int  prompt_line;
int *clr;
char fname[PATH_MAX];          /* new name for file  */

   prompt_line = (window != NULL) ? window->bottom_line : g_display.nlines;

   /*
    * first, make sure we can alloc space for line buffers.
    *  line buffers are needed for reading the config file.
    */

   line_in = (char*)malloc( MAX_LINE_LENGTH - 1 );
   if (line_in != NULL)
      line_out = (char*)malloc( g_display.ncols + 2 );
   else
      line_out = NULL;

   if (line_out != NULL) {

      need_a_redraw  = FALSE;
      need_mode_line = FALSE;
      need_rulers    = FALSE;

      if (g_status.command == LoadMacro) {
         /*
          * search path for macro file
          */
         strcpy( fname, "*.tdm" );
         if (get_name( main21, prompt_line, fname ) == OK && *fname != '\0') {
            if (is_pattern( fname ))
               rc = list_and_pick( fname, window );
            /*
             * if everything is everything, load in the file selected by user.
             */
            if (rc == OK) {
               cfg_key_tree = (window->syntax) ?
                              &window->file_info->syntax->key_tree :
                              &key_tree;
               rc = readfile( fname, prompt_line );
            }
         }
      } else {
         cfg_key_tree = &key_tree;

         /*
          * CONFIGFILE is defined in tdestr.h
          */
         strcpy( fname, g_status.argv[0] );
         strcat( fname, CONFIGFILE );
         rc = readfile( fname, prompt_line );
#if defined( __UNIX__ )
         strcpy( fname, "./" );
         strcat( fname, CONFIGFILE );
         rc = readfile( fname, prompt_line );
#else
         rc = readfile( CONFIGFILE, prompt_line );
#endif
      }
      if (need_a_redraw  &&  g_display.adapter != MDA) {
         clr = &colour.clr[1][0];
         g_display.head_color    = *clr++;
         g_display.text_color    = *clr++;
         g_display.dirty_color   = *clr++;
         g_display.mode_color    = *clr++;
         g_display.block_color   = *clr++;
         g_display.message_color = *clr++;
         g_display.help_color    = *clr++;
         g_display.diag_color    = *clr++;
         g_display.eof_color     = *clr++;
         g_display.curl_color    = *clr++;
         g_display.cross_color   = *clr++;
         g_display.ruler_color   = *clr++;
         g_display.ruler_pointer = *clr++;
         g_display.hilited_file  = *clr++;
         g_display.overscan      = *clr;
         set_overscan_color( g_display.overscan );
      }
      if (window != NULL) {
         if (need_a_redraw)
            redraw_screen( window );
         if (need_mode_line && !need_a_redraw)
            show_modes( );
         if (need_rulers)
            show_all_rulers( );
      }

      /*
       * See if there's a help file.
       */
      if (g_status.command != LoadMacro) {
#if defined( __UNIX__ )
         strcpy( fname, "./" );
         strcat( fname, HELPFILE );
#else
         strcpy( fname, HELPFILE );
#endif
         if (access( fname, F_OK ) != 0) {
            strcpy( fname, g_status.argv[0] );
            strcat( fname, HELPFILE );
         }
         if (rc == OK && access( fname, F_OK ) == 0) {
            if ((config = fopen( fname, "r" )) == NULL) {
               rc = ERROR;
               combine_strings( line_out, main7a, fname, main7b );
               error( WARNING, prompt_line, line_out );
            }
            else {
              int c, i;
              char *help_line;
               help_line = fgets( line_in, MAX_COLS, config );
               for (c=1; help_screen[c] != NULL && help_line != NULL; c++) {
                  memset( help_screen[c], ' ', 80 );
                  for (i=0; i<80 && line_in[i] != '\n'; i++)
                      help_screen[c][i] = line_in[i];
                  help_line = fgets( line_in, MAX_COLS, config );
               }
               fclose( config );
            }
         }
      }
   } else {
      /*
       * not enough memory
       */
      error( WARNING, prompt_line, main4 );
      rc = ERROR;
   }

   if (line_in != NULL)
      free( line_in );
   if (line_out != NULL)
      free( line_out );

   return( rc );
}


/*
 * Name:    readfile
 * Date:    May 31, 1998
 * Notes:   helper function for tdecfgfile
 */
int readfile( char *fname, int prompt_line )
{
FILE *config;
int  rc = OK;

   if (access( fname, F_OK ) == 0) {
      if ((config = fopen( fname, "r" )) == NULL) {
         rc = ERROR;
         combine_strings( line_out, main7a, fname, main7b );
         error( WARNING, prompt_line, line_out );
      } else {
         line_no = 1;
         process_macro = FALSE;
         while (!feof( config )  &&  !g_status.control_break) {
            if (fgets( line_in, 1500, config ) == NULL)
               break;

#if defined( __UNIX__ )
            /*
             * for convenience, let's remove the <cr><lf> pair from
             *   from MSDOS-type text files.
             */
            remove_cr( line_in );
#endif
            parse_line( line_in, prompt_line );
            ++line_no;
         }
         if (process_macro) {
            /*
             * Unexpected end of file. Tidy up the macro definition.
             */
            check_macro( mac );
            error( WARNING, prompt_line, config26 );
         }

         fclose( config );
      }
   }
   return( rc );
}


#if defined( __UNIX__ )
/*
 * Name:    remove_cr
 * Purpose: get rid of <cr>
 * Date:    June 5, 1994
 * Passed:  line:  line of text
 * Notes:   UNIX don't like <cr><lf>.  if we read MSDOS config files,
 *            we need to get rid of the <cr><lf> thing.  change <cr> to <lf>.
 */
void remove_cr( char *line )
{
   if (line != NULL) {
      while (*line) {
         if (*line == '\r')
            *line = '\n';
         ++line;
      }
   }
}
#endif


/*
 * Name:     parse_line
 * Purpose:  real work horse of the configuration utility, figure out what
 *           we need to do with each line of the config file.
 * Date:     June 5, 1994
 * Passed:   line:  line that contains the text to parse
 * jmh 980720: take into account multi-line macro definitions
 * jmh 980730: I really don't like a lot of if-else-if conditionals, so I
 *              restructured with goto bad_config.
 */
void parse_line( char *line, int prompt_line )
{
char key[1042];         /* buffer to hold any token that we parse */
char *residue;          /* pointer to next item in line, if it exists */
int key_no;             /* index into key array */
long two_key;           /* two-combination keys */
int color;              /* color field */
int mode_index;         /* index in mode array */
int func_no;            /* function number we want to assign to a key */
int color_no;           /* attribute we want to assign to a color field */
int mode_no;            /* mode number we want to assign to a mode */
int found;              /* boolean, did we find a valid key, color, or mode? */
int i;
char *errmsg = NULL;

   if (process_macro) {
      parse_macro( line, prompt_line );
      return;
   }

   /*
    * find the first token and put it in key.  residue points to next token.
    */
   residue = parse_token( line, key );
   if (*key == '\0')
      return;

   /*
    * jmh 980730: there must be more than one character with something else
    *             on the line (macros are taken care of above).
    */
   if (key[1] == '\0' || residue == NULL) {
      errmsg = config1;         /* setting without value */
      goto bad_config;
   }

   found = FALSE;
   /*
    * try to find a valid key
    */
   key_no = search( key, valid_keys, AVAIL_KEYS-1 );
   if (key_no != ERROR) {
      /*
       * find the function assignment
       */
      found = TRUE;
      residue = parse_token( residue, key );
      /*
       * clear any previous macro or key assignment.
       */
      func_no = search( key, valid_func, NUM_FUNCS+1 );
      if (func_no != ERROR) {
         clear_previous_macro( key_no );
         key_func.key[key_no] = func_no;
         if (func_no == PlayBack) {
            macro_key = key_no;
            if (parse_macro( residue, prompt_line ) == ERROR)
               errmsg = main4;          /* out of memory */
         }
      } else if (key_func.key[key_no] == TwoCharKey && residue != NULL) {
         two_key = key_no + 256;

         /*
          * was the second key one letter?
          */
         if (key[1] == '\0')
            key_no = *key;
         else {
            key_no = search( key, valid_keys, AVAIL_KEYS-1 );
            if (key_no == ERROR) {
               errmsg = config22;       /* unrecognized key */
               goto bad_config;
            }
            key_no += 256;
         }
         two_key = CREATE_TWOKEY( two_key, key_no );
         residue = parse_token( residue, key );
         func_no = search( key, valid_func, NUM_FUNCS+1 );
         if (func_no == PlayBack) {
            macro_key = two_key;
            if (parse_macro( residue, prompt_line ) == ERROR)
               errmsg = main4;
         } else
            errmsg = (func_no == ERROR) ? config3 : /* unrecognized function */
                     (func_no == TwoCharKey) ? config2 : /*two-key not allowed*/
                     (add_twokey( two_key, func_no ) == ERROR) ? main4   :
                                                                 NULL;
      } else
         errmsg = config3;
   }

   /*
    * valid key not found, now try a pseudo-macro
    */
   if (!found) {
      /*
       * Should probably search valid_func and test for PseudoMacro
       */
      if (stricmp( key, "PseudoMacro" ) == 0) {
         found = TRUE;
         residue = parse_token( residue, key );
         if (strlen( key ) != 2 || key[0] <= ' ' || key[1] <= ' ') {
            errmsg = config25;          /* invalid combination */
            goto bad_config;
         }
         macro_key = (key[0] << 8) | key[1];
         if (parse_macro( residue, prompt_line ) == ERROR)
            errmsg = main4;
      }
   }

   /*
    * pseudo-macro not found, now try a valid color
    */
   if (!found) {
      color = search( key, valid_colors, (NUM_COLORS * 2) - 1 );
      if (color != ERROR) {
         found = TRUE;
         i = (*key == 'c');
         parse_token( residue, key );

         /*
          * we found a color field and attribute.  now, make sure
          *   everything is everything before we assign the attribute
          *   to the color field.
          */
         color_no = atoi( key );
         if (color_no < 0 || color_no > 127) {
            errmsg = config6;           /* color number out of range */
            goto bad_config;
         }
         colour.clr[i][color] = color_no;
         need_a_redraw = TRUE;
      } else {
         /*
          * see if this is a color pair for Linux curses.
          */
         mode_no = search( key, valid_pairs, 7 );
         if (mode_no != ERROR) {

#if defined( __UNIX__ )
            residue = parse_token( residue, key );
            color = search( key, valid_curse, 7 );
            if (color != ERROR && residue != NULL) {
               /*
                *  so far, we got   pairx  COLOR_y.
                *    now get the "on".
                */
               residue = parse_token( residue, key );
               if (residue != NULL) {
                  /*
                   * now, get the background color.
                   */
                  parse_token( residue, key );
                  color_no = search( key, valid_curse, 7 );

                  /*
                   * we just parsed a color pair line:
                   *    pairx COLOR_y on COLOR_z.
                   */
                  if (color_no != ERROR) {
                     found = TRUE;
                     need_a_redraw = TRUE;
                     init_pair( mode_no, color, color_no );
                     mode_no *= 16;
                     for (i=0; i<8; i++)
                        tde_color_table[mode_no + i]  =  COLOR_PAIR( mode_no );

                     for (i=8; i<16; i++)
                        tde_color_table[mode_no + i]  =
                                   COLOR_PAIR( mode_no ) | A_BOLD;
                  }
               }
            }
            if (found == FALSE)
               errmsg = config24;       /* error parsing color pair */
#else
            /*
             * if we are in MSDOS, don't even bother with parsing
             *   curses junk.
             */
            found = TRUE;
#endif
         }
      }
   }

   /*
    * valid color not found, now try a valid mode
    */
   if (!found) {
      mode_index = search( key, valid_modes, NUM_MODES-1 );
      if (mode_index != ERROR) {
         found = TRUE;

         /*
          * if we find a valid mode, we need to search different
          *   option arrays before we find a valid assignment.
          */
         residue = parse_token( residue, key );
         mode_no = ERROR;
         switch ( mode_index ) {
            case Ins         :
            case Ind         :
            case Smart       :
            case Trim        :
            case Backup      :
            case Ruler       :
            case JustRM      :
            case CursorCross :
               mode_no = search( key, off_on, 1 );
               if (mode_no == ERROR) {
                  errmsg = config7;     /* off/on error */
                  goto bad_config;
               }
               switch ( mode_index ) {
                  case Ins         :
                     mode.insert = mode_no;
                     break;
                  case Ind         :
                     mode.indent = mode_no;
                     break;
                  case Smart       :
                     mode.smart_tab = mode_no;
                     break;
                  case Trim        :
                     mode.trailing = mode_no;
                     break;
                  case Backup      :
                     mode.do_backups = mode_no;
                     break;
                  case Ruler       :
                     mode.ruler = mode_no;
                     need_rulers = TRUE;
                     break;
                  case JustRM      :
                     mode.right_justify = mode_no;
                     break;
                  case CursorCross :
                     mode.cursor_cross = mode_no;
                     need_a_redraw = TRUE;
                     break;
               }
               need_mode_line = TRUE;
               break;
            case Eol      :
               mode_no = search( key, valid_eol, 2 );
               if (mode_no == ERROR) {
                  errmsg = config17;    /* eol display error */
                  goto bad_config;
               }
               mode.show_eol = mode_no;
               need_a_redraw = TRUE;
               break;
            case LTAB     :
            case PTAB     :
               mode_no = atoi( key );
               if (mode_no > 520 || mode_no < 1) {
                  errmsg = config8;     /* tab error */
                  goto bad_config;
               }
               if (mode_index == LTAB)
                  mode.ltab_size = mode_no;
               else
                  mode.ptab_size = mode_no;
               need_mode_line = TRUE;
               break;
            case Left    :
               mode_no = atoi( key );
               if (mode_no < 1 || mode_no > mode.right_margin) {
                  errmsg = config9;     /* left margin error */
                  goto bad_config;
               }
               mode.left_margin = --mode_no;
               need_rulers = TRUE;
               break;
            case Para    :
               mode_no = atoi( key );
               if (mode_no < 1 || mode_no > mode.right_margin) {
                  errmsg = config10;    /* paragraph margin error */
                  goto bad_config;
               }
               mode.parg_margin = --mode_no;
               need_rulers = TRUE;
               break;
            case Right   :
               mode_no = atoi( key );
               if (mode_no < mode.left_margin || mode_no > 1040) {
                  errmsg = config11;    /* right margin error */
                  goto bad_config;
               }
               mode.right_margin = --mode_no;
               need_rulers = TRUE;
               break;
            case InflateTabs :
               mode_no = search( key, valid_tabs, 2 );
               if (mode_no == ERROR) {
                  errmsg = config16;    /* inflate tabs error */
                  goto bad_config;
               }
               mode.inflate_tabs = mode_no;
               need_mode_line = TRUE;
               need_a_redraw  = TRUE;
               break;
            case Crlf    :
               mode_no = search( key, valid_crlf, 1 );
               if (mode_no == ERROR) {
                  errmsg = config12;    /* crlf or lf error */
                  goto bad_config;
               }
               mode.crlf = mode_no;
               break;
            case WW      :
               mode_no = search( key, valid_wraps, 2 );
               if (mode_no == ERROR) {
                  errmsg = config13;    /* word wrap error */
                  goto bad_config;
               }
               mode.word_wrap = mode_no;
               need_mode_line = TRUE;
               break;
            case Size    :
               mode_no = search( key, valid_cursor, 1 );
               if (mode_no == ERROR) {
                  errmsg = config14;    /* cursor size error */
                  goto bad_config;
               }
               mode.cursor_size = mode_no;
               break;
            case Write_Z :
               mode_no = search( key, valid_z, 1 );
               if (mode_no == ERROR) {
                  errmsg = config15;    /* control z error */
                  goto bad_config;
               }
               mode.control_z = mode_no;
               need_mode_line = TRUE;
               break;
            case Stamp   :
               strcpy( mode.stamp, key );
               break;
            case Initcase    :
               mode_no = search( key, case_modes, 1 );
               if (mode_no == ERROR) {
                  errmsg = config18;    /* initial case mode error */
                  goto bad_config;
               }
               mode.search_case = mode_no;
               need_mode_line = TRUE;
               break;
            case Match   :
               mode_no = OK;
               for (i=0; i<256; i++)
                  sort_order.match[i] = (char)i;
               new_sort_order( (unsigned char*)key,
                               (unsigned char*)sort_order.match );
               break;
            case Ignore  :
               mode_no = OK;
               for (i=0; i<256; i++)
                  sort_order.ignore[i] = (char)i;
               for (i=65; i<91; i++)
                  sort_order.ignore[i] = (char)(i + 32);
               new_sort_order( (unsigned char*)key,
                               (unsigned char*)sort_order.ignore );
               break;
            case DirSort :
               mode_no = search( key, valid_dir_sort, 1 );
               if (mode_no == ERROR) {
                  errmsg = config27;    /* directory sort error */
                  goto bad_config;
               }
               mode.dir_sort = mode_no;
               break;
            case CaseConvert :
               mode_no = OK;
               new_upper_lower( (unsigned char*)key );
               break;
            case CharDef :
               mode_no = OK;
               new_bj_ctype( (unsigned char*)key );
               break;
         }
         if (mode_no == ERROR)
            errmsg = config19;          /* unknown mode */
      }
   }

   if (!found)
      errmsg = config20;

   if (errmsg != NULL) {
bad_config:
      ERRORLINE( errmsg, key )
   }
}


/*
 * Name:    parse_token
 * Purpose: given an input line, find the first token
 * Date:    June 5, 1994
 * Passed:  line:  line that contains the text to parse
 *          token: buffer to hold token
 * Returns: pointer in line to start next token search.
 * Notes:   assume tokens are delimited by spaces.
 *
 * jmh:     September 9, 1997 - if line is NULL, *token is set to '\0'.
 *          This allows easier parsing in my syntax highlighting.
 *
 * jmh 980730: test for comments. If the line starts with ';', then token is
 *              set empty and return NULL.
 *             Test for trailing spaces and comments when returning residue.
 *             To include a ';', enclose it in quotes (eg. c+k ";" Macro ...).
 */
char *parse_token( char *line, char *token )
{
   if (line == NULL) {
      *token = '\0';
      return( NULL );
   }

   /*
    * skip over any leading spaces.
    */
   while (*line == ' ' || *line == '\t')
      ++line;

   if (*line == ';') {
      *token = '\0';
      return( NULL );
   }

   /*
    * jmh 980521: test for literal strings.
    */
   if (*line == '\"') {
      line = parse_literal( line, token );
      if (line == NULL)
         return( NULL );
   } else {
      /*
       * put the characters into the token array until we run into a space
       *   or the terminating '\0';
       */
      while (*line != ' ' && *line != '\t' && *line != '\n' && *line != '\0')
         *token++ = *line++;
      *token = '\0';
   }

   /*
    * return what's left on the line, if anything.
    */
   while (*line == ' ' || *line == '\t')
      ++line;
   if (*line != '\n' && *line != ';' && *line != '\0')
      return( line );
   else
      return( NULL );
}


/*
 * Name:    parse_literal
 * Purpose: get all letters in a literal
 * Date:    June 5, 1994
 * Passed:  line:       current line position
 *          literal:    buffer to hold literal
 * Returns: pointer in line to start next token search.
 * Notes:   a literal begins with a ".  to include a ", precede a " with a ".
 * jmh 980721: modified to return residue and display a warning.
 */
char *parse_literal( char *line, char *literal )
{
int  end_quote = 0;     /* flag to indicate the end of the literal */
char temp[10];          /* storage for the line number */
int  prompt_line;

   line++;
   /*
    * put the characters into the literal array until we run into the
    *   end of literal or terminating '\0';
    */
   while (*line != '\n' && *line != '\0') {
      if (*line == '\"') {
         line++;
         if (*line != '\"') {
            ++end_quote;
            break;
         }
      }
      *literal++ = *line++;
   }
   *literal = '\0';

   /*
    * return what's left on the line, if anything.
    */
   if (*line != '\n' && *line != '\0')
      return( line );
   else {
      if (!end_quote) {
         prompt_line = (g_status.current_window == NULL) ? g_display.nlines :
                       g_status.current_window->bottom_line;
         ERRORLINE( config21, temp )    /* unterminated quote */
      }
      return( NULL );
   }
}


/*
 * Name:    search
 * Purpose: binary search a CONFIG_DEFS structure
 * Date:    June 5, 1994
 * Passed:  token:  token to search for
 *          list:   list of valid tokens
 *          num:    number of valid tokens in list
 * Returns: value of token assigned to matching token.
 * Notes:   do a standard binary search.
 *          instead of returning mid, lets return the value of the token
 *          assigned to mid.
 */
int  search( char *token, CONFIG_DEFS list[], int num )
{
int bot;
int mid;
int top;
int rc;

   bot = 0;
   top = num;
   while (bot <= top) {
      mid = (bot + top) / 2;
      rc = stricmp( token, list[mid].key );
      if (rc == 0)
         return( list[mid].key_index );
      else if (rc < 0)
         top = mid - 1;
      else
         bot = mid + 1;
   }
   return( ERROR );
}


/*
 * Name:    clear_previous_macro
 * Purpose: clear any macro previously assigned to a key
 * Date:    June 5, 1994
 * Passed:  macro_key:  key that we are a assigning a macro to
 * Notes:   rewritten by Jason Hood, July 19, 1998.
 */
void clear_previous_macro( int macro_key )
{
   if (macro[macro_key] != NULL) {
      if (macro[macro_key]->len > 1)
         my_free( macro[macro_key]->key.keys );
      free( macro[macro_key] );
      macro[macro_key] = NULL;
   }
}


/*
 * Name:    parse_macro
 * Purpose: separate literals from keys in a macro definition
 * Date:    June 5, 1994
 * Passed:  residue: pointer to macro defs
 * Notes:   for each token in macro def, find out if it's a literal or a
 *             function key.
 * jmh 980719: allow function names to be used and multi-line definitions.
 * jmh 980820: allow literals to contain escaped function shortcuts.
 */
int  parse_macro( char *residue, int prompt_line )
{
int  rc;
char literal[1042];
char temp[42];
char *l;
long key_no = 0;
int  twokey;
int  func;
int  empty = TRUE;
TREE *twokey_func;
char *errmsg;

   /*
    * allocate storage for the keys.
    */
   if (!process_macro)
      if (initialize_macro( macro_key ) == NULL)
         return( ERROR );

   while (residue != NULL) {
      /*
       * skip over any leading spaces.
       */
      while (*residue == ' ' || *residue == '\t')
         ++residue;

      /*
       * done if we hit eol or a comment
       */
      if (*residue == '\n' || *residue == ';' || *residue == '\0')
         residue = NULL;

      /*
       * check for a literal.
       */
      else if (*residue == '\"') {
         empty   = FALSE;
         residue = parse_literal( residue, literal );
         l  = literal;
         rc = OK;
         while (*l != '\0'  &&  rc == OK) {
            func = 0;
            if (*l == '\\' && *(l+1) != '\0') {
               switch (*(++l)) {
                  case MAC_BackSpace : func = BackSpace;   break;
                  case MAC_CharLeft  : func = CharLeft;    break;
                  case MAC_CharRight : func = CharRight;   break;
                  case MAC_Pseudo    : func = PseudoMacro; break;
                  case MAC_Rturn     : func = Rturn;       break;
                  case MAC_Tab       : func = Tab;         break;
                  case '0'           : func = MacroMark;   break;
                  case '1'           : func = SetMark1;    break;
                  case '2'           : func = SetMark2;    break;
                  case '3'           : func = SetMark3;    break;
                  case '\\'          : ++l;                break;
               }
               if (func == 0)
                  --l;
            }
            rc = cfg_record_keys( mac, *l, func, prompt_line );
            ++l;
         }
      /*
       * check for a function.
       */
      } else {
         errmsg  = NULL;
         residue = parse_token( residue, literal );
         func    = search( literal, valid_func, NUM_FUNCS+1 );
         if (func != ERROR) {
            if (func == RecordMacro && process_macro) {
               empty = process_macro = FALSE;
               break;
            }
            if (func == PlayBack) {
               key_no = macro_key;
               if (key_no < MAX_KEYS)
                  key_no += 256;
               else if (key_no < 0x10000L) {
                  cfg_record_keys( mac, (int)key_no >> 8,   0, prompt_line );
                  cfg_record_keys( mac, (int)key_no & 0xff, 0, prompt_line );
                  func = PseudoMacro;
               }
               residue = NULL;
               empty = process_macro = FALSE;
            }
         } else {
            /*
             * check for a function key.
             */
            key_no = search( literal, valid_keys, AVAIL_KEYS-1 );
            if (key_no == ERROR) {
               /*
                * check for mode definition.
                */
               key_no = search( literal, valid_macro_modes, NUM_MACRO_MODES );
               if ((int)key_no != ERROR) {
                  mac->mode[(int)key_no / 2] = (int)key_no & 1;
                  continue;
               }
            } else {
               func   = key_func.key[(int)key_no];
               twokey = (int)key_no + 256;
               if (func == TwoCharKey) {
                  if (residue == NULL) {
                     key_no = ERROR;
                     errmsg = config28;
                  } else {
                     residue = parse_token( residue, literal );
                     if (literal[1] == '\0')
                        key_no = *literal;
                     else {
                        key_no = search( literal, valid_keys, AVAIL_KEYS-1 );
                        if (key_no != ERROR)
                           key_no += 256;
                     }
                     if (key_no != ERROR) {
                        key_no = CREATE_TWOKEY( twokey, key_no );
                        twokey_func = search_tree(key_no, cfg_key_tree->right);
                        func = (twokey_func == NULL) ? 0 :
                               twokey_func->type.func;
                        if ((unsigned)func > NUM_FUNCS)
                           func = PlayBack;
                     }
                  }
               } else
                  key_no = twokey;
            }
         }
         if (key_no != ERROR) {
            cfg_record_keys( mac, key_no, func, prompt_line );
            empty = FALSE;
         } else {
            ERRORLINE( (errmsg == NULL) ? config22 : errmsg, temp )
         }
      }
   }
   if (empty)
      process_macro = TRUE;

   if (!process_macro)
      check_macro( mac );

   return( OK );
}


/*
 * Name:    initialize_macro
 * Purpose: initialize the first key of a macro def
 * Date:    June 5, 1994
 * Passed:  key:  key number of macro that we are initializing
 * Returns: pointer to macro or NULL if no memory.
 * Notes:   Rewritten by Jason Hood, July 19/31, 1998.
 *          For this function, function keys start from 0, not 256 as usual.
 */
MACRO *initialize_macro( long key )
{
int rc;

   mac            = NULL;
   branch         = NULL;
   existing_macro = FALSE;

   if (key >= MAX_KEYS) {
      branch = search_tree( key, cfg_key_tree );
      if (branch == NULL) {
         branch = calloc( 1, sizeof(TREE) );
         if (branch == NULL)
            return( NULL );
         branch->key = key;
      } else {
         existing_macro = TRUE;
         if (branch->type.func >= NUM_FUNCS) {
            mac = branch->type.macro;
            if (mac->len > 1)
               my_free( mac->key.keys );
            mac->len = 0;
         }
      }
   }
   if (mac == NULL) {
      mac = calloc( 1, sizeof(MACRO) );
      if (mac == NULL) {
         if (!existing_macro)
            free( branch );
         return( NULL );
      }
      if (branch != NULL)
         branch->type.macro = mac;
   }
   /*
    * Try allocating STROKE_LIMIT first. If it fails, halve
    * it each time and try again.
    */
   stroke_count = STROKE_LIMIT;
   while ((mac->key.keys = my_malloc( stroke_count * sizeof(long), &rc ))
                         == NULL  &&  stroke_count > 1)
      stroke_count /= 2;

   if (mac->key.keys == NULL) {
      stroke_count = 0;
      free( mac );
      mac = NULL;
      if (!existing_macro)
         free( branch );
   } else                       /* set default modes to: */
      mac->mode[0] =            /* insert                */
      mac->mode[1] =            /* smart tabs            */
      mac->mode[2] = TRUE;      /* indent                */

   return( mac );
}


/*
 * Name:    check_macro
 * Purpose: see if macro def has any valid key.  if not, clear macro
 * Date:    June 5, 1994
 * Passed:  mac:  macro that we are checking
 * Notes:   Rewritten by Jason Hood, July 19, 1998.
 */
void check_macro( MACRO *mac )
{
long key;
long FAR *keys;
int  rc;

   if (mac->len == 0) {
      my_free( mac->key.keys );
      free( mac );
      if (!existing_macro)
         free( branch );

   } else {
      if (mac->len == 1) {
         key = *mac->key.keys;
         my_free( mac->key.keys );
         mac->key.key = key;

      } else {
         if ((keys = my_malloc( mac->len * sizeof(long), &rc )) == NULL)
            mac->key.keys = my_realloc( mac->key.keys, mac->len*sizeof(long) );
         else {
            my_memcpy( keys, mac->key.keys, mac->len * sizeof(long) );
            my_free( mac->key.keys );
            mac->key.keys = keys;
         }
      }

      if (branch == NULL)
         macro[(int)macro_key] = mac;
      else if (!existing_macro)
         add_branch( branch, cfg_key_tree );
   }
}


/*
 * Name:    cfg_record_keys
 * Purpose: save keystrokes in keystroke buffer
 * Date:    June 5, 1994
 * Passed:  line: line to display prompts
 * Notes:   Rewritten by Jason Hood, July 19, 1998.
 *
 * jmh 980826: when possible, store the function rather than the key.
 */
int  cfg_record_keys( MACRO *mac, long key, int func, int prompt_line )
{
int rc;
char temp[42];

   rc = OK;
   if (stroke_count-- > 0) {
      if (func != RecordMacro    && func != SaveMacro  &&
          func != ClearAllMacros && func != LoadMacro) {
         mac->key.keys[mac->len++] = (func == 0 || func == PlayBack) ? key :
                                     func + 256+MAX_KEYS;
      }
   } else {
      rc = ERROR;
      if (stroke_count == -1) {
         ERRORLINE( config23, temp )    /* no more room in macro buffer */
      }
   }
   return( rc );
}


/*
 * Name:    cfg_search_tree
 * Purpose: search the two-key tree for a function
 * Author:  Jason Hood
 * Date:    July 31, 1998
 * Passed:  func: function to find
 *          tree: pointer to tree
 * Returns: key if found; ERROR if not found
 * Notes:   recursively traverse the tree.
 */
long cfg_search_tree( int func, TREE *tree )
{
long key;

   if (tree == NULL)
      return( ERROR );

   if (tree->type.func == func)
      return( tree->key );

   key = cfg_search_tree( func, tree->left );
   if (key == ERROR)
      key = cfg_search_tree( func, tree->right );

   return( key );
}


/*
 * Name:    new_sort_order
 * Purpose: change the sort order
 * Date:    October 31, 1992
 * Notes:   New sort order starts from ! (ie. skips space and control chars)
 */
void new_sort_order( unsigned char *residue, unsigned char *sort )
{
int i;

   sort += 33;
   for (i=33; *residue != '\0'  &&  *residue != '\n' && i <= 255; i++)
      *sort++ = *residue++;
}


/*
 * Name:    add_twokey
 * Purpose: find an open slot and insert the new two-key combination
 * Date:    April 1, 1993
 * Passed:  key:  two-key
 *          func: function number to assign to this combo
 * Notes:   Rewritten by Jason Hood, July 30, 1998.
 */
int  add_twokey( long key, int func )
{
TREE *twokey;
int  rc = OK;

   twokey = search_tree( key, cfg_key_tree->right );
   if (twokey == NULL) {
      twokey = calloc( 1, sizeof(TREE) );
      if (twokey == NULL)
         rc = ERROR;
      else {
         twokey->key = key;
         twokey->type.func = func;
         add_branch( twokey, cfg_key_tree );
      }
   } else {
      if (twokey->type.func >= NUM_FUNCS) {
         if (twokey->type.macro->len > 1)
            my_free( twokey->type.macro->key.keys );
         free( twokey->type.macro );
      }
      twokey->type.func = func;
   }

   return( rc );
}


/*
 * Name:    new_upper_lower
 * Purpose: Define the conversion between upper and lower cases
 * Author:  Jason Hood
 * Date:    November 27, 1998
 * Notes:   Ignores space and the control characters.
 *          Dot ('.') is used as a place filler.
 */
void new_upper_lower( unsigned char *residue )
{
int i;

   for (i = 33; *residue != '\0'  &&  *residue != '\n' && i <= 255; i++) {
      upper_lower[i] = (*residue == '.') ? 0 : *residue;
      ++residue;
   }
   for (; i < 256; ++i)
      upper_lower[i] = 0;
}


/*
 * Name:    new_bj_ctype
 * Purpose: Define the type of a character
 * Author:  Jason Hood
 * Date:    November 27, 1998
 * Notes:   Space and the control characters are always the same.
 *          Dot ('.') is used as a place filler.
 */
void new_bj_ctype( unsigned char *residue )
{
int i;
int t;

   for (i = 33; *residue != '\0'  &&  *residue != '\n' && i <= 255; i++) {
      switch (*residue) {
         case 'L' : t = BJ_lower;             break;
         case 'U' : t = BJ_upper;             break;
         case 'D' : t = BJ_digit | BJ_xdigit; break;
         case 'S' : t = BJ_space;             break;
         case 'C' : t = BJ_cntrl;             break;
         case 'P' : t = BJ_punct;             break;
         case 's' : t = BJ_space | BJ_cntrl;  break;
         case 'X' : t = BJ_upper | BJ_xdigit; break;
         case 'x' : t = BJ_lower | BJ_xdigit; break;
         default  : t = 0;
      }
      bj_ctype[i+1] = t;
      ++residue;
   }
   for (; i < 256; ++i)
      bj_ctype[i+1] = 0;
}
