/*
 * Now that I own both MSC 7.0 and BC 3.1 and have linux, lets rearrange stuff
 * so many compilers can compile TDE.  Several implementation specific
 * functions needed for several environments were gathered into this file.
 *
 * In version 3.2, these functions changed to support unix.
 *
 * Incidentally, there is a difference between a NULL line pointer and
 * a pointer to a line that contains no characters.  For example, calling
 *
 *       line = malloc( 0 );
 *
 * will return a valid pointer to an item of 0 length in some compilers
 * and a NULL pointer in other compilers.  malloc( 0 ) will return a valid
 * pointer to an object of zero length in MSC.  malloc( 0 ) will return a
 * NULL pointer in BC.  The problem with returning a NULL pointer for
 * malloc( 0 ) is that it's a little harder to tell if the heap is out of
 * memory or if we have a valid NULL pointer.  On the other hand, the good
 * part about returning a NULL pointer for malloc( 0 ) is that extra space
 * is not wasted for an object of 0 length.  In TDE, we will test for 0
 * before calling my_malloc( ) and set an ERROR code if out of memory.
 *
 * New editor name:  TDE, the Thomson-Davis Editor.
 * Author:           Frank Davis
 * Date:             June 5, 1991, version 1.0
 * Date:             July 29, 1991, version 1.1
 * Date:             October 5, 1991, version 1.2
 * Date:             January 20, 1992, version 1.3
 * Date:             February 17, 1992, version 1.4
 * Date:             April 1, 1992, version 1.5
 * Date:             June 5, 1992, version 2.0
 * Date:             October 31, 1992, version 2.1
 * Date:             April 1, 1993, version 2.2
 * Date:             June 5, 1993, version 3.0
 * Date:             August 29, 1993, version 3.1
 * Date:             November 13, 1993, version 3.2
 * Date:             June 5, 1994, version 4.0
 * Date:             December 5, 1998, version 5.0 (jmh)
 *
 * This code is released into the public domain, Frank Davis.
 * You may use and distribute it freely.
 */

#include "tdestr.h"
#include "common.h"
#include "tdefunc.h"
#include "define.h"

#include <io.h>                         /* for attributes (_chmod) */
#define __dj_include_pc_h_
#include <dos.h>                        /* for drive (_dos_[gs]etdrive) */
#include <fcntl.h>                      /* for lfn querying (_USE_LFN) */
#include <sys/stat.h>                   /* for full path (_fixpath) */


/*
 * Name:    my_malloc
 * Purpose: malloc from the far heap
 * Date:    November 13, 1993
 * Passed:  size:  memory needed heap
 *          rc:   pointer to return code
 * Notes:   set the return code only if an ERROR occured with malloc.
 *           returning a NULL pointer is not neccessarily an ERROR.
 */
void *my_malloc( size_t size, int *rc )
{
void *mem;

   assert( size < MAX_LINE_LENGTH );

   if (size == 0)

      /*
       * if 0 bytes are requested, return NULL
       */
      mem = NULL;
   else {

      mem = malloc( size );

      /*
       * if malloc failed, return NULL and an ERROR.
       */
      if (mem == NULL)
         *rc = ERROR;
   }
   return( mem );
}


/*
 * Name:    my_free
 * Purpose: free memory from the far heap
 * Date:    November 13, 1993
 * Passed:  mem:  pointer to memory to free in far heap
 */
void my_free( void *mem )
{
   assert( mem != NULL );
   free( mem );
}


/*
 * Name:    my_heapavail
 * Purpose: available free memory from the far heap
 * Date:    July 8, 1997
 * Notes:   approximation only - there will be at least this amount available
 */
long my_heapavail( void )
{
   return( _go32_dpmi_remaining_physical_memory() );
}


/*
 * Name:    my_memcpy
 * Purpose: copy memory
 * Date:    November 13, 1993
 * Passed:  dest: pointer to destination
 *          src:  pointer to source
 *          size: number of bytes to copy
 */
void my_memcpy( void *dest, void *src, size_t size )
{
   if (size > 0) {
      assert( dest != NULL );
      assert( src  != NULL );
      memcpy( dest, src, size );
   }
}


/*
 * Name:    my_memmove
 * Purpose: move memory
 * Date:    November 13, 1993
 * Passed:  dest: pointer to destination
 *          src:  pointer to source
 *          size: number of bytes to copy
 */
void my_memmove( void *dest, void *src, size_t size )
{
   if (size > 0) {
      assert( dest != NULL );
      assert( src  != NULL );
      memmove( dest, src, size );
   }
}


static FTYPE found;

/*
 * Name:    my_findfirst
 * Purpose: find the first file matching a pattern
 * Date:    August 4, 1997
 * Passed:  path: path and pattern to search for files
 *          dta:  file finding info
 *          dirs: TRUE to search for directories and return file sizes
 * Notes:   Returns NULL for no matching files or bad pattern;
 *           otherwise a pointer to a static FTYPE, with fname holding the
 *           filename that was found.
 *          If dirs is TRUE fsize will hold the size of the file, and the
 *           name will have a trailing slash ('/') if it's a directory.
 *          DOS systems will convert the name to lower-case; LFN systems
 *           will leave it as-is.
 */
FTYPE * my_findfirst( char *path, FFIND *dta, int dirs )
{
int i;
fattr_t fattr;
char temp[PATH_MAX];

   dta->dirs = dirs;
   /*
    * Separate the path and pattern
    */
   i = strlen( path ) - 1;
   if (i == -1) {
      get_current_directory( dta->stem, 0 );
      strcpy( dta->pattern, "*" );
   }
   else if (path[i] == '/' || path[i] == '\\' || path[i] == ':') {
      strcpy( dta->stem, path );
      strcpy( dta->pattern, "*" );
   } else if (get_fattr( path, &fattr ) == OK && (fattr & SUBDIRECTORY)) {
      strcpy( dta->stem, path );
      strcat( dta->stem, "/" );
      strcpy( dta->pattern, "*" );
   } else {
      for (--i; i >= 0; --i) {
         if (path[i] == '/' || path[i] == ':')
            break;
         /* if it's a backslash, it could be an escape for the pattern */
         if (path[i] == '\\' && strchr( "!^-\\]", path[i+1] ) == NULL)
            break;
      }
      if (i >= 0) {
         strncpy( dta->stem, path, ++i );
         dta->stem[i] = '\0';
         strcpy( dta->pattern, path+i );
      } else {
         get_current_directory( dta->stem, 0 );
         strcpy( dta->pattern, path );
      }
      if (!is_valid_pattern( dta->pattern, &i ))
         return NULL;
   }
   /*
    * Start scanning the directory
    */
   strcpy( temp, dta->stem );
   strcat( temp, "*.*" );
   fattr = NORMAL | READ_ONLY | HIDDEN | SYSTEM | ARCHIVE;
   if (dirs) fattr |= SUBDIRECTORY;
   i = findfirst( temp, &dta->find_info, fattr );
   if (i != OK) return NULL; /* empty directory? */
   do {
      if (dirs && (dta->find_info.ff_attrib & SUBDIRECTORY)) break;
      if (wildcard( dta->pattern, dta->find_info.ff_name ) == TRUE) break;
      i = findnext( &dta->find_info );
   } while (i == OK);
   if (i != OK) return NULL; /* nothing matched */

   /* ignore the "." entry (assuming it to be found in this initial scan) */
   if (dirs && dta->find_info.ff_name[0] == '.' &&
               dta->find_info.ff_name[1] == '\0')
      return my_findnext( dta );

   strcpy( found.fname, dta->find_info.ff_name );
   if (!_USE_LFN) {
      for (i = 0; found.fname[i]; ++i)
         found.fname[i] = bj_tolower( found.fname[i] );
   }
   else i = strlen( found.fname );
   if (dirs) {
      found.fsize = dta->find_info.ff_fsize;
      found.fattr = dta->find_info.ff_attrib;
      if (dta->find_info.ff_attrib & SUBDIRECTORY) {
         found.fname[i] = '/';
         found.fname[i+1] = '\0';
      }
   }
   return &found;
}


/*
 * Name:    my_findnext
 * Purpose: find the next file matching a pattern
 * Date:    August 4, 1997
 * Passed:  dta: file finding info
 * Notes:   my_findfirst() MUST be called before calling this function.
 *          Returns NULL if no more matching names;
 *          otherwise same as my_findfirst.
 */
FTYPE * my_findnext( FFIND *dta )
{
int i;

   i = findnext( &dta->find_info );
   if (i != OK) return NULL;
   do {
      if (dta->dirs && (dta->find_info.ff_attrib & SUBDIRECTORY)) break;
      if (wildcard( dta->pattern, dta->find_info.ff_name ) == TRUE) break;
      i = findnext( &dta->find_info );
   } while (i == OK);
   if (i != OK) return NULL; /* all done */
   strcpy( found.fname, dta->find_info.ff_name );
   if (!_USE_LFN) {
      for (i = 0; found.fname[i]; ++i)
         found.fname[i] = bj_tolower( found.fname[i] );
   }
   else i = strlen( found.fname );
   if (dta->dirs) {
      found.fsize = dta->find_info.ff_fsize;
      found.fattr = dta->find_info.ff_attrib;
      if (dta->find_info.ff_attrib & SUBDIRECTORY) {
         found.fname[i] = '/';
         found.fname[i+1] = '\0';
      }
   }
   return &found;
}


/*
 * Name:    hw_fattrib
 * Purpose: To determine the current file attributes.
 * Date:    December 26, 1991
 * Passed:  name: name of file to be checked
 * Returns: use the function in the tdeasm file to get the DOS file
 *          attributes.  get_fattr() returns 0 or OK if no error.
 */
int  hw_fattrib( char *name )
{
register int rc;
fattr_t fattr;

   rc = get_fattr( name, &fattr );
   return( rc == OK ? rc : ERROR );
}


/*
 * Name:    change_mode
 * Purpose: To prompt for file access mode.
 * Date:    January 11, 1992
 * Passed:  name:  name of file
 *          line:  line to display message
 * Returns: OK if file could be changed
 *          ERROR otherwise
 * Notes:   function is used to change file attributes for save_as function.
 */
int  change_mode( char *name, int line )
{
int    result;
fattr_t fattr;
register int rc;

   rc = OK;
   result = get_fattr( name, &fattr );
   if (result != OK)
      rc = ERROR;
   else if (result == OK && fattr & READ_ONLY) {
      /*
       * file is write protected. overwrite anyway?
       */
      if (get_yn( main6, line, R_PROMPT | R_ABORT ) != A_YES)
         rc = ERROR;
      if (rc == OK && set_fattr( name, ARCHIVE ) != OK)
         rc = ERROR;
   }
   return( rc );
}


/*
 * Name:    change_fattr
 * Purpose: To change the file attributes
 * Date:    December 31, 1991
 * Passed:  window:  pointer to current window
 */
int  change_fattr( TDE_WIN *window )
{
file_infos   *file;
TDE_WIN      *wp;
int          prompt_line;
register int ok;
fattr_t      fattr;
char         *s;
int          rc;
char         answer[MAX_COLS+2];

   prompt_line = window->bottom_line;

   answer[0] = '\0';
   /*
    * enter new file attributes
    */
   if ((ok = get_name( utils14, prompt_line, answer )) == OK) {
      if (*answer != '\0') {
         fattr = 0;
         s = answer;

         /*
          * yes, I know lint complains about "ok = *s++".
          * jmh - added ",ok" to correct it
          */
         while (ok = bj_toupper( *(s++) ),ok) {
            switch (ok) {
               case L_DOS_ARCHIVE :
                  fattr |= ARCHIVE;
                  break;
               case L_DOS_SYSTEM :
                  fattr |= SYSTEM;
                  break;
               case L_DOS_HIDDEN :
                  fattr |= HIDDEN;
                  break;
               case L_DOS_READ_ONLY :
                  fattr |= READ_ONLY;
                  break;
               default :
                  break;
            }
         }
         file = window->file_info;
         if (set_fattr( file->file_name, fattr ))
            /*
             * new file attributes not set
             */
            error( WARNING, prompt_line, utils15 );
         else {
            file->file_attrib = fattr;
            for (wp=g_status.window_list; wp!=NULL; wp=wp->next) {
               if (wp->file_info == file && wp->visible)
                  show_window_fname( wp );
            }
         }
      }
      rc = OK;
   } else
      rc = ERROR;
   return( rc );
}


/*
 * Name:    get_fattr
 * Purpose: To get dos file attributes
 * Date:    August 20, 1997
 * Passed:  fname: ASCIIZ file name.  Null terminated file name
 *          fattr: pointer to file attributes
 * Returns: OK if successful, ERROR if error
 * Notes:   FYI, File Attributes:
 *              0x00 = Normal.  Can be read or written w/o restriction
 *              0x01 = Read-only.  Cannot be opened for write; a file with
 *                     the same name cannot be created.
 *              0x02 = Hidden.  Not found by directory search.
 *              0x04 = System.  Not found by directory search.
 *              0x08 = Volume Label.
 *              0x10 = Directory.
 *              0x20 = Archive.  Set whenever the file is changed, or
 *                     cleared by the Backup command.
 */
int  get_fattr( char *fname, fattr_t *fattr )
{
int  rc;                /* return code */

   assert( fname != NULL  &&  fattr != NULL);

   ceh.flag = OK;
   *fattr = _chmod( fname, 0 );
   if (*fattr == -1) {
      *fattr = 0;
      rc = ERROR;
   } else
      rc = OK;
   if (ceh.flag == ERROR)
      rc = ERROR;
   return( rc );
}


/*
 * Name:    set_fattr
 * Purpose: To set dos file attributes
 * Date:    August 20, 1997
 * Passed:  fname: ASCIIZ file name.  Null terminated file name
 *          fattr: file attributes
 * Returns: 0 if successful, non zero if not
 */
int  set_fattr( char *fname, fattr_t fattr )
{
int  rc;                /* return code */

   assert( fname != NULL );

   ceh.flag = OK;
   rc = _chmod( fname, 1, fattr );
   if (rc != -1)
      rc = OK;
   if (ceh.flag == ERROR)
      rc = ERROR;
   return( rc );
}


/*
 * Name:    get_current_directory
 * Purpose: get current directory
 * Date:    August 7, 1997
 * Passed:  path:  pointer to buffer to store path
 *          drive: drive to get current directory
 * Notes:   append a trailing slash ('/') if it's not root
 *          path is expected to be at least PATH_MAX long
 */
int  get_current_directory( char FAR *path, int drive )
{
register int rc;
int old_drive, dummy;

   assert( path != NULL );

   rc = OK;
   ceh.flag = OK;
   _dos_getdrive( &old_drive );
   _dos_setdrive( drive, &dummy );
   if (getcwd( path, PATH_MAX ) == NULL)
      rc = ERROR;
   if (ceh.flag == ERROR)
      rc = ERROR;
   _dos_setdrive( old_drive, &dummy );
   if (rc != ERROR) {
      old_drive = strlen( path );
      if (path[old_drive-1] != '/') {
        path[old_drive] = '/';
        path[old_drive+1] = '\0';
      }
      /*
       * In LFN, the returned directory is whatever was used to get there.
       * Whilst tolerable for the dir lister (actually, I didn't notice), it's
       * no good when comparing the path for the filename display.
       */
      if (_USE_LFN) get_full_path( path, path );
   }
   return( rc );
}


/*
 * Name:    set_current_directory
 * Purpose: set current directory
 * Date:    November 13, 1993
 * Passed:  new_path: directory path, which may include drive letter
 * Notes:   djgpp's chdir will change disk as well - let's keep the old one
 */
int  set_current_directory( char FAR *new_path )
{
register int  rc;
int old_drive, dummy;

   assert( new_path != NULL );

   rc = OK;
   ceh.flag = OK;
   _dos_getdrive( &old_drive );
   if( chdir( new_path ) == ERROR)
      rc = ERROR;
   if (ceh.flag == ERROR)
      rc = ERROR;
   _dos_setdrive( old_drive, &dummy );
   return( rc );
}


/*
 * Name:    get_full_path
 * Purpose: retrieve the fully-qualified path name for a file
 * Date:    May 3, 1998
 * Passed:  in_path:  path to be canonicalized
 *          out_path: canonicalized path
 * Notes:   out_path is assumed to be PATH_MAX characters.
 *
 * 980511:  added test for non-existant file in LFN.
 */
void get_full_path( char *in_path, char *out_path )
{
  union REGS regs;
  int i;
  int rc = ERROR;

  /*
   * _fixpath doesn't seem to handle LFN too well; it converts all to lowercase
   * and doesn't expand the extended parent directories ("..." etc).
   */
  if (_USE_LFN)
  {
     regs.x.ax = 0x7160;
     regs.x.cx = 0x8002;
     regs.x.si = (unsigned)in_path;
     regs.x.di = (unsigned)out_path;
     intdos( &regs, &regs );
     if (!regs.x.cflag) {
        /*
         * Now I have to convert to lowercase and slashes myself.
         */
        out_path[0] = bj_tolower( out_path[0] );
        for (i = strlen( out_path ) - 1; i > 1; --i)
           if (out_path[i] == '\\') out_path[i] = '/';
        rc = OK;
     } else
        /*
         * A non-existant file returns error code 3; use the truename function
         * to get the path, strip the filename, convert the path, and put the
         * filename back.
         * The path test is to prevent endless recursion.
         */
        if (regs.h.al == 3 && in_path != out_path) {
           regs.x.ax = 0x7160;
           regs.h.cl = 0;
           intdos( &regs, &regs );      /* others are already set */
           if (!regs.x.cflag) {
              /*
               * At this stage out_path has the full path, but it's probably a
               * mixture of long and short components. Find the end of the path
               * and convert it.
               */
              for (i = strlen( out_path ) - 1; i > 1; --i)
                 if (out_path[i] == '\\') break;
              if (out_path[i-1] != ':') {
                 out_path[i] = '\0';
                 get_full_path( out_path, out_path );
              } else
                 /*
                  * In the root directory - just lowercase the drive
                  */
                 out_path[0] = bj_tolower( out_path[0] );
               out_path[i] = '/';
              rc = OK;
           }
        }
  }
  if (rc == ERROR)
     _fixpath( in_path, out_path );
}
