/* ------------------------------------------------------------------------ */
/* MCG2BMP.C (C) CopyLeft Bill Buckels 1991-1999                            */
/* All Rights Reversed.                                                     */
/*                                                                          */
/* Licence Agreement                                                        */
/* -----------------                                                        */
/*                                                                          */
/* You have a royalty-free right to use, modify, reproduce and              */
/* distribute this source code in any way you find useful,                  */
/* provided that you agree that Bill Buckels has no warranty obligations    */
/* or liability resulting from said distribution in any way whatsoever.     */
/* If you don't agree, remove this source code from your computer now.      */
/*                                                                          */
/* Written by   : Bill Buckels                                              */
/*                589 Oxford Street                                         */
/*                Winnipeg, Manitoba, Canada R3M 3J2                        */
/*                                                                          */
/* Email: bbuckels@escape.ca                                                */
/* WebSite: http://www.escape.ca/~bbuckels                                  */
/*                                                                          */
/* Purpose      : This utility converts MCG Screen DUMP Files and           */
/*                320 x 200 x 256 color format PCX files to BMP files       */
/*                suitable for use with WORKBOOK Version 2.1                */
/*                                                                          */
/* Revision     : 1.0 First Release                                         */
/* ------------------------------------------------------------------------ */
/* Written in Large Model MSC Version 6.00a                                 */
/* ------------------------------------------------------------------------ */

typedef unsigned char     uchar;
typedef unsigned int      uint;
typedef unsigned long     ulong;

#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <string.h>
#include <io.h>
#include <malloc.h>
#include <conio.h>

/* ---------------------------------------------------------------------- */
/* Constants and Macros                                                   */
/* ---------------------------------------------------------------------- */
#define TRUE          1
#define FALSE         0
#define SUCCESS       0
#define VALID         SUCCESS
#define FAILURE       -1
#define INVALID       FAILURE
#define FRAMESIZE     ((unsigned)64000)
#define MAX_BUFFSIZE  ((unsigned)65000)
#define RASTERWIDTH   320

/* the vga palette */

uchar rgbinfo[256][3];
uchar far *rawbuffer;

uint byteword(uchar a, uchar b){
  return b << 8 | a;
}

/* ---------------------------------------------------------------------- */
/* Check the PCX file header for a valid 256 color MCGA Full Screen       */
/* ---------------------------------------------------------------------- */
int checkforpcx(uchar *pcxheader)
{
  uint zsoft,version,codetype,pixbits;
  uint xmin, ymin, xmax, ymax, x, y;
  uint hres, vres;
  uint no_planes, bytesperline;
  int status = VALID;
  
  /* read the file header */
  
  zsoft = pcxheader[0];
  version = pcxheader[1];
  codetype = pcxheader[2];
  pixbits = pcxheader[3];
  
  if(zsoft != 10)
    status = INVALID;
  if(version != 5)
    status = INVALID;
  if(codetype != 1)
    status = INVALID;
  if(pixbits != 8)
    status = INVALID;
  
  xmin = byteword(pcxheader[4], pcxheader[5]);
  ymin = byteword(pcxheader[6], pcxheader[7]);
  xmax = byteword(pcxheader[8], pcxheader[9]);
  ymax = byteword(pcxheader[10], pcxheader[11]);
  no_planes = pcxheader[65];
  bytesperline = byteword(pcxheader[66], pcxheader[67]);
  
  x = xmax - xmin;
  y = ymax - ymin;
  
  if(x != 319)
    status = INVALID;
  if(y != 199)
    status = INVALID;
  if(no_planes != 1)
    status = INVALID;
  if(bytesperline != RASTERWIDTH)
    status = INVALID;
  
  return status;
}


/* ---------------------------------------------------------------------- */
/* Read a 256 Color 320 x 200 PCX File Into a screen buffer.              */
/* ---------------------------------------------------------------------- */
int VCXREAD(uchar *pcxfilename)
{
  uint byteoff = 0,packet,width = 0;
  int fh,status;
  uchar *packbuffer;
  uchar byte,bytecount;
  uchar *ptr,*ptr2;
  uint temp;
  uint target;
  int x,y;
  
  if((packbuffer = _fmalloc(MAX_BUFFSIZE)) == NULL)return FAILURE;
  if((fh = open(pcxfilename, O_RDONLY | O_BINARY)) == FAILURE)
  {
    free(packbuffer);
    return FAILURE;
  }
  
  target = read(fh, packbuffer, MAX_BUFFSIZE);
  close(fh);
  ptr2 = (uchar *)&packbuffer[0];
  
  if((status = checkforpcx(ptr2)) != FAILURE)
  {
    /* read the palette */
    target -= 769;
    ptr2 = (uchar *)&packbuffer[target];
    while((temp = *ptr2++) != 12);
    for(y = 0;y < 256;y++)
      for(x = 0;x < 3;x++)
    {
      temp = *ptr2++;
      rgbinfo[y][x] = temp;
    }
    
    ptr = (uchar *)&rawbuffer[0];
    ptr2 = (uchar *)&packbuffer[128];
    
    do
    {
      bytecount = 1;                   /* start with a seed count */
      byte = *ptr2++;
      
      /* check to see if its raw */
      if(0xC0 == (0xC0 &byte))
      {
        /* if its not, run encoded */
        bytecount = 0x3f &byte;
        byte = *ptr2++;
      }
      for(packet = 0;packet < bytecount;packet++)
      {
        *ptr++ = byte;
        byteoff++;
      }
    }while(byteoff < FRAMESIZE);
    
  }
  free(packbuffer);
  return(status);
}

/* ---------------------------------------------------------------------- */
/* Read a BSAVED 256 Color 320 x 200 PCX Screen c/w Palette               */
/* ---------------------------------------------------------------------- */
int BVXREAD(uchar *filename)
{
  uchar buffer[66];
  uchar buffer1[66];
  uchar buffer2[66];
  uchar *wordptr, *bufptr, *ptr;
  int fh, fh2;
  uint temp;
  int x,y;
  
  strcpy(buffer, filename);
  strcat(buffer, ".");
  wordptr = strtok(buffer, ". \n");
  sprintf(buffer1, "%s.MCG", buffer);
  sprintf(buffer2, "%s.RGB", buffer);
  
  if((fh = open(buffer1, O_RDONLY | O_BINARY)) == FAILURE)
  {
    return FAILURE;
  }
  
  if((fh2 = open(buffer2, O_RDONLY | O_BINARY)) == FAILURE)
  {
    close(fh);
    return FAILURE;
  }
  
  /* read image */
  read(fh, rawbuffer, 7);
  read(fh, rawbuffer, FRAMESIZE);
  close(fh);
  
  /* read palette */
  read(fh2, (uchar *)&rgbinfo[0][0], 7);
  read(fh2, (uchar *)&rgbinfo[0][0], 768);
  for(y = 0;y < 256;y++)
  {
    for(x = 0;x < 3;x++)
    {
      temp = rgbinfo[y][x];
      rgbinfo[y][x] = (temp << 2);
      /* bump color from 18 bits to 24 bits */
    }
  }
  close(fh2);
  
  return SUCCESS;
}

unsigned char BMP_header[]={
0x42, 0x4D, 0x36, 0xFE, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x36, 0x04, 0x00, 0x00, 0x28, 0x00, 
0x00, 0x00, 0x40, 0x01, 0x00, 0x00, 0xC8, 0x00, 
0x00, 0x00, 0x01, 0x00, 0x08, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0xFA, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 
0x00, 0x00, 0x00, 0x01, 0x00, 0x00};

/* ---------------------------------------------------------------------- */
/* Write a 256 Color 320 x 200 Windows BMP file                           */
/* ---------------------------------------------------------------------- */

int makebmp(char *infile, char *bmpfile)
{
  unsigned int temp;
  unsigned char *screenbuffer;
  int x, y;
  FILE *fp;
  
  
  printf("Infile  : %s\n", infile);
  printf("Outfile : %s\n", bmpfile);
  printf("    <= scanline");
  
  fp = fopen(bmpfile, "wb");
  
  if(NULL == fp) 
  {
    puts("Unable to create bmp file.");
    return FAILURE;
  }
  else 
  {
    for(x = 0; x < sizeof(BMP_header); x++) 
    {
      temp = BMP_header[x];
      fputc(temp, fp);
    }
    
    for(y = 0;y < 256;y++)
    {
      /* RGB Quad Structure b,g,r,0 */
      for(x = 3;x > 0;x--)
      {
        temp = rgbinfo[y][x - 1];
        fputc(temp, fp);
      }
      fputc(0, fp);
      
    }
    for(y = 200; y > 0; y--) 
    {
      printf("\r%0003d", y);
      
      screenbuffer = (char *)&rawbuffer[((y - 1)*320)];
      for(x = 0; x < 320; x++) 
      {
        temp = screenbuffer[x];
        fputc(temp, fp);
      }
    }
    
    fclose(fp);
    printf("\rDone !                    \n");
  }
  return SUCCESS;
}


void main(int argc, char **argv)
{
  char infile[128];
  char bmpfile[128];
  int status = 1;   // non=zero
  
  char *wordptr;
  
  puts("MCG2BMP(C) v1.0 CopyLeft 1991-1999 Bill Buckels\nAll Rights Reversed.");
  
  if(argc == 2)
  {
    if((rawbuffer = malloc(MAX_BUFFSIZE)) == NULL) 
    {
      puts("Sorry... Insufficient Memory.");
      exit(1);
    }
    
    /* make file names */
    strcpy(infile, argv[1]);
    
    if(VCXREAD(infile)) 
    {
      if(BVXREAD(infile)) 
      {
        free(rawbuffer);
        puts("Sorry... unsupported format.");
        exit(status);
      }
    }
    
    strcat(infile, ".");
    wordptr = strtok(infile, ".");
    sprintf(bmpfile, "%s.BMP", infile);
    status = makebmp(argv[1], bmpfile);
    free(rawbuffer);
  }
  else
    printf("Usage is \"MCG2BMP [filename]\"\n");
  
  exit(status);
  
}
