#!/usr/bin/perl

# DBF2CSV.PL -- a program to convert .DBF files to .CSV format
#
# This program is uncopyrighted, so do with it whatever you wish.
# By Dave Burton,  v1 5/28/2000,  v2 4/14/2003,  v3 6/27/2003.
# Burton Systems Software, POB 4157, Cary, NC 27519-4157.
# email: dave@burtonsys.com

# The latest version of DBF2CSV can always be found on the Burton
# Systems Software web site, in the "downloads" area:
# http://www.burtonsys.com/


######## HISTORY ########
#
# Dbf2csv v2 handles some ideosyncratic .dbf files with non-standard
# headers, which confused v1.
#
# Dbf2csv v3 added support for fields > 255 characters thanks to
# Jeff Price, and for big-endian machines (for John McVeagh).
#
# v1 was modified (commented, corrected & enhanced to handle accented
# characters) by Jacky Bruno <jacky.bruno@free.fr> collge de Villeneuve
# sur Yonne 89500, in april 2003.  His version was included in some of
# the dbf2csv.zip distributions as dbf2csv_accentued_characters.pl.
#
# Dbf2csv v4 is the result of merging most of Jacky's changes into
# the standard version of DBF2CSV.PL, plus a few more improvements.
#
# [Shameless advertisement: of course I used the TLIB Version Control
# "migrate" command to do the merge and create v4.  See our web site,
# http://www.burtonssy.com/   -DAB]
#
#########################

# This program is uncopyrighted, so it can be modified by anyone who
# wants to.  But, out of courtesy, please add your own name and what
# you did to the history, and do not remove the previous history.
# -DAB


# Here are some descriptions of the .dbf file format:
#   http://www.e-bachmann.dk/computing/databases/xbase/dbf.html
#   http://www.klaban.torun.pl/prog/pg2xbase/contrib/dbf-format.html
#   http://support.microsoft.com:80/support/kb/articles/q98/7/43.asp


# Jacky Bruno's additional comments...
# The script structure is this one:
# lines 568 to the end : main program :
#   - it reads command line: if no parameters, then it shows help
#   - treat flags if there are some (d, p)
#   - reads the first file name given and give it to the do_a_file function
#   - do the same with other command line given files
# lines 535 to 565 : do_a_file function 'the name is clear) :
#   - verify the validity of the file (ending with .dbf)
#   - buids the .csv file from the .dbf file name given
#   - calls the cvt1file function by giving her the 2 file names
#   - get the records's number and shows informations:
#      input file   output file		treated records number
# lines 164 to 530 : cvt1file function :
#   - as his name tells, do the conversion job of input file
#   - write to screen informations of file beginning and field names
#   - save output file :
#       first field name, then each record


# In the output file, fields will be separated by $separe
#  $separe=";";
$separe=",";

$| = 1;   # predefined variable. If <> 0 then each print to the console
          # will immediatly be displayed, instead of buffered.

$debugmode = 0;  # set to 1 via '-d' for debug prints
$prompt = 0;  # set to 1 via '-p' for special interactive mode, for use with perl4w.exe

# Perl pack/unpack format representing the structure of the first
# 32 bytes in a .DBF file:
$DBF_header_fmt = "C" .   # version number at offset
                  "CCC" . # YY, MM, DD (one byte each)
                  "L" .   # Number of records in file
                  "S" .   # Length of header structure
                  "S" .   # Length of each record
                  "a20";  # 20 bytes that we don't care about

# Perl pack/unpack format representing the structure of each field descriptor
# (the 2nd-Nth 32-byte chunk):
$DBF_field_desc_fmt = "A11" .  # Field name in 0-terminated ASCII
                      "a" .  # Field type in ASCII
                      "L" .  # Field address in memory (unused)
                      "C" .  # Field length (binary)   \___/ these 2 bytes can also be a 2-byte field length,
                      "C" .  # Decimal count (binary)  /   \ 1-65535, for field type 'C' in Foxbase & Clipper.
                      "a2" .  # reserved
                      "C" .  # Work area ID
                      "a2" .  # reserved
                      "C" .  # Flag for SET FIELDS
                      "a7" . # reserved
                      "A";   # Index field flag

# For the meanings of the template letters, see Perl documentation
# (e.g., on Linux do 'man perlfunc' then read 'pack')

# Unfortunately, the "v" & "V" template characters (Vax-byte-order
# signed integers) don't come in unsigned flavors, and the "S" & "L"
# template characters (machine-order unsigned integers) won't
# work on big-endian machines.  So we do the best we can: on big
# endian machines we change the 'S' template characters to 'v'
# and the 'L' template character to 'V'.  That'll work 99.9% of the
# time -- i.e., as long as the record length doesn't exceed 32K.
# Thanks to John McVeagh (who uses AIX) for inspiring this.
$tst = pack("S",513);
$tst_big_endian = unpack("n",$tst);
$tst_little_endian = unpack("v",$tst);
if ((513==$tst_big_endian) && (258==$tst_little_endian)) {
   # this is a big endial machine
   $DBF_header_fmt =~ s/LSS/Vvv/;
   print "Note: This is a big-endian machine.  Adjusting template.\n";
}


# $cvt_failed is a side-effect result of &cvt1file.
$cvt_failed = 0;  # will be set to 1 iff cvt1file failed, or 0 for success

# handy constant
$zerobyte = pack("c",0);   # In Perl 5 this would be chr(0);


# use the '-ta' or '-tu' option to adjust $translate and $garde_accent


# By Jacky Bruno...
# The file can have accentued characters coded in DOS pc style (where, for example ""
# is coded "82h") or coded in ANSI style (linux or Windows) ( "" is coded "E9h")
# If codage is already ANSI, no need to re-code it: put $translate to 0
# If codage is pc, you can choose to re-code or let codage the way it is.
# Accent codage change? codage pc --> codage this way:
# 1  -> yes, let's change codage (another number is possible)
# 0  -> no, let's keep the codage the way it is in the original file
$translate=0;

# If codage is changed ($translate=1), do we keep accentued characters?
# Translation keeping or not accentued characters
# 1  -> keeping accentued characters (or another number)
# 0  -> don't keep accentued characters
$garde_accent=0;	#In french: keep = garde

# Conversion tables codage pc <--> codage ansi for accentued characters
# They can be completed regarding the "correspondances"
# (every character must respect the same order in every codage)
$code_pc="\x81\x82\x83\x84\x85\x87\x88\x89\x8a\x8b\x8c\x94\x96\x97";
$code_ansi="";
$code_brut="ueaaaceeeiiouu";


####################################
###  end of global variables
####################################


# Read infile (name of a .DBF file) and create outfile (name of a .CSV file).
#
sub cvt1file {
   local( $infile, $outfile ) = @_;
   local( $recnum ) = 0;
   local( $skipped ) = 0;
   local( $offset_of_0 );
   local( $buf ) = "";
   local( $reclens_look_right ) = 0;
   local( $offsets_match_fieldlens );
   local( $i );

   $cvt_failed = 0;  # side-effect result, 0 or 1

   # open dbf file
   if (!open( DBF, $infile )) {
      print "ERROR: Could not open '$infile', $!\n";
      $cvt_failed = 1;
      return 0;  # no records converted
   }
   # We must read the input .dbf file in binary mode (avoid translations of end-lines)
   binmode DBF;

   # get size of .dbf input file
   $DBF_file_size = -s DBF;

   # Remove old output file if it exists
   unlink $outfile;  # mostly in case outfile is on a buggy MARS_NWE volume, so we don't get trailing junk in the output file if it already existed and was bigger than the new output file

   if (!sysread(DBF, $buf, 32)) {
      print "ERROR: Could not read first 32 bytes from '$infile', $!\n";
      $cvt_failed = 1;
      return 0;  # no records converted
   }

   # Unpack the file header fields from the first 32 bytes of the .dbf file
   # The $DBF_header_fmt template is defined above
   ( $verNum, $yy, $mm, $dd, $numrecs, $hdrLen, $recLen ) = unpack( $DBF_header_fmt, $buf );

   print "version=$verNum  ";
   printf "yy/mm/dd=%02d/%02d/%02d  ", $yy,$mm,$dd;
   # printf "jj/mm/aa=%02d/%02d/%02d  ", $dd,$mm,$yy;  # -- French (by Jacky Bruno) --
   print "numrecs=$numrecs";
   # print "nombreEnregValides=$numrecs";  # -- French (by Jacky Bruno) --
   $calculated_numrecs = int(($DBF_file_size - $hdrLen) / $recLen);
   print ", calculated numrecs=$calculated_numrecs.\n";
   if ($numrecs != $calculated_numrecs) {
      print "ERROR: numcres from header unequal to calculated number of records.\n";
      if ($calculated_numrecs < $numrecs) {
         print "$infile might be incomplete.\n";
      } else {
         printf "Final %d records are suspect.\n", $calculated_numrecs-$numrecs;
      }
   }
   print "hdrLen=$hdrLen  ";
   # print "hdrLong=$hdrLen  ";  # -- French (by Jacky Bruno) --
   print "recLen=$recLen  ";
   # print "enregLong=$recLen  ";  # -- French (by Jacky Bruno) --
   $numfields = int(($hdrLen - 1) / 32) - 1;
   print " numfields = $numfields per record.\n";
   # print "nombreChamps=$numfields (by record)\n";  # -- French (by Jacky Bruno) --
   $extra_hdr_bytes = ($hdrLen - (1+(($numfields+1)*32)));
   if ($extra_hdr_bytes != 0) {
      print "Warning: non-standard .dbf file format, header contains $extra_hdr_bytes extra byte(s).\n";
   }
   $extra_file_bytes = ($DBF_file_size - ($hdrLen + ($calculated_numrecs * $recLen)));
   if ($extra_file_bytes > 0) {
      print "Warning: $infile contains $extra_file_bytes extra byte(s) at the end (ignored).\n";
   }

   # $recfmt will be the unpacking template for each record.
   # This template will be build by reading field's definitions
   # (32 bytes per field starting at the 33rd byte of the file)
   $recfmt = "A";  # first byte of each record is the "deleted" indicator byte (normally blank)

   # We will build arrays containing fields caracteristics
   # (name, type, width, offset).

   # The [0] array entries are for the "deleted" indicator byte:
   $fld_nam[0] = '';
   $fld_ofs[0] = 0;
   $fld_len[0] = 1;
   $fld_typ[0] = 'C';

   $running_offset = 1;  # 1, not 0, because the "deleted" indicator is 1 byte

   # read all the field definition headers (32 bytes each):

   for ($i=1; $i <= $numfields; $i++) {
      if (!sysread(DBF, $buf, 32)) {
          print "ERROR: Could not read field definition header for field $i from '$infile', $!\n";
          $cvt_failed = 1;
          return 0;  # exit with error
       }

      # Unpack field definition using $DBF_field_desc_fmt template (we keep
      # only the first 5 fields):
      ( $fldName, $fldType, $fldOffset, $fldLen, $decCnt ) = unpack( $DBF_field_desc_fmt, $buf );

      # I don't know why the dumb A11 format doesn't strip the garbage after
      # the 0-byte, but it doesn't.  The Perl documentation says, "When
      # unpacking, 'A' strips trailing spaces and nulls," but that apparently
      # doesn't mean that it truncates at the first null byte.  We could use
      # "Z11" instead of "A11" if we didn't care about Perl 4 compatibility.
      # Most .dbf files don't have trailing garbage after the 0-byte, anyhow,
      # but some do.  This is for those .dbf files.
      $offset_of_0 = index($fldName, $zerobyte);
      if (-1 != $offset_of_0) {
         $fldName = substr( $fldName, 0, $offset_of_0 );
      }

      # Some xBase variants (Clipper, Foxbase, perhaps others) permit
      # character data fields larger than 255 characters, using the
      # "Decimal Count" field as a high length byte.  (Thanks to Jeff
      # Price <jeff.price@rocketmail.com.nospam> for telling me this.)
      if (($decCnt > 0) && ('C' eq $fldType) && ($recLen >= (256 * $decCnt))) {
         $fldLen += (256 * $decCnt);
      }

      if ($debugmode) {
         printf "%3d: %-10s type='%s' offset=%d fldLen=%d\n", $i, $fldName, $fldType, $fldOffset, $fldLen;
      }
      $fld_nam[$i] = $fldName;
      $fld_ofs[$i] = $fldOffset;
      $fld_len[$i] = $fldLen;
      $fld_typ[$i] = $fldType;

      # Add another field to the template, type 'A' (text completed by spaces) with $fldLen width
      $recfmt .= "A$fldLen";

      $running_offset += $fldLen;
   } #for
   # The $recfmt unpacking template is complete

   # Jacky Bruno comments...
   # Definition of output format
   # Here will be used the field separator
   # (defined by $separe variable modifiable at the beginning of the file)
   # You can change quotes used at the left and the right of fields too
   # using another character (is it really useful?) by changing " in the
   # next variable to the wished character (you can even set a beginning
   # character and a ending character) example :
   #   $outfmt = 'Y%sZ' . ("${separe}Y%sZ" x ($numfields-1)) . "\n";
   # field names will have Y before and Z after : Yname_of_fieldZ
   # Attention to escape special characters if used
   # The () tells that ${separe}"%s" will be repeated ($numfields-1) times

   $outfmt = '"%s"' . ("${separe}\"%s\"" x ($numfields-1)) . "\n";

   # note: the DelFlg fields won't be output

   if ($running_offset != $recLen) {
      print "Warning: Summed field lengths = $running_offset, which is unequal to recLen.\n";
      $reclens_look_right = 0;
   } else {
      print "Summed field lengths=$running_offset=recLen (as expected).\n";
      $reclens_look_right = 1;
   }

   ### Begin code to fix field offsets for .dbf files in which the field
   ### offsets are incorrect or missing altogether

   # Are two or more fields at the same field offset?  If so then the .dbf
   # file definitely doesn't have correct field offsets in the header.
   $prev_fldOffset = $fld_ofs[1];
   $cnt_idential_fldOffsets = 0;
   for ($i=2; $i <= $numfields; $i++) {
      $fldOffset = $fld_ofs[$i];
      if ($fldOffset == $prev_fldOffset) {
         $cnt_idential_fldOffsets++;
      }
      $prev_fldOffset = $fldOffset;
   } #for

   # Tell the user about the identical field offsets
   if ($cnt_idential_fldOffsets > 0) {
      $cnt_idential_fldOffsets++;
      print "Warning: ";
      if ($cnt_idential_fldOffsets == $numfields) {
         print "All ";  # say "All nn fields have identical offsets."
      }
      print "$cnt_idential_fldOffsets fields have identical offsets.\n";
      $silent = 0;
      if ($cnt_idential_fldOffsets == $numfields) {
         print "Note: $infile is in a non-standard .dbf format (such as Alpha-4's),\n" .
               "in which the field offsets are missing from the header.\n" .
               "The offsets will be recalculated from the summed field lengths.\n";
         # Mark Godhelf reported that for his Alpha-4's .dbf files $fldOffset is always zero.  12/13/2002
         # Stephane Boireau had a .dbf file in which all the $fldOffsets were 383.  4/13/2003
         $silent = 1;
      }
   }

   # Check whether or not the field offsets are consistent with the field lengths,
   # and if they are not then tell the user about the problem (unless we already
   # told him that all the field offsets are identical).
   $running_offset = 1;
   $offsets_match_fieldlens = 1;
   for ($i=1; $i <= $numfields; $i++) {
      $fldLen = $fld_len[$i];
      $fldOffset = $fld_ofs[$i];
      if (($running_offset != $fldOffset) && !$silent) {
         print "ERROR: field $i: running calculated offset, $running_offset, does not match field offset from header, $fldOffset.\n";
         $offsets_match_fieldlens = 0;
      }
      $running_offset += $fldLen;
   } #for

   if ((!$offsets_match_fieldlens) && ($reclens_look_right || ($cnt_idential_fldOffsets > 0))) {
      # fix the field offsets by calculating them from the summed field lengths
      if (!$silent) {
         print "The offset(s) will be recalculated from the summed field lengths.\n";
      }
      $running_offset = 1;
      for ($i=1; $i <= $numfields; $i++) {
         $fldLen = $fld_len[$i];
         $fld_ofs[$i] = $running_offset;
         $running_offset += $fldLen;
      } #for
   }

   ### End of code to fix field offsets

   # Read last byte of header (at the end of fields definitions), which should be LF
   if (!sysread(DBF, $buf, 1)) {
      print "ERROR: Could not read terminator byte from '$infile', $!\n";
      $cvt_failed = 1;
      return 0;  # no records converted
   }
   if ("\r" ne $buf) {
      printf "ERROR: Header-terminator byte at offset %d is 0x%02x (it should be 0x0D)\n", $hdrLen-1, ord($buf);
   } elsif ($extra_hdr_bytes) {
      printf "Header-terminator byte is 0x%02x (as expected).\n", ord($buf);
   }
   if ($extra_hdr_bytes) {
      # Usually the header is correctly followed by data records.
      # But sometimes it isn't right. Sometimes there is a 0 byte,
      # sometimes there is a field that tells the link to a file.
      # Stephane Boireau had a .dbf file in which a zero byte followed the normal 0x0D terminator byte.  4/13/2003
      if (!sysread(DBF, $buf, $extra_hdr_bytes)) {
         print "ERROR: Could not read the $extra_hdr_bytes extra header bytes from '$infile', $!\n";
         $cvt_failed = 1;
         return 0;  # no records converted
      }
      print "The $extra_hdr_bytes extra header byte(s) are:";
      for ($i=0; $i<$extra_hdr_bytes; $i++) {
         $tmp = substr($buf, $i, 1);
         printf " 0x%02x", ord($tmp);
      }
      print "\n";
   }

   if (!open( OUTP, ">$outfile" )) {
      print "ERROR: Could not create '$outfile', $!\n";
      $cvt_failed = 1;
      return 0;  # no records converted
   }

   # echo field names to console:
   @tmp = @fld_nam;
   shift @tmp;  # don't output field #0 (the 'deleted' flag)
   foreach $fld (@tmp) {
      # remove leading and trailing whitespace and any embedded quote marks
      $fld =~ s/\s*(\S((.*\S)|)|)\s*/$1/;
      # The aim of this expression before is to take care of fields that have only one letter
      # It can be changed to another expression     $fld =~ s/\s*(\S.*\S)\s*/$1/;
      # but it won't care about one letter long fields
      $fld =~ s/\"/\`/g;
   }
   print "The $numfields fields in each record are named:\n";
   printf $outfmt,@tmp;

   # output field names as first line in output file:
   # Write name fields first in output file
   printf OUTP $outfmt,@tmp;

   $recnum = 0;

   # Then read & convert each record
   while (sysread(DBF, $buf, $recLen)) {
      if ((1 == $extra_file_bytes) && (1 == length($buf))) {
         # For some reason, some .dbf files seem to have an extra ctrl-Z at the end
         if (26 == ord($buf)) {
            print "Trailing ctrl-Z ignored.\n";
         } else {
            printf "Warning: ignored final (extra) character at end of %s: 0x%02x\n", $infile, ord($buf);
         }
         last;
      }

      $recnum++;

#      if ($recnum > 5000) {
#         last;  # for debugging
#      }

      # Write a dot every 2000 records
      if (0 == ($recnum % 2000)) {
         print '.';
      }
      if ($buf =~ /\~/) {
         print "Warning: changed '~' to space in record $recnum\n";
         $buf =~ tr/\~/ /;
         if ($buf =~ /\~/) {
            print "ERR: failed to change '~' to space in record $recnum\n";
            exit 1;
         }
      }
      # Translation of pc codes to ansi (or untranslated) if asked
      if ($translate) {
        if ($garde_accent) {
          eval "\$buf=~ tr /$code_pc/$code_ansi/;"; # Let's keep accentued characters
        } else {
          eval "\$buf=~ tr /$code_pc/$code_brut/;"; # Let's remove accentued characters
        }
      }

      # Unpack record according to $recfmt template
      @fields = unpack( $recfmt, $buf ); # the fields of @fields are record fields
      foreach $fld (@fields) {
         if ($fld =~ /[\s\"]/) {
            # remove leading and trailing whitespace and any embedded quote marks
            $fld =~ s/\s*(\S((.*\S)|)|)\s*/$1/;
            $fld =~ s/\"/\`/g;
         }
      }

      # Remove the first field (1-byte flag) that tells if the record is
      # deleted (value 2Ah= "*") or valid (value 20h = space, but the space was
      # eliminated above, when we removed leading and trailing whitespace).
      $deleted_flag = shift @fields;

      # If you want to include "deleted" records in the output .csv file, then
      # comment-out the next five lines:
      if ($deleted_flag ne '') {
         $skipped++;
         print "Warning: record $recnum is marked for delete; $skipped records skipped.\n";
         next;
      }
      # write the converted record, using the format built above
      printf OUTP $outfmt,@fields;

   } #while
   close OUTP;  # close output file
   close DBF;  # close input file
   if ($recnum >= 2000) {
      print "\n";  # because progress-indicator dots were printed
   }

  # $recnum -= $skipped;  # account for deleted records
  # Hmmmm... Bruno thinks that is needed here, but I don't think so.
  # The question to ask is: does the 'numrecs' field in the .dbf file
  # header include records marked as 'deleted' or not?  I think it should
  # include them, so $recnum shouldn't be decremented by $skipped.

  if ($recnum != $numrecs) {
     print "Warning: file should have had $numrecs records, but actually had $recnum records.\n" .
           "Calculated numrecs=$calculated_numrecs.\n";

     # Since I might be wrong, and Bruno might be right (at least for some
     # database systems), I added the following message:
     if (($recnum-$skipped) == $numrecs) {
        print "Note: The disparity seems to be accounted for by $skipped deleted records.\n" .
              "Please tell Dave Burton, dave\@burtonsys.com\n";
     }
  }
  return $recnum;

} #cvt1file


$errlevel = 0;

sub do_a_file {
   local( $infile ) = @_;
   local( $numRecords, $outFile );
   if ($inFile !~ /\.dbf$/i) {
      if ("-d" eq $infile) {
         $debugmode = 1 - $debugmode;
      } else {
         printf "ERROR: input file name '$inFile' does not end in '.dbf'\n";
         $errlevel = 2;
      }
   } else {
      $outFile = $inFile;
      if ($inFile =~ /\.DBF$/) {
         # if input file name was upper-case, make output file name upper-case, too
         # (the same in french) si fichier d'entre est en majuscules, fichier de sortie en majuscules aussi
         $outFile =~ s/\.DBF$/\.CSV/i;
      } else {
         $outFile =~ s/\.dbf$/\.csv/i;
      }

      # Display on the console the file names and number of records
      print "Input='$inFile'  Output='$outFile'\n";
      $numRecords = &cvt1file( $inFile, $outFile );
      if ($cvt_failed) {
         $errlevel = 2;
      } else {
         $numRecords++;  # add one for the first record, with the field names in it
         print "Created $outFile from $inFile, $numRecords records.\n";
      }
   }
} #do_a_file


# Main program
# Test if there are parameters
if (($#ARGV+1) < 1) {  # no, show help
   print "DBF2CSV v4 -- Convert .DBF file to .CSV (comma-separated) format\n" .
         "\n" .
         "Usage:\n" .
         "   perl4w32 dbf2csv.pl file.dbf ...\n" .
         "or (under MS-DOS):\n" .
         "   perl4s dbf2csv.pl file.dbf ...\n" .
         "or (if you have a 32-bit Perl installed):\n" .
         "   perl dbf2csv.pl file.dbf ...\n" .
         "or (to run interactively under Windows and prompt for the .dbf file):\n" .
         "   perl4w dbf2csv.pl -p\n" .
         "\n" .
         "For each input file.dbf, an output file.csv will be created.\n" .
         "There will be one more record in file.csv than file.dbf, because the\n" .
         "field names are written to file.csv as the first record.\n" .
         "A dot will printed for every 2000 records, as a progress indicator.\n" .
         "\n" .
         "Options:  (e.g., \"perl4w32 dbf2csv.pl -d infile.dbf\".)\n" .
         "   -d    to enable debug prints\n" .
         "   -p    special intaractive mode for use with perl4w.exe\n" .
         "   -ta   to translate DOS PC-style accented characters to ANSI\n" .
         "   -tu   to translate DOS PC-style accented characters to unaccented\n" .
         "\n" .
         "Limitations:  Memo fields are unsupported.  Also, if you use a 16-bit version\n" .
         "of Perl, such as perl4s or perl4w, then you must use \"8.3\" (short) file names.\n";
   exit 1;
} else {
   while ($#ARGV >= 0) {
      $inFile = $ARGV[0];
      shift @ARGV;
      if ("-d" eq $inFile) {
         $debugmode = 1;
      } elsif ("-p" eq $inFile) {
         # Special interactive mode for use with Perl4w.exe
         $prompt = 1;
      } elsif ("-ta" eq $inFile) {
         # translate DOS PC-style accented characters to to ANSI-style accented characters
         $translate = 1;
         $garde_accent = 1;
      } elsif ("-tu" eq $inFile) {
         # translate DOS PC-style accented characters to to unaccented characters
         $translate = 1;
         $garde_accent = 0;
      } elsif ("-tn" eq $inFile) {
         # no translations
         $translate = 0;
      } else {
         &do_a_file( $inFile );
      }
   } #while
   if ($prompt) {
      print "DBF2CSV v4 -- Convert .DBF file to .CSV (comma-separated) format\n" .
            "\n" .
            "For each .dbf input file that you specify, an output file will be\n" .
            "created with the same name but a .csv extension.\n" .
            "\n" .
            "For big files, a dot will printed for every 2000 records, as a\n" .
            "progress indicator.  There is no limit on the size of the files,\n" .
            "but big files may take a long time to convert.\n" .
	    "\n" .
            "There will be one more record in the .csv output file than in the .dbf\n" .
            "file, because the field names are written to the .csv file as the\n" .
            "first record.\n" .
            "\n" .
            "Options (you may enter an option instead of a file name):\n" .
            "   -d to enable debug prints\n" .
            "   -ta to translate DOS-style accented characters to ANSI\n" .
            "   -tu to translate DOS-style accented characters to unaccented\n" .
            "\n" .
            "Note that you might have to specify the full path of each .dbf file.\n";
      do {
         print "\n";
         if ($debugmode) {
            print "[debugmode enabled]\n";
         }
         print "Convert what .DBF file?  (or press Enter alone to quit) ";
         $inFile = <STDIN>;
         # remove leading and trailing whitespace (especially the CR at the end)
         $inFile =~ s/\s*(\S((.*\S)|)|)\s*/$1/;
         if ("-d" eq $inFile) {
            $debugmode = 1;
         } elsif ("-d0" eq $inFile) {
            $debugmode = 0;
         } elsif ("-ta" eq $inFile) {
            # translate DOS PC-style accented characters to to ANSI-style accented characters
            $translate = 1;
            $garde_accent = 1;
         } elsif ("-tu" eq $inFile) {
            # translate DOS PC-style accented characters to to unaccented characters
            $translate = 1;
            $garde_accent = 0;
         } elsif ("-tn" eq $inFile) {
            # no translations
            $translate = 0;
         } elsif ('' ne $inFile) {
            &do_a_file( $inFile );
         }
      } until ('' eq $inFile);
   }

   if (! $prompt) {
      exit $errlevel;
   }
   # An idiosyncracy of Perl4w.exe is that if you exit by dropping off
   # the end of the program it closes the Window, but if you exit by
   # calling 'exit' or 'die' then it leaves the window open.  Since we
   # want the window to close, we don't call 'exit'.
}

__END__

