/*  NDX.H   . . .   Header file for index package   */

#ifndef NDX_H
#define NDX_H

#include    <string.h>
#include    <stdio.h>
#include    <dir.h>

typedef unsigned short  WORD;
typedef unsigned long   DWORD;
typedef unsigned char   BYTE;

#define MAJOR       3
#define MINOR       4

#define HDRSIZE     512

#define POP(a)      a->pop(a)
#define PUSH(a, b)  b->push(a)
#define FIELDOFFSET(type, field) ((WORD)&(((type *)0)->field))

const unsigned stackdepth = 10;     // Maximum tree height
const unsigned nodesize =  512;     // Bytes in each node
const unsigned maxkey = 100;        // Maximum key length
const unsigned maxendkeys = nodesize - sizeof(DWORD) - sizeof(WORD);
                                    // Maximum keys data length
const unsigned NCACHE = 10;         // Number of node buffers
enum  types {NODUPS, FIFO, LIFO};   // File access modes

typedef int (*CFN)(const char *, const char *, ...);// Comparison function types

class Node;
class Index;

class Key {                 // Key structure
public:
    DWORD   lson;               // Pointer to left son (seek address in file)
    DWORD   offset;             // Record seek address or number
    char    data[maxkey];       // ASCIZ key
    Key(){lson = offset = 0L; }
    Key(const char *, DWORD);
protected:
    friend  Node;
    friend  Index;
    void    push(Node *);
    Node *  pop(Key *&);
    DWORD   insert(DWORD);
    Key     *prev(Node *);                  // Returns previous Key in the Node
    Key     *next(){return (Key *)((char *)this + length());}
    short    length() {return 2*sizeof(DWORD) + strlen(data) + 1;}
                               // Length of the Key
    Key  *keycpy(Key *s){return (Key *)(memcpy(this, s, s->length()));}
};

class Node {
    friend Key;
    friend Node;
    friend Index;
    DWORD       offset;     // Node offset or free node chain
    WORD        endkeys;    // First unused data byte in keys
    char        keys[maxendkeys];   // The Key structures
    DWORD   *rson(){return (DWORD *)(keys + endkeys);} // Pointer to right son
    split();                                    // Split node in two
    void    shiftup(Key *, unsigned);           // Make room for key
    void    shiftdn(Key *, unsigned, unsigned); // Delete key
};

class Frame {               // State stack frame
    friend Key;
    friend Node;
    DWORD   node;           // Offset of node in the index file
    WORD    offset;         // Offset of the Key in that node
};

typedef struct{             // Cache entry
    char        dirty;      // Node needs writing
    Index       *index;     // Index structure
    Node        *node;      // File block content (BTree node)
} CACHE;

class Index {               // THE index structure
    friend Key;
    friend Node;
    DWORD   root;           // File offset of the root node
    DWORD   eof;            // File offset of the end of file
    DWORD   freelist;       // File offset of free node list
    BYTE    minor,          // Version number
            major,
            mode;           // File access mode
    WORD n;             // File size (# keys, that is)
        // Fields above are stored in the index file at offset 0 //
    BYTE    stacktop;           // Current stack top index
    Frame   stack[stackdepth];  // Current state
    Key     ckey;               // Current key structure
    CFN     cmpfn;              // find(), etc., comparison function
    char    filename[MAXPATH];  // Index filename
    FILE    *handle;            // Index file handle
    static  unsigned hdrsize;       // Size of Index data written to disk
    static  unsigned cache_users;   // Cache user count
    static  CACHE    cache[NCACHE]; // The cache itself
    void    initcache();
    Node   *newnode();              // Get an empty node (maybe from freelist)
    Node   *getnode(DWORD);         // Get a specific node
    DWORD  _find(const char *, DWORD);   // Inner key search routine
    void   read(DWORD, void *, unsigned);  // Read header or node
    void   write(DWORD, void *, unsigned, Index *); // Write header or node
  public:
    Index(char *, int, CFN = (CFN)strcmp);
                                       // Constructor to create new index
    Index(char *, CFN = (CFN)strcmp);  // Constructor to open existing one
    ~Index();                          // Destructor, closes index
    const WORD count(){return n;}  // Return file size (# records)
    isvalid();                            // Test index validity (1 if valid)
    const   Key& key(){return ckey;};     // Return current key
    DWORD   insert(const char *, DWORD);  // Insert new key
    DWORD   remove(const char *, DWORD);  // Remove key
    DWORD   find(const char *);           // Find key
    DWORD   findkeyrec(const char *, DWORD); // Find key, specific record
    DWORD   first(unsigned = 0);       // First (last) key
    DWORD   next();                    // Next key
    DWORD   prev();                    // Previous key
    void    print(char *);             // Used only by TEST.CPP
};

#endif  //NDX_H
