{************************************************}
{   GFonts.pas                                   }
{   Graph Vision unit                            }
{   Sergey E. Levov, Moscow,1992-1993            }
{************************************************}

unit GFonts;
{$F+,O+,S-,D-}
interface
uses Objects,Graph;
type

   PFont = ^TFont;
   TFont = object(TObject)
      FontName : String;
      constructor Init(AName : string);
      function TextWidth(S : String) : integer; virtual;
      function TextHeight(S : String) : integer; Virtual;
      procedure OutText(S : String); virtual;
      procedure OutTextXY(Location : TPoint; S : String); virtual;
      procedure SetTextStyle(Direction,Size : word); virtual;
      function GetFontName : string;
      function CanFillBackground : boolean; virtual;
      function GetCharHeight(Ch : Char) : word; virtual;
      function GetCharWidth(Ch : Char) : word;  virtual;
   end;

   PBGIFont = ^TBGIFont;
   TBGIFont = object(TFont)
      ID : integer;
      constructor Init(AName : String; AID : integer);
      function TextWidth(S : String) : integer; virtual;
      function TextHeight(S : String) : integer; Virtual;
      procedure OutText(S : String); virtual;
      procedure OutTextXY(Location : TPoint; S : String); virtual;
      procedure SetTextStyle(Direction,Size : word); virtual;
      function CanFillBackground : boolean; virtual;
      function GetCharHeight(Ch : Char) : word; virtual;
      function GetCharWidth(Ch : Char) : word;  virtual;
   end;

   PDefaultBGIFont = ^TDefaultBGIFont;
   TDefaultBGIFont = object(TBGIFont)
      procedure OutText(S : String); virtual;
      procedure OutTextXY(Location : TPoint; S : String); virtual;
   end;

   PRasterFont = ^TRasterFont;
   TRasterFont = object(TFont)
     FirstChar    : byte;       {first char code in image table}
     CharCount    : byte;       {chars in table (one less) }
     ImageTable : Pointer;      {pointer to font definition table}
     CharHeight : word;         {real char height}
     constructor Init(AName : String; AImageTable : Pointer; First,Count : byte);
     function BytesPerChar(Ch : Char) : word; virtual;
     function GetCharImage(Ch : Char) : Pointer; virtual;
     function GetCharHeight(Ch : Char) : word; virtual;
     function TextHeight(S : String) : integer; virtual;
     function TextWidth(S : String) : integer; virtual;

   end;

   PFixedFont = ^TFixedFont;
   TFixedFont = object(TRasterFont)
     CharWidth : word;          {real char width}
     constructor Init(AName : String; AImageTable : Pointer;
                      First,Count : byte;
                      AWidth,AHeight : word);
     function BytesPerChar(Ch : Char) : word; virtual;
     function GetCharWidth(Ch : Char) : word;  virtual;
     function GetCharImage(Ch : Char) : Pointer; virtual;
     procedure OutText(S : String); virtual;
     procedure OutTextXY(Location : TPoint; S : String); virtual;
   end;

   PBiosFont = ^TBiosFont;
   TBiosFont = object(TFixedFont)
     Bytes : word;
     constructor Init(AName : String; AImageTable : Pointer; AWidth,
                      AHeight,ABytes : word);
     function BytesPerChar(Ch : Char) : word; virtual;
   end;

   P8x8BiosFont = ^T8x8BiosFont;
   T8x8BiosFont = object(TBiosFont)
     Top : Pointer;
     constructor Init(AName : string; ABase,ATop : Pointer; ABytes : word);
     function GetCharImage(Ch : Char) : Pointer; virtual;
   end;

procedure InitFonts;
procedure DoneFonts;
procedure SelectFont(Font : integer);
function CanFillBackground : boolean;
procedure OutText(S : String);
procedure OutTextXY(Location : TPoint; S : String);
procedure SetTextStyle(Font,Direction : word; CharSize : word);
function TextHeight(S : String) : integer;
function TextWidth(S : String) : integer;

var
   Fonts : PCollection;

const
   SystemFont : integer = 0;       {System font init as default font}
   CurrentFont : integer = 0;    {Current font initialises as default font}
     
implementation
uses Memory,GrDriver;

{ TFont methods }

constructor TFont.Init(AName : string);
begin
   TObject.Init;
   FontName := AName;
end;

function TFont.CanFillBackground : boolean;
begin
   CanFillBackground := true;
end;

function TFont.TextWidth(S : String) : integer;
begin
   Abstract;
end;

function TFont.TextHeight(S : String) : integer;
begin
   Abstract;
end;

procedure TFont.OutText(S : String);
begin
end;

procedure TFont.OutTextXY(Location : TPoint; S : String);
begin
end;

procedure TFont.SetTextStyle(Direction,Size : word);
begin
end;

function TFont.GetFontName : String;
begin
   GetFontName := FontName;
end;

function TFont.GetCharHeight(Ch : Char) : word;
begin
   Abstract;
end;

function TFont.GetCharWidth(Ch : Char) : word;
begin
   Abstract;
end;


{ TBGIFont methods}

constructor TBGIFont.Init(AName : String; AId : integer);
begin
   TFont.Init(AName);
   ID := AId;
end;

function TBGIFont.CanFillBackground : boolean;
begin
   CanFillBackground := false;
end;

function TBGIFont.TextWidth(S : String) : integer;
begin
   TextWidth := Graph.TextWidth(S);
end;

function TBGIFont.TextHeight(S : String) : integer;
begin
   TextHeight := Graph.TextHeight(S);
end;

procedure TBGIFont.OutText(S : String);
var
  V : ViewPortType;
  Cp : TPoint;
  i,j : integer;
begin
   Graph.OutText(S);
end;

procedure TBGIFont.OutTextXY(Location : TPoint; S : String);
var
  V : ViewPortType;
  i,j : integer;
begin
   with Location do Graph.OutTextXY(X,Y,S);
end;

procedure TBGIFont.SetTextStyle(Direction,Size : word);
begin
   Graph.SetTextStyle(Id,Direction,Size);
end;

function TBGIFont.GetCharHeight(Ch : Char) : word;
var
   S : string[1];
begin
   S[1] := Ch;
   GetCharHeight := TextHeight(S);
end;

function TBGIFont.GetCharWidth(Ch : Char) : word;
var
   S : string[1];
begin
   S[1] := Ch;
   GetCharWidth := TextWidth(S);
end;

procedure TDefaultBGIFont.OutText(S : String);
var
  V : ViewPortType;
  Cp : TPoint;
  i,j : integer;
begin
   GetViewSettings(V);
   CP.X := GetX;
   CP.Y := GetY;
   if CP.X < 0 then begin
      if Length(S) > 0 then J := (TextWidth(S) div Length(S)) else J := TextWidth('A');
      i := Abs(CP.X) div J;
      if Abs(CP.X) mod J > 0
      then inc(i);
      if i > Length(S) then i := Length(S);
      Graph.MoveTo(GetX+(i*j),GetY);
      S := Copy(S,i+1,Length(S)-i);
   end;
   Graph.OutText(S);
   Graph.MoveTo(GetX + TextWidth(S),GetY);
end;

procedure TDefaultBGIFont.OutTextXY(Location : TPoint; S : String);
var
  V : ViewPortType;
  i,j : integer;
begin
   GetViewSettings(V);
   if Location.X < 0 then begin
      if Length(S) > 0 then J := (TextWidth(S) div Length(S)) else J := TextWidth('A');
      i := Abs(Location.X) div J;
      if Abs(Location.X) mod J > 0
      then inc(i);
      inc(Location.X,i*j);
      S := Copy(S,i+1,Length(S)-i);
   end;
   with Location do Graph.OutTextXY(X,Y,S);
end;


{ TRasterFont methods }

constructor TRasterFont.Init(AName : String; AImageTable : Pointer; First,Count : byte);
begin
   TFont.Init(AName);
   ImageTable := AImageTable;
   FirstChar := First;
   CharCount := Count;
end;

function TRasterFont.BytesPerChar(Ch : Char) : word;
begin
   BytesPerChar := 0;
end;

function TRasterFont.GetCharHeight(Ch : Char) : word;
begin
   GetCharHeight := CharHeight;
end;

function TRasterFont.GetCharImage(Ch : Char) : Pointer;
begin
   GetCharImage := nil;
end;

function TRasterFont.TextHeight(S : String) : integer;
begin
   TextHeight := CharHeight;
end;

function TRasterFont.TextWidth(S : String) : integer;
var
   W : integer;
   i : integer;
begin
   W := 0;
   for i := 1 to length(S) do
     if (byte(S[i]) >= FirstChar) and (byte(S[i]) < (FirstChar + CharCount)) then
     inc(W,GetCharWidth(S[i]));
   TextWidth := W;
end;

{ TFixedFont methods }

constructor TFixedFont.Init(AName : String; AImageTable : Pointer;
                            First,Count : byte;
                            AWidth,AHeight : word);
begin
   TRasterFont.Init(AName,AImageTable,First,Count);
   CharWidth := AWidth;
   CharHeight := AHeight;
end;

function TFixedFont.BytesPerChar(Ch : Char) : word;
var
   i : word;
begin
   i := CharWidth div 8;
   if (CharWidth mod 8) <> 0 then inc(i);
   BytesPerChar := CharHeight * i;
end;

function TFixedFont.GetCharWidth(Ch : Char) : word;
begin
   GetCharWidth := CharWidth;
end;

function TFixedFont.GetCharImage(Ch : Char) : Pointer;
var
   P : Pointer;
begin
   GetCharImage := nil;
   P := ImageTable;
   if P = nil then Exit;
   if (byte(Ch) < FirstChar) or (byte(Ch) > (FirstChar + CharCount))
   then Exit;
   inc(LongInt(P),(byte(Ch) - FirstChar) * BytesPerChar(' '));
   GetCharImage := P;
end;

procedure TFixedFont.OutText(S : String);
var
   CP : TPoint;
begin
   CP.X := GetX;
   CP.Y := GetY;
   OutTextXY(CP,S);
   Graph.MoveTo(CP.X + TextWidth(S),CP.Y);
end;

procedure TFixedFont.OutTextXY(Location : TPoint; S : String);
var
   i : integer;
   Width,Height : word;
begin
   GetTextSettings(TextSettings);
   case TextSettings.Vert of
     BottomText : dec(Location.Y,CharHeight);
     CenterText : dec(Location.Y,CharHeight div 2);
   end;
   case TextSettings.Horiz of
      RightText : dec(Location.X,TextWidth(S));
      CenterText : dec(Location.X,TextWidth(S) div 2);
   end;
   GetViewSettings(ViewPort);
   if not ViewPort.Clip then begin
      ViewPort.X1 := 0;
      ViewPort.Y1 := 0;
      ViewPort.X2 := GetMaxX;
      ViewPort.Y2 := GetMaxY;
   end else begin;
      inc(Location.X,ViewPort.X1);
      inc(Location.Y,ViewPort.Y1);
   end;
   Height := TextHeight(S);
   if (Location.Y>ViewPort.Y2) or ((Location.Y+Height) < ViewPort.Y1) then Exit;
   GetFillSettings(FillSettings);
   SetController(Mode);
   Width := CharWidth;
   for i := 1 to Length(S) do begin
      if ((Location.X + Width) > ViewPort.X1) and
         (Location.X <= ViewPort.X2)
      then with Location do
         OutCharPrim(X,Y,Width,Height,GetCharImage(S[i]));
      inc(Location.X,Width);
   end;
   ResetController;
end;

{ TBiosFont Method }

constructor TBiosFont.Init(AName : String; AImageTable : Pointer;
                           AWidth,AHeight,ABytes : word);
begin
   TFixedFont.Init(AName,AImageTable,0,255,AWidth,AHeight);
   Bytes := ABytes;
end;

function TBiosFont.BytesPerChar(Ch : Char) : word;
begin
   BytesPerChar := Bytes;
end;

{ T8x8BiosFont - special case of TBiosFont }

constructor T8x8BiosFont.Init(AName : String; ABase,ATop : Pointer; ABytes : word);
begin
   TBiosFont.Init(AName,ABase,8,8,ABytes);
   Top := ATop;
end;

function T8x8BiosFont.GetCharImage(Ch : Char) : Pointer;
var
   SaveTablePtr : Pointer;
begin
   if byte(Ch) >= 128 then begin
      SaveTablePtr := ImageTable;
      ImageTable := Top;
      FirstChar := 128;
      CharCount := 127;
   end;
   GetCharImage := TBiosFont.GetCharImage(Ch);
   if byte(Ch) >= 128 then begin
      ImageTable := SaveTablePtr;
      FirstChar := 0;
      CharCount := 255;
   end;
end;

procedure InitFonts;
var
  FontCount,I : integer;
  Params : BiosFontParams;
begin
  Fonts := New(PCollection,Init(1,1));
  Fonts^.Insert(New(PDefaultBGIFont,Init('Default 8x8 font',0)));
  FontCount := GetMaxFont;
  if FontCount <= 0 then Exit;
  for I := 1 to FontCount do
  begin
     GetFontParams(i,Params);
     case Params.TwoPart of
        false : with Params do
           Fonts^.Insert(New(PBiosFont,Init(GetFontName(i),ImageTable,Width,Height,BytesPerChar)));
        true : with Params do
           Fonts^.Insert(New(P8x8BiosFont,Init(GetFontName(i),Base,Top,BytesPerChar)));
     end;
     if i = DefaultSysFont then SystemFont := Fonts^.Count-1;
  end;
end;

procedure DoneFonts;
begin
   if Fonts <> nil then Fonts^.Done;
end;

procedure SelectFont(Font : integer);
begin
   if (Font > 0) and (Font <= Fonts^.Count) then CurrentFont := Font
   else CurrentFont := 0;
end;

function CanFillBackground : boolean;
begin
   CanFillBackground := PFont(Fonts^.At(CurrentFont))^.CanFillBackground;
end;

procedure SetTextStyle(Font,Direction : word; CharSize : word);
begin
   SelectFont(Font);
   PFont(Fonts^.At(CurrentFont))^.SetTextStyle(Direction,CharSize);
end;

procedure OutText(S : String);
begin
   PFont(Fonts^.At(CurrentFont))^.OutText(S);
end;

procedure OutTextXY(Location : TPoint; S : String);
begin
   PFont(Fonts^.At(CurrentFont))^.OutTextXY(Location,S);
end;

function TextHeight(S : String) : integer;
begin
   TextHeight := PFont(Fonts^.At(CurrentFont))^.TextHeight(S);
end;

function TextWidth(S : String) : integer;
begin
   TextWidth := PFont(Fonts^.At(CurrentFont))^.TextWidth(S);
end;

end.
