{ Created : 1993-04-25

Memory checker, checks for deallocating with a different size than the
allocated size and tracks not deallocated memory.



$Author: Berend $
$Date: 96-05-07 12:59 $
$Revision: 1 $


Last changes :
93-12-08  Adapted MemCheck to TDInfo
94-10-03  Extended width of error report
          Added caller of caller to allocation item to make finding the
          memory slip easier. The caller of th caller is shown in MEMCHECK.RPT
94-10-10  Installed exit handlers could cause other deallocations after
          MemCheck called Halt (because when an error has occured). You could
          get a 204 in that case, so now MemCheck turns itself on, before
          calling Halt.
94-10-29  Changed behaviour of MemCheck. 'Breaks' existing code!
          You now have to call InitMemCheck to turn memory checking on.
          Call DoneMemCheck to turn memory checking off.
          You can now pass Options to InitMemCheck, but currently MemCheck
          does not use them. Future versions will.
95-05-20  Added support for the TBigCollection of TurboPower.
}


{.$DEFINE BigCollection}      { Enable this if you want to use TBigCollection}
                             { so you can store up to 132 million memory    }
                             { allocations at once. Without this you are    }
                             { limited to 16K, enough even for large        }
                             { programs, but if you need more, here it is.  }
                             { You can find BIGCOL.ZIP at CompuServe,       }
                             { internet and BBSs around the world.          }



{$X+,O-,S-,R-,Q-,I-}
unit MemCheck;

interface


{ flags to pass to InitMemCheck }
const
  mfStandard = 0;


const
  MemCheckDescr:string = '';      { not used yet }

const
  ReportFileName = 'MEMCHECK.RPT';


procedure InitMemCheck(AOptions : byte);
procedure DoneMemCheck;
procedure StoreAlloc(MemPtr : pointer; Size : word);
procedure FreeAlloc(MemPtr : pointer; Size : word);
procedure MemCheckReport;



implementation

uses
  Objects,
{$IFDEF BigCollection}
  BigColl,
{$ENDIF}
  BBError, BBGui, BBUtil,
  TDInfo;


type
  PAllocItem = ^TAllocItem;
  TAllocItem = record
    MemPtr : pointer;
    Caller,
    CallerItsCaller : pointer;
    Size : word;
  end;

{$IFDEF BigCollection}
  PAllocBigCollection = ^TAllocBigCollection;
  TAllocBigCollection = object(TBigCollection)
    function  Compare(Item1, Item2: pointer) : integer;  virtual;
    procedure FreeItem(Item : pointer);  virtual;
    procedure Insert(Item : pointer);  virtual;
  end;
{$ELSE}
  PAllocCollection = ^TAllocCollection;
  TAllocCollection = object(TSortedCollection)
    function  Compare(Key1, Key2 : pointer) : integer;  virtual;
    procedure FreeItem(Item : pointer);  virtual;
    procedure Insert(Item : pointer);  virtual;
    function  KeyOf(Item : pointer) : pointer;  virtual;
  end;
{$ENDIF}

  PMemCheckRec = ^TMemCheckRec;
  TMemCheckRec = record
    CheckMem : byte;
    Options : byte;
    StoreAlloc : pointer;
    FreeAlloc : pointer;
  end;

var
  MemCheckRec : PMemCheckRec;
{$IFDEF BigCollection}
  AllocCol : PAllocBigCollection;
{$ELSE}
  AllocCol : PAllocCollection;
{$ENDIF}


{****************************************************************************}
{* TAllocCollection/TAllocBigCollection                                     *}
{****************************************************************************}

{$IFDEF BigCollection}

function TAllocBigCollection.Compare(Item1, Item2 : pointer) : integer;
begin
  if longint(PAllocItem(Item1)^.MemPtr) < longint(PAllocItem(Item2)^.MemPtr)
   then  Compare := -1
   else
     if longint(PAllocItem(Item1)^.MemPtr) = longint(PAllocItem(Item2)^.MemPtr)
      then  Compare := 0
      else  Compare := 1;
end;

procedure TAllocBigCollection.FreeItem(Item : pointer);
begin
  Dispose(PAllocItem(Item));
end;

procedure TAllocBigCollection.Insert(Item : pointer);
var
  Index : longint;
begin
  if Search(Item, Index)
   then  begin
     PrintError('Attempt to allocate memory at same address.', 0);
     Halt(1);
   end
   else  begin
     AtInsert(Index, Item);
   end;
end;

{$ELSE}

function TAllocCollection.Compare(Key1, Key2 : pointer) : integer;
begin
  if longint(Key1) < longint(Key2)
   then  Compare := -1
   else
     if longint(Key1) = longint(Key2)
      then  Compare := 0
      else  Compare := 1;
end;

procedure TAllocCollection.FreeItem(Item : pointer);
begin
  Dispose(PAllocItem(Item));
end;

procedure TAllocCollection.Insert(Item : pointer);
var
  Index : integer;
begin
  if Search(KeyOf(Item), Index)
   then  begin
     PrintError('Attempt to allocate memory at same address.', 0);
     Halt(1);
   end
   else  begin
     AtInsert(Index, Item);
   end;
end;

function TAllocCollection.KeyOf(Item : pointer) : pointer;
begin
  KeyOf := PAllocItem(Item)^.MemPtr;
end;

{$ENDIF}


{****************************************************************************}
{* MemCheckOn and Off                                                       *}
{****************************************************************************}

procedure MemCheckOn;  assembler;
asm
  les  di,MemCheckRec
  mov  al,1
  mov  es:[di].TMemCheckRec.CheckMem,al
end;

procedure MemCheckOff;  assembler;
asm
  les  di,MemCheckRec
  xor  al,al
  mov  es:[di].TMemCheckRec.CheckMem,al
end;


{****************************************************************************}
{* InitMemCheck and DoneMemCheck                                            *}
{****************************************************************************}

var
  SaveExitProc : pointer;

procedure ExitMemCheck;  far;
begin
  ExitProc := SaveExitProc;
  if MemCheckRec <> nil then
    DoneMemCheck;
end;


procedure InitMemCheck(AOptions : byte);
{$IFDEF BigCollection}
const
  Sorted = True;
  NoDuplicates = False;
{$ENDIF}
begin
  if MemCheckRec <> nil then  begin
    MemCheckOff;
  {$IFDEF BigCollection}
    AllocCol := New(PAllocBigCollection, Init(Sorted, NoDuplicates));
  {$ELSE}
    AllocCol := New(PAllocCollection, Init(4096,4096));
  {$ENDIF}
    MemCheckRec^.Options := AOptions;
    MemCheckRec^.StoreAlloc := @StoreAlloc;
    MemCheckRec^.FreeAlloc := @FreeAlloc;
    MemCheckOn;
    SaveExitProc := ExitProc;
    ExitProc := @ExitMemCheck;
  end;
end;

procedure DoneMemCheck;
begin
  if MemCheckRec <> nil then  begin
    MemCheckOff;
    Discard(AllocCol);
    MemCheckRec := nil;
  end;
end;


{****************************************************************************}
{* StoreAlloc and FreeAlloc                                                 *}
{****************************************************************************}

procedure StoreAlloc(MemPtr : pointer; Size : word);
var
  AllocItem : PAllocItem;
begin

{ turn MemChecking off to avoid recursive loops }
  asm
    les  di,MemCheckRec
    xor  ax,ax
    mov  es:[di].TMemCheckRec.CheckMem,al
  end;

{ allocate memory tracking item }
  New(AllocItem);
  if AllocItem = nil then  begin
    FatalError('Cannot allocate memory to store memory allocations.', 0);
  end;

{ store data about current allocation in it }
  asm
    les  di,AllocItem
    mov  bx,[bp]
    ror  bx,1
    rol  bx,1
    jnc  @@1
    dec  bx
  @@1:
    mov  ax,word ptr ss:[bx+02h]
    mov  word ptr es:[di].TAllocItem.Caller,ax
    mov  ax,word ptr ss:[bx+04h]
    mov  word ptr es:[di].TAllocItem.Caller+2,ax
    mov  bx,ss:[bx]
    ror  bx,1
    rol  bx,1
    jnc  @@2
    dec  bx
  @@2:
    cmp  word ptr ss:[bx],0
    je   @@end_of_stack
    mov  ax,word ptr ss:[bx+02h]
    mov  word ptr es:[di].TAllocItem.CallerItsCaller,ax
    mov  ax,word ptr ss:[bx+04h]
    mov  word ptr es:[di].TAllocItem.CallerItsCaller+2,ax
    jmp  @@3
  @@end_of_stack:
    xor  ax,ax
    mov  word ptr es:[di].TAllocItem.CallerItsCaller,ax
    mov  word ptr es:[di].TAllocItem.CallerItsCaller+2,ax
  @@3:
    push ds
    lds  si,MemPtr
    mov  word ptr es:[di].TAllocItem.MemPtr,si
    mov  word ptr es:[di].TAllocItem.MemPtr+2,ds
    pop  ds
    mov  ax,Size
    mov  word ptr es:[di].TAllocItem.Size,ax
  end;

{ insert allocation tracking item }
  AllocCol^.Insert(AllocItem);

  asm
{ turn MemChecking on }
    les  di,MemCheckRec
    mov  al,1
    mov  es:[di].TMemCheckRec.CheckMem,al

{ and restore ax and dx }
    mov  ax,word ptr &MemPtr
    mov  dx,word ptr &MemPtr+2
  end;
end;


procedure FreeAlloc(MemPtr : pointer; Size : word);

  function LowerMemoryCheck(Item : PAllocItem) : Boolean;
  {* checks only first four bytes... *}
  var
    p : pointer;
  begin
    LowerMemoryCheck := FALSE;
    with Item^ do  begin
      if Size <= 65536-8-16 then  begin
        if MemL[PtrRec(MemPtr).Seg:PtrRec(MemPtr).Ofs-4] <> $CCCCCCCC then
          Exit;
      end;
    end; { of with }
    LowerMemoryCheck := TRUE;
  end;

  function UpperMemoryCheck(Item : PAllocItem) : Boolean;
  {* checks only first four bytes... *}
  var
    p : pointer;
  begin
    UpperMemoryCheck := FALSE;
    with Item^ do  begin
      if Size <= 65536-8-8 then  begin
        if MemL[PtrRec(MemPtr).Seg:PtrRec(MemPtr).Ofs+Size] <> $CCCCCCCC then
          Exit;
      end;
    end; { of with }
    UpperMemoryCheck := TRUE;
  end;

var
{$IFDEF BigCollection}
  Index : longint;
  AllocItem: TAllocItem;
{$ELSE}
  Index : integer;
{$ENDIF}
begin

{ turn memory checking off }
  asm
    les  di,MemCheckRec
    xor  ax,ax
    mov  es:[di].TMemCheckRec.CheckMem,al
  end;

  with AllocCol^ do  begin
  {$IFDEF BigCollection}
    AllocItem.MemPtr := MemPtr;
    if not Search(@AllocItem, Index) then  begin
  {$ELSE}
    if not Search(MemPtr, Index) then  begin
  {$ENDIF}
      PrintError('Attempt to dispose a non-allocated block.', 0);
      MemCheckOn;  { installed exit handlers might dispose here after }
      Halt(1);
    end;
    if PAllocItem(At(Index))^.Size <> Size then  begin
      PrintError('Attempt to dispose a memory block with wrong block size. ' +
                 'Expected block size: ' + StrW(PAllocItem(At(Index))^.Size) +
                 '. Got: ' + StrW(Size), 0);
      MemCheckOn;  { installed exit handlers might dispose here after }
      Halt(1);
    end;
    if not LowerMemoryCheck(PAllocItem(At(Index))) then  begin
      PrintError('Memory before allocated area corrupt!', 0);
      MemCheckOn;  { installed exit handlers might dispose here after }
      Halt(1);
    end;
    if not UpperMemoryCheck(PAllocItem(At(Index))) then  begin
      PrintError('Memory after allocated area corrupt!', 0);
      MemCheckOn;  { installed exit handlers might dispose here after }
      Halt(1);
    end;
    AtFree(Index);
  end;

  asm
{ turn MemChecking on }
    les  di,MemCheckRec
    mov  al,1
    mov  es:[di].TMemCheckRec.CheckMem,al

{ and restore ax, bx and cx }
    mov  ax,Size
    mov  cx,word ptr &MemPtr
    mov  bx,word ptr &MemPtr+2
  end;
end;


{****************************************************************************}
{* MemCheckReport                                                           *}
{****************************************************************************}

function GetAddress(Address : pointer) : string;
var
  LogicalAddr : pointer;
  LineNumber : PLineNumber;
  Symbol : PSymbol;
  s : string;
begin
  LogicalAddr := GetLogicalAddr(Address);
  if TDInfoPresent(nil)
   then  begin
     New(LineNumber, AtAddr(LogicalAddr));
     if LineNumber = nil
      then  begin
        s := HexStr(PtrRec(LogicalAddr).Seg) + ':' + HexStr(PtrRec(LogicalAddr).Ofs);
      end
      else  begin
        s := LineNumber^.ItsCorrelation^.ItsSourceFile^.ItsName + ' (' + StrW(LineNumber^.Value) + ') ';
        New(Symbol, AtAddr(LogicalAddr));
        if Symbol <> nil then  begin
          if Symbol^.ItsType^.ReturnType = 1
           then  s := s + 'procedure '
           else  s := s + 'function ';
          if Symbol^.ItsType^.ID = tid_SpecialFunc then  begin
            s := s + Symbol^.ItsType^.ItsClassType^.ItsName + '.';
          end;
          s := s + Symbol^.ItsName + ';';
          Dispose(Symbol, Done);
        end;
        Dispose(LineNumber, Done);
      end;
   end
   else
     s := HexStr(PtrRec(LogicalAddr).Seg) + ':' + HexStr(PtrRec(LogicalAddr).Ofs);
  GetAddress := s;
end;


const
  CallerWidth = 70;
var
  t : text;
  Amount : longint;


{$IFDEF BigCollection}
procedure Print(Item : pointer);  far;
begin
  with PAllocItem(Item)^ do  begin
    writeln(t, LeftJustify(GetAddress(Caller), CallerWidth), '  ', Size:5);
    if CallerItsCaller = nil
      then  writeln(t, '  n/a')
      else  writeln(t, '  ', LeftJustify(GetAddress(CallerItsCaller), CallerWidth-2));
    Inc(Amount, Size);
  end;
end;
{$ENDIF}


procedure MemCheckReport;

{$IFNDEF BigCollection}
  procedure Print(Item : PAllocItem);  far;
  begin
    with Item^ do  begin
      writeln(t, LeftJustify(GetAddress(Caller), CallerWidth), '  ', Size:5);
      writeln(t, '  ', LeftJustify(GetAddress(CallerItsCaller), CallerWidth-2));
      Inc(Amount, Size);
    end;
  end;
{$ENDIF}

const
  BufSize = 1024;
var
  Buffer : array[1..BufSize] of char;
begin
  if MemCheckRec = nil then  begin
    PrintError('Memory checking not initalized. ' +
               'You either didn''t call InitMemCheck or you have the wrong system unit.', 0);
    Halt(1);
  end;
  MemCheckOff;
  if AllocCol = nil then  Exit;
  Assign(t, ReportFileName);
  Rewrite(t);
  SetTextBuf(t, Buffer, BufSize);
  writeln(t, 'Not disposed memory report. Date: ', GetDateStr, '  Time: ', GetTimeStr);
  writeln(t);
  writeln(t, LeftJustify('Caller', CallerWidth), '   Size');
  writeln(t);
  Amount := 0;
{$IFDEF BigCollection}
  AllocCol^.ForEach(Print);
{$ELSE}
  AllocCol^.ForEach(@Print);
{$ENDIF}
  writeln(t);
  writeln(t);
  writeln(t, 'Total not disposed memory: ', Amount, ' bytes');
{$IFDEF BigCollection}
  writeln(t, 'Total items: ', AllocCol^.bcCount);
{$ELSE}
  writeln(t, 'Total items: ', AllocCol^.Count);
{$ENDIF}
  Close(t);
  MemCheckOn;
end;


begin
  MemCheckRec := ErrorAddr;
end.  { of unit MemCheck }
