Program mp1tompx;  { WorldMap data file preprocessor }
{ convert .MP1 text file with map data to .MP0, .MP2 or .MP3 binary file     }
{ .MP3 format : precomputed values for fast MAPVIEW execution (4 byte reals) }
{ .MP2 format : precomputed values for fast MAPVIEW execution (6 byte reals) }
{ .MP0 format : very compact data format                                     }

{ For file formats, see below: TYPEs mp2rec rsp. mp0rec                      }
{ For further description, see MAPVIEW.DOC                                   }
{ Specify /2, /3 or /0 on command line; default is /2.                       }
{ A file name must be specified on the command line. If the extension is     }
{ MP1, it will be processed; if it is LST, it is assumed to be a list file   }
{ as used by MapView. If the extension is missing, LST will be assumed.      }
{ Any other extension is an error.                                           }

{ Original conversion programmes by:                                         }
{ Ton van den Bogert  <WWDONIC@HEITUE5.BITNET>                               }
{ Kenneth van Camp    <kvancamp@ARDEC.ARPA>                                  }
{ Combined conversion programmes, changed MP0 format,                        }
{ tweaked and adapted to TP5.0:                                              }
{ Gisbert W.Selke  <RECK@DBNUAMA1.BITNET> 17 Dec 1988                        }

{$R-,B-,S+,I+,N+,E+}
{$M 65500,65500,560000}

  Const bufsize = 63500;

  Type
    mp2rec = Record rectyp: byte;       { record type: 1 is start of segment }
                    lon, lat: real;     { longitude and latitude }
                    merclat: real;      { mercator function of latitude }
                    x,y,z: real;        { globe coordinates }
             End;

    mp3rec = Record rectyp: byte;       { record type: 1 is start of segment }
                    lon, lat: single;   { longitude and latitude }
                    merclat: single;    { mercator function of latitude }
                    x,y,z: single;      { globe coordinates }
             End;

    mp0rec = Record ilon, ilat : integer;
      { 100*longitude, offset by 20000 if start of segment; then 100*latitude }
             End;

  Var
    filename, mapfilnam : string;
    listfile, mapfile : text;
    line : string[80];
    onefile : boolean;
    retcode : integer;
    p1, p2, format : byte;
    count : word;
    inbuf : Array [1..bufsize] Of byte;
    m1, m2 : real;
    raddeg : real;                   { radians in one degree }
    rlat, rlon : real;               { lat&lon in radians }
    coslat : real;                   { cos of latitude }

  Procedure abort(t : string; i : byte);
  { output error message and abort with error code                           }
  Begin                                                              { abort }
    writeln(t);
    Halt(i);
  End;                                                               { abort }

  Procedure init;
  { initialize variables                                                     }
    Var i : byte;
        cmd : string;

    Procedure usage;
    { give usage hints                                                       }
    Begin                                                            { usage }
      writeln;
      writeln('Usage:  convert  [/0 | /2 | /3]  <filename>');
      writeln('        where /2 is the default.');
      writeln;
      writeln('<filename> must have either extension LST or MP1.');
      writeln('If it is MP1, the file with this name will be processed;');
      writeln('if it is LST, it is assumed to be a text file containing the ',
              'names of');
      writeln('.MP1 raw data files, complete with path, one per line.');
      writeln;
      writeln('/3 yields large data files for fast 80x87 MapView execution;');
    writeln('/2 yields large data files for fast non-80x87 MapView execution;');
      writeln('/0 yields very compact data files with slower MapView ',
              'execution.');
      Halt(1);
    End;                                                             { usage }

  Begin                                                               { init }
    If ParamCount = 0 Then usage;
    format := 99;
    filename := '';
    For i := 1 To ParamCount Do
    Begin
      cmd := ParamStr(i);
      If cmd[1] = '?' Then usage;
      If (cmd[1] In ['-','/']) And (Length(cmd) = 2 ) Then
      Begin
        If format <> 99 Then abort('Conflicting switches on command line',2);
        Case cmd[2] Of
          '3' : format := 3;
          '2' : format := 2;
          '0' : format := 0;
          Else abort('Unrecognized command line switch "'+cmd+'"',2);
        End;
      End Else
      Begin
        If filename = '' Then filename := cmd
                         Else abort('Multiple file names not supported',2);
      End;
    End;
    If format = 99 Then format := 2;
    If filename = '' Then abort('File name missing',2);
    For i := 1 To Length(filename) Do filename[i] := UpCase(filename[i]);
    cmd := filename;
    While Pos('.',cmd) > 0 Do Delete(cmd,1,Pos('.',cmd));
    If (cmd <> 'LST') And (cmd <> 'MP1') Then
                                    abort('Illegal file name extension',2);
    If cmd = 'MP1' Then
    Begin
      onefile := True;
      mapfilnam := filename;
    End Else
    Begin
      onefile := False;
      Assign(listfile,filename);
      {$I- } Reset(listfile);  {$I+ }
      If IOResult <> 0 Then abort('Cannot open list file '+filename,3);
      mapfilnam := '';
    End;
    m1 := Pi / 4.0;
    m2 := Pi / 360.0;
    raddeg := Pi / 180.0;
  End;                                                                { init }

  Procedure  makemp3;
  { prepare .MP3 file                                                        }
    Var binfile : file of mp3rec;
        buffer : mp3rec;
        minlat, minlon, maxlat, maxlon : single;
  Begin                                                            { makemp3 }
    minlat :=  9000.0;
    minlon :=  9000.0;
    maxlat := -9000.0;
    maxlon := -9000.0;
    Assign(binfile,ConCat(mapfilnam,'.MP3'));
    Rewrite(binfile);
    With buffer Do
    Begin
      rectyp := $FE;
      lon := 0.0; lat := 0.0; merclat := 0.0;
      x := 0.0;   y := 0.0;   z := 0.0;
    End;
    write(binfile,buffer);  { dummy record, to be filled in later }
    line := ' ';  { start with blank line }
    count := 0;
    Repeat
      With buffer Do
      Begin
        Inc(count);
        If lo(count) = 0 Then write(count,#13);
        { blank line: start new curve segment }
        If (length(line) = 0) or (line[1] = ' ') Then
        Begin
          rectyp := 1;
          readln(mapfile,line);
        End Else rectyp := 0;
        { extract two reals }
        line := line + ' ';
        p1 := Pos(' ', line);
        Val(Copy(line,1,Pred(p1)),lat,retcode);
        p2 := Pos(' ', Copy(line,Succ(p1),100)) + p1;
        Val(Copy(line,Succ(p1),p2-p1-1),lon,retcode);
        x := m1 + m2*lat;
        merclat := Ln(Sin(x)/Cos(x));
        rlat := raddeg*lat;
        coslat := Cos(rlat);
        rlon := raddeg*lon;
        x := Cos(rlon)*coslat;
        y := Sin(rlon)*coslat;
        z := Sin(rlat);
        If lat < minlat Then minlat := lat;
        If lat > maxlat Then maxlat := lat;
        If lon < minlon Then minlon := lon;
        If lon > maxlon Then maxlon := lon;
      End; { With }
      write(binfile,buffer);
      If (Not Eof(mapfile)) Then readln(mapfile,line);
    Until Eof(mapfile);
    write(count,#13);
    With buffer Do
    Begin
      rectyp := $FE;
      lon     := minlon;  lat := minlat;
      merclat := maxlon;  x   := maxlat;
      y := 0.0;           z := 0.0;
    End;
    Seek(binfile,0);
    write(binfile,buffer);
    Close(binfile);
  End;                                                             { makemp3 }

  Procedure  makemp2;
  { prepare .MP2 file                                                        }
    Var binfile : file of mp2rec;
        buffer : mp2rec;
        minlat, minlon, maxlat, maxlon : real;
  Begin                                                            { makemp2 }
    minlat :=  9000.0;
    minlon :=  9000.0;
    maxlat := -9000.0;
    maxlon := -9000.0;
    Assign(binfile,ConCat(mapfilnam,'.MP2'));
    Rewrite(binfile);
    With buffer Do
    Begin
      rectyp := $FF;
      lon := 0.0; lat := 0.0; merclat := 0.0;
      x := 0.0;   y := 0.0;   z := 0.0;
    End;
    write(binfile,buffer);  { dummy record, to be filled in later }
    line := ' ';  { start with blank line }
    count := 0;
    Repeat
      With buffer Do
      Begin
        Inc(count);
        If lo(count) = 0 Then write(count,#13);
        { blank line: start new curve segment }
        If (length(line) = 0) or (line[1] = ' ') Then
        Begin
          rectyp := 1;
          readln(mapfile,line);
        End Else rectyp := 0;
        { extract two reals }
        line := line + ' ';
        p1 := Pos(' ', line);
        Val(Copy(line,1,Pred(p1)),lat,retcode);
        p2 := Pos(' ', Copy(line,Succ(p1),100)) + p1;
        Val(Copy(line,Succ(p1),p2-p1-1),lon,retcode);
        x := m1 + m2*lat;
        merclat := Ln(Sin(x)/Cos(x));
        rlat := raddeg*lat;
        coslat := Cos(rlat);
        rlon := raddeg*lon;
        x := Cos(rlon)*coslat;
        y := Sin(rlon)*coslat;
        z := Sin(rlat);
        If lat < minlat Then minlat := lat;
        If lat > maxlat Then maxlat := lat;
        If lon < minlon Then minlon := lon;
        If lon > maxlon Then maxlon := lon;
      End; { With }
      write(binfile,buffer);
      If (Not Eof(mapfile)) Then readln(mapfile,line);
    Until Eof(mapfile);
    write(count,#13);
    With buffer Do
    Begin
      rectyp := $FF;
      lon     := minlon;  lat := minlat;
      merclat := maxlon;  x   := maxlat;
      y := 0.0;           z := 0.0;
    End;
    Seek(binfile,0);
    write(binfile,buffer);
    Close(binfile);
  End;                                                             { makemp2 }

  Procedure  makemp0;
  { prepare .MP0 file                                                        }
    Var binfile : file of mp0rec;
        buffer : mp0rec;
        offset : integer;
        lat, lon : real;
        minlat, minlon, maxlat, maxlon : real;
  Begin                                                            { makemp0 }
    minlat :=  9000.0;
    minlon :=  9000.0;
    maxlat := -9000.0;
    maxlon := -9000.0;
    Assign(binfile,ConCat(mapfilnam,'.MP0'));
    Rewrite(binfile);
    buffer.ilon := 0;
    buffer.ilat := 0;
    write(binfile,buffer);  { two records, to be filled in later }
    write(binfile,buffer);
    line := ' ';  { start with blank line }
    count := 0;
    Repeat
      With buffer Do
      Begin
        Inc(count);
        If lo(count) = 0 Then write(count,#13);
        { blank line: start new curve segment }
        If (length(line) = 0) or (line[1] = ' ') Then
        Begin
          offset := 20000;
          readln(mapfile,line);
        End Else offset := 0;
        { extract two reals }
        line := line + ' ';
        p1 := Pos(' ', line);
        Val(Copy(line,1,Pred(p1)),lat,retcode);
        p2 := Pos(' ', Copy(line,Succ(p1),100)) + p1;
        Val(Copy(line,Succ(p1),p2-p1-1),lon,retcode);
        If lat < minlat Then minlat := lat;
        If lat > maxlat Then maxlat := lat;
        If lon < minlon Then minlon := lon;
        If lon > maxlon Then maxlon := lon;
        If (100.0*lat > 12000.0) Or (100.0*lon > 32000) Then
        Begin
          writeln('Range error in line ',count,':');
          writeln('latitude/longitude ',lat:15:5,lon:15:5);
          close(binfile);
          Halt(5);
        End;
        ilat := Round (lat * 100.0) + offset;
        ilon := Round (lon * 100.0);
      End; { With }
      write(binfile,buffer);
      If (Not Eof(mapfile)) Then readln(mapfile,line);
    Until Eof(mapfile);
    write(count,#13);
    Seek(binfile,0);
    buffer.ilat := Round(minlat * 100.0);
    buffer.ilon := Round(minlon * 100.0);
    write(binfile,buffer);
    buffer.ilat := Round(maxlat * 100.0);
    buffer.ilon := Round(maxlon * 100.0);
    write(binfile,buffer);
    Close(binfile);
  End;                                                             { makemp0 }

Begin                                                                 { main }
  writeln('MP1TOMPX 1.2 - preprocess MapView raw data files');
  init;
  If onefile Then
       write('        Converting ',mapfilnam,' to .MP',format:1,' format',#13)
    Else writeln('Converting files in ',filename,' to .MP',format:1,' format');
  While mapfilnam <> 'END' Do
  Begin
    If Not onefile Then readln(listfile,mapfilnam);
    While (mapfilnam <> '') And (mapfilnam[1] In [' ',#9]) Do
                                                       Delete(mapfilnam,1,1);
    If Pos(' ',mapfilnam) > 0 Then Delete(mapfilnam,Pos(' ',mapfilnam),255);
    For p1 := 1 To Length(mapfilnam) Do mapfilnam[p1] := UpCase(mapfilnam[p1]);
    If Not onefile Then write('        Processing ',mapfilnam,#13);
    If (Copy(mapfilnam,1,3) <> 'END') Then
    Begin
      If Pos('.',mapfilnam) <> 0 Then Delete(mapfilnam,Pos('.',mapfilnam),999);
      Assign(mapfile,mapfilnam+'.MP1');
      {$I- } Reset(mapfile); {$I+ }
      If IOResult <> 0 Then abort('Cannot open map file '+mapfilnam+'.MP1',4);
      SetTextBuf(mapfile,inbuf);
      Case format Of
        3 : makemp3;
        2 : makemp2;
        0 : makemp0;
      End;
      Close(mapfile);
    End;
    If onefile Then mapfilnam := 'END'
              Else If eof(listfile) Then mapfilnam := 'END';
    writeln;
  End; { While }
  If Not onefile Then Close(listfile);
End.
