/****************************************************************************

    PROGRAM: WinOem.c
             by Bill Buckels 1994

             WinOem was adapted from the EditFile program that
             was included in the Windows SDK (1992).

             It now bears little resemblance to its progenitor.

             I made some additions, such as providing
             clipboard services, an undo feature,
             and the saving of text files to .BMP files.

             Several changes were made to make WinOem handle
             MS-DOS Ascii Characters in an editing session
             under Windows.

             Most of this stuff is in the SDK or implied
             by the documentation, but you may have to look
             closely to find it.

             In my memory allocation I am assuming
             that I can get and lock my resources. I don't make
             the same assumptions regarding my file i/o,
             and I have cleaned-up that part of the SDK example.
             I haven't put-in any special checks for the
             single character entry of High Ascii overflowing
             memory, but I tried not miss any other limit checks.

             I also use explicit types in some cases and more
             standard C than is usually found in Windows code.

             I wasn't able to get the edit window to allow the
             entry of high ascii from the alt-numeric keypad,
             so I did two things to compensate for this problem:

             1. I set-up the F1 through F10 keys as "FAST KEYS"
             to allow the entry of line drawing characters and
             allow the entry of High Ascii - MS-DOS style.

             2. I set-up the normal numeric keys to work similarily
             to the numeric keypad... i.e. when the alt key is pressed
             the numeric value of the keys is the ascii value of the
             entry...

             An additional step is necessary... the Insert
             Key must be pressed after the Alt-Numeric keypress is
             made to register the entry.

             If anyone would care to share a reliable method of
             reading the numeric keypad for ascii values under windows,
             I would be grateful for it. At this point I am
             convinced that such a thing can't be done reliably under
             Windows. The ascii codes that Windows returns to me
             have no meaningful raw value above or below
             the usual printable characters.

             WinOem needs the Insert Key Press at the end of
             the entry in WinOem as a "make" code. I would like to
             do away with this step if I can.

             At any rate, I am satisfied even with the
             minor nits to release WinOem. Enjoy!
             And let me know if the code has any REAL BUGS thanks.

             Bill Buckels 1995

             Added support for translation of Clarion Reports
             For Rene Cyr - April 1996

    PURPOSE: Loads, saves, and edits text files using the Windows OEM Font
             Copies The Edit Buffer to The Clipboard
             Pastes The Text From The Clipboard Into The Edit Buffer
             Alternately Saves The Text From The Editbuffer
                         as a Monochrome BMP file.

    FUNCTIONS:

        WinMain() - calls initialization function, processes message loop
        InitApplication() - initializes window data and registers window
        InitInstance() - saves instance handle and creates main window
        MainWndProc() - processes messages
        SaveFile() - Save current file
        QuerySaveFile() - Called when some action might lose current contents
        SetNewBuffer() - Set new buffer for edit window

        MyCopy()  - Copies Editbuffer to the Clipboard
        MyPaste() - Copies Clipboard to the Editbuffer
        MyBmp()   - Saves Editbuffer as a Windows Monochrome BITMAP

        MyHelp()    - Winhelp for WinOem
        MyAbout()   - About WinOem
        MyLicence() - WinOem Licence
        MyUndo()    - Undo For Editing Commands
        AddAscii()  - Add High Ascii Character To Edit Buffer

        TranslateReport() - Translate Clarion Reports to Ascii Text

****************************************************************************/

#include <windows.h>
#include <commdlg.h>
#include <stdio.h>
#include <fcntl.h>
#include <io.h>
#include <string.h>
#include <stdlib.h>
#include <malloc.h>
#include "winoem.h"


#define E_ASCIIZ     0
#define E_FIRST      0
#define E_SECOND     1
#define E_CR         13
#define E_LF         10
#define E_FF         12
#define E_WHITESPACE 32
#define E_MAXBUFFER  512
#define E_MINBUFFER  2

FILE *fp, *fp2;

HICON myicon;


typedef struct tagMYBITMAPINFO
{
    BITMAPINFOHEADER bmiHeader;
    RGBQUAD          bmiColors[2];
} MYBITMAPINFO;

BITMAPFILEHEADER bmfhead;
MYBITMAPINFO     bmp;

// the 8 x 16 OEM character set bitmaps
typedef struct{
    unsigned char b[256][16];
}RAMFONT;

RAMFONT far *lpRamFont;                  // bitmap conversion font resource
HRSRC   hMyFontloc;
HGLOBAL hMyFontRes;

// the FastKeys keyboard codes
typedef struct{
    unsigned char b[12][10];
}FASTKEYS;

FASTKEYS far *lpFastKeys;                // keycode substitution resource
HRSRC   hMyFastloc;
HGLOBAL hMyFastRes;

HANDLE hInst;
HFONT hfnt;                              // handle for the OEM font

HANDLE hData;                            // handle to clip data
LPSTR lpData;                            // pointer to clip data

HANDLE hInsert;
LPSTR InsertText;

HWND hEditWnd;                           // handle to edit window
HANDLE hAccTable;                        // handle to accelerator table

HBITMAP hMenuBitmap0;
HBITMAP hMenuBitmap1;
HBITMAP hMenuBitmap2;
HBITMAP hMenuBitmap3;
HBITMAP hMenuBitmap4;
HBITMAP hMenuBitmap5;
HBITMAP hMenuBitmap6;
HBITMAP hMenuBitmap7;
HBITMAP hMenuBitmap8;
HBITMAP hMenuBitmap9;
HBITMAP hMenuBitmap10;
HBITMAP hMenuBitmap11;
HBITMAP hMenuBitmap12;

WPARAM  IDM_CURRENTBMP=IDM_BITMAP1;
WORD DosAscii;
BOOL MakeCode  = ALT_BREAK;
// Additional includes needed for the fstat() function

#include <sys\types.h>
#include <sys\stat.h>

// new variables for common dialogs
static char szFilterSpec [128]
          = "All Files(*.*)\0*.*\0";      // filter string for dir. listings
char szCustFilterSpec[MAXCUSTFILTER];     // custom filter buffer
char szOpenDlgTitle[] = "Open File";      // title of File open dialog
char szSaveDlgTitle[] = "Save File";      // title of File saveas dialog
char szSaveBmpTitle[] = "Save BMP";       // title of BITMAP saveas dialog

OPENFILENAME ofn;             // struct. passed to GetOpenFileName

static char PathName[128] = "\0";
static char FileName[128] = "\0";
static char SaveName[128] = "\0";
static char SaveText[128] = "\0";

char str[255];
char wintitle[]="WinOem Copyright \251 Bill Buckels 1994-1999\0";

unsigned char szBuffer[E_MAXBUFFER];

HANDLE hEditBuffer;                       // handle to editing buffer
HANDLE hHourGlass;                        // handle to hourglass cursor
HANDLE hSaveCursor;                       // current cursor handle

int hFile;                                // file handle
OFSTRUCT OfStruct;                        // information from OpenFile()
struct stat FileStatus;                   // information from fstat()
BOOL bChanges = FALSE;                    // TRUE if the file is changed
PSTR pEditBuffer;                         // address of the edit buffer
HWND hwnd;                                // handle to main window

char Untitled[] =                         // default window title
     "WinOem - (untitled)";

/****************************************************************************

    FUNCTION: WinMain(HANDLE, HANDLE, LPSTR, int)

    PURPOSE: calls initialization function, processes message loop

****************************************************************************/

int PASCAL WinMain(hInstance, hPrevInstance, lpCmdLine, nCmdShow)
HANDLE hInstance;
HANDLE hPrevInstance;
LPSTR lpCmdLine;
int nCmdShow;
{
    MSG msg;

    if (!hPrevInstance)
        if (!InitApplication(hInstance))
            return (FALSE);

    if (!InitInstance(hInstance, nCmdShow))
        return (FALSE);

    if(hMyFontloc=FindResource(hInstance,"MYFONT",RT_RCDATA))
     {
        if(hMyFontRes=LoadResource(hInstance,hMyFontloc))
        {
            lpRamFont=(RAMFONT far *)LockResource(hMyFontRes);
        }
     }

    if(hMyFastloc=FindResource(hInstance,"FASTKEYS",RT_RCDATA))
     {
        if(hMyFastRes=LoadResource(hInstance,hMyFastloc))
        {
            lpFastKeys=(FASTKEYS far *)LockResource(hMyFastRes);
        }
     }

    hInsert = GlobalAlloc(GMEM_MOVEABLE,2L);

    while (GetMessage(&msg, NULL, 0, 0))
       {
         if (!TranslateAccelerator(hwnd, hAccTable, &msg))
            {
            TranslateMessage(&msg);
            DispatchMessage(&msg); 
            }
        }
     
    GlobalFree(hInsert);
    UnlockResource(hMyFastRes);
    FreeResource(hMyFastRes);
    UnlockResource(hMyFontRes);
    FreeResource(hMyFontRes);

    return (msg.wParam);
}


/****************************************************************************

    FUNCTION: InitApplication(HANDLE)

    PURPOSE: Initializes window data and registers window class

****************************************************************************/

BOOL InitApplication(HANDLE hInstance)
{
    WNDCLASS  wc;

    myicon=LoadIcon(hInstance, "MyIcon");

    wc.style = NULL;
    wc.lpfnWndProc = MainWndProc;
    wc.cbClsExtra = 0;
    wc.cbWndExtra = 0;
    wc.hInstance = hInstance;
    wc.hIcon = myicon;
    wc.hCursor = LoadCursor(NULL, IDC_ARROW);
    wc.hbrBackground = GetStockObject(WHITE_BRUSH); 
    wc.lpszMenuName =  "WinOemMenu";
    wc.lpszClassName = "WinOemWClass";

    return (RegisterClass(&wc));
}


/****************************************************************************

    FUNCTION:  InitInstance(HANDLE, int)

    PURPOSE:  Saves instance handle and creates main window

****************************************************************************/

BOOL InitInstance(HANDLE hInstance,int nCmdShow)
{
    RECT            Rect;

    hInst = hInstance;

    hAccTable = LoadAccelerators(hInst, "WinOemAcc");

    hwnd = CreateWindow(
        "WinOemWClass",
        wintitle,
        WS_OVERLAPPEDWINDOW,
        CW_USEDEFAULT,
        CW_USEDEFAULT,
        CW_USEDEFAULT,
        CW_USEDEFAULT,
        NULL,
        NULL,
        hInstance,
        NULL
    );

    if (!hwnd)return (FALSE);

    GetClientRect(hwnd, (LPRECT) &Rect);

    // Create a child window

    hEditWnd = CreateWindow("Edit",
        NULL,
        WS_CHILD | WS_VISIBLE |
        ES_MULTILINE |
        WS_VSCROLL | WS_HSCROLL |
        ES_AUTOHSCROLL | ES_AUTOVSCROLL,
        0,
        0,
        (Rect.right-Rect.left),
        (Rect.bottom-Rect.top),
        hwnd,
        IDC_EDIT,                          // Child control i.d.
        hInst,
        NULL);
    
    if (!hEditWnd)
        {
        DestroyWindow(hwnd);
        return FALSE;
        }

    // Get an hourglass cursor to use during file transfers

    hHourGlass = LoadCursor(NULL, IDC_WAIT);

    // fill in non-variant fields of OPENFILENAME struct.
    ofn.lStructSize	  = sizeof(OPENFILENAME);
    ofn.hwndOwner	  = hwnd;
    ofn.lpstrFilter	  = szFilterSpec;
    ofn.lpstrCustomFilter = szCustFilterSpec;
    ofn.nMaxCustFilter	  = MAXCUSTFILTER;
    ofn.nFilterIndex	  = 1;
    ofn.lpstrFile	  = FileName;
    ofn.nMaxFile	  = MAXFILENAME;
    ofn.lpstrInitialDir   = PathName;
    ofn.Flags		  = 0L;
    ofn.lpfnHook	  = NULL;


    ShowWindow(hwnd, nCmdShow);
    hfnt = GetStockObject(OEM_FIXED_FONT);

    SendMessage(hwnd,WM_SETFONT,hfnt,TRUE);
    SendMessage(hEditWnd,WM_SETFONT,hfnt,TRUE);
    UpdateWindow(hwnd);

    return (TRUE);

}

/****************************************************************************

    FUNCTION: MainWndProc(HWND, UINT, WPARAM, LPARAM)

    PURPOSE:  Processes messages

    MESSAGES:

        WM_COMMAND    - read the code...
        WM_DESTROY    - destroy window
        WM_SIZE       - window size has changed
        WM_QUERYENDSESSION - willing to end session?
        WM_ENDSESSION - end Windows session
        WM_CLOSE      - close the window
        WM_SIZE       - window resized

    COMMENTS:

        (left in from the Microsoft Examples and Updated Where Necessary)

        WM_COMMAND processing:

            IDM_NEW - query to save current file if there is one and it has
                      been changed, clear buffer and start new file.

            IDM_OPEN - query to save current file if there is one and it
                       has been changed, open a new file.

            IDM_SAVE - save current file, prompt for name if none exists.

            IDM_SAVEAS - prompt for new filename to save to, save file.

            IDC_EDIT - change "bChanges" flag to indicate if edit buffer has
                      been modified.  Affects actions of IDM_NEW and
                      IDM_OPEN.  Reset when file is saved.

            IDM_EXIT - query to save current file if there is one and it
                       has been changed, then exit.

        After the size of the file is determined, only enough memory to store
        the file is allocated for the Edit buffer.  The edit control will
        automatically expand this memory as needed.  Once the file has been
        read into the edit buffer, unlock the memory.  Use whatever was
        obtained from the read() function, even if an error occured.  This
        allows partial salvage of a file with a bad sector.

****************************************************************************/

long FAR PASCAL MainWndProc(hWnd, message, wParam, lParam)
HWND hWnd;
UINT message;
WPARAM wParam;
LPARAM lParam;
{
    HMENU hMenu;
    unsigned IOStatus;               // result of file i/o
    WORD wFastSet, wFastChar;
    unsigned char c;

    switch (message) {

        case WM_CREATE:

            DosAscii  = 0;
            MakeCode  = ALT_BREAK;

            hMenu = CreateMenu();
            // Use bitmaps for menu items
            hMenuBitmap0  = LoadBitmap(hInst, "own0");
            hMenuBitmap1  = LoadBitmap(hInst, "own1");
            hMenuBitmap2  = LoadBitmap(hInst, "own2");
            hMenuBitmap3  = LoadBitmap(hInst, "own3");
            hMenuBitmap4  = LoadBitmap(hInst, "own4");
            hMenuBitmap5  = LoadBitmap(hInst, "own5");
            hMenuBitmap6  = LoadBitmap(hInst, "own6");
            hMenuBitmap7  = LoadBitmap(hInst, "own7");
            hMenuBitmap8  = LoadBitmap(hInst, "own8");
            hMenuBitmap9  = LoadBitmap(hInst, "own9");
            hMenuBitmap10 = LoadBitmap(hInst, "own10");
            hMenuBitmap11 = LoadBitmap(hInst, "own11");
            hMenuBitmap12 = LoadBitmap(hInst, "own12");
            AppendMenu(hMenu, MF_BITMAP|MF_DISABLED,IDM_BITMAP0,
                       (LPSTR)(LONG) hMenuBitmap0);
            AppendMenu(hMenu, MF_BITMAP, IDM_BITMAP1,
                       (LPSTR)(LONG) hMenuBitmap1);
            AppendMenu(hMenu, MF_BITMAP, IDM_BITMAP2,
                       (LPSTR)(LONG) hMenuBitmap2);
            AppendMenu(hMenu, MF_BITMAP, IDM_BITMAP3,
                       (LPSTR)(LONG) hMenuBitmap3);
            AppendMenu(hMenu, MF_BITMAP, IDM_BITMAP4,
                       (LPSTR)(LONG) hMenuBitmap4);
            AppendMenu(hMenu, MF_BITMAP, IDM_BITMAP5,
                       (LPSTR)(LONG) hMenuBitmap5);
            AppendMenu(hMenu, MF_BITMAP, IDM_BITMAP6,
                       (LPSTR)(LONG) hMenuBitmap6);
            AppendMenu(hMenu, MF_BITMAP, IDM_BITMAP7,
                       (LPSTR)(LONG) hMenuBitmap7);
            AppendMenu(hMenu, MF_BITMAP, IDM_BITMAP8,
                       (LPSTR)(LONG) hMenuBitmap8);
            AppendMenu(hMenu, MF_BITMAP, IDM_BITMAP9,
                       (LPSTR)(LONG) hMenuBitmap9);
            AppendMenu(hMenu, MF_BITMAP, IDM_BITMAP10,
                       (LPSTR)(LONG) hMenuBitmap10);
            AppendMenu(hMenu, MF_BITMAP, IDM_BITMAP11,
                       (LPSTR)(LONG) hMenuBitmap11);
            AppendMenu(hMenu, MF_BITMAP, IDM_BITMAP12,
                       (LPSTR)(LONG) hMenuBitmap12);


            ModifyMenu(GetMenu(hWnd),2, MF_POPUP | MF_BYPOSITION,
                       (WORD) hMenu, "Fast &Keys");
            CheckMenuItem(GetMenu(hWnd), IDM_CURRENTBMP, MF_CHECKED);
            return 0L;

        case WM_INITMENU:

            DosAscii  = 0;
            MakeCode  = ALT_BREAK;

            // if the clipboard is available enable the clipboard commands
            if (wParam == GetMenu(hWnd)) 
            {
                if (OpenClipboard(hWnd)) 
                {
                    if (IsClipboardFormatAvailable(CF_OEMTEXT)||
                        IsClipboardFormatAvailable(CF_TEXT))
                      {
                        EnableMenuItem(wParam, IDM_PASTE, MF_ENABLED);
                      }
                    else
                      {
                        EnableMenuItem(wParam, IDM_PASTE, MF_GRAYED);
                      }
                    CloseClipboard();
                    return (LONG)(TRUE);
                }
                else                    // Clipboard is not available
                    return (LONG)(FALSE);

            }
            break;

        case WM_COMMAND:

            // alt numeric key strokes
            for(;;)
            {
              if(wParam < IDK_HIGHSET0  || wParam > IDK_HIGHSET9)break;
              DosAscii*=10;
              DosAscii+=(wParam-IDK_HIGHSET0);
              if(DosAscii > 255)DosAscii=256;
              MakeCode = ALT_MAKE;
              return 0L;
             }

            if(wParam == IDK_HIGHMAKE)
            {
              if(MakeCode == ALT_BREAK)
              {
                DosAscii = 0;
                return 0L;
              }

              switch(DosAscii)
              {
                case 13:
                case 10:
                case 9 :
                case 8 :
                case 0 :
                case 0x1a: break;
                default:   if(DosAscii > 254)break;
                           AddAscii();

               }
               DosAscii  = 0;
               MakeCode  = ALT_BREAK;
               return 0L;
             }

            for(;;)
            {
              if(wParam<IDM_BITMAP1 || wParam>IDM_BITMAP12)break;
              CheckMenuItem(GetMenu(hWnd), IDM_CURRENTBMP, MF_UNCHECKED);
              IDM_CURRENTBMP = wParam;
              CheckMenuItem(GetMenu(hWnd), IDM_CURRENTBMP, MF_CHECKED);
              DosAscii  = 0;
              MakeCode  = ALT_BREAK;
              return 0L;
            }

            for(;;)
            {
                if(wParam<IDK_FASTSET1 || wParam>IDK_FASTSET10)break;
                wFastSet  = IDM_CURRENTBMP - IDM_BITMAP1;
                wFastChar = wParam         - IDK_FASTSET1;
                c=lpFastKeys[0].b[wFastSet][wFastChar];
                DosAscii =  (WORD)c;
                AddAscii();
                DosAscii  = 0;
                MakeCode  = ALT_BREAK;
                return 0L;
            }

            switch (wParam)
                {
                case IDM_UNDO:
                  if(MyUndo(hWnd)==FALSE)break; return 0L;

                case IDM_ABOUT:
                  if(MyAbout(hWnd)==FALSE)break; return 0L;

                case IDM_LICENCE:
                   if(MyLicence(hWnd)==FALSE)break; return 0L;

                case IDM_HELP:
                    if(MyHelp(hWnd)==FALSE)break; return 0L;

                case IDM_BMP:
                    if(SaveBmp(hWnd)==FALSE)break; return 0L;

                case IDM_COPY:
                    if(MyCopy(hWnd)==FALSE)break;  return 0L;

                case IDM_PASTE:
                    if(MyPaste(hWnd)==FALSE)break; return 0L;

                case IDM_NEW:

                    // If current file has been modified, query user
                    if (!QuerySaveFile(hWnd))return 0L;

                    // bChanges is set to FALSE to indicate there have been
                    // no changes since the last file save.

                    bChanges = FALSE;
                    FileName[0] = 0;

                    // Update the edit buffer

                    SetNewBuffer(hWnd, 0, Untitled);
                    break;

                case IDM_IMPORT:

                  DosAscii  = 0;
                  MakeCode  = ALT_BREAK;

                    if (!QuerySaveFile(hWnd))return 0L;

            // fill in title field of OPENFILENAME struct.
            // and show dialog box
		    ofn.lpstrTitle	  = (LPSTR)szOpenDlgTitle;
            if (!GetOpenFileName ((LPOPENFILENAME)&ofn))return 0L;


            // Open the file and get its handle
		    hFile = OpenFile (FileName, (LPOFSTRUCT)&OfStruct,
						  OF_READ);
            if (!hFile)return 0L;
            close(hFile);

            if(TranslateReport(hWnd)==FALSE)return 0L;
            hFile = open("WINOEM.$$$",O_RDONLY|O_BINARY);
            if(hFile==-1)
            {
                hFile = 0;
                return 0L;
            }

            // Allocate edit buffer to the size of the file + 1
		    fstat(hFile, &FileStatus);
            if(FileStatus.st_size > MAXFILESIZE)
            {
                close(hFile);
                // remove work file
                remove("WINOEM.$$$");
                sprintf(str, "File %s is %ld bytes in size.\n\n"
                             "This exceeds the capacity of WinOem\n",
                              FileName,
                              FileStatus.st_size);
                 MessageBox(hWnd, str,wintitle,MB_ICONHAND);
                 return 0L;
             }

             hEditBuffer = LocalAlloc(LMEM_MOVEABLE | LMEM_ZEROINIT,
                           (WORD)(FileStatus.st_size+1));

             if (!hEditBuffer)
             {
                MessageBox(hWnd, "Not enough memory.",wintitle,MB_ICONHAND);
                close(hFile);
                // remove work file
                remove("WINOEM.$$$");
                return 0L;
              }

              hSaveCursor = SetCursor(hHourGlass);
              pEditBuffer = LocalLock(hEditBuffer);

              IOStatus = read(hFile,
                              pEditBuffer, (unsigned int)FileStatus.st_size);
              close(hFile);
              // remove work file
              remove("WINOEM.$$$");
              // # bytes read must equal file size

              if (IOStatus != (unsigned)FileStatus.st_size)
                {
                sprintf(str, "Error reading %s.", FileName);
                SetCursor(hSaveCursor);
                // Remove the hourglass
                MessageBox(hWnd, str,wintitle,MB_ICONEXCLAMATION);
                }

            LocalUnlock(hEditBuffer);


            wParam = (WPARAM) MAXFILESIZE-IOStatus;
            lParam = 0L;
            SendMessage(hEditWnd, EM_LIMITTEXT,wParam,lParam);

            // Set up a new buffer and window title

            sprintf(str, "WinOem - %s", FileName);
            SetNewBuffer(hWnd, hEditBuffer, str);
            wParam = (WPARAM) MAXFILESIZE-IOStatus;
            lParam = 0L;

            // Limit the size of the edit buffer
            SendMessage(hEditWnd, EM_LIMITTEXT,wParam,lParam);
            SetCursor(hSaveCursor);            // restore the cursor
            break;



                case IDM_OPEN:

                  DosAscii  = 0;
                  MakeCode  = ALT_BREAK;

                    if (!QuerySaveFile(hWnd))return 0L;

            // fill in title field of OPENFILENAME struct.
            // and show dialog box
		    ofn.lpstrTitle	  = (LPSTR)szOpenDlgTitle;
            if (!GetOpenFileName ((LPOPENFILENAME)&ofn))return 0L;

            // Open the file and get its handle
		    hFile = OpenFile (FileName, (LPOFSTRUCT)&OfStruct,
						  OF_READ);
            if (!hFile)return 0L;

            // Allocate edit buffer to the size of the file + 1
		    fstat(hFile, &FileStatus);
            if(FileStatus.st_size > MAXFILESIZE)
            {
                close(hFile);
                sprintf(str, "File %s is %ld bytes in size.\n\n"
                             "This exceeds the capacity of WinOem\n",
                              FileName,
                              FileStatus.st_size);
                 MessageBox(hWnd, str,wintitle, MB_ICONEXCLAMATION);
                 return 0L;
             }
             hEditBuffer = LocalAlloc(LMEM_MOVEABLE | LMEM_ZEROINIT,
                           (WORD)(FileStatus.st_size+1));

             if (!hEditBuffer)
             {
            MessageBox(hWnd, "Not enough memory.",wintitle,MB_ICONHAND);
            close(hFile);
            return 0L;
              }

              hSaveCursor = SetCursor(hHourGlass);
              pEditBuffer = LocalLock(hEditBuffer);

              IOStatus = read(hFile,
                              pEditBuffer, (unsigned int)FileStatus.st_size);
              close(hFile);

              // # bytes read must equal file size

              if (IOStatus != (unsigned)FileStatus.st_size)
                {
                sprintf(str, "Error reading %s.", FileName);
                SetCursor(hSaveCursor);
                // Remove the hourglass
                MessageBox(hWnd, str,wintitle,MB_ICONEXCLAMATION);
                }
                LocalUnlock(hEditBuffer);


            wParam = (WPARAM) MAXFILESIZE-IOStatus;
            lParam = 0L;
            SendMessage(hEditWnd, EM_LIMITTEXT,wParam,lParam);

            // Set up a new buffer and window title

            sprintf(str, "WinOem - %s", FileName);
            SetNewBuffer(hWnd, hEditBuffer, str);
            wParam = (WPARAM) MAXFILESIZE-IOStatus;
            lParam = 0L;

            // Limit the size of the edit buffer
            SendMessage(hEditWnd, EM_LIMITTEXT,wParam,lParam);
            SetCursor(hSaveCursor);            // restore the cursor
            break;

            case IDM_SAVE:
                    // If there is no filename, use the saveas command
                    // to get one.
                    // Otherwise, save the file using the current filename.

                  DosAscii  = 0;
                  MakeCode  = ALT_BREAK;

            if (FileName[0])
                {
                    if (bChanges)SaveFile(hWnd);
                    break;
                 }
                    // else fall through for Saveas processing

            case IDM_SAVEAS:

            // fill in title field of OPENFILENAME struct.
            // and show  dialog box


                  DosAscii  = 0;
                  MakeCode  = ALT_BREAK;

            ofn.lpstrTitle    = (LPSTR)szSaveDlgTitle;
            if (!GetSaveFileName ((LPOPENFILENAME)&ofn))return (LONG)FALSE;
            // User canceled

            // If successful, update the window title, save the file
            sprintf(str, "WinOem - %s", FileName);
		    SetWindowText(hWnd, str);
		    SaveFile(hWnd);
		    break;

            case IDM_EXIT:

                    DosAscii  = 0;
                    MakeCode  = ALT_BREAK;
                    QuerySaveFile(hWnd);
                    DestroyWindow(hWnd);
                    break;

            case IDC_EDIT:

            switch (HIWORD (lParam))
                {
			case EN_ERRSPACE:
                MessageBox (GetFocus (), "Out of memory.\n\n"
                                         "Please Quit.\n",
                            wintitle,MB_ICONHAND);
			    break;

			case EN_CHANGE:
                // contents of edit control buffer have changed
			    if (!bChanges)
				bChanges = TRUE;
			    break;

                }
               break;

            } 

            break;

        case WM_SETFOCUS:
            SetFocus (hEditWnd);
            break;

        case WM_SIZE:

            DosAscii  = 0;
            MakeCode  = ALT_BREAK;
            MoveWindow(hEditWnd, 0, 0, LOWORD(lParam), HIWORD(lParam), TRUE);
            break;

        case WM_QUERYENDSESSION:
            // message: to end the session?
            return (long)(QuerySaveFile(hWnd));

        case WM_CLOSE:
            // message: close the window
            if (QuerySaveFile(hWnd))
                DestroyWindow(hWnd);
            break;

        case WM_DESTROY:

            DosAscii  = 0;
            MakeCode  = ALT_BREAK;

            WinHelp(hWnd,"WINOEM.HLP",HELP_QUIT,0L);
            DeleteObject(hMenuBitmap0);
            DeleteObject(hMenuBitmap1);
            DeleteObject(hMenuBitmap2);
            DeleteObject(hMenuBitmap3);
            DeleteObject(hMenuBitmap4);
            DeleteObject(hMenuBitmap5);
            DeleteObject(hMenuBitmap6);
            DeleteObject(hMenuBitmap7);
            DeleteObject(hMenuBitmap8);
            DeleteObject(hMenuBitmap9);
            DeleteObject(hMenuBitmap10);
            DeleteObject(hMenuBitmap11);
            DeleteObject(hMenuBitmap12);
            PostQuitMessage(0);
            break;

        default:
            return (DefWindowProc(hWnd, message, wParam, lParam));
    }
    return 0L;
}

/****************************************************************************

    FUNCTION: SaveFile(HWND)

    PURPOSE: Save current file

    COMMENTS:

        This saves the current contents of the Edit buffer, and changes
        bChanges to indicate that the buffer has not been changed since the
        last save.

        Before the edit buffer is sent, you must get its handle and lock it
        to get its address.  Once the file is written, you must unlock the
        buffer.  This allows Windows to move the buffer when not in immediate
        use.

****************************************************************************/
BOOL SaveFile(HWND hWnd)
{
    BOOL bSuccess;
    unsigned IOStatus;     // result of a file write

    DosAscii  = 0;
    MakeCode  = ALT_BREAK;

    if ((hFile =
        OpenFile(FileName, &OfStruct,OF_PROMPT | OF_CANCEL | OF_CREATE)) < 0)
        {

        // If the file can't be saved
        sprintf(str, "Cannot write to %s.", FileName);
        MessageBox(hWnd, str,wintitle,MB_ICONEXCLAMATION);
        SetFocus(hEditWnd);
        return (FALSE);
        }

    hEditBuffer = (HANDLE)SendMessage(hEditWnd, EM_GETHANDLE, 0, 0L);
    pEditBuffer = LocalLock(hEditBuffer);

    // Set the cursor to an hourglass during the file transfer

    hSaveCursor = SetCursor(hHourGlass);
    IOStatus = write(hFile, pEditBuffer, strlen(pEditBuffer));
    close(hFile);
    SetCursor(hSaveCursor);
    if (IOStatus != (unsigned)strlen(pEditBuffer))
        {
        sprintf(str, "Error writing to %s.", FileName);
        MessageBox(hWnd, str,wintitle,MB_ICONHAND);
        bSuccess = FALSE;
    }
    else {
        bSuccess = TRUE;                // Indicates the file was saved
        bChanges = FALSE;               // Indicates changes have been saved
    }

    LocalUnlock(hEditBuffer);
    SetFocus(hEditWnd);
    return (bSuccess);
}

/****************************************************************************/
BOOL MyCopy(HWND hWnd)
{
    DWORD dwMySize;          // number of characters to copy to clipboard
    DWORD dwResult;
    WORD  wStart, wStop;
    unsigned i,target;

    DosAscii  = 0;
    MakeCode  = ALT_BREAK;

    if (!OpenClipboard(hWnd))
    {
      MessageBox(hWnd, "The Clipboard is not available.",
                       wintitle,MB_ICONEXCLAMATION);
      SetFocus(hEditWnd);
      return FALSE;
    }


    if(hEditBuffer = (HANDLE)SendMessage(hEditWnd, EM_GETHANDLE, 0, 0L))
    {
       // if there was no text selected, copy the entire
       // file to the clipboard, otherwise just copy the
       // selected portion.
       dwResult = SendMessage(hEditWnd, EM_GETSEL, 0, 0L);
       wStart = LOWORD(dwResult);
       wStop  = HIWORD(dwResult);
       pEditBuffer = LocalLock(hEditBuffer);
       if(wStart == wStop)
       {
        wStart      = 0;
        wStop       = (unsigned)strlen(pEditBuffer);
        dwMySize    = (DWORD)wStop+1;
       }
       else
       {
        dwMySize     =  (DWORD)(wStop-wStart)+1;
       }
    }
    else
    {
      MessageBox(hWnd, "There is no text to copy.",
                        wintitle,MB_ICONEXCLAMATION);
      CloseClipboard();
      SetFocus(hEditWnd);
      return FALSE;
    }

    if(!(hData = GlobalAlloc(GMEM_MOVEABLE,dwMySize)))
    {
      MessageBox(hWnd, "Unable To Allocate Memory.",
                       wintitle,MB_ICONEXCLAMATION);
      CloseClipboard();
      LocalUnlock(hEditBuffer);
      SetFocus(hEditWnd);
      return FALSE;
    }

    if (!(lpData = GlobalLock(hData)))
    {
      MessageBox(hWnd, "Unable To Lock Memory.",
                       wintitle,MB_ICONEXCLAMATION);
      GlobalFree(hData);
      hData=NULL;
      CloseClipboard();
      LocalUnlock(hEditBuffer);
      SetFocus(hEditWnd);
      return FALSE;
     }

     // Clear the current contents of the clipboard, and set
     // the data handle to the new string.
     EmptyClipboard();
     target=0;
     for(i=wStart;i<wStop;i++)
     {
        lpData[target]=pEditBuffer[i];target++;
     }
     lpData[target]=0;
     GlobalUnlock(hData);
     LocalUnlock(hEditBuffer);
     SetClipboardData(CF_OEMTEXT, hData);
     CloseClipboard();
     hData = NULL;
     EnableMenuItem(GetMenu(hWnd), IDM_PASTE, MF_ENABLED);
     SetFocus(hEditWnd);
     return TRUE;
}

/****************************************************************************/
BOOL MyPaste(HWND hWnd)
{
    DWORD dwMySize;          // number of characters to paste from clipboard
    DWORD dwResult;
    LPARAM lParam;
    WPARAM wParam;
    WORD  wStart, wStop;
    unsigned i,target;

    DosAscii  = 0;
    MakeCode  = ALT_BREAK;

    if (!OpenClipboard(hWnd))
    {

      MessageBox(hWnd, "The Clipboard is not available.",
                       wintitle, MB_ICONEXCLAMATION);
      SetFocus(hEditWnd);
      return FALSE;
    }

    if (!(hData = GetClipboardData(CF_OEMTEXT)))
    {
       MessageBox(hWnd, "There is no text in the Clipboard.",
                        wintitle, MB_ICONEXCLAMATION);
       CloseClipboard();
       SetFocus(hEditWnd);
       return FALSE;

    }

    if (!(lpData = GlobalLock(hData)))
    {
      MessageBox(hWnd, "Unable To Lock Memory.",
                       wintitle, MB_ICONEXCLAMATION);
      hData=NULL;
      CloseClipboard();
      SetFocus(hEditWnd);
      return FALSE;
     }

     dwMySize = GlobalSize (hData);

     if(dwMySize > MAXFILESIZE)
     {
      MessageBox(hWnd, "Not Enough Memory To Edit Clipboard Text.",
                       wintitle,MB_ICONEXCLAMATION);
      GlobalUnlock(hData);
      hData=NULL;
      CloseClipboard();
      SetFocus(hEditWnd);
      return FALSE;
     }

      hEditBuffer = (HANDLE)SendMessage(hEditWnd, EM_GETHANDLE, 0, 0L);
      if(!hEditBuffer)
      {
        hEditBuffer = LocalAlloc(LMEM_MOVEABLE | LMEM_ZEROINIT,1);
        SendMessage(hEditWnd, EM_SETHANDLE, hEditBuffer, 0L);
      }

      pEditBuffer = LocalLock(hEditBuffer);
      wStop       = (unsigned)strlen(pEditBuffer);
      LocalUnlock(hEditBuffer);
      dwMySize    += wStop;

     if(dwMySize > MAXFILESIZE)
     {
      MessageBox(hWnd, "Not Enough Memory To Edit Clipboard Text.",
                       wintitle,MB_ICONEXCLAMATION);
      GlobalUnlock(hData);
      hData=NULL;
      CloseClipboard();
      SetFocus(hEditWnd);
      return FALSE;
     }

     // save the insertion point for the pasted text
     dwResult = SendMessage(hEditWnd, EM_GETSEL, 0, 0L);
     wStart = LOWORD(dwResult);

     wParam = 0;
     lParam = (LPARAM) (LPCSTR) lpData;
     SendMessage(hEditWnd, EM_REPLACESEL,wParam,lParam);
     bChanges = TRUE;

     // now that the edit window has expanded its buffer
     // at the insertion point transfer the untranslated
     // version to ensure 8-bit conversion of the OEM Text
     hEditBuffer = (HANDLE)SendMessage(hEditWnd, EM_GETHANDLE, 0, 0L);
     if(hEditBuffer)
     {
      pEditBuffer = LocalLock(hEditBuffer);
      wStop       = (unsigned)strlen(pEditBuffer);
      target=0;
      for(i=wStart;lpData[target]!=0;i++)
       {
          pEditBuffer[i]=lpData[target];target++;
       }
      LocalUnlock(hEditBuffer);
      wParam = (WPARAM) MAXFILESIZE-wStop;
      lParam = 0L;
      SendMessage(hEditWnd, EM_LIMITTEXT,wParam,lParam);
     }

     GlobalUnlock(hData);
     hData=NULL;
     CloseClipboard();
     SetFocus(hEditWnd);
     return TRUE;

}


/****************************************************************************

    FUNCTION: QuerySaveFile(HWND);

    PURPOSE: Called when some action might lose current contents

    COMMENTS:

        This function is called whenever we are about to take an action that
        would lose the current contents of the edit buffer.

****************************************************************************/

BOOL QuerySaveFile(HWND hWnd)
{
    int Response;

    DosAscii  = 0;
    MakeCode  = ALT_BREAK;

    if (bChanges)
    {
      ofn.lpstrTitle = (LPSTR)szSaveDlgTitle;
      sprintf(str, "Save current changes: %s", FileName);
      Response = MessageBox(hWnd, str,wintitle,
                             MB_YESNOCANCEL | MB_ICONEXCLAMATION);

      switch(Response)
      {
        case IDCANCEL: SetFocus(hEditWnd);
                       return FALSE;
        case IDNO    : SetFocus(hEditWnd);
                       return TRUE;
      }
      // Make sure there is a filename to save to
      while (!FileName[0])
      {
        if (!GetSaveFileName ((LPOPENFILENAME)&ofn))return FALSE;
       }

      SaveFile(hWnd);
    }

return TRUE;
}

/****************************************************************************

    FUNCTION: SetNewBuffer(HWND, HANDLE, PSTR)

    PURPOSE: Set new buffer for edit window

    COMMENTS:

        Point the edit window to the new buffer, update the window title,
        and redraw the edit window.

        If hNewBuffer is 0, then create an empty 1 byte buffer

****************************************************************************/

void SetNewBuffer(HWND hWnd, HANDLE hNewBuffer, PSTR Title)
{
    WPARAM wParam;
    LPARAM lParam;
    HANDLE hOldBuffer;

    DosAscii  = 0;
    MakeCode  = ALT_BREAK;

    hOldBuffer = (HANDLE)SendMessage(hEditWnd, EM_GETHANDLE, 0, 0L);
    if(hOldBuffer)LocalFree(hOldBuffer);
    if (!hNewBuffer)
     {
        // Allocates a buffer if none exists
        hNewBuffer = LocalAlloc(LMEM_MOVEABLE | LMEM_ZEROINIT, 1);
        // Updates the buffer and displays new buffer
        SendMessage(hEditWnd, EM_SETHANDLE, hNewBuffer, 0L);
        wParam = (WPARAM) MAXFILESIZE-1;
        lParam = 0L;
        SendMessage(hEditWnd, EM_LIMITTEXT,wParam,lParam);

    }
    else
    {
        SendMessage(hEditWnd, EM_SETHANDLE, hNewBuffer, 0L);
    }
    SetWindowText(hWnd, Title);
    SetFocus(hEditWnd);
    bChanges = FALSE;
}


/****************************************************************************/
BOOL  SaveBmp(HWND hWnd)
{
    BOOL Werror;
    WPARAM wParam;
    LPARAM lParam;
    WPARAM cLines, cLength, cIndex;
    unsigned bytesperline, width, height;
    unsigned char *ptr, *crtbuf, c;
    int y,x,i;
    int result;

    DosAscii  = 0;
    MakeCode  = ALT_BREAK;

    strcpy(SaveName,FileName);
    GetWindowText(hWnd,SaveText,127);

    // step 1. - get the  number of lines in the edit buffer
    //           this gives us the height of our bitmap.
    cLines = (WPARAM)SendMessage(hEditWnd, EM_GETLINECOUNT,0, 0L);

    if(cLines < 0)
    {
      MessageBox(hWnd, "Can't Convert Text To Bitmap!\n\n"
                       "No Text In Edit Buffer.\n",
                       wintitle, MB_ICONEXCLAMATION);
      SetFocus(hEditWnd);
      return FALSE;

    }

    // step 2. - create a conversion buffer the length of the longest line
    //           plus 1 for the terminating 0.
    bytesperline = 0;
    for(wParam=0;wParam<cLines;wParam++)
    {
        cIndex =  (WPARAM) SendMessage(hEditWnd, EM_LINEINDEX, wParam,0L);
        cLength = (WPARAM) SendMessage(hEditWnd, EM_LINELENGTH,cIndex,0L);
        if(cLength>bytesperline)bytesperline=cLength;

    }

    if(bytesperline==0)
    {
         MessageBox(hWnd,"Can't Convert Text To Bitmap!\n\n"
                         "No Text In Edit Buffer.\n",
                         wintitle,MB_ICONEXCLAMATION);
         return FALSE;
    }

    // lock the edit buffer before we try to get more memory
    hEditBuffer = (HANDLE)SendMessage(hEditWnd, EM_GETHANDLE, 0, 0L);
    pEditBuffer = LocalLock(hEditBuffer);

    height = cLines*19;
    width = bytesperline*8;
    // pad for double word boundaries
    while((bytesperline%4)!=0)bytesperline++;
    if((crtbuf=malloc(bytesperline))==NULL)
    {
         MessageBox(hWnd,"Out Of Memory.\n\nOperation Aborted!\n\n"
                         "Please Quit The Program.",
                         wintitle, MB_ICONHAND);
         LocalUnlock(hEditBuffer);
         SetFocus(hEditWnd);
         return FALSE;

    }

    ofn.lpstrTitle = (LPSTR)szSaveBmpTitle;
    if(FileName[0]!=0)
    {
        i=0;
        for(;;)
        {
            if(FileName[i]=='.' || FileName[i]==0)
            {
                FileName[i]   = '.';
                FileName[i+1] = 'b';
                FileName[i+2] = 'm';
                FileName[i+3] = 'p';
                FileName[i+4] =  0 ;
                break;
            }
            i++;
        }
    }

    for(;;)
      {
        if (!GetSaveFileName ((LPOPENFILENAME)&ofn))
        {
            free(crtbuf);
            LocalUnlock(hEditBuffer);
            strcpy(FileName,SaveName);
            return FALSE;
        }
        if (FileName[0])break;
      }

    if ((hFile =
        OpenFile(FileName, &OfStruct,OF_PROMPT | OF_CANCEL | OF_CREATE)) < 0)
        {
        // If the file can't be saved
        sprintf(str, "Cannot write to %s.", FileName);
        MessageBox(hWnd, str, wintitle, MB_ICONEXCLAMATION);
        free(crtbuf);
        LocalUnlock(hEditBuffer);
        strcpy(FileName,SaveName);
        SetFocus(hEditWnd);
        return (FALSE);
        }

    sprintf(str,"WinOem - %s",FileName);
    SetWindowText(hWnd,str);

    hSaveCursor = SetCursor(hHourGlass);

    // step 3. - read from last  line to first line through the edit buffer
    //           since the BMP file must be written from bottom to top.

    // create and write the header
    memset((char *)&bmfhead.bfType,0,sizeof(BITMAPFILEHEADER));
    memset((char *)&bmp.bmiHeader.biSize,0,sizeof(MYBITMAPINFO));

    ptr=(char *)&bmfhead.bfType;
    ptr[0] = 'B';                        // signature = BM
    ptr[1] = 'M';
    bmfhead.bfSize += bytesperline;      // 8 pixels per byte DW boundary
    bmfhead.bfSize *= height;            // X number of rasters
    bmfhead.bfSize +=  62L;              // + Header
    bmfhead.bfOffBits =62L;              // standard monochrome header
    bmp.bmiHeader.biSize     = 40L;      // info header size
    bmp.bmiHeader.biWidth    += width;   // pixels
    bmp.bmiHeader.biHeight   += height;  // rasters
    bmp.bmiHeader.biPlanes        = 1;         // device planes (always 1)
    bmp.bmiHeader.biBitCount      = 1;         // bits per pixel = 1
    bmp.bmiHeader.biCompression   = BI_RGB;    // no compression
    bmp.bmiHeader.biSizeImage     = bmfhead.bfSize - 62;
    bmp.bmiColors[1].rgbRed       = 255;       // color 1 is white
    bmp.bmiColors[1].rgbGreen     = 255;
    bmp.bmiColors[1].rgbBlue      = 255;

    Werror = FALSE;

    // Write the header
    result = write(hFile,(char *)&bmfhead.bfType,sizeof(BITMAPFILEHEADER));
    if(result < sizeof(BITMAPFILEHEADER))Werror = TRUE;
    result = write(hFile,(char *)&bmp.bmiHeader.biSize,sizeof(MYBITMAPINFO));
    if(result < sizeof(MYBITMAPINFO))Werror = TRUE;

    // loop backwards
    wParam = cLines;
    for(;;)
    {
       wParam--;
       cIndex =  (WPARAM) SendMessage(hEditWnd, EM_LINEINDEX, wParam,0L);
       cLength = (WPARAM) SendMessage(hEditWnd, EM_LINELENGTH,cIndex,0L);
       memset(crtbuf,0xff,bytesperline); // clear the buffer to white
       for(y=15;y>-1;y--)
       {
         i = cIndex;
         if(cLength>0)
         {
           for(x=0;x<cLength;x++)
           {
            c=pEditBuffer[i];i++;
            if(c==13||c==10||c==0||c=='\x1a')break;
            crtbuf[x] = lpRamFont[0].b[c][y];
            crtbuf[x]^=0xff;
           }
         }

         result = write(hFile,crtbuf,bytesperline);
         if(result < bytesperline)Werror = TRUE;

         if(y==2||y==8||y==13)result = write(hFile,crtbuf,bytesperline);
         if(result < bytesperline)Werror = TRUE;

       }
       // finished last line
       if(wParam < 1)break;
    }

    close(hFile);
    free(crtbuf);
    LocalUnlock(hEditBuffer);    // free memory
    SetCursor(hSaveCursor);
    SetWindowText(hWnd,SaveText);
    if(Werror==TRUE)
        sprintf(str, "Error writing to %s.", FileName);
    else
        sprintf(str, "Bitmap File %s was succesfully written.", FileName);

    MessageBox(hWnd, str,wintitle,MB_ICONEXCLAMATION);
      
    Strcpy(FileName,SaveName);
    SetFocus(hEditWnd);
    return TRUE;
}

BOOL MyHelp(HWND hWnd)
{
    DosAscii  = 0;
    MakeCode  = ALT_BREAK;

    hSaveCursor = SetCursor(hHourGlass);
    WinHelp(hWnd,"WINOEM.HLP",HELP_CONTENTS,0L);
    SetCursor(hSaveCursor);
    return TRUE;
}

BOOL MyAbout(HWND hWnd)
{
    DosAscii  = 0;
    MakeCode  = ALT_BREAK;

    MessageBox(hWnd,"WinOem Version 3.0 "
                    "Copyright \251 Bill Buckels 1994-1999.\n\n"
                    "WinOem is a limited function\n"
                    "Windows OEM Text Editor and "
                    "monochrome BMP File Writer.",
                    "About WinOem Copyright \251 Bill Buckels 1994-1999",
                     MB_ICONINFORMATION);

    SetFocus(hEditWnd);
    return TRUE;
}

BOOL MyLicence(HWND hWnd)
{
    DosAscii  = 0;
    MakeCode  = ALT_BREAK;

    MessageBox(hWnd,"WinOem Version 3.0 "
      "Copyright \251 Bill Buckels 1994-1999.\n\n"
      "WINOEM is distributed as ShareWare. Registration is $10.00 per computer. "
      "You must register with the Author after 30 days, or you must remove "
      "WINOEM and all associated files from your computer. Send registration in "
      "the form of cheque or money order to:\n\n"
      "\t\tBill Buckels\n"
      "\t\t589 Oxford Street\n"
      "\t\tWinnipeg, Manitoba, Canada R3M 3J2\n\n"
      "Email: bbuckels@escape.ca\n"
      "WebSite: http://www.escape.ca/~bbuckels\n\n"
      "Distribution of this program for profit without the express permission "
      "of the author is not permitted.",
      "WinOem Licencing Agreement",
      MB_ICONINFORMATION);

    SetFocus(hEditWnd);
    return TRUE;

}


BOOL MyUndo(HWND hWnd)
{
    DosAscii  = 0;
    MakeCode  = ALT_BREAK;

   if(SendMessage(hEditWnd, EM_CANUNDO,0,0L))
   {
      SendMessage(hEditWnd, EM_UNDO,0,0L);
      SetFocus(hEditWnd);
      return TRUE;
    }
    else
    {
      MessageBox(hWnd,"Sorry... \n\nUnable to Undo Last Operation.\n\n",
                      wintitle,MB_OK);

      SetFocus(hEditWnd);
      return FALSE;
    }

}

// add high ascii dos characters to the edit buffer
void AddAscii()
{
      LPARAM lParam;
      WPARAM wParam;
      WORD wStart;
      DWORD dwResult;

      InsertText = GlobalLock(hInsert);
      InsertText[0]=(unsigned char)DosAscii;
      InsertText[1]=0;

      hEditBuffer = (HANDLE)SendMessage(hEditWnd, EM_GETHANDLE, 0, 0L);
      if(!hEditBuffer)
      {
        hEditBuffer = LocalAlloc(LMEM_MOVEABLE | LMEM_ZEROINIT,1);
        SendMessage(hEditWnd, EM_SETHANDLE, hEditBuffer, 0L);
      }

     // save the insertion point
     dwResult = SendMessage(hEditWnd, EM_GETSEL, 0, 0L);
     wStart = LOWORD(dwResult);

     wParam = 0;
     lParam = (LPARAM) (LPCSTR) InsertText;
     SendMessage(hEditWnd, EM_REPLACESEL,wParam,lParam);
     bChanges = TRUE;

     // now that the edit window has expanded its buffer
     // at the insertion point transfer the untranslated
     // version to ensure 8-bit conversion of the OEM Text
     hEditBuffer = (HANDLE)SendMessage(hEditWnd, EM_GETHANDLE, 0, 0L);
     if(hEditBuffer)
     {
      pEditBuffer = LocalLock(hEditBuffer);
      pEditBuffer[wStart] = InsertText[0];
      LocalUnlock(hEditBuffer);
     }
     GlobalUnlock(hInsert);
     SetFocus(hEditWnd);
     DosAscii  = 0;
     MakeCode  = ALT_BREAK;


}

// ------------------------------------------------------------------------
// translate a clarion report and import it
// this may not work in cases that I haven't anticipated (grin).
//
//   we need to basically emulate a line printer in this routine
//   we have no way of knowing what kind of garbage we will get
//     but we attempt to handle FF, LF, and CR since these are common
//     printer control characters.
//   other characters below ascii 32 (whitespace) are ignored since these
//     are usually printer control characters.
//   since a CR may be received before or after a LF or FF we try to
//     behave the way a printer will... we "look-ahead" for the character
//     that follows the CR.
//     - if the next character is a FF or a LF we print the line...
//     - if the next character is another CR etc. we just throw it away...
//     - if the next character is a printable character the line is
//       reset to 1 character and we carry-on.
//     - ignores TAB and BACKSPACE etc. in this version.
// ------------------------------------------------------------------------
BOOL TranslateReport(HWND hWnd)
{
    int c, i;

    // remove work file
    remove("WINOEM.$$$");

    // open text file
    if((fp=fopen(FileName,"rb"))==NULL)
    {
         sprintf(str, "Unable to open %s.\n",FileName);
         MessageBox(hWnd, str,wintitle,MB_ICONHAND);
         return FALSE;

    }

    // open work file
    if((fp2=fopen("WINOEM.$$$","w"))==NULL)
    {
        fclose(fp);
        MessageBox(hWnd,"Unable to open workfile.\n",wintitle,MB_ICONHAND);
        return FALSE;
    }

    // clarion reports use printer control characters
    // instead of normal text formatting

    // start with a clean buffer
    for(i=E_FIRST;i<E_MAXBUFFER;i++)szBuffer[i]=(BYTE)E_ASCIIZ;
    // set counter to 0
    i=0;

    // read a byte, write a buffer
    while((c=fgetc(fp))!=EOF)
    {
       // if a printer control character ignore it, except for FF,CR, and LF
       if(c<E_WHITESPACE)
       {
          if(c!=E_FF && c!=E_LF && c!=E_CR)continue;

       }
       else  // if a printable character print-it until the carriage returns
       {     // or the line or the form feeds...
             // this may result in some garbage at the head of lines
             // but this can be cleaned-up in the editor
          if(i==E_MAXBUFFER)
          {
            fprintf(fp2,"%s",szBuffer);
            for(i=E_FIRST;i<E_MAXBUFFER;i++)szBuffer[i]=(BYTE)E_ASCIIZ;
            szBuffer[E_FIRST] = (BYTE)c;
            i=1;
          }
          else
          {
           szBuffer[i]=(BYTE)c;
           i++;
          }
          continue;
       }

       // -----------------------------------------------------------------
       // only FF, LF, and CR get to here...

       // if we get a page feed or a line feed
       // then print the contents of the buffer
       // ...then get another pass...
       // -----------------------------------------------------------------
       if(c==E_FF||c==E_LF)
       {
          fprintf(fp2,"%s\n",szBuffer);
          for(i=E_FIRST;i<E_MAXBUFFER;i++)szBuffer[i]=(BYTE)E_ASCIIZ;
          i=E_FIRST;
          continue;

       }

       // exclusive case... we have found a carriage return
       // if we get a carriage return...
       // read the next character... the result of the next character
       // will determine what we should do with the contents of the buffer

       for(;;)
       {
         c=fgetc(fp);
         if(c==EOF)break;
         if(c==E_LF)break;
         if(c==E_FF)break;
         if(c==E_WHITESPACE||c>E_WHITESPACE)break;
                               // ignore redundant returns...
                               // 2 or more in a row just homes the
                               // carriage on a printer and doesn't feed
                               // the line...
       }

       if(c==EOF)
       {
          fprintf(fp2,"%s\n",szBuffer);  // special case for end-of-file
          break;                     // flush the buffer
       }

        // if the character following the carriage return is a formfeed
        // or a line feed we print the buffer and reset it to 0 bytes
       if(c==E_FF||c==E_LF)
       {
          fprintf(fp2,"%s\n",szBuffer);
          for(i=E_FIRST;i<E_MAXBUFFER;i++)szBuffer[i]=(BYTE)E_ASCIIZ;
          i=E_FIRST;                 // set counter to 0
          continue;

       }

       // put the character into the buffer at position 0 since we
       // are starting the line over again...
       for(i=E_FIRST;i<E_MAXBUFFER;i++)szBuffer[i]=(BYTE)E_ASCIIZ;
       szBuffer[E_FIRST]=(BYTE)c;
       i=E_SECOND;                // set counter to 1 because we have 1 byte

     }
     // assume we have been successful and return TRUE;
     fclose(fp);
     fclose(fp2);

return TRUE;
}
