/* libmdate 
 * libmdate is a library that allows manipulation of mayan dates */

/*
 * Copyright (C) 2000 Mdate Development Team, Project Admin:
 * Sean Dwyer <ewe2@users.sourceforge.net> . All Rights Reserved.
 * This code is distributed under the terms of the GPL. Some GPL'd code by
 * others is included in this module.
 */

/* main decls and macros */

/* everything we need MUST be in here! */
#include "mdate.h"

BOOL jdate_from_double_jd(double from, julian_date * to)
{
  if (!is_valid_jdate(from)) {
/*    fprintf(stderr, _("Bounds error for epoch, invalid data.\n")); */
    return FALSE;
  }
  *to = from;
  return TRUE;
}

void jdate_to_str(julian_date from, char *to)
{
#ifdef HAVE_SNPRINTF
  snprintf(to, 255, "%.1f", from);
#else
  sprintf(to, "%.1f", from);
#endif
}

/* These functions are derived from mayalib.py, Ivan Van Laningham's excellent
 * mayan date routines in python, so attributed below.
 * Modified for bounds-checking the epoch.
 */

/*
Copyright & Disclaimer

(C) 1997,1998, Copyright by Ivan Van Laningham; All Rights Reserved.
Mail to: ivanlan@callware.com

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee or royalty is hereby granted,
provided that the above copyright notice appear in all copies and that both
the copyright notice and this permission notice appear in supporting
documentation or portions thereof, including modifications, that you make.

THE AUTHOR IVAN VAN LANINGHAM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL
DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE!
*/

BOOL greg_date_from_jdate(julian_date from, greg_date * to)
{
  int d, m, y;
  julian_date jn;
  int jalpha, ja, jb, jc, jd, je;

  jn = from;

  if (!is_valid_jdate(from)) {
/*    fprintf(stderr, _("Bounds error for epoch, invalid data.\n")); */
    return FALSE;
  }

  jalpha = (int) (((jn - 1867216) - 0.25) / 36524.25);
  ja = (int) (jn + 1 + jalpha - ((int) ((0.25 * jalpha))));
  jb = (int) (ja + 1524);
  jc = (int) (6680.0 + ((jb - 2439870.0) - 122.1) / 365.25);
  jd = (int) (365.0 * jc + (0.25 * jc));
  je = (int) ((jb - jd) / 30.6001);

  d = (jb - jd - ((int) (30.6001 * je)));

  while (d < 0) {
    d = d + 7;
  }

  m = je - 1;

  if (m > 12) {
    m = m - 12;
  }

  while (m < 0) {
    m = m + 12;
  }

  y = jc - 4715;

  if (m > 2) {
    y = y - 1;
  }

  to->day = d;
  to->month = m;
  to->year = y;
  return TRUE;
}

/* XXX: don't test validity here, perhaps? -- ewe2 */

BOOL jdate_from_greg_date(greg_date from, julian_date * to)
    /* double jdate_from_greg_date(int d, int m, int y) */
{
  int mm, dd, yy, jm;
  double jul, ja, jul2, temp1, temp2;

  /* XXX: presumed bug -- ewe2 */
  if (!(is_valid_greg_date(from)))
    return FALSE;

  mm = from.month;

  if (mm > 2) {
    yy = from.year;
    jm = mm + 1;
  }
  else {
    yy = from.year - 1;
    jm = mm + 13;
  }

  dd = from.day;
  temp1 = 365.25 * yy;
  temp2 = (30.6001 * jm) + (dd + 1720995.0);
  jul = (int) (floor(temp1)) + (floor(temp2));

  ja = (int) (0.01 * yy);
  jul2 = (int) (jul + (2 - ja + ((int) (0.25 * ja))));
  *to = jul2;
  return TRUE;
}

/* end of ivan's routines */

/* try to pick the right mod to use regardless of sign */

/* added to shut bcc32 up */
extern double drem(double,double);

static double my_mod(double x, double y)
{
  double result;

  if ((x < 0) && (y < 0))
    /* both -ve */
    result = fmod(x, y);
  else if ((x > 0) && (y < 0))
    /* x is +ve, y is -ve */
    result = drem(x, y);
  else if ((x < 0) && (y > 0))
    /* x is -ve, y is +ve */
    result = drem(x, y);
  else
    /* both +ve */
    result = fmod(x, y);

  return result;
}

/* take a julian day number and convert it to a haab_date. */
BOOL haab_date_from_jdate(julian_date from, haab_date * to)
{
  double lc = from - JD_CORRELATION;
  double doh;
  int day, month;

  doh = my_mod((lc + 8.0 + 20.0 * (18.0 - 1.0)), 365.0);
  day = floor(my_mod(doh, 20.0));
  month = floor(doh / 20.0) + 1;

  to->day = day;
  to->month = month;
  return TRUE;
}

BOOL haab_month_to_str(int from, char *to)
{
  /* Leave untranslated, although current Mayan usage is different */
  static const char *mnth[] = {
    "START", "Pop", "Uo", "Zip", "Zotz", "Tzec", "Xul", "Yaxkin",
    "Mol", "Chen", "Yax", "Zac", "Ceh", "Mac", "Kankin", "Muan",
    "Pax", "Kayab", "Cumku", "Uayeb"
  };

  if (from > 19) {
    return FALSE;
  }

  strcpy(to, mnth[from]);

  return TRUE;
}

BOOL haab_date_to_str(haab_date from, char *to)
{
  char month[256];

  if (!haab_month_to_str(from.month, month)) {
    return FALSE;
  }
#ifdef HAVE_SNPRINTF
  snprintf(to, 256, "%d %s", from.day, month);
#else
  sprintf(to, "%d %s", from.day, month);
#endif
  return TRUE;
}

BOOL tzolkin_month_to_str(int from, char *to)
{
  static const char *mnth[] = {
    "START", "Imix", "Ik", "Akbal", "Kan", "Chicchan", "Cimi",
    "Manik", "Lamat", "Muluc", "Oc", "Chuen", "Eb", "Ben", "Ix", "Men",
    "Cib", "Caban", "Eznab", "Cauac", "Ahau"
  };

  if (from > 20) {
    return FALSE;
  }

  strcpy(to, mnth[from]);
  return TRUE;
}

#define AMOD(x,y) (1 + (((x) - 1) % (y)))

/* take a julian day number and convert it to a tzolkin_date */
BOOL tzolkin_date_from_jdate(julian_date from, tzolkin_date * to)
{
  double lc = from - JD_CORRELATION;
  int number, name, ilc;

  ilc = floor(lc);
  number = AMOD((ilc + 4), 13);
  name = AMOD((ilc + 20), 20);

  to->month = name;
  to->day = number;

  return TRUE;
}

BOOL tzolkin_date_to_str(tzolkin_date from, char *to)
{
  char month[256];

  if (!tzolkin_month_to_str(from.month, month)) {
    return FALSE;
  }
#ifdef HAVE_SNPRINTF
  snprintf(to, 256, "%d %s", from.day, month);
#else
  sprintf(to, "%d %s", from.day, month);
#endif
  return TRUE;
}

void long_count_to_str(long_count from, char *to)
{
#ifdef HAVE_SNPRINTF
  snprintf(to, 256, "%.2d.%.2d.%.2d.%.2d.%.2d",
	   from.bak, from.kat, from.tun, from.uin, from.kin);
#else
  sprintf(to, "%.2d.%.2d.%.2d.%.2d.%.2d",
	  from.bak, from.kat, from.tun, from.uin, from.kin);
#endif
  return;
}

static int gregorian_leap_year(int g_year)
{
  return !(g_year % 100 ? g_year % 4 : g_year % 400);
}

/* Take a Julian Day Number, and convert it to a Long Count */

BOOL long_count_from_jdate(julian_date from, long_count * to)
{
  double lc;
  double r;

#if 0				/* How's this different from from - JD_CORRELATION? 
				 * If from == JD_CORRELATION, then from - JD_CORRELATION 
				 * will equal 0.0 */
  if (from > JD_CORRELATION) {
    lc = from - JD_CORRELATION;
  }
  else if (from == JD_CORRELATION) {
    lc = 0.0;
  }
  else {
    lc = from - JD_CORRELATION;
  }
#else
  lc = from - JD_CORRELATION;
#endif

  if (lc < 0.0) {
/*    fprintf(stderr, _("Bounds error for epoch, invalid data.\n")); */
    return FALSE;
  }
  else {
    to->bak = floor(lc / 144000.0);
    r = my_mod(lc, 144000.0);
    to->kat = floor(r / 7200);
    r = my_mod(r, 7200.0);
    to->tun = floor(r / 360);
    r = my_mod(r, 360.0);
    to->uin = floor(r / 20);
    to->kin = my_mod(r, 20.0);
    return TRUE;
  }
}

/* Take a Long Count and convert it to a Julian Day Number */
BOOL jdate_from_long_count(long_count from, julian_date * to)
{
  double result;

  result = (double) ((from.bak * 144000.0) +
		     (from.kat * 7200.0) +
		     (from.tun * 360.0) +
		     (from.uin * 20.0) + (double) from.kin);

  *to = result + JD_CORRELATION;

  return TRUE;
}

/* short-circuit bad dates before the epoch */

BOOL is_valid_jdate(double jul)
{
  if (jul < JD_CORRELATION)
    return FALSE;

  return TRUE;
}

/* this works now -- ewe2 */
BOOL is_valid_greg_date(greg_date date)
{
  static const int daysinmonth[] =
    { 0, 31, 29, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 };

  if (date.month < 1 || date.month > 12)
    return FALSE;

  if ((date.month == 2) && (!gregorian_leap_year(date.year))
      && (date.day > 28))
    return FALSE;

  if (date.day < 1 || date.day > daysinmonth[date.month])
    return FALSE;

  return TRUE;
}

#if defined(WIN32)
/* this is back because of the problems with the function below under Win32. works fine. */
const char *gmonth_str (int month)
{
  static const char *names[] = { 
     "January", "February", "March", "April", "May", "June", "July",
     "August", "September", "October", "November", "December"
  };
  return names[month - 1];
}
#else /* this should be ifdef'd out if under Win32 */
void greg_date_to_str(greg_date from, char *to)
{
  /* The first three are arbitrary, and the next to last two are wrong, but
   * hopefully irrelevant. */

  /* XXX - altered because only GNU C allows non-constant initializers, Borland C/C++
   * blows chunks. We still have a problem with -ve dates, i think strftime is to blame :(
   */
  struct tm gtm;
/*
  int y, m, d;
  y = from.year - 1900;
  m = from.month - 1;
  d = from.day;
*/
  gtm = {12, 12, 12, from.day, from.month - 1, from.year - 1900, 0, 0, 0};
/*
  gtm.tm_year = y;
  gtm.tm_mon = m;
  gtm.tm_mday = d;
*/
  strftime(to,256, "%d-%B-%Y (%d/%m/%Y)", &gtm);
}
#endif /* defined(WIN32) */

BOOL jdate_from_tm(struct tm from, julian_date * to)
{
  greg_date gdate;

  gdate.day = from.tm_mday;
  gdate.month = from.tm_mon + 1;
  gdate.year = from.tm_year + 1900;
  jdate_from_greg_date(gdate, to);
  return TRUE;
}

/*
 * $Id: libmdate.c,v 1.1.1.1 2000/04/06 15:33:36 ewe2 Exp $
 * $Log: libmdate.c,v $
 * Revision 1.1.1.1  2000/04/06 15:33:36  ewe2
 * libmdate 0.0.3 import
 *
 *
 * Revision dvdeug branch 1 pre-alpha 2000/03/31 18:45:00 dvdeug
 * Mdate branch for new API
 * 
 * Revision 1.1.1.2  2000/03/27 00:58:39  ewe2
 * Mdate 1.1.0 tentative version
 *
 * Revision 1.2  2000/03/26 06:38:47  ewe2
 * Mdate 1.1.0 tentative version
 *
 * Revision 1.1.1.1  2000/03/16 03:39:17  ewe2
 * Initial import
 *
 * $State: Exp $
 */
