/*====================================================================================================================================
DbGen.c

(c) 1999-2001, Jim Gray (Gray@MICROSOFT.com), Andrej Barkhatov (abarkhatov@yahoo.com)

These programs are provided "as is", without any express or implied warranty.
The program is freeware.  It is freely distributed subject to the restriction that this Copyright be included with any distribution
(so that we can continue to disclaim responsibility for any problems that may arise from it).

For explanations see "Quickly Generating Billion-Record Synthetic Databases",
Jim Gray, Prakash Sundaresan, Susanne Englert, Ken Baclawski, Peter J. Weinberger, SIGMOD Conference 1994: 243-252.
An earlier version of the paper is at the web site http://research.microsoft.com/~Gray/DbGen/SyntheticDataGen.doc.
=====================================================================================================================================*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <limits.h>
#include <float.h>
#include <math.h>
#include <ctype.h>
#include <stdarg.h>

#include <windows.h>
#include <windowsx.h>
#include <commctrl.h>
#include <shellapi.h>

#include <sql.h>
#include <sqlext.h>

#pragma warning( disable : 4100)
#pragma warning( disable : 4127)

#ifndef _I64_MAX
#define _I64_MAX 9223372036854775807LL
#endif

#define I31 0x80000000
#define I31_1 0x7FFFFFFF

#define I62 0x4000000000000000
#define I62_1 0x3FFFFFFFFFFFFFFF

#define IS_DBL_ZERO(x) (fabs(x) < 4 * DBL_EPSILON)

#ifndef XLARGEBUFF
#define XLARGEBUFF 1024
#endif
#ifndef LARGEBUFF
#define LARGEBUFF 300
#endif
#ifndef SMALLBUFF
#define SMALLBUFF 128
#endif
#ifndef TINYBUFF
#define TINYBUFF 40
#endif

#define X_SHT 2
#define Y_SHT 54
#define W_SHT 780
#define H_SHT 290
#define W_MAIN 790
#define H_MAIN 408

#define COLOR_NUM 40

#define INT_LEN 12
#define SMALLINT_LEN 6
#define TINYINT_LEN 3
#define I64_LEN 19

#define TIME_LEN 16

#define CHAR95_LEN 9

#define DECIMAL_PREC 18
#define DECIMAL_LEN (DECIMAL_PREC+2)
#define REAL_PREC 7
#define REAL_LEN (REAL_PREC+2)
#define FLOAT_PREC 15
#define FLOAT_LEN (FLOAT_PREC+2)

#define GEN_DEC 16

#define PRIM_MAX_1 11327587540653820

#define GRAN_SZ 1000

#define COL_NUM 12

#define COL_NAME 0
#define COL_TYPE 1
#define COL_LENGTH 2
#define COL_PREC 3
#define COL_SCALE 4
#define COL_DISTR 5
#define COL_STDDEV 6
#define COL_MEAN 7
#define COL_MIN 8
#define COL_MAX 9
#define COL_KEY 10
#define COL_UNIQUE 11

#define TOK_NUM	21

#define TOK_DBGEN 12
#define TOK_CONNECT	13
#define TOK_TABLE 14
#define TOK_FLDNUM 15
#define TOK_ROWNUM 16
#define TOK_DESTFILE 17
#define TOK_DESTODBC 18
#define TOK_H 19
#define TOK_A 20

#define DISTR_NUM 7

#define DISTR_RAND 0
#define DISTR_ORD 1
#define DISTR_GAUS 2
#define DISTR_EXPN 3
#define DISTR_POIS 4
#define DISTR_SSIM 5
#define DISTR_ZIPF 6

#define TYPE_NUM 10
#define TYPE_INT 0
#define TYPE_SMALLINT 1
#define TYPE_TINYINT 2
#define TYPE_REAL 3
#define TYPE_FLOAT 4
#define TYPE_DECIMAL 5
#define TYPE_CHAR 6
#define TYPE_VARCHAR 7
#define TYPE_DATETIME 8
#define TYPE_BIGINT 9

#define MONTH_0 0
#define MONTH_1 (31*24*60)
#define MONTH_2 (MONTH_1 + 28*24*60)
#define MONTH_3 (MONTH_2 + 31*24*60)
#define MONTH_4 (MONTH_3 + 30*24*60)
#define MONTH_5 (MONTH_4 + 31*24*60)
#define MONTH_6 (MONTH_5 + 30*24*60)
#define MONTH_7 (MONTH_6 + 31*24*60)
#define MONTH_8 (MONTH_7 + 31*24*60)
#define MONTH_9 (MONTH_8 + 30*24*60)
#define MONTH_10 (MONTH_9 + 31*24*60)
#define MONTH_11 (MONTH_10 + 30*24*60)
#define MONTH_12 (365*24*60)

#define IDC_OK 1100
#define IDC_LEFT 1101
#define IDC_RIGHT 1102
#define IDC_README 1103

#define PAR_NUM 3
#define PAR_MEAN 0
#define PAR_MIN	1
#define PAR_MAX	2

#define CONNECT_STR "DRIVER={SQL Server};UID=sa;PWD=;SERVER=(local);DATABASE=tempdb;"
#define DBGENINI_STR "dbgen.ini"
#define README_STR "readme.rtf"
#define COL_DBL(t) ((t) == TYPE_REAL || (t) == TYPE_FLOAT || (t) == TYPE_DECIMAL)

typedef struct _RandT {
	__int64 Root;
	__int64 Prim;
	__int64 iRand;
}
RandT;

typedef struct _GraphT {
	double *Val;
	double ValMin;
	double ValMax;
	int *Bar;
	int	BarMax;
}
GraphT;

typedef struct _ColT {
	char Name[SMALLBUFF];
	int Type;
	int Length;
	int Prec;
	int Scale;
	int Distr;
	double Dev;
	double DPar[PAR_NUM];
	__int64 LPar[PAR_NUM];
	int fPrimKey;
	int fUnique;
	RandT Rand;
	GraphT Graph;
}
ColT;

typedef struct _OdbcT {
	char Schem[LARGEBUFF];
	char Cat[LARGEBUFF];
	char Quot[2];
	char CatSep[TINYBUFF];
	char BigName[XLARGEBUFF];
	char *Stmt;
	SQLINTEGER **Len;
}
OdbcT;

typedef struct _VecT {
	char *h;
	int cur;
	int	max;
	int	sz;
}
VecT;

typedef struct _VecColT {
	ColT *h;
	int	cur;
	int	max;
	int	sz;
}
VecColT;

#define DEF_INT {{0},TYPE_INT, INT_LEN, 0, 0,DISTR_RAND,0,{0},{0,0,INT_MAX},0,0,{0}}
#define DEF_SMALLINT {{0},TYPE_SMALLINT,SMALLINT_LEN,0,0,DISTR_RAND,0,{0},{0,0,SHRT_MAX},0,0,{0},{0}}
#define DEF_TINYINT {{0},TYPE_TINYINT,TINYINT_LEN,0,0,DISTR_RAND,0,{0},{0,0,UCHAR_MAX},0,0,{0},{0}}
#define DEF_REAL {{0},TYPE_REAL,REAL_LEN,REAL_PREC,0,DISTR_RAND,0,{0,0,FLT_MAX},{0},0,0,{0},{0}}
#define DEF_FLOAT {{0},TYPE_FLOAT,FLOAT_LEN,FLOAT_PREC,0,DISTR_RAND,0,{0,0,DBL_MAX},{0},0,0,{0},{0}}
#define DEF_DECIMAL {{0},TYPE_DECIMAL,DECIMAL_LEN,DECIMAL_PREC,0,DISTR_RAND,0,{0,0,INT_MAX},{0},0,0,{0},{0}}
#define DEF_CHAR {{0},TYPE_CHAR,CHAR95_LEN,0,0,DISTR_RAND,0,{0},{0,0,PRIM_MAX_1},0,0,{0},{0}}
#define DEF_VARCHAR {{0},TYPE_VARCHAR,CHAR95_LEN,0,0,DISTR_RAND,0,{0},{0,0,PRIM_MAX_1},0,0,{0},{0}}
#define DEF_DATETIME {{0},TYPE_DATETIME,TIME_LEN,0,0,DISTR_RAND,0,{0},{0,INT_MAX/16,INT_MAX/10},0,0,{0},{0}}
#define DEF_BIGINT {{0},TYPE_BIGINT,I64_LEN,0,0,DISTR_RAND,0,{0},{0,0,PRIM_MAX_1},0,0,{0},{0}}

ColT DefCol[TYPE_NUM] = {
	DEF_INT, DEF_SMALLINT, DEF_TINYINT, DEF_REAL, DEF_FLOAT, DEF_DECIMAL, DEF_CHAR, DEF_VARCHAR, DEF_DATETIME, DEF_BIGINT
};

char *DistrName[DISTR_NUM] = {
	"random", "ordinal", "normal", "exp(neg)", "poisson", "self-sim", "zipf"
};

char *TypeName[TYPE_NUM] = {
	"int", "smallint", "tinyint", "real", "float", "decimal", "char", "varchar", "datetime", "bigint"
};

int TypeSz[TYPE_NUM] = {
	sizeof(int), sizeof(short), sizeof(char), sizeof(float), sizeof(double), 0, 0, 0, sizeof(TIMESTAMP_STRUCT), sizeof(__int64)
	};

__int64 Prime[GEN_DEC] = {
	11, 101, 1009, 10007, 100003, 1000003, 10000019, 100000007, 2147483647,
	27095900237, 114535788533, 1051514334749, 34112825814573, 241407036416013, 1969087082879949, 11327587540653821
};

__int64  Root[GEN_DEC] = {
	2, 7, 26, 59, 242, 568, 1792, 5649, 16807,
	104581, 296029, 836661, 3346853, 9465541, 53545221, 75724373
};

int DayNum[12] = {
	31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31
};

int MonthMin[13] = {
	MONTH_0, MONTH_1, MONTH_2, MONTH_3, MONTH_4, MONTH_5, MONTH_6, MONTH_7, MONTH_8, MONTH_9, MONTH_10, MONTH_11, MONTH_12
};

__int64 Pow95[CHAR95_LEN] = {
	1, 95, 9025, 857375, 81450625, 7737809375, 735091890625, 69833729609375, 6634204312890625
};

char *MonthName[12] = {
	"jan", "feb", "mar", "apr", "may", "jun", "jul", "aug", "sep", "oct", "nov", "dec"
};

char *TokName[TOK_NUM] = {
	"Name", "Type", "Length", "Prec", "Scale", "Distr", "StdDev", "Mean", "Min", "Max", "Key", "Unique",
	"DbGen", "Connect", "Table", "Col_Number", "Record_Number", "Dest_File", "Dest_Odbc", "Param_h", "Param_a"
};

SQLSMALLINT SqlType[TYPE_NUM] = {
	SQL_INTEGER, SQL_SMALLINT, SQL_TINYINT, SQL_REAL, SQL_FLOAT, SQL_DECIMAL, SQL_CHAR, SQL_VARCHAR, SQL_TYPE_TIMESTAMP, SQL_BIGINT
};

BYTE gColor[3 * COLOR_NUM] = {
	0x00,0x00,0x00, 0xA5,0x2A,0x00, 0x00,0x40,0x40, 0x00,0x55,0x00, 0x00,0x00,0x5E, 0x00,0x00,0x8B, 0x4B,0x00,0x82, 0x28,0x28,0x28,
	0x8B,0x00,0x00, 0xFF,0x68,0x20, 0x8B,0x8B,0x00, 0x00,0x93,0x00, 0x38,0x8E,0x8E, 0x00,0x00,0xFF, 0x7B,0x7B,0xC0, 0x66,0x66,0x66,
	0xFF,0x00,0x00, 0xFF,0xAD,0x5B, 0x32,0xCD,0x32, 0x3C,0xB3,0x71, 0x7F,0xFF,0xD4, 0x7D,0x9E,0xC0, 0x80,0x00,0x80, 0x7F,0x7F,0x7F,
	0xFF,0xC0,0xCB, 0xFF,0xD7,0x00, 0xFF,0xFF,0x00, 0x00,0xFF,0x00, 0x40,0xE0,0xD0, 0xC0,0xFF,0xFF, 0x48,0x00,0x48, 0xC0,0xC0,0xC0,
	0xFF,0xE4,0xE1, 0xD2,0xB4,0x8C, 0xFF,0xFF,0xE0, 0x98,0xFB,0x98, 0xAF,0xEE,0xEE, 0x68,0x83,0x8B, 0xE6,0xE6,0xFA, 0xFF,0xFF,0xFF
};

HANDLE ghInst;
HWND ghMain, ghConnect, ghTable, ghColNum, ghRowNum, ghDestFile, ghDestOdbc, ghLv, ghStatus, ghTip, ghTab, ghGraph, ghHelp;
DLGTEMPLATE *gTmpl;
DLGITEMTEMPLATE *gItemTmpl;
WNDPROC OldLvProc, OldColNumProc, OldInputCtrlProc;
SQLHANDLE ghEnv, ghDbc;
SQLHSTMT ghStmt;
char gConnect[XLARGEBUFF], gTable[SMALLBUFF];
int gLvHdHeight, gLviHeight, gLvCur, gLvCurSub, gfDestOdbc, gfDestFile, gfTab, gfUSADate = 1;
__int64 gRowNum;
VecColT	gCol;

void TrimStr(char *s)
{
	char *z;
	int  i;

	if (!s || !*s)
		return;
	z = s;
	while (*z && ((*z == ' ') || (*z == '\t')))
		z++;
	if (*z && s != z)
		memmove(s, z, strlen(z) + 1);
	i = strlen(s) - 1;
	while (i >= 0 && (s[i] == ' ' || s[i] == '\t')) {
		s[i] = '\0';
		i--;
	}
}

int PrintI64(char *buf, __int64 i)
{
	char a[I64_LEN + 1], *p, f = 0;

	if (i < 0) {
		i = -i;
		f = 1;
	}
	a[I64_LEN] = '\0';
	p = a + I64_LEN - 1;
	if (i) {
		while (i) {
			*p-- = (char) (i % 10 + '0');
			i /= 10;
		}
		if (f)
			*p = '-';
		else
			p++;
		memmove(buf, p, strlen(p) + 1);
	}
	else
		buf[0] = '0';
	return 1;
}

__int64 AtoI64(char *buf)
{
	char *p;
	__int64 ret = 0, ten = 1, d;
	int i;

	TrimStr(buf);
	if (!buf || !*buf || (i = strlen(buf)) > I64_LEN)
		return 0;
	p = buf + i - 1;
	while (p >= buf) {
		if (*p < '0' || *p > '9')
			break;
		d = (*p-- - '0') * ten;
		if (_I64_MAX - ret < d)
			break;
		ret += d;
		ten *= 10;
	}
	if (buf[0] == '-')
		return -ret;
	return ret;
}

void VecInit(VecT *v, int cur, int sz)
{
	v->sz = sz;
	v->cur = cur;
	if (!v->h && v->cur >= 0) {
		v->max = v->cur + 1;
		if ((v->h = (char *) malloc(v->max * sz)) == NULL)
			exit(-1);
	}
	if (v->cur >= 0)
		memset(v->h, 0, (v->cur + 1) * v->sz);
}

void VecPush(VecT *v, void *b)
{
	v->cur++;
	if (v->cur > v->max - 1) {
		v->max += 100;
		if ((v->h = (char *) realloc(v->h, v->max * v->sz)) == NULL)
			exit(-1);
		memset(v->h + v->sz * (v->max - 100), 0, 100 * v->sz);
	}
	if (b)
		memcpy(v->h + v->cur * v->sz, b, v->sz);
}

void VecDel(VecT *v)
{
	free(v->h);
	memset(v, 0, sizeof(VecT));
	v->cur = -1;
}

void DbGenOnSize(HWND hwnd, UINT state, int cx, int cy)
{
	RECT rect;

	GetWindowRect(ghMain, &rect);
	MoveWindow(ghMain, rect.left, rect.top, W_MAIN, H_MAIN, 1);
}

LPWORD Align(LPWORD p)
{
	ULONG           ul;

	ul = (ULONG) p;
	ul += 3;
	ul >>= 2;
	ul <<= 2;
	return (LPWORD) ul;
}

int CALLBACK SetSmallFont(HWND hwnd, LPARAM lParam)
{
	if (hwnd != ghLv)
		SendMessage(hwnd, WM_SETFONT, (WPARAM) lParam, 1L);
	return 1;
}

void ColSetName(VecColT *Col)
{
	int i = 0, j = 1;
	char buf[TINYBUFF];

	while (1) {
		sprintf(buf, "F%d", j);
		if (!Col->cur)
			break;
		for (i = 0; i < Col->cur; i++)
			if (!strcmp(Col->h[i].Name, buf))
				break;
		if (i < Col->cur)
			j++;
		else
			break;
	}
	strcpy(Col->h[i].Name, buf);
}

int MainInit(HWND hwnd)
{
	int i, cx[COL_NUM] = {
		56, 56, 50, 50, 50, 60, 74, 74, 100, 100, 40, 46																																								};
	int sb[4] = {
		340, 400, 528, -1																																								};
	HD_LAYOUT layoutHd;
	LV_COLUMN lvCol;
	RECT rect;
	WINDOWPOS wpos;

	CreateWindowEx(0, "static", TokName[TOK_CONNECT], WS_CHILD|WS_VISIBLE, 4, 10, 40, 18, hwnd, NULL, ghInst, NULL);
	CreateWindowEx(0, "static", TokName[TOK_TABLE], WS_CHILD|WS_VISIBLE, 4, 36, 90, 18, hwnd, NULL, ghInst, NULL);
	CreateWindowEx(0, "static", TokName[TOK_FLDNUM], WS_CHILD|WS_VISIBLE, 120, 36, 70, 18, hwnd, NULL, ghInst, NULL);
	CreateWindowEx(0, "static", TokName[TOK_ROWNUM], WS_CHILD|WS_VISIBLE, 234, 36, 78, 18, hwnd, NULL, ghInst, NULL);
	CreateWindowEx(0, "static", TokName[TOK_DESTFILE], WS_CHILD|WS_VISIBLE, 404, 36, 90, 18, hwnd, NULL, ghInst, NULL);
	CreateWindowEx(0, "static", TokName[TOK_DESTODBC], WS_CHILD|WS_VISIBLE, 470, 36, 90, 18, hwnd, NULL, ghInst, NULL);
	CreateWindowEx(WS_EX_STATICEDGE, "button", "?", WS_CHILD|WS_VISIBLE, 760, 7, 16, 16, hwnd, (HMENU) IDC_README, ghInst, NULL);
	ComboBox_AddString(ghConnect = CreateWindowEx(0, "ComboBox", CONNECT_STR, WS_CHILD|WS_VISIBLE|WS_TABSTOP|WS_GROUP|CBS_DROPDOWN|WS_VSCROLL, 46, 6, 492, 64, hwnd, NULL, ghInst, NULL), CONNECT_STR);
	ghTable = CreateWindowEx(WS_EX_STATICEDGE, "edit", TokName[TOK_DBGEN], WS_CHILD|WS_VISIBLE|WS_TABSTOP|WS_GROUP|ES_AUTOHSCROLL, 32, 34, 80, 16, hwnd, NULL, ghInst, NULL);
	ghColNum = CreateWindowEx(WS_EX_STATICEDGE, "edit", "1", WS_CHILD|WS_VISIBLE|WS_TABSTOP|WS_GROUP|ES_AUTOHSCROLL|ES_NUMBER, 186, 34, 40, 16, hwnd, NULL, ghInst, NULL);
	ghRowNum = CreateWindowEx(WS_EX_STATICEDGE, "edit", "1", WS_CHILD|WS_VISIBLE|WS_TABSTOP|WS_GROUP|ES_AUTOHSCROLL|ES_NUMBER, 314, 34, 80, 16, hwnd, NULL, ghInst, NULL);
	ghDestFile = CreateWindowEx(0, "button", "", WS_CHILD|WS_VISIBLE|BS_AUTOCHECKBOX|WS_TABSTOP|WS_GROUP, 450, 36, 16, 16, hwnd, NULL, ghInst, NULL);
	ghDestOdbc = CreateWindowEx(0, "button", "", WS_CHILD|WS_VISIBLE|BS_AUTOCHECKBOX|WS_TABSTOP|WS_GROUP, 525, 36, 16, 16, hwnd, NULL, ghInst, NULL);
	CreateWindowEx(0, "button", "Generate", WS_CHILD|WS_VISIBLE|WS_TABSTOP|WS_GROUP, 660, 7, 90, 24, hwnd, (HMENU) IDC_OK, ghInst, NULL);
	ghLv = CreateWindowEx(WS_EX_STATICEDGE, WC_LISTVIEW, "", WS_CHILD|WS_VISIBLE|WS_TABSTOP|WS_GROUP|WS_VSCROLL|LVS_REPORT|LVS_OWNERDATA|LVS_ICON, X_SHT, Y_SHT, W_SHT, H_SHT, hwnd, NULL, ghInst, NULL);
	ghStatus = CreateStatusWindow(WS_CHILD|WS_VISIBLE, "", hwnd, 0);
	ghTip = CreateWindowEx(0, "static", "", WS_VISIBLE|WS_CHILD, 0, 0, 0, 0, hwnd, NULL, ghInst, NULL);
	ghTab = CreateWindowEx(0, "tab", "", WS_CHILD|WS_VISIBLE, 2, H_MAIN - 64, 124, 14, hwnd, NULL, ghInst, NULL);
	memset(&lvCol, 0, sizeof(LV_COLUMN));
	lvCol.mask = LVCF_FMT|LVCF_WIDTH|LVCF_TEXT|LVCF_SUBITEM;
	lvCol.fmt = LVCFMT_LEFT;
	for (i = 0; i < COL_NUM; i++) {
		lvCol.iSubItem = i;
		lvCol.cx = cx[i];
		lvCol.pszText = TokName[i];
		ListView_InsertColumn(ghLv, i, &lvCol);
	}
	gTmpl = (DLGTEMPLATE *) malloc(16 + sizeof(DLGTEMPLATE) + sizeof(DLGITEMTEMPLATE));
	memset(gTmpl, 0, 16 + sizeof(DLGTEMPLATE) + sizeof(DLGITEMTEMPLATE));
	gTmpl->style = WS_VISIBLE|WS_CHILD|WS_POPUP|WS_BORDER|DS_SETFOREGROUND;
	gTmpl->cdit = 1;
	gItemTmpl = (DLGITEMTEMPLATE *) ((unsigned char *) (gTmpl + 1) + 6);
	gItemTmpl = (DLGITEMTEMPLATE *) Align((LPWORD) gItemTmpl);
	*(WORD *) ((unsigned char *) (gItemTmpl + 1)) = 0xFFFF;
	layoutHd.prc = &rect;
	layoutHd.pwpos = &wpos;
	rect.left = rect.top = 0;
	rect.right = W_SHT;
	rect.bottom = H_SHT;
	SendMessage((HWND) SendMessage(ghLv, (0x1000 + 31), 0, 0L), HDM_LAYOUT, 0, (LPARAM) (HD_LAYOUT FAR *) & layoutHd);
	gLvHdHeight = (short) wpos.cy;
	SendMessage(ghLv, LVM_SETEXTENDEDLISTVIEWSTYLE, 0, LVS_EX_GRIDLINES|LVS_EX_ONECLICKACTIVATE|LVS_EX_FULLROWSELECT);
	SendMessage(ghStatus, SB_SETPARTS, (WPARAM) 4, (LPARAM) sb);
	SendMessage(ghStatus, SB_SETTEXT, (WPARAM) 1, (LPARAM) "v beta");
	SendMessage(ghStatus, SB_SETTEXT, (WPARAM) 2, (LPARAM) " (c) 1999-2001, J.Gray, Barkhatov A.");
	SendMessage(ghStatus, SB_SETTEXT, (WPARAM) 3, (LPARAM) "  http://research.Microsoft.com/~Gray/DbGen");
	EnumChildWindows(hwnd, SetSmallFont, (LPARAM) GetStockObject(DEFAULT_GUI_FONT));
	return 1;
}

void LvGrow(int NewExt)
{
	int i, OldExt;
	LVITEM lvItem;
	char buf[TINYBUFF];

	OldExt = ListView_GetItemCount(ghLv);
	if (OldExt != NewExt) {
		for (i = NewExt; i < OldExt; i++)
			ListView_DeleteItem(ghLv, NewExt);
		sprintf(buf, "%d", NewExt);
		Edit_SetText(ghColNum, buf);
		memset(&lvItem, 0, sizeof(LVITEM));
		lvItem.mask = LVIF_TEXT;
		lvItem.pszText = LPSTR_TEXTCALLBACK;
		for (i = OldExt; i < NewExt; i++) {
			lvItem.iItem = i;
			ListView_InsertItem(ghLv, &lvItem);
		}
		ListView_SetItemCount(ghLv, NewExt);
		ListView_SetItemState(ghLv, ListView_GetNextItem(ghLv, -1, LVNI_SELECTED), 0, 0xFFFF);
		ListView_SetItemState(ghLv, NewExt - 1, LVIS_FOCUSED|LVIS_SELECTED, 0xF);
	}
}

int CALLBACK ColNumProc(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	int n;
	char buf[TINYBUFF];

	switch (uMsg) {
	case WM_KILLFOCUS:
		GetWindowText(hwnd, buf, TINYBUFF);
		n = max(1, atoi(buf));
		while (gCol.cur + 1 < n) {
			VecPush((VecT *) &gCol, &DefCol[TYPE_INT]);
			ColSetName(&gCol);
		}
		gCol.cur = n - 1;
		LvGrow(n);
		break;
	}
	return CallWindowProc(OldColNumProc, hwnd, uMsg, wParam, lParam);
}

int IntToStr(__int64 iVal, char *buf, int length)
{
	int i, n = CHAR95_LEN - 1;

	if (!iVal) {
		buf[0] = ' ';
		return 0;
	}
	if (iVal > PRIM_MAX_1)
		iVal = PRIM_MAX_1;
	while (n > 0 && iVal < Pow95[n])
		n--;
	for (i = 0; i <= n && i < length; i++) {
		buf[i] = (char) (iVal / Pow95[n - i]);
		iVal -= buf[i] * Pow95[n - i];
		buf[i] += ' ';
	}
	return 1;
}

__int64 StrToInt(char *buf)
{
	int	i, n;
	__int64	ret = 0;

	if (!buf || !*buf)
		return 0;
	n = strlen(buf);
	if (n > CHAR95_LEN)
		return PRIM_MAX_1;
	for (i = 0; i < n; i++)
		ret += (buf[n - 1 - i] - ' ') * Pow95[i];
	return ret;
}

__int64 ValidDate(char *buf, TIMESTAMP_STRUCT *ts)
{
	short i, hour, mint;
	char *p, APM[3], buf0[LARGEBUFF];

/* for Day-Mon-Year use gfUSADate=0  */

	TrimStr(buf);
	if (!buf || !*buf)
		return 0;
	memset(ts, 0, sizeof(TIMESTAMP_STRUCT));
	memset(buf0, 0, LARGEBUFF);
	strcpy(buf0, buf);
	if ((p = strtok(buf0, " -./")) == NULL)
		return 0;
	if (gfUSADate)
		goto lmon;
	ts->day = atoi(p);
lday:
	if ((p = strtok(NULL, " -./")) == NULL)
		return 0;
	if (gfUSADate && ts->month)
		goto lend;
lmon:
	if (isalpha(*p)) {
		for (i = 0; i < 12; i++)
			if (!strncmp(p, MonthName[i], 3))
				break;
		ts->month = i + 1;
	}
	else
		ts->month = atoi(p);
	if (gfUSADate)
		goto lday;
lend:
	if (ts->day < 1 || ts->day > DayNum[ts->month - 1])
		return 0;
	if ((p = strtok(NULL, " ")) == NULL)
		return 0;
	ts->year = (short) atoi(p);
	if (ts->year > 9999)
		return 0;
	if (ts->year < 100)
		ts->year += 1900;
	if (ts->year < 1753)
		return 0;
	if ((p = strtok(NULL, " ")) == NULL)
		return 1;
	memset(APM, 0, 3);
	sscanf(p, "%2d:%2d%2c", &hour, &mint, APM);
	if (hour < 0 || hour > 23)
		return 0;
	ts->hour = hour;
	if (mint < 0 || mint > 59)
		return 0;
	ts->minute = mint;
	if ((APM[0] == 'p' || APM[0] == 'P') && ts->hour < 12)
		ts->hour += 12;
	return 1;
}

int IntToTime(__int64 iVal, char *TextBuf, char *BinBuf)
{
	TIMESTAMP_STRUCT ts;
	int i, dYear, dMonth, dDay, dHour;

	memset(&ts, 0, sizeof(TIMESTAMP_STRUCT));
	dYear = (short) (iVal / MONTH_12);
	ts.year = (short) (1753 + dYear);
	dMonth = (int) (iVal - dYear * MONTH_12);
	for (i = 0; i < 12; i++) {
		if (dMonth < MonthMin[i])
			break;
	}
	ts.month = (unsigned short) i;
	dDay = dMonth - MonthMin[i - 1];
	ts.day = (short) (dDay / (24 * 60) + 1);
	dHour = dDay - (ts.day - 1) * (24 * 60);
	ts.hour = (short) (dHour / 60);
	ts.minute = (short) (dHour - ts.hour * 60);
	if (TextBuf) {
		if (gfUSADate)
			sprintf(TextBuf, "%02d/%02d/%04d %02d:%02d", ts.month, ts.day, ts.year, ts.hour, ts.minute);
		else
			sprintf(TextBuf, "%02d/%02d/%04d %02d:%02d", ts.day, ts.month, ts.year, ts.hour, ts.minute);
	}
	if (BinBuf)
		memcpy(BinBuf, &ts, sizeof(TIMESTAMP_STRUCT));
	return 1;
}

__int64 TimeToInt(char *buf)
{
	TIMESTAMP_STRUCT ts;

	if (!buf || !*buf)
		return 0;
	if (!ValidDate(buf, &ts))
		return 0;
	return ((__int64) MONTH_12 * (ts.year - 1753) + MonthMin[ts.month - 1] + 24 * 60 * (ts.day - 1) + 60 * ts.hour + ts.minute);
}

int ColEnable(int Type, int Distr, int i)
{
	if ((Type == TYPE_CHAR || Type == TYPE_VARCHAR) && (i == COL_PREC || i == COL_SCALE)
			|| Type == TYPE_DECIMAL && i == COL_LENGTH
			|| (Type == TYPE_TINYINT || Type == TYPE_SMALLINT || Type == TYPE_INT || Type == TYPE_BIGINT
			|| Type == TYPE_DATETIME || Type == TYPE_REAL || Type == TYPE_FLOAT) && (i == COL_PREC || i == COL_SCALE || i == COL_LENGTH)
			|| (Distr == DISTR_RAND || Distr == DISTR_ORD) && (i == COL_MEAN || i == COL_STDDEV)
			|| (Distr == DISTR_EXPN || Distr == DISTR_POIS || Distr == DISTR_SSIM || Distr == DISTR_ZIPF) && i == COL_MEAN)
		return 0;
	return 1;
}

void ParToStr(ColT *Col, int i, char *buf, int sz)
{
	if (COL_DBL(Col->Type))
		sprintf(buf, "%g", Col->DPar[i]);
	else if (Col->Type == TYPE_TINYINT || Col->Type == TYPE_SMALLINT || Col->Type == TYPE_INT || Col->Type == TYPE_BIGINT)
		PrintI64(buf, Col->LPar[i]);
	else if(Col->Type == TYPE_CHAR || Col->Type == TYPE_VARCHAR)
		IntToStr(Col->LPar[i], buf, sz);
	else if(Col->Type == TYPE_DATETIME)
		IntToTime(Col->LPar[i], buf, NULL);
}

void ColToStr(ColT *Col, int iSubItem, char *buf, int bufSz)
{
	memset(buf, 0, bufSz);
	if (!ColEnable(Col->Type, Col->Distr, iSubItem)) {
		strcpy(buf, "...");
		return;
	}
	switch (iSubItem) {
	case COL_NAME:
		strcpy(buf, Col->Name);
		break;
	case COL_TYPE:
		strcpy(buf, TypeName[Col->Type]);
		break;
	case COL_LENGTH:
		sprintf(buf, "%d", Col->Length);
		break;
	case COL_PREC:
		sprintf(buf, "%d", Col->Prec);
		break;
	case COL_SCALE:
		sprintf(buf, "%d", Col->Scale);
		break;
	case COL_DISTR:
		strcpy(buf, DistrName[Col->Distr]);
		break;
	case COL_STDDEV:
		sprintf(buf, "%g", Col->Dev);
		break;
	case COL_MEAN:
	case COL_MIN:
	case COL_MAX:
		ParToStr(Col, iSubItem - COL_MEAN, buf, bufSz);
		break;
	case COL_KEY:
		if (Col->fPrimKey)
			strcpy(buf, "On");
		break;
	case COL_UNIQUE:
		if (Col->fUnique)
			strcpy(buf, "On");
		break;
	}
}

void StrToPar(ColT *Col, int i, char *buf)
{
	if (COL_DBL(Col->Type)) {
		Col->DPar[i] = atof(buf);
		Col->LPar[i] = (__int64) Col->DPar[i];
	}
	else if (Col->Type == TYPE_TINYINT || Col->Type == TYPE_SMALLINT || Col->Type == TYPE_INT) {
		Col->LPar[i] = atoi(buf);
		Col->DPar[i] = (double) Col->LPar[i];
	}
	else if (Col->Type == TYPE_BIGINT) {
		Col->LPar[i] = AtoI64(buf);
		Col->DPar[i] = (double) Col->LPar[i];
	}
	else if (Col->Type == TYPE_CHAR || Col->Type == TYPE_VARCHAR) {
		Col->LPar[i] = StrToInt(buf);
		Col->DPar[i] = (double) Col->LPar[i];
	}
	else if (Col->Type == TYPE_DATETIME) {
		Col->LPar[i] = TimeToInt(buf);
		Col->DPar[i] = (double) Col->LPar[i];
	}
}

void ColFromStr(char *buf, ColT *Col, int iSubItem)
{
	int i;
	char buf0[SMALLBUFF];

	if (!ColEnable(Col->Type, Col->Distr, iSubItem))
		return;
	switch (iSubItem) {
	case COL_NAME:
		strcpy(Col->Name, buf);
		break;
	case COL_TYPE:
		i = *(int *) buf;
		if (i != Col->Type) {
			strcpy(buf0, Col->Name);
			memcpy(Col, &DefCol[i], sizeof(ColT));
			strcpy(Col->Name, buf0);
			Col->Type = i;
		}
		break;
	case COL_LENGTH:
		Col->Length = atoi(buf);
		break;
	case COL_PREC:
		Col->Prec = atoi(buf);
		if (Col->Type == TYPE_DECIMAL) {
			Col->Length = Col->Prec + 2;
			Col->Scale = (Col->Scale <= Col->Prec ? Col->Scale : Col->Prec);
		}
		break;
	case COL_SCALE:
		Col->Scale = atoi(buf);
		if (Col->Type == TYPE_DECIMAL)
			Col->Scale = (Col->Scale <= Col->Prec ? Col->Scale : Col->Prec);
		break;
	case COL_DISTR:
		Col->Distr = *(int *) buf;
		break;
	case COL_STDDEV:
		Col->Dev = atof(buf);
		break;
	case COL_MEAN:
	case COL_MIN:
	case COL_MAX:
		StrToPar(Col, iSubItem - COL_MEAN, buf);
		break;
	case COL_KEY:
		if (!Col->fPrimKey)
			for (i = 0; i < gCol.cur + 1; i++)
				gCol.h[i].fPrimKey = 0;
		Col->fUnique = 0;
		Col->fPrimKey = !(*(int *) buf);
		break;
	case COL_UNIQUE:
		Col->fUnique = !(*(int *) buf);
		Col->fPrimKey = 0;
		break;
	}
}

void LvGetInput(HWND hWndCtl, int iItem, int iSubItem)
{
	char buf[SMALLBUFF];

	memset(buf, 0, SMALLBUFF);
	if (iSubItem == COL_TYPE || iSubItem == COL_DISTR)
		*(int *) buf = ComboBox_GetCurSel(hWndCtl);
	else
		Edit_GetText(hWndCtl, buf, SMALLBUFF);
	ColFromStr(buf, &gCol.h[iItem], iSubItem);
}

int CALLBACK InputCtrlProc(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	switch (uMsg) {
	case WM_GETDLGCODE:
		return DLGC_WANTALLKEYS;
	case WM_KEYDOWN:
		switch (wParam) {
		case VK_RETURN:
			LvGetInput(hwnd, gLvCur, gLvCurSub);
			DestroyWindow(GetParent(hwnd));
			break;
		case VK_ESCAPE:
			DestroyWindow(GetParent(hwnd));
			break;
		case VK_TAB:
			if (GetKeyState(VK_SHIFT) & 0x8000)
				SendMessage(ghMain, WM_COMMAND, IDC_LEFT, 0);
			else
				SendMessage(ghMain, WM_COMMAND, IDC_RIGHT, 0);
			break;
		}
		break;
	case WM_KILLFOCUS:
		LvGetInput(hwnd, gLvCur, gLvCurSub);
		MoveWindow(ghTip, 0, 0, 0, 0, 1);
		InvalidateRect(ghTip, NULL, 1);
		UpdateWindow(ghTip);
		DestroyWindow(GetParent(hwnd));
		break;
	case WM_SETFOCUS:
		Edit_SetSel(hwnd, 0, -1);
		break;
	}
	return CallWindowProc(OldInputCtrlProc, hwnd, uMsg, wParam, lParam);
}

int CALLBACK InputDlgProc(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	int i;
	char buf[SMALLBUFF];
	RECT rc;
	POINT pt;
	HWND hwndFocus = (HWND) wParam;

	switch (uMsg) {
	case WM_INITDIALOG:
		SendMessage(hwndFocus, WM_SETFONT, (WPARAM) GetStockObject(DEFAULT_GUI_FONT), 1L);
		if (gLvCurSub == COL_TYPE) {
			for (i = 0; i < TYPE_NUM; i++)
				ComboBox_AddString(hwndFocus, TypeName[i]);
			ComboBox_SetCurSel(hwndFocus, gCol.h[gLvCur].Type);
		}
		else if (gLvCurSub == COL_DISTR) {
			for (i = 0; i < DISTR_NUM; i++)
				ComboBox_AddString(hwndFocus, DistrName[i]);
			ComboBox_SetCurSel(hwndFocus, gCol.h[gLvCur].Distr);
		}
		else if (gLvCurSub != COL_KEY && gLvCurSub != COL_UNIQUE) {
			ColToStr(&gCol.h[gLvCur], gLvCurSub, buf, SMALLBUFF);
			Edit_SetText(hwndFocus, buf);
		}
		OldInputCtrlProc = (WNDPROC) SetWindowLong(hwndFocus, GWL_WNDPROC, (long) InputCtrlProc);
		if (gLvCurSub == COL_STDDEV && (gCol.h[gLvCur].Distr == DISTR_SSIM || gCol.h[gLvCur].Distr == DISTR_ZIPF)) {
			GetWindowRect(hwnd, &rc);
			pt.x = rc.left;
			pt.y = rc.bottom + 1;
			ScreenToClient(ghMain, &pt);
			MoveWindow(ghTip, pt.x, pt.y, 44, 12, 1);
			SetWindowText(ghTip, TokName[TOK_H + gCol.h[gLvCur].Distr - DISTR_SSIM]);
			SetForegroundWindow(ghTip);
			ShowWindow(ghTip, SW_SHOW);
		}
		return 1;
	case WM_COMMAND:
		if (HIWORD(wParam) == CBN_CLOSEUP) {
			LvGetInput((HWND) lParam, gLvCur, gLvCurSub);
			DestroyWindow(hwnd);
		}
		break;
	}
	return 0;
}

void TmplSet(int iItem, int iSubItem)
{
	int i, u;

	u = GetDialogBaseUnits();
	gTmpl->x = X_SHT;
	for (i = 0; i < iSubItem; i++)
		gTmpl->x = (short) (gTmpl->x + ListView_GetColumnWidth(ghLv, i));
	gTmpl->x = (short) (1 + gTmpl->x * 4 / LOWORD(u));
	gTmpl->cx = (short) (ListView_GetColumnWidth(ghLv, iSubItem) * 4 / LOWORD(u) - 1);
	gTmpl->y = (short) (2 + (Y_SHT + gLvHdHeight + (iItem - ListView_GetTopIndex(ghLv)) * gLviHeight) * 8 / HIWORD(u));
	gTmpl->cy = (short) (gLviHeight * 8 / HIWORD(u) - 1);
	gItemTmpl->cx = (short) (gTmpl->cx + 4);
	gItemTmpl->x = -2;
	gItemTmpl->y = -3;
	if (iSubItem == COL_TYPE || iSubItem == COL_DISTR) {
		gItemTmpl->cy = 50;
		*(WORD *) ((unsigned char *) (gItemTmpl + 1) + 2) = 0x0085;
		gItemTmpl->style = WS_VISIBLE|WS_CHILD|WS_OVERLAPPED|CBS_DROPDOWNLIST|WS_VSCROLL;
	}
	else {
		gItemTmpl->cy = (short) (gTmpl->cy + 4);
		*(WORD *) ((unsigned char *) (gItemTmpl + 1) + 2) = 0x0081;
		gItemTmpl->style = WS_VISIBLE|WS_CHILD|WS_OVERLAPPED|ES_AUTOHSCROLL|WS_BORDER;
		if (iSubItem == COL_LENGTH || iSubItem == COL_PREC || iSubItem == COL_SCALE)
			gItemTmpl->style |= ES_NUMBER;
	}
	ListView_SetItemState(ghLv, ListView_GetNextItem(ghLv, -1, LVNI_SELECTED), 0, 0xFFFF);
	ListView_SetItemState(ghLv, iItem, LVIS_FOCUSED|LVIS_SELECTED, 0xF);
}

int CALLBACK LvProc(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	switch (uMsg) {
	case WM_KEYDOWN:
		if (wParam == VK_RETURN) {
			gLvCur = ListView_GetNextItem(ghLv, -1, LVNI_SELECTED);
			gLvCurSub = 0;
			TmplSet(gLvCur, gLvCurSub);
			CreateDialogIndirect(ghInst, gTmpl, ghMain, InputDlgProc);
		}
		else if (wParam == VK_TAB)
			SetFocus(GetNextWindow(hwnd, GW_HWNDNEXT));
		break;
	case WM_GETDLGCODE:
		return DLGC_WANTALLKEYS;
	}
	return CallWindowProc(OldLvProc, hwnd, uMsg, wParam, lParam);
}

int ConfInit(char *name)
{
	FILE *fh;
	int  i, j, f = 0;
	char *p, *p1, buf[XLARGEBUFF], buf0[XLARGEBUFF];

	sprintf(gConnect, CONNECT_STR);
	strcpy(gTable, TokName[TOK_DBGEN]);
	gRowNum = 1;
	VecInit((VecT *) &gCol, -1, sizeof(ColT));
	VecPush((VecT *) &gCol, &DefCol[TYPE_INT]);
	gCol.h[0].Distr = DISTR_ORD;
	ColSetName(&gCol);
	if ((fh = fopen(name, "r")) == NULL) {
		gfDestFile = 1;
		gfDestOdbc = 1;
		for (i = 0; i < TYPE_NUM - 1; i++) {
			VecPush((VecT *) &gCol, &DefCol[i]);
			ColSetName(&gCol);
		}
		return 0;
	}
	while (fgets(buf, XLARGEBUFF, fh)) {
		p = strtok(buf, "\r\n");
		while (p) {
			p = strtok(p, "=");
			strcpy(buf0, p);
			TrimStr(buf0);
			for (i = 0; i < TOK_NUM; i++)
				if (!strcmp(TokName[i], buf0))
					break;
			if (i == TOK_NUM)
				break;
			p = strtok(NULL, "");
			if (!p)
				break;
			p1 = p;
			while (*p1) {
				for (j = 0; j < TOK_NUM; j++)
					if (!strncmp(p1, TokName[j], strlen(TokName[j])))
						break;
				if (j < TOK_NUM)
					break;
				p1++;
			}
			if (*p1)
				*(p1 - 1) = 0;
			strcpy(buf0, p);
			TrimStr(buf0);
			switch (i) {
			case TOK_CONNECT:
				if (strlen(buf0) < LARGEBUFF)
					strcpy(gConnect, buf0);
				break;
			case TOK_TABLE:
				if (strlen(buf0) < SMALLBUFF)
					strcpy(gTable, buf0);
				break;
			case TOK_ROWNUM:
				gRowNum = AtoI64(buf0);
				break;
			case TOK_DESTFILE:
				gfDestFile = (*(short *) buf0 == *(short *) "On") ? 1 : 0;
				break;
			case TOK_DESTODBC:
				gfDestOdbc = (*(short *) buf0 == *(short *) "On") ? 1 : 0;
				break;
			case COL_NAME:
				if (!f)
					f = 1;
				else
					VecPush((VecT *) &gCol, &DefCol[TYPE_INT]);
				if (strlen(buf0) < SMALLBUFF)
					strcpy(gCol.h[gCol.cur].Name, buf0);
				break;
			case COL_TYPE:
				for (j = 0; j < TYPE_NUM; j++)
					if (!strcmp(buf0, TypeName[j]))
						break;
				if (j < TYPE_NUM)
					gCol.h[gCol.cur].Type = j;
				break;
			case COL_DISTR:
				for (j = 0; j < DISTR_NUM; j++)
					if (!strcmp(buf0, DistrName[j]))
						break;
				if (j < DISTR_NUM)
					gCol.h[gCol.cur].Distr = j;
				break;
			case COL_LENGTH:
			case COL_PREC:
			case COL_SCALE:
			case COL_STDDEV:
			case COL_MEAN:
			case COL_MIN:
			case COL_MAX:
			case TOK_A:
			case TOK_H:
				i = i == TOK_A || i == TOK_H ? COL_STDDEV : i;
				ColFromStr(buf0, &gCol.h[gCol.cur], i);
				break;
			case COL_KEY:
			case COL_UNIQUE:
				if (*(short *) buf0 == *(short *) "On")
					gCol.h[gCol.cur].fUnique = 1;
				break;
			}
			if (*p1)
				p = p1;
		}
	}
	fclose(fh);
	return 1;
}

int DbGenOnCreate(HWND hwnd, CREATESTRUCT *lpCreateStruct)
{
	RECT rect;
	INITCOMMONCONTROLSEX icc;
	char buf[LARGEBUFF];

	icc.dwSize = sizeof(INITCOMMONCONTROLSEX);
	icc.dwICC = ICC_USEREX_CLASSES|ICC_LISTVIEW_CLASSES|ICC_BAR_CLASSES;
	InitCommonControlsEx(&icc);
	if (!MainInit(hwnd))
		return 0;
	ConfInit(DBGENINI_STR);
	OldColNumProc = (WNDPROC) SetWindowLong(ghColNum, GWL_WNDPROC, (long) ColNumProc);
	OldLvProc = (WNDPROC) SetWindowLong(ghLv, GWL_WNDPROC, (long) LvProc);
	SetWindowText(ghConnect, gConnect);
	Button_SetCheck(ghDestOdbc, gfDestOdbc);
	Button_SetCheck(ghDestFile, gfDestFile);
	Edit_SetText(ghTable, gTable);
	PrintI64(buf, gRowNum);
	Edit_SetText(ghRowNum, buf);
	LvGrow(gCol.cur + 1);
	ListView_GetItemRect(ghLv, 0, &rect, LVIR_LABEL);
	gLviHeight = (short) (rect.bottom - rect.top);
	return 1;
}

int DbGenOnNotify(HWND hwnd, int idFrom, NMHDR * pnmhdr)
{
	int x, i, j;
	POINT pt;
	LV_DISPINFO *di;
	char buf[SMALLBUFF];

	memset(buf, 0, SMALLBUFF);
	switch (pnmhdr->code) {
	case LVN_GETDISPINFO:
		di = (LV_DISPINFO *) pnmhdr;
		ColToStr(&gCol.h[di->item.iItem], di->item.iSubItem, buf, SMALLBUFF);
		di->item.pszText = buf;
		return 0;
	case NM_CLICK:
		if (pnmhdr->hwndFrom == ghLv) {
			GetCursorPos(&pt);
			ScreenToClient(hwnd, &pt);
			i = (pt.y - Y_SHT - gLvHdHeight) / gLviHeight + ListView_GetTopIndex(ghLv);
			if (i < 0 || i >= gCol.cur + 1)
				return 0;
			x = X_SHT;
			for (j = 0; j < COL_NUM; j++) {
				x += ListView_GetColumnWidth(ghLv, j);
				if (x >= pt.x)
					break;
			}
			if (j == COL_NUM)
				return 0;
			gLvCur = i;
			gLvCurSub = j;
			if (j < COL_NUM - 2) {
				if (!ColEnable(gCol.h[gLvCur].Type, gCol.h[gLvCur].Distr, gLvCurSub))
					return 0;
				TmplSet(gLvCur, gLvCurSub);
				CreateDialogIndirect(ghInst, gTmpl, ghMain, InputDlgProc);
			}
			else {
				*(int *) buf = (gLvCurSub == COL_KEY ? gCol.h[gLvCur].fPrimKey : gCol.h[gLvCur].fUnique);
				ColFromStr(buf, &gCol.h[gLvCur], gLvCurSub);
				InvalidateRect(ghLv, NULL, 1);
				UpdateWindow(ghLv);
			}
		}
		break;
	}
	return 0;
}

int ConfSave(char *name)
{
	char buf[SMALLBUFF];
	FILE *fh;
	int  i, j, k;

	gConnect[0] = '\0';
	ComboBox_GetText(ghConnect, gConnect, XLARGEBUFF);
	gfDestOdbc = Button_GetCheck(ghDestOdbc);
	gfDestFile = Button_GetCheck(ghDestFile);
	gTable[0] = '\0';
	Edit_GetText(ghTable, gTable, SMALLBUFF);
	buf[0] = '\0';
	Edit_GetText(ghRowNum, buf, SMALLBUFF);
	gRowNum = AtoI64(buf);
	if ((fh = fopen(name, "w")) == NULL)
		return 0;
	fprintf(fh, "%s\n", TokName[TOK_DBGEN]);
	if (gConnect[0])
		fprintf(fh, "%s=%s\n", TokName[TOK_CONNECT], gConnect);
	if (gTable[0])
		fprintf(fh, "%s=%s\n", TokName[TOK_TABLE], gTable);
	fprintf(fh, "%s=%s\n", TokName[TOK_ROWNUM], buf);
	if (gfDestOdbc)
		fprintf(fh, "%s=On\n", TokName[TOK_DESTODBC]);
	if (gfDestFile)
		fprintf(fh, "%s=On\n", TokName[TOK_DESTFILE]);
	for (i = 0; i < gCol.cur + 1; i++) {
		for (j = 0; j < COL_NUM; j++) {
			if (ColEnable(gCol.h[i].Type, gCol.h[i].Distr, j)) {
				ColToStr(&gCol.h[i], j, buf, SMALLBUFF);
				k = (j == COL_STDDEV && (gCol.h[i].Distr == DISTR_SSIM || gCol.h[i].Distr == DISTR_ZIPF))
					? (gCol.h[i].Distr == DISTR_SSIM ? TOK_H : TOK_A) :
					j;
				if (buf[0])
					fprintf(fh, "%s=%s ", TokName[k], buf);
			}
		}
		fprintf(fh, "\n");
	}
	if (fclose(fh) == EOF)
		return 0;
	return 1;
}

void DistrInit(ColT *Col)
{
	int i = 0;

	if (!COL_DBL(Col->Type) && Col->Distr == DISTR_RAND) {
		i = 0;
		while (i < GEN_DEC && (Col->LPar[PAR_MAX] - Col->LPar[PAR_MIN]) > Prime[i] - 1)
			i++;
		i = min(i, GEN_DEC - 1);
	}
	else if (Col->Distr == DISTR_ZIPF)
		i = 3;
	else
		i = 5;
	Col->Rand.Prim = Prime[i];
	Col->Rand.Root = Root[i];
	Col->Rand.iRand = Col->Rand.Root;
	return;
}

__int64 AddMod(__int64 i, __int64 j, __int64 m)
{
	__int64 k;

	if (i > I62_1 - j) {
		if (i > j) {
			k = i - m;
			k += j;
		}
		else {
			k = j - m;
			k += i;
		}
	}
	else
		k = i + j;
	if (k > m)
		k -= m;
	return k;
}

__int64 MulMod(__int64 x, __int64 y, __int64 m)
{
	__int64 i, j;
	int x0, y0, x1, y1;

	if (x > I62_1 / y) {
		x0 = (int) (x & I31_1);
		x1 = (int) ((unsigned __int64) x >> 31);
		y0 = (int) (y & I31_1);
		y1 = (int) ((unsigned __int64) y >> 31);
		i = x1 * y1 % m;
		i = MulMod(i, I31, m);
		j = x0 * y1 + x1 * y0;
		i = AddMod(i, j, m);
		i = MulMod(i, I31, m);
		j = x0 * y0;
		i = AddMod(i, j, m);
		return i;
	}
	else
		return x * y % m;
}

double RanDbl(RandT *Rand)
{
	double r;

	r = (double) Rand->iRand / (double) (Rand->Prim - 1);
	Rand->iRand = MulMod(Rand->iRand, Rand->Root, Rand->Prim);
	return r;
}

double Gauss(ColT *Col)
{
	int i;
	double ans = 0;

	for (i = 0; i < 12; i++)
		ans += RanDbl(&Col->Rand) - 0.5;
	return Col->DPar[PAR_MEAN] + Col->Dev * ans;
}

double ExpNeg(ColT *Col)
{
	return Col->DPar[PAR_MIN] - Col->Dev * log(RanDbl(&Col->Rand));
}

double Poisson(ColT *Col)
{
	int i = -1;
	double c, p = 1;

	c = pow(2.718282, -Col->Dev);
	if (IS_DBL_ZERO(c))
		return 0;
	while (p >= c) {
		p *= RanDbl(&Col->Rand);
		i++;
	}
	return Col->DPar[PAR_MIN] + (double) i / Col->Rand.Prim * (Col->DPar[PAR_MAX] - Col->DPar[PAR_MIN]);
}

double SelfSim(ColT *Col)
{
	double r;

	r = RanDbl(&Col->Rand);
	return Col->DPar[PAR_MIN] + (Col->DPar[PAR_MAX] - Col->DPar[PAR_MIN]) * r * pow(r, log(Col->Dev) / log(1 - Col->Dev));
}

double zeta(int n, double theta)
{
	int i;
	double ret = 0.0;

	for (i = 1; i < n; i++)
		ret += pow(i, -theta);
	return ret;
}

double Zipf(ColT *Col)
{
	int n;
	double alpha, zetan, eta, uz, u;

	n = (int) Col->Rand.iRand;
	u = RanDbl(&Col->Rand);
	alpha = 1 / (1 - Col->Dev);
	zetan = zeta(n, Col->Dev);
	uz = u * zetan;
	if (uz < 1)
		return Col->DPar[PAR_MIN];
	if (uz < 1 + pow(0.5, Col->Dev))
		return  1.0 / Col->Rand.Prim * (Col->DPar[PAR_MAX] - Col->DPar[PAR_MIN]);
	eta = (1 - pow(2.0 / n, 1 - Col->Dev)) * (1 - zeta(2, Col->Dev) / zetan);
	return Col->DPar[PAR_MIN] + (1 + n * pow(eta * u - eta + 1, alpha)) / Col->Rand.Prim * (Col->DPar[PAR_MAX] - Col->DPar[PAR_MIN]);
}

void FillBuf(ColT *Col, char *Buf, __int64 k, double d)
{
	int n, rAlign;
	char buf[XLARGEBUFF];

	switch (Col->Type) {
	case TYPE_INT:
	case TYPE_SMALLINT:
	case TYPE_TINYINT:
	case TYPE_BIGINT:
		if (Col->Distr != DISTR_RAND && Col->Distr != DISTR_ORD)
			k = (__int64) d;
		memcpy(Buf, &k, TypeSz[Col->Type]);
		break;
	case TYPE_CHAR:
	case TYPE_VARCHAR:
		if (Col->Distr != DISTR_RAND && Col->Distr != DISTR_ORD)
			k = (__int64) d;
		IntToStr(k, Buf, Col->Length);
		if (Col->Type == TYPE_CHAR) {
			rAlign = min(CHAR95_LEN, Col->Length);
			n = (int) strlen(Buf);
			memmove(Buf + rAlign - n, Buf, n);
			memset(Buf, ' ', rAlign - n);
		}
		n = (int) strlen(Buf);
		if (Col->Length > n)
			memset(Buf + n, ' ', Col->Length - n);
		break;
	case TYPE_DECIMAL:
		sprintf(buf, "%*.*f", Col->Length, Col->Scale, d);
		buf[Col->Length] = 0;
		strcpy(Buf, buf);
		break;
	case TYPE_FLOAT:
		*(double *) Buf = d;
		break;
	case TYPE_REAL:
		*(float *) Buf = (float) d;
		break;
	case TYPE_DATETIME:
		if (Col->Distr != DISTR_RAND && Col->Distr != DISTR_ORD)
			k = (__int64) d;
		IntToTime(k, NULL, Buf);
		break;
	}
}

void DistrSet(ColT *Col, char *Buf, __int64 CurGran, int GranSz)
{
	int j;
	__int64 k = 0;
	double d = 0, s;

	s = (Col->DPar[PAR_MAX] - Col->DPar[PAR_MIN]) / gRowNum;
	k = CurGran ? Col->LPar[PAR_MIN] + GRAN_SZ * CurGran - 1 : Col->LPar[PAR_MIN] - 1;
	d = CurGran ? Col->DPar[PAR_MIN] + s * (GRAN_SZ * CurGran - 1) : Col->DPar[PAR_MIN] - s;
	for (j = 0; j < GranSz; j++) {
		while (1) {
			switch (Col->Distr) {
			case DISTR_ORD:
				if (!COL_DBL(Col->Type)) {
					k++;
					if (k > Col->LPar[PAR_MAX])
						k = Col->LPar[PAR_MIN];
				}
				else {
					d += s;
					if (d > Col->DPar[PAR_MAX])
						d = Col->DPar[PAR_MIN];
				}
				break;
			case DISTR_RAND:
				if (!COL_DBL(Col->Type))
					k = Col->Rand.iRand + Col->LPar[PAR_MIN];
				else
					d = Col->DPar[PAR_MIN] + (double) Col->Rand.iRand / Col->Rand.Prim * (Col->DPar[PAR_MAX] - Col->DPar[PAR_MIN]);
				Col->Rand.iRand = MulMod(Col->Rand.iRand, Col->Rand.Root, Col->Rand.Prim);
				break;
			case DISTR_GAUS:
				d = Gauss(Col);
				break;
			case DISTR_EXPN:
				d = ExpNeg(Col);
				break;
			case DISTR_POIS:
				d = Poisson(Col);
				break;
			case DISTR_SSIM:
				d = SelfSim(Col);
				break;
			case DISTR_ZIPF:
				d = Zipf(Col);
				break;
			}
			if ((Col->Distr == DISTR_RAND || Col->Distr == DISTR_ORD) && !COL_DBL(Col->Type)) {
				if (k >= Col->LPar[PAR_MIN] && k <= Col->LPar[PAR_MAX])
					break;
			}
			else if (d >= Col->DPar[PAR_MIN] && d <= Col->DPar[PAR_MAX])
				break;
		}
		FillBuf(Col, Buf, k, d);
		Buf += TypeSz[Col->Type] ? TypeSz[Col->Type] : Col->Length + 1;
	}
}

void TextWrite(char **Buf, int GranSz, FILE *File)
{
	int i, j;
	TIMESTAMP_STRUCT *ts;
	char b[TINYBUFF];

	for (i = 0; i < GranSz; i++) {
		for (j = 0; j < gCol.cur + 1; j++) {
			switch (gCol.h[j].Type) {
			case TYPE_CHAR:
			case TYPE_VARCHAR:
			case TYPE_DECIMAL:
				fprintf(File, "%s", Buf[j] + i * (gCol.h[j].Length + 1));
				break;
			case TYPE_REAL:
				fprintf(File, "%g", *((float *) Buf[j] + i));
				break;
			case TYPE_FLOAT:
				fprintf(File, "%g", *((double *) Buf[j] + i));
				break;
			case TYPE_DATETIME:
				ts = (TIMESTAMP_STRUCT *) Buf[j] + i;
				fprintf(File, "%02d/%02d/%04d %02d:%02d", ts->month, ts->day, ts->year, ts->hour, ts->minute);
				break;
			case TYPE_INT:
				fprintf(File, "%d", *((int *) Buf[j] + i));
				break;
			case TYPE_SMALLINT:
				fprintf(File, "%d", *((short *) Buf[j] + i));
				break;
			case TYPE_TINYINT:
				fprintf(File, "%d", *(Buf[j] + i));
				break;
			case TYPE_BIGINT:
				PrintI64(b, *((__int64 *) Buf[j] + i));
				fprintf(File, "%s", b);
				break;
			}
			if (j != gCol.cur)
				fprintf(File, ",");
			else
				fprintf(File, "\n");
		}
	}
}

void OdbcError(short hdlType, SQLHANDLE hdl)
{
	char state[6], msg[SQL_MAX_MESSAGE_LENGTH];
	char buf[XLARGEBUFF], DispBuf[XLARGEBUFF] = {
		'\0'															};
	short len, n = 1;
	UDWORD native;
	SQLRETURN rc = SQL_SUCCESS;

	while ((rc = SQLGetDiagRec(hdlType, hdl, n++, (SQLCHAR *) state, (SQLINTEGER *) & native, (SQLCHAR *) msg, SQL_MAX_MESSAGE_LENGTH, &len)) != SQL_NO_DATA) {
		if (rc == SQL_ERROR || rc == SQL_INVALID_HANDLE)
			break;
		if (58 + strlen(state) + strlen(msg) >= XLARGEBUFF)
			break;
		sprintf(buf, "SQL Error State:%s, Native Error Code: %6x, ODBC Error: %s\n",
			state, native, msg);
		if (strlen(DispBuf) + strlen(buf) >= XLARGEBUFF)
			break;
		strcat(DispBuf, buf);
	}
	MessageBox(ghMain, DispBuf, NULL, MB_ICONEXCLAMATION);
}

int OdbcErr(RETCODE rc, int fFatal)
{
	short hdlType = 0;
	SQLHANDLE hdl = NULL;
	if (rc == SQL_SUCCESS || rc == SQL_SUCCESS_WITH_INFO)
		return 0;
	else if (fFatal) {
		if (ghStmt != SQL_NULL_HSTMT) {
			hdl = ghStmt;
			hdlType = SQL_HANDLE_STMT;
		}
		else if (ghDbc != SQL_NULL_HANDLE) {
			hdl = ghDbc;
			hdlType = SQL_HANDLE_DBC;
		}
		else if (ghEnv != SQL_NULL_HANDLE) {
			hdl = ghEnv;
			hdlType = SQL_HANDLE_ENV;
		}
		OdbcError(hdlType, hdl);
		SQLFreeHandle(SQL_HANDLE_STMT, ghStmt);
		SQLDisconnect(ghDbc);
		SQLFreeHandle(SQL_HANDLE_DBC, ghDbc);
		SQLFreeHandle(SQL_HANDLE_ENV, ghEnv);
		ghEnv = SQL_NULL_HANDLE;
		ghDbc = SQL_NULL_HANDLE;
		ghStmt = SQL_NULL_HSTMT;
	}
	return 1;
}

int OdbcConnect(void)
{
	short c;
	char buf[XLARGEBUFF];

	if (OdbcErr(SQLAllocHandle(SQL_HANDLE_ENV, SQL_NULL_HANDLE, &ghEnv), 1)
			|| OdbcErr(SQLSetEnvAttr(ghEnv, SQL_ATTR_ODBC_VERSION, (SQLPOINTER) SQL_OV_ODBC3, 0), 1)
			|| OdbcErr(SQLAllocHandle(SQL_HANDLE_DBC, ghEnv, &ghDbc), 1)
			|| OdbcErr(SQLDriverConnect(ghDbc, NULL, (SQLCHAR *) gConnect, (short) (strlen(gConnect) + 1), (UCHAR *) buf, (UWORD) XLARGEBUFF, &c,
		SQL_DRIVER_NOPROMPT), 1)
			|| OdbcErr(SQLAllocHandle(SQL_HANDLE_STMT, ghDbc, &ghStmt), 1))
		return 0;
	return 1;
}

int OdbcClearStmt(void)
{
	if (OdbcErr(SQLFreeStmt(ghStmt, SQL_RESET_PARAMS), 1)
			|| OdbcErr(SQLFreeStmt(ghStmt, SQL_UNBIND), 1)
			|| OdbcErr(SQLFreeStmt(ghStmt, SQL_CLOSE), 1))
		return 0;
	return 1;
}

int OdbcBigName(OdbcT *Db)
{
	SQLINTEGER ci, f;
	short c, f0;
	char buf[LARGEBUFF];

	if (OdbcErr(SQLGetInfo(ghDbc, SQL_QUALIFIER_USAGE, &f, sizeof(f), &c), 1)
			|| f & SQL_QU_TABLE_DEFINITION && OdbcErr(SQLGetConnectAttr(ghDbc, SQL_CURRENT_QUALIFIER, Db->Cat, LARGEBUFF, &ci), 1)
			|| OdbcErr(SQLGetInfo(ghDbc, SQL_QUALIFIER_NAME_SEPARATOR, Db->CatSep, TINYBUFF, &c), 1))
		return 0;
	if (!c)
		Db->CatSep[0] = '\0';
	if (OdbcErr(SQLGetInfo(ghDbc, SQL_OWNER_USAGE, &f, sizeof(f), &c), 1)
			|| f & SQL_OU_TABLE_DEFINITION && OdbcErr(SQLGetInfo(ghDbc, SQL_USER_NAME, Db->Schem, LARGEBUFF, &c), 1)
			|| OdbcErr(SQLGetInfo(ghDbc, SQL_IDENTIFIER_QUOTE_CHAR, Db->Quot, 2, &c), 1))
		return 0;
	if (Db->Quot[0] == ' ' || c > 1)
		Db->Quot[0] = '\0';
	if (OdbcErr(SQLGetInfo(ghDbc, SQL_QUALIFIER_LOCATION, &f0, sizeof(f0), &c), 1))
		return 0;
	memset(Db->BigName, 0, XLARGEBUFF);
	if (*Db->Schem)
		sprintf(Db->BigName, "%s%s%s.", Db->Quot, Db->Schem, Db->Quot);
	sprintf(buf, "%s%s%s", Db->Quot, gTable, Db->Quot);
	strcat(Db->BigName, buf);
	if (*Db->Cat) {
		if (f0 == SQL_QL_END) {
			sprintf(buf, "%s%s%s%s", Db->CatSep, Db->Quot, Db->Cat, Db->Quot);
			strcat(Db->BigName, buf);
		}
		else {
			sprintf(buf, "%s%s%s%s", Db->Quot, Db->Cat, Db->Quot, Db->CatSep);
			memmove(Db->BigName + strlen(buf), Db->BigName, strlen(Db->BigName) + 1);
			memcpy(Db->BigName, buf, strlen(buf));
		}
	}
	return 1;
}

int OdbcDropTable(OdbcT *Db)
{
	SQLINTEGER c;
	char buf[LARGEBUFF], srchName[LARGEBUFF] = {
		'\0'															};

	if (!OdbcClearStmt()
			|| OdbcErr(SQLTables(ghStmt, (SQLCHAR *) Db->Cat, SQL_NTS, (SQLCHAR *) Db->Schem, SQL_NTS, (SQLCHAR *) gTable, SQL_NTS, NULL, 0), 1)
			|| SQLFetch(ghStmt) == SQL_ERROR)
		return 0;
	SQLGetData(ghStmt, 3, SQL_CHAR, srchName, LARGEBUFF, &c);
	if (!OdbcClearStmt())
		return 0;
	if (strcmp(srchName, gTable) == 0) {
		sprintf(buf, "drop table %s", Db->BigName);
		if (OdbcErr(SQLExecDirect(ghStmt, (SQLCHAR *) buf, SQL_NTS), 1))
			return 0;
	}
	return 1;
}

int OdbcCreateStmt(OdbcT *Db)
{
	int i, sz = XLARGEBUFF;
	char buf0[TINYBUFF], buf[XLARGEBUFF];

	if ((Db->Stmt = (char *) malloc(sz)) == NULL)
		return 0;
	sprintf(Db->Stmt, "create table %s (", Db->BigName);
	for (i = 0; i < gCol.cur + 1; i++) {
		sprintf(buf, "%s %s", gCol.h[i].Name, TypeName[gCol.h[i].Type]);
		if (gCol.h[i].Type == TYPE_CHAR || gCol.h[i].Type == TYPE_VARCHAR || gCol.h[i].Type == TYPE_DECIMAL) {
			sprintf(buf0, "(%d", gCol.h[i].Type == TYPE_DECIMAL ? gCol.h[i].Prec : gCol.h[i].Length);
			strcat(buf, buf0);
			if (gCol.h[i].Type == TYPE_DECIMAL) {
				sprintf(buf0, ",%d", gCol.h[i].Scale);
				strcat(buf, buf0);
			}
			strcat(buf, ")");
		}
		strcat(buf, " not NULL");
		if (gCol.h[i].fUnique) {
			strcat(buf, " ");
			strcat(buf, "unique");
		}
		if (gCol.h[i].fPrimKey) {
			strcat(buf, " ");
			strcat(buf, "primary key");
		}
		if (i != gCol.cur)
			strcat(buf, ",");
		else
			strcat(buf, ")");
		if ((int) strlen(buf) + (int) strlen(Db->Stmt) >= sz) {
			sz += XLARGEBUFF;
			if ((Db->Stmt = (char *) realloc(Db->Stmt, sz)) == NULL)
				return 0;
		}
		strcat(Db->Stmt, buf);
	}
	SendMessage(ghStatus, SB_SETTEXT, 0, (LPARAM) Db->Stmt);
	return 1;
}

int OdbcInit(OdbcT *Db, int GranSz, char **Buf)
{
	int i, j;

	memset(Db, 0, sizeof(OdbcT));
	ghEnv = SQL_NULL_HANDLE;
	ghDbc = SQL_NULL_HANDLE;
	ghStmt = SQL_NULL_HSTMT;
	if (!OdbcConnect()
			|| !OdbcBigName(Db)
			|| !OdbcDropTable(Db)
			|| !OdbcCreateStmt(Db)
			|| OdbcErr(SQLExecDirect(ghStmt, (SQLCHAR *) Db->Stmt, SQL_NTS), 1))
		return 0;
	free(Db->Stmt);
	if ((Db->Stmt = (char *) malloc(XLARGEBUFF)) == NULL)
		return 0;
	sprintf(Db->Stmt, "insert into %s values (", Db->BigName);
	if (!OdbcClearStmt()
			|| OdbcErr(SQLSetStmtAttr(ghStmt, SQL_ATTR_PARAMSET_SIZE, (SQLPOINTER) GranSz, 0), 1))
		return 0;
	if ((Db->Len = (SQLINTEGER **) malloc((gCol.cur + 1) * sizeof(SQLINTEGER *))) == NULL)
		return 0;
	memset(Db->Len, 0, (gCol.cur + 1) * sizeof(SQLINTEGER *));
	for (i = 0; i < gCol.cur + 1; i++) {
		strcat(Db->Stmt, "?");
		if (i != gCol.cur)
			strcat(Db->Stmt, ",");
		else
			strcat(Db->Stmt, ")");
		if ((Db->Len[i] = (SQLINTEGER *) malloc(GranSz * sizeof(SQLINTEGER))) == NULL)
			return 0;
		memset(Db->Len[i], 0, GranSz * sizeof(SQLINTEGER));
		for (j = 0; j < GranSz; j++)
			Db->Len[i][j] = gCol.h[i].Type == TYPE_CHAR || gCol.h[i].Type == TYPE_VARCHAR || gCol.h[i].Type == TYPE_DECIMAL
				? SQL_NTS : TypeSz[gCol.h[i].Type];
		if (OdbcErr(SQLBindParameter(ghStmt, (SQLUSMALLINT) (i + 1), SQL_PARAM_INPUT, SQL_DEFAULT, (SQLSMALLINT) SqlType[gCol.h[i].Type],
			gCol.h[i].Type == TYPE_DECIMAL ? gCol.h[i].Prec : gCol.h[i].Length,
			(SQLSMALLINT) (gCol.h[i].Type == TYPE_DECIMAL ? gCol.h[i].Scale : 0),
			Buf[i],
			TypeSz[gCol.h[i].Type] ? TypeSz[gCol.h[i].Type] : gCol.h[i].Length + 1, Db->Len[i]), 1))
			return 0;
	}
	return 1;
}

int OdbcDone(OdbcT *Db)
{
	int i;

	if (OdbcErr(SQLFreeHandle(SQL_HANDLE_STMT, ghStmt), 1)
			|| OdbcErr(SQLDisconnect(ghDbc), 1)
			|| OdbcErr(SQLFreeHandle(SQL_HANDLE_DBC, ghDbc), 1)
			|| OdbcErr(SQLFreeHandle(SQL_HANDLE_ENV, ghEnv), 1))
		return 0;
	for (i = 0; i < gCol.cur + 1; i++)
		free(Db->Len[i]);
	free(Db->Len);
	free(Db->Stmt);
	return 1;
}

int Generate(void)
{
	__int64 i, n, k = 0;
	int j, f = 0, GranSz;
	char ProgMsg[SMALLBUFF], buf0[SMALLBUFF];
	char **Buf = NULL;
	FILE *File = NULL;
	OdbcT Db = {
		0	};

	if (gfDestFile && (File = fopen(gTable, "w")) == NULL)
		goto l;
	if ((Buf = (char **) malloc((gCol.cur + 1) * sizeof(void *))) == NULL)
		goto l;
	memset(Buf, 0, (gCol.cur + 1) * sizeof(void *));
	for (j = 0; j < gCol.cur + 1; j++)
		if ((Buf[j] = (char *) malloc((TypeSz[gCol.h[j].Type] ? TypeSz[gCol.h[j].Type] : gCol.h[j].Length + 1) * GRAN_SZ)) == NULL)
			goto l;
	n = gRowNum / GRAN_SZ + (gRowNum % GRAN_SZ != 0);
	GranSz = (int) (n == 1 && (gRowNum % GRAN_SZ != 0) ? gRowNum % GRAN_SZ : GRAN_SZ);
	if (gfDestOdbc && !OdbcInit(&Db, GranSz, Buf))
		goto l;
	for (i = 0; i < n; i++) {
		if (i == n - 1) {
			GranSz = (int) (gRowNum % GRAN_SZ != 0 ? gRowNum % GRAN_SZ : GRAN_SZ);
			if (gfDestOdbc && OdbcErr(SQLSetStmtAttr(ghStmt, SQL_ATTR_PARAMSET_SIZE, (SQLPOINTER) GranSz, 0), 1))
				goto l;
		}
		for (j = 0; j < gCol.cur + 1; j++) {
			memset(Buf[j], 0, (TypeSz[gCol.h[j].Type] ? TypeSz[gCol.h[j].Type] : gCol.h[j].Length + 1) * GRAN_SZ);
			DistrSet(&gCol.h[j], Buf[j], i, GranSz);
		}
		if (gfDestFile)
			TextWrite(Buf, GranSz, File);
		if (gfDestOdbc && OdbcErr(SQLExecDirect(ghStmt, (SQLCHAR *) Db.Stmt, SQL_NTS), 1))
			goto l;
		k += GranSz;
		PrintI64(buf0, k);
		sprintf(ProgMsg, "%s records generated, remains ", buf0);
		PrintI64(buf0, gRowNum - k);
		strcat(ProgMsg, buf0);
		SendMessage(ghStatus, SB_SETTEXT, 0, (LPARAM) ProgMsg);
	}
	f = 1;
l:
	for (j = 0; j < gCol.cur + 1; j++)
		free(Buf[j]);
	free(Buf);
	if (gfDestFile && File)
		fclose(File);
	if (gfDestOdbc)
		OdbcDone(&Db);
	return f;
}

void CursSet(HWND hwnd, char *i, int f)
{
	HCURSOR hCurs;
	if (i == IDC_ARROW || i == IDC_WAIT || i == IDC_CROSS)
		hCurs = LoadCursor(NULL, i);
	else
		hCurs = LoadCursor(NULL, i);
	SetClassLong(hwnd, GCL_HCURSOR, (LONG) hCurs);
	SetCursor(hCurs);
	if (f) {
		if (i != IDC_ARROW)
			SetCapture(hwnd);
		else
			ReleaseCapture();
	}
}

void DbGenOnCommand(HWND hwnd, int id, HWND hwndCtl, UINT codeNotify)
{
	int i;
	char buf[MAX_PATH];

	switch (id) {
	case IDC_OK:
		SetFocus(ghMain);
		CursSet(ghMain, IDC_WAIT, 1);
		ConfSave(DBGENINI_STR);
		for (i = 0; i < gCol.cur + 1; i++)
			DistrInit(&gCol.h[i]);
		if (Generate())
			SendMessage(ghStatus, SB_SETTEXT, (WPARAM) 0, (LPARAM) "Output -- Ok.");
		else
			SendMessage(ghStatus, SB_SETTEXT, (WPARAM) 0, (LPARAM) "Output -- Failed.");
		if (ghGraph)
			DestroyWindow(ghGraph);
		ghGraph = 0;
		CursSet(ghMain, IDC_ARROW, 1);
		break;
	case IDCANCEL:
		PostQuitMessage(0);
		break;
	case IDC_LEFT:
		LvGetInput(GetFocus(), gLvCur, gLvCurSub);
		DestroyWindow(GetParent(GetFocus()));
		do {
			if (!gLvCur && !gLvCurSub)
				return;
			if (gLvCurSub > 0)
				gLvCurSub--;
			else {
				gLvCurSub = COL_NUM - 3;
				if (gLvCur > 0)
					gLvCur--;
				else
					return;
			}
		}
		while (!ColEnable(gCol.h[gLvCur].Type, gCol.h[gLvCur].Distr, gLvCurSub));
		if (!ColEnable(gCol.h[gLvCur].Type, gCol.h[gLvCur].Distr, gLvCurSub))
			return;
		ListView_EnsureVisible(ghLv, gLvCur, 0);
		TmplSet(gLvCur, gLvCurSub);
		CreateDialogIndirect(ghInst, gTmpl, ghMain, InputDlgProc);
		break;
	case IDC_RIGHT:
		LvGetInput(GetFocus(), gLvCur, gLvCurSub);
		DestroyWindow(GetParent(GetFocus()));
		do {
			if (gLvCur == gCol.cur + 1 - 1 && gLvCurSub == COL_NUM - 3)
				return;
			if (gLvCurSub < COL_NUM - 3)
				gLvCurSub++;
			else {
				gLvCurSub = 0;
				if (gLvCur < gCol.cur + 1 - 1)
					gLvCur++;
				else
					return;
			}
		}
		while (!ColEnable(gCol.h[gLvCur].Type, gCol.h[gLvCur].Distr, gLvCurSub));
		if (!ColEnable(gCol.h[gLvCur].Type, gCol.h[gLvCur].Distr, gLvCurSub))
			return;
		ListView_EnsureVisible(ghLv, gLvCur, 0);
		TmplSet(gLvCur, gLvCurSub);
		CreateDialogIndirect(ghInst, gTmpl, ghMain, InputDlgProc);
		break;
	case IDC_README:
		buf[0] = 0;
		GetCurrentDirectory(MAX_PATH, buf);
		strcat(buf, "\\");
		strcat(buf, README_STR);
		if (ShellExecute(hwnd, "open", buf, NULL, NULL, SW_SHOWDEFAULT) <= (HINSTANCE) 32)
			MessageBox(hwnd, "Unable to find " README_STR, "DbGen error", MB_ICONEXCLAMATION);
		break;
	}
}

void DbGenOnDestroy(HWND hwnd)
{
	if (ghGraph)
		DestroyWindow(ghGraph);
	VecDel((VecT *) &gCol);
	PostQuitMessage(0);
}

int CALLBACK DbGenProc(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	switch (uMsg) {
		HANDLE_MSG(hwnd, WM_DESTROY, DbGenOnDestroy);
		HANDLE_MSG(hwnd, WM_SIZE, DbGenOnSize);
		HANDLE_MSG(hwnd, WM_CREATE, DbGenOnCreate);
		HANDLE_MSG(hwnd, WM_COMMAND, DbGenOnCommand);
		HANDLE_MSG(hwnd, WM_NOTIFY, DbGenOnNotify);
	default:
		return DefWindowProc(hwnd, uMsg, wParam, lParam);
	}
}

void TabDraw(HDC hdc, char *str, int bSelected, int x)
{
	POINT pts[4];
	RECT rc = {
		0, 0, 0, 0										};

	SelectObject(hdc, GetStockObject(bSelected ? WHITE_BRUSH : NULL_BRUSH));
	DrawText(hdc, str, -1, &rc, DT_CALCRECT);
	rc.left += x;
	rc.right += x + 20;
	rc.bottom = rc.top + GetSystemMetrics(SM_CYHSCROLL) - 2;
	pts[0].x = rc.left;
	pts[0].y = rc.top;
	pts[1].x = rc.left + 8;
	pts[1].y = rc.bottom - 1;
	pts[2].x = rc.right - 7;
	pts[2].y = rc.bottom - 1;
	pts[3].x = rc.right - 1;
	pts[3].y = rc.top;
	Polygon(hdc, pts, 4);
	if (!bSelected) {
		pts[2].x--;
		pts[3].x--;
		MoveToEx(hdc, pts[2].x, pts[2].y, NULL);
		LineTo(hdc, pts[3].x, pts[3].y);
	}
	else {
		MoveToEx(hdc, pts[0].x, pts[0].y, NULL);
		LineTo(hdc, pts[3].x, pts[3].y);
	}
	DrawText(hdc, str, -1, &rc, DT_CENTER|DT_VCENTER|DT_SINGLELINE);
}

void TabOnPaint(HWND hwnd)
{
	PAINTSTRUCT ps;

	BeginPaint(hwnd, &ps);
	SetBkMode(ps.hdc, TRANSPARENT);
	SelectObject(ps.hdc, GetStockObject(DEFAULT_GUI_FONT));
	if (gfTab == 0) {
		TabDraw(ps.hdc, "  Graph  ", 0, 60);
		TabDraw(ps.hdc, "   Main    ", 1, 1);
	}
	else if (gfTab == 1) {
		TabDraw(ps.hdc, "   Main    ", 0, 1);
		TabDraw(ps.hdc, "  Graph  ", 1, 60);
	}
	EndPaint(hwnd, &ps);
}

void TabOnLButtonDown(HWND hwnd, BOOL fDoubleClick, int x, int y, UINT keyFlags)
{
	if (x > 60 && x < 126 && gfTab != 1) {
		CursSet(ghMain, IDC_WAIT, 1);
		gfTab = 1;
		ShowWindow(ghLv, SW_HIDE);
		if (ghGraph)
			DestroyWindow(ghGraph);
		ghGraph = CreateWindow("graph", "", WS_CHILD, X_SHT, Y_SHT, W_SHT, H_SHT, ghMain, NULL, ghInst, NULL);
		SetForegroundWindow(ghGraph);
		ShowWindow(ghGraph, SW_SHOWNORMAL);
		InvalidateRect(ghGraph, NULL, 1);
		UpdateWindow(ghGraph);
		CursSet(ghMain, IDC_ARROW, 1);
	}
	else if (x < 60 && gfTab != 0) {
		gfTab = 0;
		DestroyWindow(ghGraph);
		ShowWindow(ghLv, SW_SHOWNORMAL);
		SetForegroundWindow(ghLv);
		InvalidateRect(ghLv, NULL, 1);
		UpdateWindow(ghLv);
	}
	else
		return;
	InvalidateRect(ghTab, NULL, 1);
	UpdateWindow(ghTab);
}

int CALLBACK TabProc(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	switch (uMsg) {
		HANDLE_MSG(hwnd, WM_PAINT, TabOnPaint);
		HANDLE_MSG(hwnd, WM_LBUTTONDOWN, TabOnLButtonDown);
	default:
		return DefWindowProc(hwnd, uMsg, wParam, lParam);
	}
}

void DblScan(double *d, int n, double *emin, double *emax)
{
	int i;

	*emin = d[0];
	*emax = d[n - 1];
	for (i = 0; i < n; i++, d++) {
		if (*emin > *d)
			*emin = *d;
		if (*emax < *d)
			*emax = *d;
	}
}

int TextRead(void)
{
	char *p, buf[XLARGEBUFF];
	int i = 0, j, k;
	double s;
	FILE *File;
	GraphT *Graph;

	if ((File = fopen(gTable, "r")) == NULL)
		return 0;
	while (i < gRowNum && fgets(buf, XLARGEBUFF, File)) {
		p = strtok(buf, "\r\n,;");
		for (j = 0; j < gCol.cur + 1; j++) {
			if (p && *p) {
				switch (gCol.h[j].Type) {
				case TYPE_CHAR:
				case TYPE_VARCHAR:
					gCol.h[j].Graph.Val[i] = (double) StrToInt(p);
					break;
				case TYPE_DATETIME:
					gCol.h[j].Graph.Val[i] = (double) TimeToInt(p);
					break;
				default:
					gCol.h[j].Graph.Val[i] = atof(p);
				}
			}
			p = strtok(NULL, "\r\n,;");
		}
		i++;
	}
	if (fclose(File) == EOF)
		return 0;
	for (i = 0; i < gCol.cur + 1; i++) {
		DblScan(gCol.h[i].Graph.Val, (int) gRowNum, &gCol.h[i].Graph.ValMin, &gCol.h[i].Graph.ValMax);
		Graph = &gCol.h[i].Graph;
		s = 999 / (Graph->ValMax - Graph->ValMin);
		for (j = 0; j < gRowNum; j++) {
			k = (int) (s * (Graph->Val[j] - Graph->ValMin));
			k = max(0, min(999, k));
			Graph->Bar[k]++;
			if (Graph->Bar[k] > Graph->BarMax)
				Graph->BarMax = Graph->Bar[k];
		}
	}
	return 1;
}

int GraphOnCreate(HWND hwnd, CREATESTRUCT * lpCreateStruct)
{
	int i;

	for (i = 0; i < gCol.cur + 1; i++) {
		memset(&gCol.h[i].Graph, 0, sizeof(GraphT));
		if ((gCol.h[i].Graph.Val = (double *) malloc((int) gRowNum * sizeof(double))) == NULL)
			return 0;
		if ((gCol.h[i].Graph.Bar = (int *) malloc(1000 * sizeof(int))) == NULL)
			return 0;
		memset(gCol.h[i].Graph.Val, 0, (int) gRowNum * sizeof(double));
		memset(gCol.h[i].Graph.Bar, 0, 1000 * sizeof(int));
	}
	if (!TextRead()) {
		MessageBox(ghMain, "Unable to read text data file, \"Generate\" it, please.", "DbGen error", MB_ICONEXCLAMATION);
		return 0;
	}
	return 1;
}

void GraphOnPaint(HWND hwnd)
{
	PAINTSTRUCT ps;
	int i, j, x, y;
	double s;
	HBRUSH hBr;
	GraphT *Graph;

	BeginPaint(hwnd, &ps);
	hBr = SelectObject(ps.hdc, GetStockObject(LTGRAY_BRUSH));
	SelectObject(ps.hdc, GetStockObject(BLACK_PEN));
	Rectangle(ps.hdc, 0, 0, ps.rcPaint.right, ps.rcPaint.bottom);
	SelectObject(ps.hdc, hBr);
	for (i = 0; i < gCol.cur + 1; i++) {
		j = 3 * (i > COLOR_NUM ? 0 : (i + 15) % COLOR_NUM);
		SelectObject(ps.hdc, CreatePen(PS_SOLID, 1, RGB(gColor[j], gColor[j + 1], gColor[j + 2])));
		Graph = &gCol.h[i].Graph;
		s = (double) ps.rcPaint.right / 1000;
		if (IS_DBL_ZERO(s))
			break;
		for (j = 0; j < 1000; j++) {
			x = (int) (s * j);
			x = max(0, min(ps.rcPaint.right - 1, x));
			y =  (int) (ps.rcPaint.bottom * (1 - (double) Graph->Bar[j] / Graph->BarMax));
			y = max(0, min(ps.rcPaint.bottom - 1, y));
			if (j)
				LineTo(ps.hdc, x, y);
			else
				MoveToEx(ps.hdc, x, y, NULL);
		}
	}
	EndPaint(hwnd, &ps);
}

void GraphOnDestroy(HWND hwnd)
{
	int i;

	for (i = 0; i < gCol.cur + 1; i++) {
		free(gCol.h[i].Graph.Val);
		free(gCol.h[i].Graph.Bar);
		memset(&gCol.h[i].Graph, 0, sizeof(GraphT));
	}
}

int CALLBACK GraphProc(HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message) {
		HANDLE_MSG(hwnd, WM_CREATE, GraphOnCreate);
		HANDLE_MSG(hwnd, WM_DESTROY, GraphOnDestroy);
		HANDLE_MSG(hwnd, WM_PAINT, GraphOnPaint);
	}
	return DefWindowProc(hwnd, message, wParam, lParam);
}

int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrev, LPTSTR CmdLine, int CmdShow)
{
	MSG msg;
	WNDCLASSEX wcex;

	memset(&wcex, 0, sizeof(WNDCLASSEX));
	wcex.cbSize = sizeof(WNDCLASSEX);
	wcex.style = CS_OWNDC|CS_HREDRAW|CS_VREDRAW;
	wcex.hInstance = hInstance;
	wcex.lpfnWndProc = (WNDPROC) DbGenProc;
	wcex.lpszClassName = TokName[TOK_DBGEN];
	wcex.hCursor = LoadCursor(NULL, IDC_ARROW);
	wcex.hbrBackground = (HBRUSH) (COLOR_BTNFACE + 1);
	RegisterClassEx(&wcex);
	wcex.lpfnWndProc = (WNDPROC) TabProc;
	wcex.lpszClassName = "tab";
	RegisterClassEx(&wcex);
	wcex.lpfnWndProc = (WNDPROC) GraphProc;
	wcex.lpszClassName = "graph";
	RegisterClassEx(&wcex);
	ghInst = hInstance;
	ghMain = CreateWindowEx(WS_EX_STATICEDGE, TokName[TOK_DBGEN], "Database generator", WS_CAPTION|WS_SYSMENU|WS_THICKFRAME|WS_MINIMIZEBOX,
		5, 5, W_MAIN, H_MAIN, NULL, NULL, ghInst, NULL);
	ShowWindow(ghMain, CmdShow);
	InvalidateRect(ghMain, NULL, 1);
	UpdateWindow(ghMain);
	while (GetMessage(&msg, NULL, 0, 0)) {
		if (!IsDialogMessage(ghMain, &msg)) {
			TranslateMessage(&msg);
			DispatchMessage(&msg);
		}
	}
	return 0;
}
