{$INCLUDE cHeader.inc}
unit cStream;

interface

uses
  // System units
  SysUtils,             // Exception, fmxxx constants
  Classes,              // TFileStream

  // Simple types (L0)
  cStrings;             // CharSet, Pack



{                                                                              }
{ Streams unit v0.10 (L0)                                                      }
{                                                                              }
{    Stream abstraction and an implementations for primary and secondary       }
{    memory.                                                                   }
{                                                                              }
{ This unit is copyrighted  1999 by David Butler (david@e.co.za)              }
{                                                                              }
{ I invite you to use this unit, free of charge.                               }
{ I invite you to distibute this unit, but it must be for free.                }
{ I also invite you to contribute to its development, but do not distribute    }
{ a modified copy of this file, send suggestions and bug reports to            }
{ david@e.co.za                                                                }
{                                                                              }
{ Revision history:                                                            }
{   26/06/99  v0.01   Moved AStream from cDataStruct unit.                     }
{             v0.02   Added ATBinaryStream.                                    }
{                     Added ATTextStream from TTextParser in unit cTextParser. }
{                     Added Peek, Skip, OctetsRemaing and OctetsAvailable to   }
{                     AStream.                                                 }
{   17/07/99  v0.03   Added Col property.                                      }
{   03/11/99  v0.04   Revised AStream documentation.                           }
{   06/11/99  v0.10   Revisited whole unit.                                    }
{                     Added general functions to ATStream.                     }
{                     Renamed TStringStream to TMemoryStream.                  }
{                     433 lines interface, 933 lines implementation.           }
{                                                                              }
{                                                                              }
{ * To-do: Do default implementations for most abstract methods..              }
{ * Derive from Delphi's TStream? and implement.                               }
{ * Implement TTCPIPStream using fpiette's unit.                               }
{                                                                              }



{                                                                              }
{ AStream                                                                      }
{                                                                              }
{   AStream is the abstraction of a stream, like a file or a communication     }
{   line. Principally streams read and write octets (bytes), usually best or   }
{   only in a sequential manner.                                               }
{                                                                              }
{   Note that most stream implementation are for devices which can and as      }
{   standard operation do fail. When writing code that uses streams, bear      }
{   in mind that any function could raise an exception if, for example, the    }
{   communications line is dropped or if a file is locked.                     }
{                                                                              }
{   All read/write functions in AStream are blocking. To avoid blocking,       }
{   use OctetsAvailable to find out how many octets can be read without        }
{   having to block.                                                           }
{   EOF becomes true when at EOF and no more octets can become available.      }
{                                                                              }
type
  TOctet = Byte;
  EStream = class (Exception);
  TStreamNotifyEvent = Procedure (const Sender : TObject) of object;
  AStream = class
    FOnOctetAvailable : TStreamNotifyEvent;

    Function Read : TOctet; overload; virtual; abstract;
    { Read one octet from the stream. Blocks until data becomes available.     }
    { Reading past EOF raises an exception.                                    }
    Function Read (const Count : Integer) : String; overload; virtual; abstract;
    { Reads Length octets in from the stream. Blocks until data is available.  }
    { If Count is past EOF then a string shorter than Count is returned, ie if }
    { EOF then it will immediately return with ''. If Count <= 0 returns ''.   }

    Procedure Skip; overload; virtual; abstract;
    { Skip one octet. Blocks. Skip past EOF raises an exception.               }
    Procedure Skip (const Count : Integer); overload; virtual; abstract;
    { Skip Count octets. Blocks until EOF if Count is past EOF.                }

    Function Peek : TOctet; overload; virtual; abstract;
    { Returns one octet without removing it from the buffer. Blocks.           }
    { Peeking past EOF raises an exception.                                    }
    Function Peek (const Count : Integer) : String; overload; virtual; abstract;
    { Returns octets without removing them from the buffer. Blocks.            }
    { If Count is past EOF then a string shorter than Count is returned, ie if }
    { EOF then it will immediately return with ''.                             }

    Procedure Reset; virtual; abstract;
    { Resets the stream to it's startup state.                                 }
    { For files, seek to bof, for comms, reset connection to startup state     }
    { (equivalent to a reconnect). Raises an exception if not supported or     }
    { unsuccessful.                                                            }
    Procedure MoveEnd; virtual; abstract;
    { Move the stream to the end of the data.                                  }
    { For files, move to EOF, from where data can be appended.                 }
    { Raises an exception if not supported.                                    }
    Function EOF : Boolean; virtual; abstract;
    { False if more octects *can* become available. If True then no more       }
    { octets will become available.                                            }
    { For example, for files, this means EOF, for a modem line it would mean   }
    { not Carrier Detect, for a TCP/IP connection whether the socket is open.  }
    Procedure SetPosition (const NewPosition : Int64); virtual; abstract;
    { Seeking function. Raises an exception if NewPosition is past EOF or if   }
    { function is not supported.                                               }
    Function GetPosition : Int64; virtual; abstract;
    { Returns position in stream where 0 = bof.                                }
    Property Position : Int64 read GetPosition write SetPosition;
    { Reads / Writes the position in a random access stream.                   }
    Function Size : Int64; virtual; abstract;
    { Returns -1 if indeterminite.                                             }
    Function OctetsRemaining : Int64; virtual; abstract;
    { Returns the number of octets remaining until EOF, if known.              }
    { Returns -1 if indeterminite.                                             }
    { Returns 0 if NotEOF = False                                              }
    Procedure Truncate; virtual; abstract;
    { Truncates the stream at the current position.                            }
    { Ignored if not applicable to specific implementation.                    }

    Function OctetsAvailable : Int64; virtual; abstract;
    { Returns number of octets available for retrieval without blocking.       }
    { For files this would usually be the bytes left in the file, for a comms  }
    { stream, the number of bytes in the receive buffer.                       }
    Property OnOctetAvailable : TStreamNotifyEvent read FOnOctetAvailable write FOnOctetAvailable;
    { Event gets fired when new octets become available, but won't             }
    { necessarilly be fired for every new octet that becomes available.        }
    Function WaitAvailable (const Count : Integer) : Boolean; virtual; abstract;
    { Blocks until (OctetsAvailable >= Count) or known that Count is past EOF, }
    { ie (OctetsRemaining = OctetsAvailable) and (Count > OctetsRemaining)     }
    { Returns True if (OctetsAvailable >= Count)                               }
    Function WaitSequence (const S : String) : Integer; virtual; abstract;
    { Blocks until sequence S appears somewhere in available characters or it  }
    { is known that it appears nowhere. Raises an exception if it can't be     }
    { determined (eg buffer overflow).                                         }
    { Returns 0 if it appears nowhere (ie from current position to EOF),       }
    { otherwise returns the position of the sequence relative to current pos.  }

    Procedure Write (const Data : TOctet); overload; virtual; abstract;
    { Write one octet to the stream. Blocks until the data has been sent/      }
    { written/placed in a buffer, otherwise, raises an exception.              }
    Procedure Write (const Data : String); overload; virtual; abstract;
    { Write sequence of octets to the stream.                                  }
    { Exit from this procedure assumes the data has been sent/written/placed   }
    { in a buffer.                                                             }
  end;



{                                                                              }
{ ATBinaryStream                                                               }
{   Implements functionality to read/write binary data types from streams.     }
{   Uses internal Delphi formats:                                              }
{                                                                              }
{     Integer       4 byte integer (LSB first)                                 }
{     Int64         8 byte integer                                             }
{     Single        4 byte FPU Single                                          }
{     Double        8 byte FPU Double                                          }
{     Extended      10 byte FPU Extended                                       }
{     Currency      8 byte FPU fixed point                                     }
{     Boolean       1 byte, 0 = False                                          }
{     DateTime      8 byte FPU Double                                          }
{     String        Integer length + String                                    }
{     ShortString   Byte length + String                                       }
{                                                                              }
{   Implementations call inherited Read/ReadOctet/Write/WriteOctect.           }
{                                                                              }
{                                                                              }
{ Revision history:                                                            }
{   26/06/99  v0.01  Added ReadPacked functions.                               }
{   04/11/99  v0.10  Overloaded methods. Added Int64.                          }
{   09/11/99  v0.11  Moved WritePacked's implementations to cStrings unit as   }
{                    functions called Pack.                                    }
{                                                                              }
type
  ATBinaryStream = class (AStream)
    Function ReadPackedInteger : Integer; virtual;
    Function ReadPackedString : String; virtual;
    Function ReadPackedShortString : ShortString; virtual;
    Function ReadPackedSingle : Single; virtual;
    Function ReadPackedDouble : Double; virtual;
    Function ReadPackedExtended : Extended; virtual;
    Function ReadPackedCurrency : Currency; virtual;
    Function ReadPackedBoolean : Boolean; virtual;
    Function ReadPackedDateTime : TDateTime; virtual;
    Function ReadPackedInt64 : Int64; virtual;

    Procedure WritePacked (const D : Integer); overload; virtual;
    Procedure WritePacked (const D : Extended); overload; virtual;
    Procedure WritePackedSingle (const D : Single); overload; virtual;
    Procedure WritePackedDouble (const D : Double); overload; virtual;
    Procedure WritePackedCurrency (const D : Currency); overload; virtual;
    Procedure WritePackedDateTime (const D : TDateTime); overload; virtual;
    Procedure WritePacked (const D : Boolean); overload; virtual;
    Procedure WritePacked (const D : Int64); overload; virtual;
    Procedure WritePacked (const D : String); overload; virtual;
    Procedure WritePackedShortString (const D : ShortString); overload; virtual;
  end;



{                                                                              }
{ ATTextStream                                                                 }
{                                                                              }
{                                                                              }
{ Description:                                                                 }
{    Functions for parsing text from a stream.                                 }
{                                                                              }
{ Revision history:                                                            }
{      03/99  v0.01  Initial version (Skip, Extract and Match methods)         }
{   11/05/99  v0.02  Whitespace functions                                      }
{   20/06/99  v0.03  Added MatchString                                         }
{                    Added Line, Col, LineDelim properties                     }
{                    199 lines interface, 207 lines implementation             }
{   26/06/99  v0.04  Added ExtractPacked functions                             }
{             v0.10  Moved from cTextParser and renamed TTextParser to         }
{                    ATTextStream. It now inherits from AStream to allow all   }
{                    derived streams to use it.                                }
{   03/07/99  v0.11  Added MetaCharacter support (machine dependant charsets), }
{                    WhiteSpace functions are now implemented using these.     }
{                    Added MetaString support.                                 }
{   05/11/99  v0.12  Removed MetaCharacter support.                            }
{   06/11/99  v0.20  Overloaded methods. Revisited class.                      }
{                                                                              }
{                                                                              }
type
  ATTextStream = class (ATBinaryStream)
    protected
    FLineNr        : Integer;
    FLastNewLine   : Integer;
    FLineDelim     : String;

    Procedure DoReset;
    Function GetCol : Integer;

    public
    Constructor Create;
    Procedure Reset; override;

    { The Skip functions return True if any characters were skipped; and exit  }
    {   without raising an exception if EOF is reached.                        }
    { SkipCharSet skips while character matches SkipSet.                       }
    { SkipDelim skips up to *and over Delimiter* or EOF.                       }
    { SkipLine is equivalent to SkipDelim (FLineDelim), but also increases the }
    {   line counter.                                                          }
    Function  SkipChars (const SkipSet : CharSet) : Boolean;
    Function  SkipChar (const SkipSet : CharSet) : Boolean;
    Function  SkipDelim (const Delim : String) : Boolean;
    Function  SkipLine : Boolean;
    Function  SkipWhiteSpace : Boolean;

    { Match functions return True if next characters match, without moving the }
    {   current position. If the match overruns EOF then the function returns  }
    {   False, except MatchLine which will tolerate the FLineDelim past EOF.   }
    { MatchText is a case insensitive, locale sensitive match.                 }
    Function  Match (const MatchChars : CharSet) : Boolean; overload;
    Function  Match (const Ch : Char) : Boolean; overload;
    Function  Match (const S : String) : Boolean; overload;
    Function  MatchText (const Txt : String) : Boolean;
    Function  MatchLine (const Txt : String) : Boolean;
    Function  MatchWhiteSpace : Boolean;

    { The Extract functions return the extracted text, unchanged. Returns ''   }
    {   if a match could not be made.                                          }
    { All functions exit when EOF is reached without raising an exception,     }
    {   except Function Extract : Char.                                        }
    { ExtractText is a case insensitive, locale sensitive match.               }
    { ExtractRest extracts until EOF.                                          }
    Function Extract : Char; overload;
    Function Extract (const MatchChars : CharSet) : String; overload;
    Function Extract (const S : String) : String; overload;
    Function ExtractText (const Txt : String) : String;
    Function ExtractRest : String;

    { ExtractDelim extracts a string delimited by Delimiter or EOF and returns }
    {   the text up to but excluding the delimiter.                            }
    { ExtractLine is equivalent to ExtractDelim (FLineDelim); SkipLine;        }
    { ExtractKeyword uses MatchText to match from a list of possible keywords. }
    {   The keyword must be followed by DelimSet or EOF. Returns index of      }
    {   extracted keyword, or -1 if none.                                      }
    { ExtractQuotedText extracts text enclosed in a pair of the same from      }
    {   QuoteSet. Quotes can be included inside the text by using two          }
    {   consequetive quotes,                                                   }
    {   eg for QuoteSet = ['"', '''']  "A'BC"=A'BC, 'Dave''s'=Dave's, """"="   }
    Function ExtractDelim (const Delimiter : CharSet) : String; overload;
    Function ExtractDelim (const Delimiter : String) : String; overload;
    Function ExtractLine : String;
    Function ExtractKeyword (const Keywords : Array of String; const DelimSet : CharSet) : Integer;
    Function ExtractQuotedText (const QuoteSet : CharSet) : String;

    Property LineDelim : String read FLineDelim write FLineDelim;
    Property Line : Integer read FLineNr;
    Property Col : Integer read GetCol;
  end;



{                                                                              }
{ ATStream                                                                     }
{   ATStream is the base class from which stream implementations inherit.      }
{   It includes all the ATxStream implementations and some general operations. }
{                                                                              }
type
  ATStream = class (ATTextStream)
    private
    Procedure CopyFrom (const S : AStream);

    public
    Function Size : Int64; override;

    Procedure Assign (const S : AStream); virtual;
    Procedure Append (const S : AStream); virtual;

    Function GetAsString : String; virtual;
    Procedure SetAsString (const S : String); virtual;

    Function IsEqual (const S : AStream) : Boolean; overload; virtual;
    Function IsEqual (const S : String) : Boolean; overload; virtual;
    Function IsEmpty : Boolean; virtual;
  end;



{                                                                              }
{ ETStream                                                                     }
{   Encapsulation of a ATStream class in another ATStream class.               }
{   Useful as a base class for classes that act as a proxy for another         }
{   stream.                                                                    }
{                                                                              }
type
  ETStream = class (ATStream)
    protected
    FOnOctetAvailable : TStreamNotifyEvent;
    FS : ATStream;

    Procedure OnOctetAvailableHandler (const Sender : TObject); virtual;

    public
    Constructor Create (const ES : ATStream);
    Destructor Destroy; override;

    Procedure Skip (const Count : Integer); overload; override;
    Procedure Skip; overload; override;
    Function  Read : TOctet; overload; override;
    Function  Read (const Count : Integer) : String; overload; override;
    Function  Peek : TOctet; overload; override;
    Function  Peek (const Count : Integer) : String; overload; override;
    Procedure Reset; override;
    Function  GetPosition : Int64; override;
    Procedure SetPosition (const NewPosition : Int64); override;
    Procedure MoveEnd; override;
    Function  Size : Int64; override;
    Function  OctetsRemaining : Int64; override;
    Function  EOF : Boolean; override;
    Function  OctetsAvailable : Int64; override;
    Procedure Truncate; override;
    Function  WaitAvailable (const Count : Integer) : Boolean; override;
    Function  WaitSequence (const S : String) : Integer; override;
    Procedure Write (const Data : TOctet); overload; override;
    Procedure Write (const Data : String); overload; override;

    Property OnOctetAvailable : TStreamNotifyEvent read FOnOctetAvailable write FOnOctetAvailable;
  end;



{                                                                              }
{ TFileStream                                                                  }
{   A file implemented as a stream.                                            }
{                                                                              }
type
  TFileStream = class (ATStream)
    protected
    FFileStream : Classes.TFileStream;

    public
    Constructor Create (const FilePath : String; const CreateFile : Boolean);
    Destructor Destroy; override;

    Procedure Skip (const Count : Integer); overload; override;
    Procedure Skip; overload; override;
    Function  Read : TOctet; overload; override;
    Function  Read (const Count : Integer) : String; overload; override;
    Function  Peek : TOctet; overload; override;
    Function  Peek (const Count : Integer) : String; overload; override;
    Procedure Reset; override;
    Function  GetPosition : Int64; override;
    Procedure SetPosition (const NewPosition : Int64); override;
    Procedure MoveEnd; override;
    Function  Size : Int64; override;
    Function  OctetsRemaining : Int64; override;
    Function  EOF : Boolean; override;
    Function  OctetsAvailable : Int64; override;
    Procedure Truncate; override;
    Function  WaitAvailable (const Count : Integer) : Boolean; override;
    Function  WaitSequence (const S : String) : Integer; override;
    Procedure Write (const Data : TOctet); overload; override;
    Procedure Write (const Data : String); overload; override;
  end;



{                                                                              }
{ TMemoryStream                                                                }
{   A implementation of a stream that stores/retrieves data from memory.       }
{                                                                              }
type
  TMemoryStream = class (ATStream)
    protected
    FData : String;
    FPos : Integer;

    Procedure DoReset;

    public
    Constructor Create (const S : String);

    Procedure Skip (const Count : Integer); overload; override;
    Procedure Skip; overload; override;
    Function  Read : TOctet; overload; override;
    Function  Read (const Count : Integer) : String; overload; override;
    Function  Peek : TOctet; overload; override;
    Function  Peek (const Count : Integer) : String; overload; override;
    Procedure Reset; override;
    Function  GetPosition : Int64; override;
    Procedure SetPosition (const NewPosition : Int64); override;
    Procedure MoveEnd; override;
    Function  Size : Int64; override;
    Function  OctetsRemaining : Int64; override;
    Function  EOF : Boolean; override;
    Function  OctetsAvailable : Int64; override;
    Procedure Truncate; override;
    Function  WaitAvailable (const Count : Integer) : Boolean; override;
    Function  WaitSequence (const S : String) : Integer; override;
    Procedure Write (const Data : TOctet); overload; override;
    Procedure Write (const Data : String); overload; override;
  end;



implementation



uses
  // System units
  Math;              // Max


{                                                                              }
{ ATBinaryStream                                                               }
{                                                                              }
Function ATBinaryStream.ReadPackedInteger : Integer;
  Begin
    Move (Read (Sizeof (Result)) [1], Result, Sizeof (Result));
  End;

Function ATBinaryStream.ReadPackedInt64 : Int64;
  Begin
    Move (Read (Sizeof (Result)) [1], Result, Sizeof (Result));
  End;

Function ATBinaryStream.ReadPackedSingle : Single;
  Begin
    Move (Read (Sizeof (Result)) [1], Result, Sizeof (Result));
  End;

Function ATBinaryStream.ReadPackedDouble : Double;
  Begin
    Move (Read (Sizeof (Result)) [1], Result, Sizeof (Result));
  End;

Function ATBinaryStream.ReadPackedExtended : Extended;
  Begin
    Move (Read (Sizeof (Result)) [1], Result, Sizeof (Result));
  End;

Function ATBinaryStream.ReadPackedCurrency : Currency;
  Begin
    Move (Read (Sizeof (Result)) [1], Result, Sizeof (Result));
  End;

Function ATBinaryStream.ReadPackedBoolean : Boolean;
  Begin
    Move (Read (Sizeof (Result)) [1], Result, Sizeof (Result));
  End;

Function ATBinaryStream.ReadPackedDateTime : TDateTime;
  Begin
    Move (Read (Sizeof (Result)) [1], Result, Sizeof (Result));
  End;

Function ATBinaryStream.ReadPackedString : String;
  Begin
    SetLength (Result, ReadPackedInteger);
    Move (Read (Length (Result)) [1], Result [1], Length (Result));
  End;

Function ATBinaryStream.ReadPackedShortString : ShortString;
  Begin
    SetLength (Result, Read);
    Move (Read (Length (Result)) [1], Result [1], Length (Result));
  End;

Procedure ATBinaryStream.WritePacked (const D : Integer);
  Begin
    Write (Pack (D));
  End;

Procedure ATBinaryStream.WritePacked (const D : Int64);
  Begin
    Write (Pack (D));
  End;

Procedure ATBinaryStream.WritePacked (const D : String);
  Begin
    Write (Pack (D));
  End;

Procedure ATBinaryStream.WritePackedShortString (const D : ShortString);
  Begin
    Write (PackShortString (D));
  End;

Procedure ATBinaryStream.WritePackedSingle (const D : Single);
  Begin
    Write (PackSingle (D));
  End;

Procedure ATBinaryStream.WritePackedDouble (const D : Double);
  Begin
    Write (PackDouble (D));
  End;

Procedure ATBinaryStream.WritePacked (const D : Extended);
  Begin
    Write (Pack (D));
  End;

Procedure ATBinaryStream.WritePackedCurrency (const D : Currency);
  Begin
    Write (PackCurrency (D));
  End;

Procedure ATBinaryStream.WritePacked (const D : Boolean);
  Begin
    Write (Pack (D));
  End;

Procedure ATBinaryStream.WritePackedDateTime (const D : TDateTime);
  Begin
    Write (PackDateTime (D));
  End;



{                                                                              }
{ ATTextStream                                                                 }
{                                                                              }
Procedure ATTextStream.DoReset;
  Begin
    FLineNr := 1;
    FLastNewLine := GetPosition;
  End;

Function ATTextStream.GetCol : Integer;
  Begin
    Result := GetPosition - FLastNewLine + 1;
  End;

Constructor ATTextStream.Create;
  Begin
    inherited Create;
    FLineDelim := c_EOL;
    DoReset;
  End;

Procedure ATTextStream.Reset;
  Begin
    DoReset;
  End;

{ Skip                                                                         }
Function ATTextStream.SkipChar (const SkipSet : CharSet) : Boolean;
  Begin
    Result := not EOF and (Char (Peek) in SkipSet);
    if Result then
      Skip;
  End;

Function ATTextStream.SkipChars (const SkipSet : CharSet) : Boolean;
  Begin
    Result := SkipChar (SkipSet);
    if Result then
      Repeat
      Until not SkipChar (SkipSet);
  End;

Function ATTextStream.SkipDelim (const Delim : String) : Boolean;
var R : Boolean;
  Begin
    Result := False;
    if Length (Delim) = 0 then
      exit;

    Repeat
      While not EOF and not (Char (Peek) = Delim [1]) do
        begin
          Skip;
          Result := True;
        end;
      if EOF then
        exit;

      R := cStrings.Match (Delim, Peek (Length (Delim)));
      if not R then
        Skip;
    Until R;
    Result := True;
    Skip (Length (Delim));
  End;

Function ATTextStream.SkipWhiteSpace : Boolean;
  Begin
    Result := SkipChars (cs_WhiteSpace)
  End;

Function ATTextStream.SkipLine : Boolean;
  Begin
    Result := SkipDelim (FLineDelim);
    if Result then
      begin
        Inc (FLineNr);
        FLastNewLine := GetPosition;
      end;
  End;

{ Match                                                                        }
Function ATTextStream.Match (const MatchChars : CharSet) : Boolean;
  Begin
    Result := not EOF and (Char (Peek) in MatchChars);
  End;

Function ATTextStream.MatchWhiteSpace : Boolean;
  Begin
    Result := Match (cs_WhiteSpace);
  End;

Function ATTextStream.Match (const Ch : Char) : Boolean;
  Begin
    Result := not EOF and (Char (Peek) = Ch);
  End;

Function ATTextStream.Match (const S : String) : Boolean;
var A, L, R : Integer;
  Begin
    L := Length (S);
    R := OctetsRemaining;
    if (R >= 0) and (R < L) then
      begin
        Result := False;
        exit;
      end;

    A := OctetsAvailable;
    While A < L do
      begin
        if (Peek (A) <> Copy (S, 1, A)) or not WaitAvailable (A + 1) then
          begin
            Result := False;
            exit;
          end;
        A := OctetsAvailable;
      end;
    Result := Peek (L) = S;
  End;

Function ATTextStream.MatchText (const Txt : String) : Boolean;
var A, L, R : Integer;
  Begin
    L := Length (Txt);
    R := OctetsRemaining;
    if (R >= 0) and (R < L) then
      begin
        Result := False;
        exit;
      end;

    A := OctetsAvailable;
    While A < L do
      begin
        if (AnsiCompareText (Copy (Txt, 1, A), Peek (A)) <> 0) or not WaitAvailable (A + 1) then
          begin
            Result := False;
            exit;
          end;
        A := OctetsAvailable;
      end;
    Result := AnsiCompareText (Txt, Peek (L)) <> 0;
  End;

Function ATTextStream.MatchLine (const Txt : String) : Boolean;
var I, L : Integer;
  Begin
    if FLineDelim = '' then
      raise EStream.Create ('Line delimiter not set');

    I := WaitSequence (FLineDelim);
    L := Length (Txt);
    Result := ((I = 0) and (OctetsRemaining = L) and MatchText (Txt)) or
              ((I = L) and MatchText (Txt));
  End;

{ Extract                                                                      }
Function ATTextStream.Extract (const MatchChars : CharSet) : String;
  Begin
    Result := '';
    While not EOF and (Char (Peek) in MatchChars) do
      Result := Result + Char (Read);
  End;

Function ATTextStream.Extract : Char;
  Begin
    Result := Char (Read);
  End;

Function ATTextStream.Extract (const S : String) : String;
  Begin
    if Match (S) then
      begin
        Skip (Length (S));
        Result := S;
      end else
      Result := '';
  End;

Function ATTextStream.ExtractText (const Txt : String) : String;
  Begin
    if MatchText (Txt) then
      Result := Read (Length (Txt)) else
      Result := '';
  End;

Function ATTextStream.ExtractRest : String;
  Begin
    Result := '';
    While not EOF do
      Result := Result + Read (Max (1, OctetsAvailable));
  End;

{ ExtractDelim                                                                 }
Function ATTextStream.ExtractDelim (const Delimiter : CharSet) : String;
  Begin
    Result := '';
    While not EOF and not (Char (Peek) in Delimiter) do
      Result := Result + Char (Read);
  End;

Function ATTextStream.ExtractDelim (const Delimiter : String) : String;
  Begin
    Result := Read (WaitSequence (Delimiter) - 1);
  End;

Function ATTextStream.ExtractQuotedText (const QuoteSet : CharSet) : String;
var Quote : Char;
    Fin   : Boolean;
  Begin
    if Match (QuoteSet) then
      begin
        Quote := Extract;
        Result := '';
        Repeat
          Result := Result + ExtractDelim ([Quote]);
          if EOF then
            raise EStream.Create ('No closing quote (' + Quote + ' expected)');
          Skip;
          Fin := not Match (Quote);
          if not Fin then
            begin
              Result := Result + Quote;
              Skip;
            end;
        Until Fin;
      end else
      Result := '';
  End;

Function ATTextStream.ExtractLine : String;
var I : Integer;
  Begin
    if FLineDelim = '' then
      raise EStream.Create ('Line delimiter not set');

    I := WaitSequence (FLineDelim);
    if I = 0 then
      Result := Read (OctetsAvailable) else
      begin
        Result := Read (I);
        Read (Length (FLineDelim));
      end;
  End;

Function ATTextStream.ExtractKeyword (const Keywords : Array of String; const DelimSet : CharSet) : Integer;
var I : Integer;
    R : Boolean;
    S : String;
  Begin
    For I := 0 to High (Keywords) do
      if MatchText (Keywords [I]) then
        begin
          S := Peek (Length (Keywords [I]) + 1);
          R := (Length (S) = Length (Keywords [I])) or (DelimSet = []);
          if R or (S [Length (S)] in DelimSet) then
            begin
              Result := I;
              Skip (Length (Keywords [I]));
              if not R then
                Skip;
              exit;
            end;
        end;
    Result := -1;
  End;



{                                                                              }
{ ATStream                                                                     }
{                                                                              }
Procedure ATStream.CopyFrom (const S : AStream);
const BufSize = 16384;
  Begin
    S.Reset;
    Write (S.Read (S.OctetsAvailable));   // non-blocking
    While not S.EOF do
      Write (S.Read (BufSize));           // blocking
  End;

Procedure ATStream.Assign (const S : AStream);
  Begin
    Reset;
    CopyFrom (S);
    Truncate;
  End;

Procedure ATStream.Append (const S : AStream);
  Begin
    MoveEnd;
    CopyFrom (S);
  End;

Function ATStream.GetAsString : String;
  Begin
    Reset;
    Result := ExtractRest;
  End;

Procedure ATStream.SetAsString (const S : String);
  Begin
    Reset;
    Write (S);
    Truncate;
  End;

Function ATStream.Size : Int64;
var L : Integer;
  Begin
    L := OctetsRemaining;
    if L = -1 then
      Result := -1 else
      Result := GetPosition + L;
  End;

// It would be better if the non-blocking part compared in smaller chunks. This
// will avoid unnecassary loading from streams if a mismatch occur early on.
Function ATStream.IsEqual (const S : AStream) : Boolean;
var L1, L2 : Int64;
  Begin
    Result := False;

    L1 := Size;
    L2 := S.Size;
    if (L1 >= 0) and (L2 >= 0) and (L1 <> L2) then
      exit;

    Reset;
    S.Reset;
    L1 := Min (OctetsAvailable, S.OctetsAvailable);
    if Read (L1) <> S.Read (L1) then      // non-blocking
      exit;

    While not EOF do
      if Read <> S.Read then              // blocking
        exit;

    Result := S.EOF;
  End;

Function ATStream.IsEqual (const S : String) : Boolean;
var L1 : Int64;
    L2, I : Integer;
  Begin
    Result := False;

    L1 := Size;
    L2 := Length (S);
    if (L1 >= 0) and (L1 <> L2) then
      exit;

    I := Min (OctetsAvailable, L2);
    if Read (I) <> CopyLeft (S, I) then   // non-blocking
      exit;

    Inc (I);
    While not EOF do
      if Char (Read) <> S [I] then        // blocking
        exit else
        Inc (I);

    Result := I = L2 + 1;
  End;

Function ATStream.IsEmpty : Boolean;
var L : Integer;
  Begin
    L := Size;
    if L >= 0 then
      Result := L = 0 else
      begin
        Reset;
        Result := EOF;
      end;
  End;



{                                                                              }
{ TFileStream                                                                  }
{                                                                              }
Constructor TFileStream.Create (const FilePath : String; const CreateFile : Boolean);
  Begin
    inherited Create;
    if CreateFile then
      FFileStream := Classes.TFileStream.Create (FilePath, fmCreate) else
      FFileStream := Classes.TFileStream.Create (FilePath, fmOpenReadWrite);
  End;

Destructor TFileStream.Destroy;
  Begin
    FFileStream.Free;
    FFileStream := nil;
    inherited Destroy;
  End;

Procedure TFileStream.Reset;
  Begin
    inherited Reset;
    FFileStream.Seek (0, soFromBeginning);
  End;

Function TFileStream.Read : TOctet;
  Begin
    FFileStream.ReadBuffer (Result, Sizeof (Result));
  End;

Function TFileStream.Read (const Count : Integer) : String;
var I : Integer;
  Begin
    if Count <= 0 then
      Result := '' else
      begin
        I := Min (Count, OctetsRemaining);
        SetLength (Result, 0);
        SetLength (Result, FFileStream.Read (Result [1], I));
      end;
  End;

Procedure TFileStream.Write (const Data : TOctet);
  Begin
    FFileStream.WriteBuffer (Data, Sizeof (Data));
  End;

Procedure TFileStream.Write (const Data : String);
  Begin
    FFileStream.WriteBuffer (Data [1], Length (Data));
  End;

Function TFileStream.OctetsAvailable : Int64;
  Begin
    Result := FFileStream.Size - FFileStream.Position;
  End;

Function TFileStream.EOF : Boolean;
  Begin
    Result := not (FFileStream.Position < FFileStream.Size);
  End;

Procedure TFileStream.Skip;
  Begin
    if EOF then
      raise EStream.Create ('Skip past EOF');
    FFileStream.Seek (FFileStream.Position + 1, soFromBeginning);
  End;

Procedure TFileStream.Skip (const Count : Integer);
  Begin
    FFileStream.Seek (Min (FFileStream.Position + Count, FFileStream.Size), soFromBeginning);
  End;

Function TFileStream.Peek : TOctet;
var P : Integer;
  Begin
    P := FFileStream.Position;
    FFileStream.Read (Result, Sizeof (Result));
    FFileStream.Seek (P, soFromBeginning);
  End;

Function TFileStream.Peek (const Count : Integer) : String;
var P : Integer;
  Begin
    P := FFileStream.Position;
    Result := Read (Count);
    FFileStream.Seek (P, soFromBeginning);
  End;

Function TFileStream.GetPosition : Int64;
  Begin
    Result := FFileStream.Position;
  End;

Function TFileStream.Size : Int64;
  Begin
    Result := FFileStream.Size;
  End;

Procedure TFileStream.SetPosition (const NewPosition : Int64);
  Begin
    FFileStream.Position := NewPosition;
  End;

Procedure TFileStream.Truncate;
  Begin
    FFileStream.Size := FFileStream.Position;
  End;

Procedure TFileStream.MoveEnd;
  Begin
    FFileStream.Seek (0, soFromEnd);
  End;

Function TFileStream.OctetsRemaining : Int64;
  Begin
    Result := FFileStream.Size - FFileStream.Position;
  End;

Function TFileStream.WaitAvailable (const Count : Integer) : Boolean;
  Begin
    Result := OctetsRemaining >= Count;
  End;

Function TFileStream.WaitSequence (const S : String) : Integer;

  Function ReadBuf (const Len : Integer) : String;
    Begin
      SetLength (Result, Len);
      SetLength (Result, FFileStream.Read (Result [1], Len));
    End;

const BufSize = 1024;

var Buf, B : String;
    P, I   : Integer;

  Begin
    P := FFileStream.Position;
    try
      Buf := ReadBuf (Length (S));
      Result := Pos (S, Buf);
      if Result > 0 then
        exit;
      I := 0;
      While FFileStream.Position < FFileStream.Size do
        begin
          B := ReadBuf (BufSize);
          Buf := Buf + B;
          Result := I + Pos (S, Buf);
          if Result > I then
            exit;
          Delete (Buf, 1, Length (B));
          Inc (I, Length (B));
        end;
      Result := 0;
    finally
      FFileStream.Seek (P, soFromBeginning);
    end;
  End;



{                                                                              }
{ TMemoryStream                                                                }
{                                                                              }
Constructor TMemoryStream.Create (const S : String);
  Begin
    inherited Create;
    FData := S;
    DoReset;
  End;

Procedure TMemoryStream.DoReset;
  Begin
    FPos := 0;
  End;

Procedure TMemoryStream.Reset;
  Begin
    inherited Reset;
    DoReset;
  End;

Function TMemoryStream.Read : TOctet;
  Begin
    if FPos = Length (FData) then
      raise EStream.Create ('Read past EOF');
    Inc (FPos);
    Result := TOctet (FData [FPos]);
  End;

Function TMemoryStream.Read (const Count : Integer) : String;
var L : Integer;
  Begin
    if FPos + Count > Length (FData) then
      L := Length (FData) - FPos else
      L := Count;
    Result := Copy (FData, FPos, L);
    Inc (FPos, L);
  End;

Procedure TMemoryStream.Write (const Data : TOctet);
  Begin
    if FPos = Length (FData) then
      FData := FData + Char (Data) else
      begin
        Inc (FPos);
        FData [FPos] := Char (Data);
      end;
  End;

Procedure TMemoryStream.Write (const Data : String);
var C : Integer;
  Begin
    if FPos = Length (FData) then
      FData := FData + Data else
      begin
        if FPos + Length (Data) > Length (FData) then
          C := Length (FData) - FPos else
          C := Length (Data);
        Move (Data [1], FData [FPos + 1], C);
        Inc (FPos, C);
        if Length (Data) - C > 0 then
          FData := FData + Copy (Data, C + 1, Length (Data) - C);
      end;
  End;

Function TMemoryStream.EOF : Boolean;
  Begin
    Result := FPos >= Length (FData);
  End;

Procedure TMemoryStream.Skip;
  Begin
    if EOF then
      raise EStream.Create ('Skip past EOF');
    Inc (FPos);
  End;

Procedure TMemoryStream.Skip (const Count : Integer);
  Begin
    if FPos + Count > Length (FData) then
      FPos := Length (FData) else
      Inc (FPos, Count);
  End;

Function TMemoryStream.Peek : TOctet;
  Begin
    if EOF then
      raise EStream.Create ('Peek past EOF');
    Result := TOctet (FData [FPos + 1]);
  End;

Function TMemoryStream.Peek (const Count : Integer) : String;
  Begin
    Result := Copy (FData, FPos + 1, Count);
  End;

Function TMemoryStream.GetPosition : Int64;
  Begin
    Result := FPos;
  End;

Procedure TMemoryStream.SetPosition (const NewPosition : Int64);
  Begin
    if NewPosition > Length (FData) then
      raise EStream.Create ('Seek past EOF');
    FPos := NewPosition;
  End;

Procedure TMemoryStream.MoveEnd;
  Begin
    FPos := Length (FData);
  End;

Function TMemoryStream.OctetsRemaining : Int64;
  Begin
    Result := Length (FData) - FPos;
  End;

Function TMemoryStream.OctetsAvailable : Int64;
  Begin
    Result := Length (FData) - FPos;
  End;

Function TMemoryStream.Size : Int64;
  Begin
    Result := Length (FData);
  End;

Procedure TMemoryStream.Truncate;
  Begin
    SetLength (FData, FPos);
  End;

Function TMemoryStream.WaitAvailable (const Count : Integer) : Boolean;
  Begin
    Result := OctetsAvailable >= Count;
  End;

Function TMemoryStream.WaitSequence (const S : String) : Integer;
  Begin
    Result := Pos (S, Copy (FData, FPos + 1, Length (FData) - FPos));
  End;

{ ETStream                                                                    }
Constructor ETStream.Create (const ES : ATStream);
  Begin
    inherited Create;
    FS := ES;
    FS.OnOctetAvailable := OnOctetAvailableHandler;
  End;

Destructor ETStream.Destroy;
  Begin
    FS.Free;
    FS := nil;
    inherited Destroy;
  End;

Procedure ETStream.OnOctetAvailableHandler (const Sender : TObject);
  Begin
    if Assigned (FOnOctetAvailable) then
      FOnOctetAvailable (Sender);
  End;

Function ETStream.Read : TOctet;
  Begin
    Result := FS.Read;
  End;

Function ETStream.Read (const Count : Integer) : String;
  Begin
    Result := FS.Read (Count);
  End;

Procedure ETStream.Skip;
  Begin
    FS.Skip;
  End;

Procedure ETStream.Skip (const Count : Integer);
  Begin
    FS.Skip (Count);
  End;

Function ETStream.Peek : TOctet;
  Begin
    Result := FS.Peek;
  End;

Function ETStream.Peek (const Count : Integer) : String;
  Begin
    Result := FS.Peek (Count);
  End;

Procedure ETStream.Reset;
  Begin
    FS.Reset;
  End;

Function ETStream.GetPosition : Int64;
  Begin
    Result := FS.GetPosition;
  End;

Procedure ETStream.SetPosition (const NewPosition : Int64);
  Begin
    FS.SetPosition (NewPosition);
  End;

Function ETStream.OctetsRemaining : Int64;
  Begin
    Result := FS.OctetsRemaining;
  End;

Function ETStream.Size : Int64;
  Begin
    Result := FS.Size;
  End;

Function ETStream.EOF : Boolean;
  Begin
    Result := FS.EOF;
  End;

Function ETStream.OctetsAvailable : Int64;
  Begin
    Result := FS.OctetsAvailable;
  End;

Function ETStream.WaitAvailable (const Count : Integer) : Boolean;
  Begin
    Result := FS.WaitAvailable (Count);
  End;

Function ETStream.WaitSequence (const S : String) : Integer;
  Begin
    Result := FS.WaitSequence (S);
  End;

Procedure ETStream.MoveEnd;
  Begin
    FS.MoveEnd;
  End;

Procedure ETStream.Truncate;
  Begin
    FS.Truncate;
  End;

Procedure ETStream.Write (const Data : TOctet);
  Begin
    FS.Write (Data);
  End;

Procedure ETStream.Write (const Data : String);
  Begin
    FS.Write (Data);
  End;

end.

