// Event Handlers Demo.
// Given script is based on a Tom Swan program (Delphi 4 Bible)

program WhatsNew; 
const
  maxIndex = 50; { Maximum number of lines visible }
  dx1: Integer = 4;       { "Delta" values for controlling  }
  dy1: Integer = 10;      {  the animation's personality.   }
  dx2: Integer = 3;
  dy2: Integer = 9;

type
  LineRec = record     { Line ends and color }
    X1, Y1, X2, Y2 : Integer;
    Color: TColor;
  end;

type
  TMainForm = class(TForm)
  private
    Timer: TTimer;
    LineArray: array[0 .. maxIndex - 1] of LineRec;
    Index: Integer;     { Index for LineArray }
    Erasing: Boolean;   { True if erasing old lines }
    SignTimer: boolean; 
    procedure InitLineArray;
    procedure MakeNewLine(R: TRect; Index: Integer);
    procedure DrawLine(Index: Integer);
// Event handlers
    procedure MainFormTimer(Sender: TObject);
    procedure MainFormResize(Sender: TObject);
    procedure MainFormPaint(Sender: TObject);
    procedure MainFormKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
    procedure MainFormClose(Sender: TObject; 
                      var Action: TCloseAction);

  public
    constructor Create(AOwner: TComponent); override;
  end;

function Sign(N: Integer): Integer;
begin
  if N < 0 then Sign := -1 else Sign := 1;
end;

{- Initialize globals and LineArray }
procedure TMainForm.Create(AOwner: TComponent);
begin
  inherited;

  Caption := 'PolyFlow';

  Timer := TTimer.Create(Self);
  Timer.Interval := 1;

  with Canvas.Pen do
  begin
    Style := psSolid;
    Width := 3;
    Mode := pmXor;
  end;

  InitLineArray;

  OnClose := MainFormClose;
  OnPaint := MainFormPaint;
  OnResize := MainFormResize;
  OnKeyDown := MainFormKeyDown;
  Timer.OnTimer := MainFormTimer;

  SignTimer := true;
end;

procedure TMainForm.InitLineArray;
var
  I: Integer;
begin
  Index := 0;
  Erasing := False;
  FillChar(LineArray, SizeOf(LineArray), 0);
  for I := 0 to maxIndex - 1 do
    LineArray[I].X1 := -1;
end;

{- Create new line, direction, and color }
procedure TMainForm.MakeNewLine(R: TRect; Index: Integer);
procedure NewCoord(var C, Change: Integer; Max: Integer; var Color: TColor);
var
  Temp, I: Integer;
begin
  Temp := C + Change;
  if (Temp < 0) or (Temp > Max) then
    Change := Sign(-Change) * (3 + Random(12))
  else
    C := Temp;
  I := Random(5);
  case I of
   0:Color := clRed;
   1:Color := clGray;
   2:Color := clYellow;
   3:Color := clWhite;
   4:Color := clGreen;
  end;
end;
begin
  with LineArray[Index] do
  begin
    NewCoord(X1, dx1, R.Right, Color);
    NewCoord(Y1, dy1, R.Bottom, Color);
    NewCoord(X2, dx2, R.Right, Color);
    NewCoord(Y2, dy2, R.Bottom, Color)
  end
end;

{- Draw or erase a line identified by Index }
procedure TMainForm.DrawLine(Index: Integer);
begin
  with Canvas, LineArray[Index] do
  begin
    Canvas.Pen.Color := Color;
    MoveTo(X1, Y1);
    LineTo(X2, Y2);
  end;
end;

{- Draw some lines at each timer interval }
procedure TMainForm.MainFormTimer(Sender: TObject);
var
  I, OldIndex: Integer;
begin
  for I := 1 to 10 do  { 10 = number of lines }
  begin
    OldIndex := Index;
    Inc(Index);
    if Index = maxIndex - 1 then
    begin
      Index := 0;       { Wrap Index around to start }
      Erasing := True;  { True until window size changes }
    end;
    if Erasing then
      DrawLine(Index);  { Erase old line }
    LineArray[Index] := LineArray[OldIndex];
    MakeNewLine(ClientRect, Index);
    DrawLine(Index);    { Draw new line }
  end;
end;

{- Paint or repaint screen using data in LineArray }
procedure TMainForm.MainFormPaint(Sender: TObject);
var
  I: Integer;
begin
  Canvas.Brush.Color := clBlue;
  Canvas.FillRect(ClientRect);

  with Canvas do
    for I := 0 to maxIndex - 1 do
      if LineArray[I].X1 >= 0 then  { Draw non-flagged lines }
        DrawLine(I);
end;

{- Start new lines when window size changes }
procedure TMainForm.MainFormResize(Sender: TObject);
begin
  Canvas.FillRect(ClientRect);
  InitLineArray;  { Erase LineArray and reset globals }
end;

procedure TMainForm.MainFormClose(Sender: TObject; 
                         var Action: TCloseAction);
begin
  writeln('OnClose event was handled. Action = ', Action);
  Timer.Free;
end;

procedure TMainForm.MainFormKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
begin
  writeln('Key =', Key, ' ShiftState=', Shift);
  if Key = 32 then
  begin
    if SignTimer then
      Timer.OnTimer := nil     {pause}
    else
      Timer.OnTimer := MainFormTimer; {resume}
    SignTimer := not SignTimer;            
  end; 
end;

var
  MainForm: TMainForm;
begin
  Randomize;

  MainForm := TMainForm.Create(nil);
  MainForm.Show;

  repeat
  until not MainForm.Visible;

  MainForm.Free;
end.