unit Alien;

{  ******
   *
   * Module:    Alien
   * Author:    Joe Kessler
   *            IntegrationWare - A New Generation of Extraordinary PC Solutions
   *            www.integrationware.com
   *
   * Purpose:
   *
   *    This module defines the look and behavior of the Alien.  The Alien
   *    appears toward the end of the levels, shooting randomly at asteroids
   *    and at Bart.
   *
   ****** }

interface
uses SoundDev, Visible, Missle, Constant, WinTypes, WinProcs, Graphics, Classes,
     Global;

type TAlien = class(TVisibleObject)
   { *** Our constructor *** }
   constructor Create(listObjects: TList; iCollisionID: Integer; fFireProb: Real);

    { Customer movement behavior. }
   procedure Move; override;

   { Missle cannon control. }
   procedure FireMissle;

private
   procedure SetInitialPosition;

private
   m_fFireProb: Real;           { Frequency at which the alien will fire. }
   m_fGeneralDirection: Real;   { General direction of the alien onscreen. }
   m_bBeenOnScreen: Boolean;    { TRUe if the alien has been visible to the user. }
end;

implementation

uses General;

constructor TAlien.Create(listObjects: TList; iCollisionID: Integer; fFireProb: Real);
begin
    // Default processing.
    inherited Create(listObjects, iCollisionID);
    m_fFireProb := fFireProb;
    m_bBeenOnScreen := False;

    { Set initial position.  Notice the the alien always starts off-screen, and
      wanders around firing until he goes off the screen again. }
    SetInitialPosition;
    mtrxTransform.fScale := 2.0;

    { Set up our initial movement. }
    mtrxTransform.fDirection := m_fGeneralDirection;
    mtrxTransform.fSpeed := 2.0;

    { Declare the shape of alien. }
    IncludeShape('Alien', 0, 0, clYellow, True);
 end;

procedure TAlien.SetInitialPosition;
var
   iPosY: Integer;
begin
    { Find a good starting vertical position. }
    iPosY := Round(Random(iScreenHeight) * 0.60 + Random(iScreenHeight) * 0.20);

    { Start on either the left or right side of the screen.  Give the alien an
      initial direction that will take him onto the screen. }
    if Random >= 0.5 then
        begin
        mtrxTransform.SetTranslation(-10, iPosY);
        m_fGeneralDirection := PI / 2;
        end
    else
        begin
        mtrxTransform.SetTranslation(iScreenWidth + 10, iPosY);
        m_fGeneralDirection := 3 * PI / 2;
        end;
end;

procedure TAlien.Move;
var
   bOnScreen: Boolean;
begin
    { Do the usual moving stuff. }
    inherited Move;

    { Occasionally change directions slightly.  The alien will have a general
      direction but will deviation from the course a little bit. }
    if Random < 0.15 then
        mtrxTransform.fDirection := (mtrxTransform.fDirection + Random * 0.40 - Random * 0.40);

    { Fire off a missle at random for maximum craziness. }
    FireMissle;

    { If the alien is goes off the screen then kill him, but don't kill him
      unless he's appeared at least once on the screen. }
    bOnScreen := not bIsRectOffScreen(rectGetBoundingRect);
    if (bOnScreen = False) and (m_bBeenOnScreen = True) then
        KillObject(False, 0, 0, 0, 0);

    // Remember if the alien has come onto the screen yet.
    m_bBeenOnScreen := bOnScreen or m_bBeenOnScreen;
end;

procedure TAlien.FireMissle;
var
    mslFire: TMissle;
begin
    // Fire based upon the current probability.
    if (Random <= m_fFireProb) then
    begin
        { Create the sound of an explosion. }
        g_envEnviron.devGetSoundDevice.PlaySound('Laser2.wav', 0);

        { Create a new missle object. }
        mslFire := TMissle.Create(lstUniverse, iCollisionID);

        { Set initial orientation, size, and direction. }
        mslFire.mtrxTransform.fScale := 0.5;
        mslFire.mtrxTransform.SetTranslation(mtrxTransform.fTranslationX,
                                             mtrxTransform.fTranslationY);
        mslFire.mtrxTransform.fDirection := (Random * TWOPI);

        { Add the missle object to the global object list. }
        lstUniverse.Add(mslFire);
    end;
end;

end.
