unit Asteroid;

{  ******
   *
   * Module:    Asteroid
   * Author:    Joe Kessler
   *            IntegrationWare - A New Generation of Extraordinary PC Solutions
   *            www.integrationware.com
   *
   * Purpose:
   *
   *    This module defines the look and behavior of an asteroid in Rocks.
   *    Asteroid are nothing but floating chunks of rock, but they have a
   *    special look and behavior as defined here.
   *
   ****** }

interface
uses SoundDev, Visible, Vertex, Edge, Constant, General, Score,
     Forms, WinTypes, Graphics, Classes, Global;

type TAsteroid = class(TVisibleObject)
public
   { *** Our constructor *** }
   constructor Create(listObjects: TList; fMass: Real;
                      scrScoreBoard: TScoreBoard; iCollisionID: Integer);

   procedure DefineAsteroidShape;

   procedure Move; Override;
   procedure HandleCollision(objOtherObject: TVisibleObject); Override;

   procedure SplitIntoChunk(fScale: Real; fDirection, fSpeed: Real);

protected
    m_scrScoreBoard: TScoreBoard;   { Reference to the scoreboard object. }
end;

implementation

constructor TAsteroid.Create(listObjects: TList; fMass: Real;
                             scrScoreBoard: TScoreBoard; iCollisionID: Integer);
begin
    inherited Create(listObjects, iCollisionID);

    { Save a reference to the scoreboard object.  This is so that the player can
      be rewarded when the asteroid is destroyed. }
    m_scrScoreBoard := scrScoreBoard;

    { Establish the mass of the asteroid up front, since the shape relies upon
      this.  Then, generate a reasonable asteroid shape algorithmically. }
    mtrxTransform.fScale := fMass;
    DefineAsteroidShape;
end;

procedure TAsteroid.DefineAsteroidShape;
var
   vtxFirst, vtxPrevious, vtxCurrent: TVertex;
   edgeNew: TEdge;
   iVertexCount: Integer;

   fCurrentRadian, fRadianStep: Real;
   fBaseRadius, fActualRadius: Real;
begin
    { Start off at 0 degrees and go in a complete circle.  The asteroid must be
      connected all of the way around. }
    fCurrentRadian := 0.0;

    { Choose a default step angle based upon the mass of the asteroid. }
    { Ensure that the asteroid always has at least four sides. }
    fRadianStep := 0.50 + Random * 0.50;

    { Start off at a random initial rockface height from thecenter. }
    fBaseRadius := 8 + (Random * 5);

    { Count the number of edges created while we go around the center. }
    iVertexCount := 0;
    while (fCurrentRadian < TWOPI) do
    begin
        { Move to a radius relative to the previous one. }
        fActualRadius := fBaseRadius + (Random * fBaseRadius) - (Random * fBaseRadius * 0.75);

        { Create a new vertex.  Record the first one that is created. }
        vtxCurrent := vtxDefineNewVertex(Sin(fCurrentRadian) * fActualRadius,
                                         Cos(fCurrentRadian) * fActualRadius);

        if iVertexCount = 0 then
            vtxFirst := vtxCurrent;

        { Increment the total vertex count. }
        iVertexCount := iVertexCount + 1;

        if iVertexCount > 1 then
        begin
            { Define an edge linking the new vertex with the previous one. }
            edgeNew := edgeDefineNewEdge(vtxPrevious, vtxCurrent, clGray, True);

            { Set up default coloring and visiblity information. }
            edgeNew.m_bVisible := True;
            edgeNew.m_iShapeID := 0;
        end;

        { Move to the next vertex comprising the asteroid. }
        fCurrentRadian := fCurrentRadian + fRadianStep + (Random * 0.10);

        { Store the current vertex for the next iteration. }
        vtxPrevious := vtxCurrent;
    end;

    { Define an edge linking the last vertex with the first one.  This makes }
    { the asteroid always a closed shape. }
    edgeNew := edgeDefineNewEdge(vtxPrevious, vtxFirst, clGray, True);

    { Set up default coloring and visiblity information. }
    edgeNew.m_bVisible := True;
    edgeNew.m_iShapeID := 0;
end;

procedure TAsteroid.Move;
begin
    { Do the usual moving stuff. }
    inherited Move;

    { Wrap around if we've gone off the screen. }
    Wrap;
end;

procedure TAsteroid.HandleCollision(objOtherObject: TVisibleObject);
var
    fMassRemaining: Real;

    fScaleChunk1, fScaleChunk2: Real;
    fSpeed, fSpeedChunk1, fSpeedChunk2: Real;
    fDirection, fDirectionChunk1, fDirectionChunk2: Real;

    astNew1, astNew2: TAsteroid;
    fScoreIncrease: Real;
begin
    { Find the total mass of the asteroid left after the blast.  This can be
      anywhere between 80%-95% of the original mass, }
    fMassRemaining := mtrxTransform.fScale * (0.80 + Random * 0.15);

    { Split the remaining mass into two pieces.  The first chunk will be between
      25%-50% of the original mass.  The second chunk will be the difference. }
    fScaleChunk1 := (Random * 0.25 + 0.25) * fMassRemaining;
    fScaleChunk2 := fMassRemaining - fScaleChunk1;

    { Get the speed of the new chunks as a combination of the current object's
      speed, plus some energy from the object we've collided with. }
    fSpeed := mtrxTransform.fSpeed + objOtherObject.mtrxTransform.fSpeed / 3;
    fSpeed := fMax(fSpeed, 1.0);
    fSpeed := fMin(fSpeed, 3.0);

    fSpeedChunk1 := fSpeed * (0.5 + Random * 0.4);
    fSpeedChunk2 := fSpeed * (0.5 + Random * 0.4);

    { Set the movement direction of the chunks based upon asteroid's current
      direction. }
    fDirection := mtrxTransform.fDirection;
    fDirectionChunk1 := fDirection + (0.30 + Random * 0.40);
    fDirectionChunk2 := fDirection - (0.30 + Random * 0.40);

    if (fScaleChunk1 >= 0.5) then
        SplitIntoChunk(fScaleChunk1, fDirectionChunk1, fSpeedChunk1);

    if (fScaleChunk2 >= 0.5) then
        SplitIntoChunk(fScaleChunk2, fDirectionChunk2, fSpeedChunk2);

    { Destroy the asteroid that was just split apart, since it will now be
      replaced by one or two smaller chunks. }
    KillObject(True, 3 + Random(4), 0.15, 0, 0.5);

    { Increase the player's score by a fixed amount. }
    m_scrScoreBoard.iCurrentScore := m_scrScoreBoard.iCurrentScore + 50;

     { Make an exploding sound. }
    g_envEnviron.devGetSoundDevice.PlaySound('HAsteroid.wav', 0);
end;

procedure TAsteroid.SplitIntoChunk(fScale: Real; fDirection, fSpeed: Real);
var
    astNew: TAsteroid;
begin
    { Create a new asteroid chunk of the requested size! }
    astNew := TAsteroid.Create(lstUniverse, fScale, m_scrScoreBoard, iCollisionID);

    { Establish the position of the new chunk as given. }
    astNew.mtrxTransform.SetTranslation(mtrxTransform.fTranslationX, mtrxTransform.fTranslationY);

    { Set up the chunk's direction and speed. }
    astNew.mtrxTransform.fDirection := fDirection;
    astNew.mtrxTransform.fSpeed := fSpeed;

    { Add the chunk to the global object list. }
    lstUniverse.Add(astNew);
end;

end.

