unit Keys;

{  ******
   *
   * Module:    Keys
   * Author:    Joe Kessler
   *            IntegrationWare - A New Generation of Extraordinary PC Solutions
   *            www.integrationware.com
   *
   * Purpose:
   *
   *    Interface for customizing player controls.
   *
   * Details:
   *
   *    This form allows the user to customize the keyboard layout.  The form
   *    can be called up at any time (even in the middle of a game) to change
   *    which keys will control Bart.  Note that this customization is possible
   *    because of the architecture of the input objects.  The inputs that
   *    game objects receive are connected only indirectly to actual user inputs.
   *
   ****** }

interface

uses
  RocksKbd, InpCodes,
  SysUtils, WinProcs, WinTypes, Messages, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, ExtCtrls;

type
  TKeySelection = class(TForm)
    GroupBox1: TGroupBox;
    GroupBox2: TGroupBox;
    Label1: TLabel;
    Label2: TLabel;
    m_cmdOK: TButton;
    m_cmdCancel: TButton;
    m_cboTurnLeft: TComboBox;
    m_cmdDefault: TButton;
    Image1: TImage;
    Label3: TLabel;
    m_cboTurnRight: TComboBox;
    Image2: TImage;
    Label4: TLabel;
    m_cboThrust: TComboBox;
    Image3: TImage;
    Label5: TLabel;
    m_cboFire: TComboBox;
    Image4: TImage;
    Label6: TLabel;
    m_cboHyperSpace: TComboBox;
    Image5: TImage;
    Label7: TLabel;
    procedure m_cmdOKClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure m_cmdCancelClick(Sender: TObject);
    procedure m_cmdDefaultClick(Sender: TObject);
    procedure m_cboTurnLeftEnter(Sender: TObject);
    procedure m_cboTurnRightEnter(Sender: TObject);
    procedure m_cboThrustEnter(Sender: TObject);
    procedure m_cboFireEnter(Sender: TObject);
    procedure m_cboHyperSpaceEnter(Sender: TObject);
    procedure m_cboTurnLeftKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
  private
    { Private declarations }
   procedure FillComboWithKeyChoices(cboKeys: TComboBox);

  public

    { Method to initialize the form to customize a specific input object. }
    procedure UseKeyboard(kbRocks: TRocksKeyboard);

  private

    { Methods to select keys based upon text our an input code. }
    procedure SelectComboItem(cboList: TComboBox; iInputCode: Integer);
    procedure SelectComboString(cboList: TComboBox; szSearchStr: String);

  private

    m_kbRocks: TRocksKeyboard;      { Input object being customized. }
    m_cboLastActive: TComboBox;     { Last combo box to have focus. }
end;

var
  KeySelection: TKeySelection;

implementation

uses General;

{$R *.DFM}

procedure TKeySelection.m_cmdOKClick(Sender: TObject);
begin
   { Stash the new associations in the keyboard input object.  }
   m_kbRocks.AssociateKey(m_kbRocks.wGetKeyCodeFromText(m_cboTurnLeft.Text), BART_TURN_LEFT);
   m_kbRocks.AssociateKey(m_kbRocks.wGetKeyCodeFromText(m_cboTurnRight.Text), BART_TURN_RIGHT);
   m_kbRocks.AssociateKey(m_kbRocks.wGetKeyCodeFromText(m_cboThrust.Text), BART_THRUST);
   m_kbRocks.AssociateKey(m_kbRocks.wGetKeyCodeFromText(m_cboFire.Text), BART_FIRE);
   m_kbRocks.AssociateKey(m_kbRocks.wGetKeyCodeFromText(m_cboHyperSpace.Text), BART_HYPERSPACE);

   { Close up the form. }
   Close;
end;

procedure TKeySelection.FormCreate(Sender: TObject);
begin
   FillComboWithKeyChoices(m_cboTurnLeft);
   FillComboWithKeyChoices(m_cboTurnRight);
   FillComboWithKeyChoices(m_cboThrust);
   FillComboWithKeyChoices(m_cboFire);
   FillComboWithKeyChoices(m_cboHyperSpace);

   m_cboLastActive := nil;
end;

procedure TKeySelection.FillComboWithKeyChoices(cboKeys: TComboBox);
var
   iIndex: Integer;
begin
   { Add the letters A thru Z. }
   for iIndex := ord('A') to ord('Z') do
      cboKeys.Items.Add(chr(iIndex));

   { Add numeric values. }
   for iIndex := 0 to 9 do
      cboKeys.Items.Add(IntToStr(iIndex));

   for iIndex := 0 to 9 do
      cboKeys.Items.Add('NumPad ' + IntToStr(iIndex));

   for iIndex := 1 to 12 do
      cboKeys.Items.Add('F' + IntToStr(iIndex));

   { Add other misc. control keys. }
   cboKeys.Items.Add('SpaceBar');
   cboKeys.Items.Add('Enter');
   cboKeys.Items.Add('Tab');
   cboKeys.Items.Add('Caps Lock');
   cboKeys.Items.Add('Shift');
   cboKeys.Items.Add('Left Arrow');
   cboKeys.Items.Add('Right Arrow');
   cboKeys.Items.Add('Up Arrow');
   cboKeys.Items.Add('Down Arrow');
   cboKeys.Items.Add('Insert');
   cboKeys.Items.Add('Delete');
   cboKeys.Items.Add('Home');
   cboKeys.Items.Add('End');
   cboKeys.Items.Add('Page Up');
   cboKeys.Items.Add('Page Down');
   cboKeys.Items.Add('Num Lock');
   cboKeys.Items.Add('Scroll Lock');
   cboKeys.Items.Add('Pause');
end;

procedure TKeySelection.UseKeyboard(kbRocks: TRocksKeyboard);
var
   wKeyCode: Word;
begin
   { Stash a reference to the keyboard input object. }
   m_kbRocks := kbRocks;

   { Show the current selections in the text areas. }
   SelectComboItem(m_cboTurnLeft, BART_TURN_LEFT);
   SelectComboItem(m_cboTurnRight, BART_TURN_RIGHT);
   SelectComboItem(m_cboThrust, BART_THRUST);
   SelectComboItem(m_cboFire, BART_FIRE);
   SelectComboItem(m_cboHyperSpace, BART_HYPERSPACE);
end;

procedure TKeySelection.SelectComboItem(cboList: TComboBox; iInputCode: Integer);
var
   szKeyText: String;
begin
   { Get text corresponding to the given input code. }
   szKeyText := m_kbRocks.szGetTextFromKeyCode(m_kbRocks.wGetKeyCodeFromInput(iInputCode));

   { Select the corresponding listbox item. }
   SelectComboString(cboList, szKeyText);
end;

procedure TKeySelection.SelectComboString(cboList: TComboBox; szSearchStr: String);
var
   iNewIndex: Integer;
begin
   { Get the index of the given string. }
   iNewIndex := cboList.Items.IndexOf(szSearchStr);

   { If it was located, then make it the currently selected string. }
   if iNewIndex >= 0 then
      cboList.ItemIndex := iNewIndex;
end;

procedure TKeySelection.m_cmdCancelClick(Sender: TObject);
begin
   Close;
end;

procedure TKeySelection.m_cmdDefaultClick(Sender: TObject);
var
   szKeyStr: String;
   iInputCode: Integer;
begin
   { If no combo box has yet been focused then ignore this button. }
   if m_cboLastActive = nil then
      Exit;

   { Determine which input we're getting the default of. }
   if m_cboLastActive = m_cboTurnLeft then iInputCode := BART_TURN_LEFT;
   if m_cboLastActive = m_cboTurnRight then iInputCode := BART_TURN_RIGHT;
   if m_cboLastActive = m_cboThrust then iInputCode := BART_THRUST;
   if m_cboLastActive = m_cboFire then iInputCode := BART_FIRE;
   if m_cboLastActive = m_cboHyperSpace then iInputCode := BART_HYPERSPACE;

   { Get the default key mapping for the given input.  }
   szKeyStr:= m_kbRocks.szGetTextFromKeyCode(m_kbRocks.wGetDefaultMapping(iInputCode));
   SelectComboString(m_cboLastActive, szKeyStr);
end;

procedure TKeySelection.m_cboTurnLeftEnter(Sender: TObject);
begin
   m_cboLastActive := m_cboTurnLeft;
end;

procedure TKeySelection.m_cboTurnRightEnter(Sender: TObject);
begin
   m_cboLastActive := m_cboTurnRight;
end;

procedure TKeySelection.m_cboThrustEnter(Sender: TObject);
begin
   m_cboLastActive := m_cboThrust;
end;

procedure TKeySelection.m_cboFireEnter(Sender: TObject);
begin
   m_cboLastActive := m_cboFire;
end;

procedure TKeySelection.m_cboHyperSpaceEnter(Sender: TObject);
begin
   m_cboLastActive := m_cboHyperSpace;
end;

procedure TKeySelection.m_cboTurnLeftKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
var
   szKeyStr: String;
begin
   { Get the text version of the given key code. }
   szKeyStr:= m_kbRocks.szGetTextFromKeyCode(Key);

   { If there is a translation then change the current combo box text. }
   if szKeyStr <> '' then
      SelectComboString(m_cboTurnLeft, szKeyStr);
end;

end.
