unit MatCtrl;

{  ******
   *
   * Module:    MatCtrl
   * Author:    Joe Kessler
   *            IntegrationWare - A New Generation of Extraordinary PC Solutions
   *            www.integrationware.com
   *
   * Purpose:
   *
   *    This module builds upon the functionality of TMatrix2D by encapsulating
   *    qualities such as speed, rotation, and direction.  These are changes in
   *    a matrix's values that occur over time.
   *
   ****** }

interface

uses Matrix2D;

type TTemporalMatrix2D = class(TMatrix2D)
    public
        { Method to apply matrix change values. }
        procedure Move;

    private
        { Transformation increment values. }
        m_fDirection: Real;             { Direction in radians. }
        m_fRotation: Real;              { Current rotation in radians. }
        m_fSpeed: Real;                 { Current speed in pixels/tick. }
        m_fChangeX, m_fChangeY: Real;   { Position change per tick as X & Y deltas. }

        { Methods for getting/setting property values. }
        procedure SetDirection(fRadians: Real);
        procedure SetSpeed(fSpeed: Real);

    public

        { Exposed properties. }
        property fDirection: Real       Read m_fDirection           Write SetDirection;
        property fSpeed: Real           Read m_fSpeed               Write SetSpeed;
        property fRotation: Real        Read m_fRotation            Write m_fRotation;
end;

implementation

procedure TTemporalMatrix2D.Move;
begin
    { Change our orientation by the current rotation factor. }
    fOrientation := fOrientation + m_fRotation;

    { Set the new translation value depending upon whether the change value is nonzero. }
    SetTranslation(fTranslationX + m_fChangeX, fTranslationY - m_fChangeY);
end;

procedure TTemporalMatrix2D.SetDirection(fRadians: Real);
begin
    { Store our new direction in radians. }
    m_fDirection := fRadians;

    { For speed, convert the direction in X & Y deltas only once here. }
    m_fChangeX := sin(m_fDirection) * m_fSpeed;
    m_fChangeY := cos(m_fDirection) * m_fSpeed;
end;

procedure TTemporalMatrix2D.SetSpeed(fSpeed: Real);
begin
    { Record the new speed in pixels. }
    m_fSpeed := fSpeed;

    { When speed changes, the X & Y deltas change per tick. }
    m_fChangeX := sin(m_fDirection) * m_fSpeed;
    m_fChangeY := cos(m_fDirection) * m_fSpeed;
end;

end.
