unit Matrix2D;

{  ******
   *
   * Module:    Matrix2D
   * Author:    Joe Kessler
   *            IntegrationWare - A New Generation of Extraordinary PC Solutions
   *            www.integrationware.com
   *
   * Purpose:
   *
   *    This module encapsulates scaling, rotation, and movement of vertices
   *    using 2-dimensional matrix math.
   *
   ****** }

interface

type TMatrix2D = class(TObject)
    public
        constructor Create;

        { Methods for operating upon the matrix. }
        procedure Reset;
        procedure SetTranslation(iX, iY: Real);

        { Method to determine if the matrix has changed. }
        function bTransformChanged: Boolean;
        function bTranslateChanged: Boolean;
        function bAppearanceChanged: Boolean;

        { Method for transforming a single vertex through the matrix. }
        procedure TransformVertex(fInputX, fInputY: Real; var fOutputX, fOutputY: Real);

        { Method for translating a local vertex to a new scaling system. }
        procedure TranslateVertex(fInputX, fInputY: Real; var fOutputX, fOutputY: Real);

        { Method for refreshing the matrix with current values. }
        procedure ConstructMatrix;
        procedure RecordMatrixState;

    private

        { Master tranformation matrix. }
        m_afMasterMatrix: array[0..2, 0..2] of Real;

        { Scaling and rotation matrices. }
        m_afScalingMatrix: array[0..2, 0..2] of Real;
        m_afRotationMatrix: array[0..2, 0..2] of Real;

        { TRUE when the matrix has been initialized. }
        m_bMatrixInitialized: Boolean;

        { Current Transformation Values. }
        m_fTranslationX, m_fTranslationY: Real;
        m_fOrientation: Real;
        m_fScale: Real;

        { Previous transformation values. }
        m_fLastTranslationX, m_fLastTranslationY: Real;
        m_fLastOrientation: Real;
        m_fLastScale: Real;

        { Methods for getting/setting property values. }
        procedure SetOrientation(fRadians: Real);
        procedure SetScale(fScale: Real);
        procedure SetTranslationX(iX: Real);
        procedure SetTranslationY(iY: Real);
        function fGetTranslationX: Real;
        function fGetTranslationY: Real;

    public

        { Exposed properties. }
        property fOrientation: Real     Read m_fOrientation     Write SetOrientation;
        property fScale: Real           Read m_fScale           Write SetScale;
        property fTranslationX: Real    Read m_fTranslationX    Write SetTranslationX;
        property fTranslationY: Real    Read m_fTranslationY    Write SetTranslationY;
end;

implementation

constructor TMatrix2D.Create;
begin
    inherited Create;

    { Establish default tranformation values. }
    m_fTranslationX := 0;
    m_fTranslationY := 0;
    SetOrientation(0);
    SetScale(1.0);

    { Record our initial state. }
    RecordMatrixState;

    { Initially, the matrix has not been initialized. }
    m_bMatrixInitialized := False;
end;

procedure TMatrix2D.Reset;
begin
    { Reset master matrix to an identity state. }
    m_afMasterMatrix[0][0] := 1.0; m_afMasterMatrix[0][1] := 0.0; m_afMasterMatrix[0][2] := 0.0;
    m_afMasterMatrix[1][0] := 0.0; m_afMasterMatrix[1][1] := 1.0; m_afMasterMatrix[1][2] := 0.0;
    m_afMasterMatrix[2][0] := 0.0; m_afMasterMatrix[2][1] := 0.0; m_afMasterMatrix[2][2] := 1.0;
end;

procedure TMatrix2D.SetTranslation(iX, iY: Real);
begin
    { Record the new translation values. }
    m_fTranslationX := iX;
    m_fTranslationY := iY;
end;

procedure TMatrix2D.SetTranslationX(iX: Real);
begin
    { Record the new translation values. }
    m_fTranslationX := iX;
end;

procedure TMatrix2D.SetTranslationY(iY: Real);
begin
    { Record the new translation values. }
    m_fTranslationY := iY;
end;

procedure TMatrix2D.SetOrientation(fRadians: Real);
begin
    { Record the new orientation. }
    m_fOrientation := fRadians;

    { Initialize the rotation matrix. }
    m_afRotationMatrix[0,0] := cos(m_fOrientation); m_afRotationMatrix[0,1] := -sin(m_fOrientation); m_afRotationMatrix[0,2]:=0;
    m_afRotationMatrix[1,0] := sin(m_fOrientation); m_afRotationMatrix[1,1] := cos(m_fOrientation);  m_afRotationMatrix[1,2]:=0;
    m_afRotationMatrix[2,0] := 0;                   m_afRotationMatrix[2,1] := 0;                    m_afRotationMatrix[2,2]:=1;
end;

procedure TMatrix2D.SetScale(fScale: Real);
begin
    { Record the new scale. }
    m_fScale := fScale;

    { Initialize the scaling matrix. }
    m_afScalingMatrix[0,0] := m_fScale; m_afScalingMatrix[0,1] := 0;        m_afScalingMatrix[0,2]:=0;
    m_afScalingMatrix[1,0] := 0;        m_afScalingMatrix[1,1] := m_fScale; m_afScalingMatrix[1,2]:=0;
    m_afScalingMatrix[2,0] := 0;        m_afScalingMatrix[2,1] := 0;        m_afScalingMatrix[2,2]:=1;
end;

function TMatrix2D.fGetTranslationX: Real;
begin
    fGetTranslationX := m_fTranslationX;
end;

function TMatrix2D.fGetTranslationY: Real;
begin
    fGetTranslationY := m_fTranslationY;
end;

procedure TMatrix2D.TransformVertex(fInputX, fInputY: Real; var fOutputX, fOutputY: Real);
begin
    fOutputX := fInputX * m_afMasterMatrix[0, 0] + fInputY * m_afMasterMatrix[1, 0] + m_afMasterMatrix[2, 0];
    fOutputY := fInputX * m_afMasterMatrix[0, 1] + fInputY * m_afMasterMatrix[1, 1] + m_afMasterMatrix[2, 1];
end;

procedure TMatrix2D.TranslateVertex(fInputX, fInputY: Real; var fOutputX, fOutputY: Real);
begin
    fOutputX := m_fTranslationX + fInputX;
    fOutputY := m_fTranslationY - fInputY;
end;

procedure TMatrix2D.ConstructMatrix;
var
    i, j, k: Integer;
begin
    { Start back with an identity matrix. }
    Reset;

    { Concatenate the rotation and scaling matrices into the master. }
    for i := 0 to 2 do
        for j:= 0 to 2 do
        begin
            m_afMasterMatrix[i,j] := 0;

            for k := 0 to 2 do
                m_afMasterMatrix[i,j] := m_afMasterMatrix[i,j] + m_afScalingMatrix[i,k] * m_afRotationMatrix[k,j];
        end;

    { The matrix has now been constructed at least once. }
    m_bMatrixInitialized := True;
end;

function TMatrix2D.bTransformChanged: Boolean;
begin
    { Check if the orientation of scaling has changed. }
    Result := (m_fLastOrientation <> m_fOrientation) or
              (m_fLastScale <> m_fScale) or
              (m_bMatrixInitialized = False);
end;

function TMatrix2D.bTranslateChanged: Boolean;
begin
    { Check if the position has changed. }
    Result := (m_fLastTranslationX <> m_fTranslationX) or
              (m_fLastTranslationY <> m_fTranslationY) or
              (m_bMatrixInitialized = False);
end;

function TMatrix2D.bAppearanceChanged: Boolean;
begin
    { Check if the position, orientation, or scaling has changed. }
    Result := (bTransformChanged = True) or (bTranslateChanged = True);
end;

procedure TMatrix2D.RecordMatrixState;
begin
    { Record transformation parms for later comparision. }
    m_fLastOrientation := m_fOrientation;
    m_fLastScale := m_fScale;
    m_fLastTranslationX := m_fTranslationX;
    m_fLastTranslationY := m_fTranslationY;
end;

end.
