unit StateCh;

{  ******
   *
   * Module:    StateCh
   * Author:    Joe Kessler
   *            IntegrationWare - A New Generation of Extraordinary PC Solutions
   *            www.integrationware.com
   *
   * Purpose:
   *
   *    The TStateChannel class is a single state channel as used by the
   *    TMultiChannelStateAware class.  It uses the functionality of the
   *    TStateAware class, but augments it by recording its activities to
   *    be monitored by derived classes.
   *
   ****** }

interface
uses State;

{ Queue activity constants. }
const ACTION_STATE_NONE = 0;
const ACTION_STATE_ENTERED = 1;
const ACTION_STATE_EXITED = 2;
const ACTION_STATE_PROCESSED = 3;

{ Class definition for a state channel of a multi-channel state object. }
type TStateChannel = class(TStateAware)
    public

        { Virtual method called to process object movements. }
        procedure Move; Override;

        { Methods to get the last state change performed. }
        function iGetLastAction: Integer;
        function pstGetLastSubject: PState;

    protected

        { Virtual methods for customizing state behavior. }
        procedure EnterState(pstCurrent: PState); Override;
        procedure ExitState(pstCurrent: PState); Override;
        procedure ProcessState(pstCurrent: PState); Override;

    private

        { Values to record what happened during the last state movement. }
        m_iStateAction: Integer;
        m_stSubject: TState;
end;

implementation

uses General;

procedure TStateChannel.Move;
begin
    { Reset our activity flags for this movement iteration. }
    m_iStateAction := ACTION_STATE_NONE;

    { Do the default movement activity. }
    inherited Move;
end;

procedure TStateChannel.EnterState(pstCurrent: PState);
begin
    { Record the state that was entered. }
    m_iStateAction := ACTION_STATE_ENTERED;
    m_stSubject := pstCurrent^;

    inherited EnterState(pstCurrent);
end;

procedure TStateChannel.ExitState(pstCurrent: PState);
begin
    { Record the state that was exited. }
    m_iStateAction := ACTION_STATE_EXITED;
    m_stSubject := pstCurrent^;

    inherited ExitState(pstCurrent);
end;

procedure TStateChannel.ProcessState(pstCurrent: PState);
begin
    { Record the state that was processed. }
    m_iStateAction := ACTION_STATE_PROCESSED;
    m_stSubject := pstCurrent^;

    inherited ProcessState(pstCurrent);
end;

function TStateChannel.iGetLastAction: Integer;
begin
    Result := m_iStateAction;
end;

function TStateChannel.pstGetLastSubject: PState;
begin
    Result := @m_stSubject;
end;

end.
