/* Serializer 
 * Copyright (C) 1999 Dmitry Rogatkin.  All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 *  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 *  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 *  ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 *  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 *  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 *  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package rogatkin;

import java.util.Hashtable;
import java.util.Enumeration;
import java.util.StringTokenizer;
import java.util.NoSuchElementException;
import java.io.*;
/**
  Parser of Windows style ini file
  [section]
  parameter=value,...
  */

public class Serializer {
    final static String HOMEDIR = "po.home";
    final static String INIEXT     = ".ini";
    final static char COMMENT      = '#';
    final static char ENDLN        = '\n';
    final static char STARTSEC     = '[';
    final static char ENDSEC       = ']';
    final static char EQ           = '=';
    final static char COMMA        = ',';

    Hashtable directory;
    String    inifolder;
    String    programname;
    
    public Serializer(String name) {
	this.programname = name;
        inifolder = System.getProperty(HOMEDIR);
        if (inifolder == null)
            inifolder = System.getProperty("user.dir");
        if (inifolder == null)
            inifolder = System.getProperty("user.home");
        directory = new Hashtable();
    }

    public String getHomeDirectory() {
        return inifolder;
    }

    public Object getProperty(Object topic, Object entry) {
        Hashtable chapter = (Hashtable)directory.get(topic);
        if (chapter == null)
            return null;
        return chapter.get(entry);
    }

    /** set and remove a property as well, if property value is null
    */
    public void setProperty(Object topic, Object entry, Object value) {
        Hashtable chapter = (Hashtable)directory.get(topic);
        if (chapter == null) {
            if (value == null)
                return ; // nothing to do here
            chapter = new Hashtable();
            directory.put(topic, chapter);
        }
        if (value == null)
            chapter.remove(entry);
        else
            chapter.put(entry, value);
    }

    public void load() {
        String line, el;
        Hashtable chapter = null;
        Object [] params, tempa;
        StringBuffer sb = new StringBuffer();
        try {
            BufferedReader fr = new BufferedReader(new FileReader(new File(inifolder, programname+INIEXT)));
            boolean eof;
            do {
                eof = readString(sb, fr);
                //line = sb.toString();
		if (sb.length() > 0) {
		    if (sb.charAt(0) == COMMENT)
			continue; // skip comment
		    if (sb.charAt(0) == STARTSEC) {
                        line = sb.toString().substring(1, sb.length()-1);
                        chapter = (Hashtable)directory.get(line);
                        if (chapter == null) {
                            chapter = new Hashtable();
                            directory.put(line, chapter);
                        }
                    } else {
                        if (chapter == null)
                            continue;
                        StringTokenizer st = new StringTokenizer(sb.toString(), ""+EQ);
                        if (st.hasMoreTokens()) {
                            line = st.nextToken();
                            params = new Object[0];
                            try {
                                el = st.nextToken(""+COMMA).substring(1); // eat '='
                                params = new Object[1];
                                try {
                                    params[0] = new Integer(el);
                                } catch(NumberFormatException nfe) {
                                    params[0] = el;
                                }
                            } catch (NoSuchElementException nsee) {
                                params = new Object[0];
                            }
                            while(st.hasMoreTokens()) {
                                el = st.nextToken();
                                tempa = new Object[params.length+1];
                                System.arraycopy(params, 0, tempa, 0, params.length);
                                params = tempa;
                                try {
                                    params[params.length-1] = new Integer(el);
                                } catch(NumberFormatException nfe) {
                                    params[params.length-1] = el;
                                }
                            }
                            if (params.length == 1)
                                chapter.put(line, params[0]);
                            else if (params.length > 1)
                                chapter.put(line, params);
                        }
                    }
		}
            } while (eof);
            fr.close();
        } catch (IOException ioe) {
        }
    }
    
    boolean readString(StringBuffer sb, BufferedReader rd) {
        sb.setLength(0);
        int c;
        for(;;) {
            try {
                c = rd.read();
                if (c == '\n')
                    break;
                if (c == '\r')
                    continue;
                if (c == -1) // eof
                    return false;
                sb.append((char)c);
            } catch (IOException ioe) {
                return false;
            }
        }
        return true;
    }

    public void save() {
        try {
            BufferedWriter fw = new BufferedWriter(new FileWriter(new File(inifolder, programname+INIEXT)));
            fw.write(COMMENT+programname+ENDLN);
            Enumeration ed = directory.keys();
            Hashtable chapter;
            Object chaptername, entryname;
            Object entry;
            while (ed.hasMoreElements()) {
                chaptername = ed.nextElement();
                if (chaptername instanceof String)
                    fw.write(STARTSEC+(String)chaptername+ENDSEC);
                else
                    fw.write(STARTSEC+chaptername.toString()+ENDSEC);
                fw.newLine();
                chapter = (Hashtable)directory.get(chaptername);
                Enumeration ec = chapter.keys();
                while (ec.hasMoreElements()) {
                    entryname = ec.nextElement();
                    if (entryname instanceof String)
                        fw.write((String)entryname+EQ);
                    else
                        fw.write(entryname.toString()+EQ);
                    entry = chapter.get(entryname);
                    if (entry instanceof Object[]) {
                        fw.write(((Object[])entry)[0].toString());
                        for (int i = 1; i < ((Object [])entry).length; i++) {
                            fw.write(COMMA+((Object[])entry)[i].toString());
                        }
                        fw.newLine();
                    } else {
                        if (entry instanceof String)
                            fw.write((String)entry+ENDLN);
                        else
                            fw.write(entry.toString()+ENDLN);
                    }
                }
            }
	    fw.flush();
            fw.close();
        } catch (IOException ioe) {
        }
    }

    public static String arrayToString(Object o) {
	String result;
        if (o != null) {
            if (o instanceof Object[]) {
                result = (String)((Object[])o)[0];
                for(int j=1; j<((Object[])o).length; j++)
                    result += ","+(String)((Object[])o)[j];
            } else
                result = (String)o;
	    return result;
	}
	return "";
    }

    public static int getInt(Object o, int defval) {
        if (o != null && o instanceof Integer)
            return ((Integer)o).intValue();
        return defval;
    }
}