// <><><><><><><><><><><><><>  Plot3D.h  <><><><><><><><><><><><><><>
//
// ----------------------------------------------------
// (C) Copyright Clark Dailey, Inc. 1999
//
// C++ Class Name:
//    Plot3D
//
// Purpose:
//    Used as a base class for several Plot3D classes
//
// ----------------------------------------------------

#ifndef	_PLOT3D_H_	 	// avoid repeated including
#define	_PLOT3D_H_	1	// has been included

#include <math.h>
#include <assert.h>

#ifdef _DEBUG
 #define PLOT_ASSERT(v)	assert(v)
#else
 #define PLOT_ASSERT(v)	// do nothing
#endif
#define	DBGPLOT	TRACE	// TRACE for Windows95, printf for DOS

// degrees/radians conversions
inline double Deg2Rad(double deg) { return((deg*2.*3.1415925359)/360.);   }
inline double Rad2Deg(double rad) { return((rad*360.)/(2.*3.1415925359)); }
#define	PLOT3D_DBL_MAX	(1.e300)	// largest double value

// color values
#define	PLOT3D_MIN_COLOR	(0)
#define	PLOT3D_MAX_COLOR	(100)
#define	WIN_MIN_COLOR		(0)
#define	WIN_MAX_COLOR		(255)
// convert plot color component to windows color component
#define	PCOLOR_TO_WINCOLOR(d)		((int)((double)d*(double)WIN_MAX_COLOR/(double)PLOT3D_MAX_COLOR))

// disable annoying compiler warnings
#pragma warning( disable : 4100 )  // unreferenced formal parameter

// ----------
// axis info
// ----------
typedef struct axis_s
{
	unsigned nxticks;		// number of x tick marks
	unsigned nyticks;		// number of y tick marks
	unsigned nzticks;		// number of z tick marks
	long	 axis_color;	// color of axis
	char	 font_name[128];// name of font (needs to be True-Type for rotating)
	int		 font_height;	// pixels
	int		 font_width;	// pixels
	unsigned tick_length;	// 1-100
	unsigned tick_spacing;	// 1-100
} AXIS_PLOT_INFO;


// ------------------------------------------------------------------------------------------
// abstract class - must be derived from to use
class Plot3D // base class for several Plot3D classes
{
public:
	Plot3D() { Init(); }	// constructor
	virtual ~Plot3D() { }	// destructor

	// pure virtual functions must be defined by derived class
	virtual void MoveTo(int xv,int yv)=0;	// move to a location
	virtual void DrawTo(int xv,int yv)=0;	// draw to a location
	virtual void SetPixelColor(int xv,int yv,long color)=0;	// set the color of a pixel
	virtual void PlotText(AXIS_PLOT_INFO* pAxisInfo,double degrees,int xv,int yv,CString text)=0;	// draw text at location
	// justification: 1=left, 2=center, 3=right

	// overridable color functions
	virtual void ColorTo(long color) { }
	virtual long Color(double x,double y,double z) { return(0); }

	// settings
	int SetRanges(double xmin=-1.,double xmax=1.,
				  double ymin=-1.,double ymax=1.,
				  double zmin=-1.,double zmax=1.)
	{
		int rc=0;
		if (xmin>xmax || ymin>ymax || zmin>zmax) return(1);
		m_xmin = xmin;
		m_xmax = xmax;
		m_ymin = ymin;
		m_ymax = ymax;
		m_zmin = zmin;
		m_zmax = zmax;
		m_xmid = (m_xmin + m_xmax)/2.;
		m_ymid = (m_ymin + m_ymax)/2.;
		m_zmid = (m_zmin + m_zmax)/2.;
		return(rc);
	}

	// set angle viewing position
	int SetViewingPosition(double xytilt=60.,double zrotate=45.)
	{
		m_xytilt = xytilt;
		m_zrotate = zrotate;
		return(0);
	}

	// set video resolution
	int SetVideoRes(int xvmin=0,   int yvmin=0,
					int xvmax=640, int yvmax=480,double scale=1.0)
	{
		if (xvmin>=xvmax || yvmin>=yvmax) return(1);
		if (scale>1.0) scale=1.0;
		if (scale<.01) scale=.01;
		m_xvmin = xvmin;
		m_yvmin = yvmin;
		m_xvmax = xvmax;
		m_yvmax = yvmax;
		m_xvmid = (m_xvmin + m_xvmax)/2;
		m_yvmid = (m_yvmin + m_yvmax)/2;
		m_scale = scale;
		return(0);
	}

	// set axis parameters
	void SetAxisParameters(unsigned ShowAxis=0,AXIS_PLOT_INFO* pAxisInfo=0)
	{
		m_show_axis  = ShowAxis;
		if (pAxisInfo==0) return;
	
		memcpy((char*)&m_axis,(char*)pAxisInfo,sizeof(m_axis));
	}

	// set color parameters
	void SetColorMode(int ShowColor=0) { m_is_color = ShowColor; }
	void SetMonochromePen(long color=RGB(0,0,0)) { m_monochrome_color = color; }
	void SetBackgroundColor(long bk_color=RGB(255,255,255)) { m_background_color = bk_color; }
	long GetBackgroundColor() { return(m_background_color); }

	// call back routines
	virtual void cbBegPlot() { }	// called at beginning of plotting
	virtual void cbEndPlot() { }	// called when plotting is complete
	virtual void cbBegDrawAxis() { }// called at beginning of drawing axis
	virtual void cbEndDrawAxis() { }// called when draw axis plotting is complete
	virtual void cbCheckUserAbort() {  }	// called periodically to check for user abort plotting
	void  DoAbort() { m_abort_plotting=1; }	// call this from cbCheckUserAbort() to abort plotting

	// current pen location status
	int  CurX()     { return(m_xv_cur);    }	// returns current x location
	int  CurY()     { return(m_yv_cur);    }	// returns current y location
	long CurColor() { return(m_color_cur); }	// returns current plotting color

	// ------------------------------------------------------------
	//             Transformation Equations
	// ------------------------------------------------------------
	// object space to plot space
	double xpf(double xo,double yo,double zo) { return(tm[0][0]*xo + tm[0][1]*yo + tm[0][2]*zo); }
	double ypf(double xo,double yo,double zo) { return(tm[1][0]*xo + tm[1][1]*yo + tm[1][2]*zo); }
	double zpf(double xo,double yo,double zo) { return(tm[2][0]*xo + tm[2][1]*yo + tm[2][2]*zo); } 
	
	// plot space to object space
	double xof(double xp,double yp,double zp) { return(tm[0][0]*xp + tm[1][0]*yp + tm[2][0]*zp); }
	double yof(double xp,double yp,double zp) { return(tm[0][1]*xp + tm[1][1]*yp + tm[2][1]*zp); }
	double zof(double xp,double yp,double zp) { return(tm[0][2]*xp + tm[1][2]*yp + tm[2][2]*zp); }
	
	// plot space to video space
	int xvf(double xp) { return( (int)((m_p2v*(xp-m_xpmid) + (double)m_xvmid)) ); }
	int yvf(double yp) { return( (int)((m_p2v*(yp-m_ypmid) + (double)m_yvmid)) ); }
	void MapPoint(double x,double y,double z,int* xv,int* yv)
	{
		double	xp,yp;
		xp = xpf( x, y, z);
		yp = ypf( x, y, z);
		*xv = xvf(xp);
		*yv = yvf(yp);
	}

	void MapAndDrawLine(double x1,double y1,double z1,double x2,double y2,double z2)
	{
		int	xv1,yv1,xv2,yv2;
	
		// Map points
		MapPoint(x1,y1,z1,&xv1,&yv1);
		MapPoint(x2,y2,z2,&xv2,&yv2);
	
		// draw the line
		FilterMoveTo(xv1,yv1);
		FilterDrawTo(xv2,yv2);
	}

	// video space to plot space
	double xv2xp(int xv) { return( (double)(xv-m_xvmid)/m_p2v + m_xpmid ); }
	double yv2yp(int yv) { return( (double)(yv-m_yvmid)/m_p2v + m_ypmid ); }

	// ouput control
	void PlotOutputOff() { m_to_plotter=0; }
	void PlotOutputOn()  { m_to_plotter=1; }

	// debugging
	void Dump();

protected:	// routines

	void Init()	// initialization
	{
		m_abort_plotting = 0;
		// default axis info
		m_show_axis = 0;	// no axis
		memset((char*)&m_axis,0,sizeof(m_axis));
		m_axis.nxticks = 3;
		m_axis.nyticks = 3;
		m_axis.nzticks = 3;
		m_axis.axis_color = RGB(0,0,0);	// black
		strcpy(m_axis.font_name,"Arial"); // needs to be TrueType
		m_axis.font_height = 12;
		m_axis.font_width = 7;
		m_axis.tick_length = 20;
		m_axis.tick_spacing = 20;
	
		// set to defaults
		SetRanges();
		SetViewingPosition();
		SetVideoRes();
		PlotOutputOn();
		CalcTM();
		CalcCorners();
	
		m_pending_moveto = 0;
		m_xv_cur = 0;
		m_yv_cur = 0;
		m_color_cur = 1;
	}

	// calculate transformation matrix
	void CalcTM()
	{
		double	zrotate,sin_zrotate,cos_zrotate;
		double	xytilt,sin_xytilt,cos_xytilt;
	
		zrotate = Deg2Rad(m_zrotate);
		xytilt  = Deg2Rad(m_xytilt);
		sin_zrotate = sin(zrotate);
		cos_zrotate = cos(zrotate);
		sin_xytilt  = sin(xytilt);
		cos_xytilt  = cos(xytilt);
	
		tm[0][0] =  cos_zrotate;
		tm[0][1] =  sin_zrotate;
		tm[0][2] =  0;
		tm[1][0] = -sin_zrotate * cos_xytilt;
		tm[1][1] =  cos_zrotate * cos_xytilt;
		tm[1][2] =                sin_xytilt;
		tm[2][0] =  sin_zrotate * sin_xytilt;
		tm[2][1] = -cos_zrotate * sin_xytilt;
		tm[2][2] =                cos_xytilt;
	}

	// filter move/draw commands
	void FilterMoveTo(int xv,int yv)
	{
		if (m_xv_cur==xv && m_yv_cur==yv) return; // already there
		if (m_to_plotter)
			MoveTo(xv,yv);
		else
			m_pending_moveto = 1;
		m_xv_cur = xv;
		m_yv_cur = yv;
	}

	void FilterDrawTo(int xv,int yv)
	{
		if (m_xv_cur==xv && m_yv_cur==yv) return; // already there
		if (m_to_plotter)
		{
			if (m_pending_moveto) MoveTo(m_xv_cur,m_yv_cur);
			m_pending_moveto = 0;
			DrawTo(xv,yv);
		}
		m_xv_cur = xv;
		m_yv_cur = yv;
	}

	void FilterColorTo(long color)
	{
		if (color==m_color_cur) return; // already got this color
		if (m_to_plotter) ColorTo(color);
		m_color_cur = color;
	}

	// calculate projections of corners
	void CalcCorners();

protected:	// user input data
	// x,y,z min/max ranges
	double	m_xmin, m_xmax, m_ymin, m_ymax, m_zmin, m_zmax;

	// viewing angles
	double	m_xytilt, m_zrotate;	// degrees

	// x,y video min/max ranges
	int m_xvmin, m_xvmax, m_yvmin, m_yvmax;
	double m_scale;	// plotting scale (.01 < scale < 1)

	// axis parameters
	unsigned       m_show_axis;  // 0=no, 1=draw axis
	AXIS_PLOT_INFO m_axis;

	// color info
	long	m_line_color;
	long	m_monochrome_color;
	long	m_background_color;
	int		m_is_color;

protected:	// internal data
	double	m_xmid, m_ymid, m_zmid;

	// transformation matrix
	double tm[3][3];

	// min/mid/max projection values
	double m_xpmin,  m_xpmid,  m_xpmax;
	double m_ypmin,  m_ypmid,  m_ypmax;
	double m_zpmin,  m_zpmid,  m_zpmax;

	// mid points of video coordinate system
	int m_xvmid, m_yvmid;
	double m_p2v;	// maps plot coordinates to video coordinates

	// corner coordinates
	double m_xcorner [8],m_ycorner [8],m_zcorner [8];	// corner values
	double m_xpcorner[8],m_ypcorner[8],m_zpcorner[8];	// projection of corners

	// current pen location and color
	int	 m_xv_cur, m_yv_cur;
	long m_color_cur;

	// output variables
	int m_to_plotter;	 	// 0=no, 1=output to plotter
	int m_pending_moveto;	// 0=no, 1=need to move before drawing

	// control flags
	int	m_abort_plotting;	// 0=no, 1=abort plotting
};

#endif	/* if _PLOT3D_H_ included */

/*	<><><><><><><><><><><><><>  Plot3D.h  <><><><><><><><><><><><><><> */



