/*  
 * Copyright (c) 2002-2003 MIIK Ltd. All rights reserved.  
 *  
 * Use is subject to license terms.  
 *   
 * The complete licence text can be found at   
 * http://www.jniwrapper.com/license.jsp?prod=winpack  
 */
package com.jniwrapper.win32.io;

import com.jniwrapper.*;
import com.jniwrapper.util.EnumItem;
import com.jniwrapper.util.Enums;
import com.jniwrapper.util.FlagSet;
import com.jniwrapper.win32.FunctionName;
import com.jniwrapper.win32.Kernel32;

import java.io.File;

/**
 *
 * @author Alexander Evsukov
 */
public class FileSystem
{
    static final FunctionName FUNCTION_GET_DRIVE_TYPE = new FunctionName("GetDriveType");
    static final FunctionName FUNCTION_GET_DISK_FREE_SPACE = new FunctionName("GetDiskFreeSpaceEx");
    static final FunctionName FUNCTION_GET_FILE_ATTRIBUTES = new FunctionName("GetFileAttributes");
    static final FunctionName FUNCTION_SET_FILE_ATTRIBUTES = new FunctionName("SetFileAttributes");

    /**
     *  Class FileAttributes represent file attributes set.
     */
    public static class FileAttributes extends FlagSet
    {
        public static final int READONLY = 0x00000001;
        public static final int HIDDEN = 0x00000002;
        public static final int SYSTEM = 0x00000004;
        public static final int DIRECTORY = 0x00000010;
        public static final int ARCHIVE = 0x00000020;
        public static final int ENCRYPTED = 0x00000040;
        public static final int NORMAL = 0x00000080;
        public static final int TEMPORARY = 0x00000100;
        public static final int SPARSE_FILE = 0x00000200;
        public static final int REPARSE_POINT = 0x00000400;
        public static final int COMPRESSED = 0x00000800;
        public static final int OFFLINE = 0x00001000;
        public static final int NOT_CONTENT_INDEXED = 0x00002000;

        public FileAttributes(long flags)
        {
            super(flags);
        }

        public boolean isReadOnly()
        {
            return contains(READONLY);
        }

        public boolean isArchive()
        {
            return contains(ARCHIVE);
        }

        public boolean isSystem()
        {
            return contains(SYSTEM);
        }
    }

    /**
     *  DriveTypes class represents enumeration of drive types.
     */
    public static class DriveType extends EnumItem
    {
        public static final DriveType UNKNOWN = new DriveType(0);
        public static final DriveType NO_ROOT_DIR = new DriveType(1);
        public static final DriveType REMOVABLE = new DriveType(2);
        public static final DriveType FIXED = new DriveType(3);
        public static final DriveType REMOTE = new DriveType(4);
        public static final DriveType CDROM = new DriveType(5);
        public static final DriveType RAMDISK = new DriveType(6);

        private DriveType(int value)
        {
            super(value);
        }
    }

    static Function getFunction(Object functionName)
    {
        return Kernel32.get(functionName);
    }

    /**
     * Returns drive type of spefied disk.
     * @param diskName is a drive letter like A:\, C:\ etc. i.e. root folder.
     *        If the passed path is not root folder function wont determine a drive
     *        type and will return DRIVE_NO_ROOT_DIR type.
     * @return drive type specified in the {@link DriveType} enumeration class
     */
    public static DriveType getDriveType(String diskName)
    {
        Function function = getFunction(FUNCTION_GET_DRIVE_TYPE);
        UInt result = new UInt();
        function.invoke(result, Kernel32.getInstance().stringParam(diskName));
        return (DriveType)Enums.getItem(DriveType.class, (int) result.getValue());
    }

    /**
     * Returns drive type of spefied disk.
     * @param drive is a root file like A:\, C:\ etc., otherwise
     *        If the passed file is not root folder function wont determine a drive
     *        type and will return DRIVE_NO_ROOT_DIR type.
     * @return drive type specified in the {@link DriveType} enumeration class
     */
    public static DriveType getDriveType(File drive)
    {
        return getDriveType(drive.getAbsolutePath());
    }

    /**
     * Returns amount of space available on a disk volume in bytes.
     * @param path specifies directory on the disk
     * @return number of free bytes available on disk
     */
    public static long getDiskFreeSpace(String path)
    {
        Function function = getFunction(FUNCTION_GET_DISK_FREE_SPACE);
        Bool result = new Bool();
        Int64 freeToCaller = new Int64();
        function.invoke(result,
                new Pointer(Kernel32.getInstance().stringParam(path)),
                new Pointer(freeToCaller),
                new Pointer(new Int64()),
                new Pointer.Const(new Int64(), true));
        return freeToCaller.getValue();
    }

    /**
     * Returns amount of space available on a disk volume in bytes.
     * @param path specifies directory on the disk
     * @return number of free bytes available on disk
     */
    public static long getDiskFreeSpace(File path)
    {
        return getDiskFreeSpace(path.getAbsolutePath());
    }

    /**
     * Retrieves {@link FileAttributes} from file specified by fileName.
     * @param fileName is a file name
     * @return FileAttributes of necessary file
     */
    public static FileAttributes getFileAttributes(String fileName)
    {
        Function function = getFunction(FUNCTION_GET_FILE_ATTRIBUTES);
        UInt32 result = new UInt32();
        function.invoke(result, Kernel32.getInstance().stringParam(fileName));
        return new FileAttributes(result.getValue());
    }

    /**
     * Retrieves {@link FileAttributes} from specified file.
     * @param file
     * @return FileAttributes of necessary file
     */
    public static FileAttributes getFileAttributes(File file)
    {
        return getFileAttributes(file.getAbsolutePath());
    }

    /**
     * Sets attributes for specified file.
     * @param fileName
     * @param attributes new file attributes
     * @return is function executed successfully the result is true; otherwise false.
     */
    public static boolean setFileAttributes(String fileName, FileAttributes attributes)
    {
        Function function = getFunction(FUNCTION_SET_FILE_ATTRIBUTES);
        Bool result = new Bool();
        function.invoke(result, Kernel32.getInstance().stringParam(fileName), new UInt32(attributes.getFlags()));
        return result.getValue();
    }

    /**
     * Sets attributes for specified file.
     * @param file
     * @param attributes new file attributes
     * @return is function executed successfully the result is true; otherwise false.
     */
    public static boolean setFileAttributes(File file, FileAttributes attributes)
    {
        return setFileAttributes(file.getAbsolutePath(), attributes);
    }
}
