/*  
 * Copyright (c) 2002-2003 MIIK Ltd. All rights reserved.  
 *  
 * Use is subject to license terms.  
 *   
 * The complete licence text can be found at   
 * http://www.jniwrapper.com/license.jsp?prod=winpack  
 */
package com.jniwrapper.win32.ui;

import com.jniwrapper.*;
import com.jniwrapper.win32.FunctionName;
import com.jniwrapper.win32.Handle;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

/**
 * This class represents default WINDOWPROC callback.
 * Also it provides abiliti to substitute window proc. of specified window to enable custom
 * message handling and processing.
 *
 * @author Serge Piletsky
 */
public class WindowProc extends Callback
{
    private static final FunctionName FUNCTION_DEF_WINDOW_PROC = new FunctionName("DefWindowProc");

    protected Wnd _wnd = new Wnd();
    protected UInt _msg = new UInt();
    protected UInt32 _wParam = new UInt32();
    protected UInt32 _lParam = new UInt32();
    protected UInt32 _lResult = new UInt32();
    private Function _defWindowProc;
    private boolean _substituted = false;
    private Handle _wndProc;
    private List _messageListeners = new LinkedList();

    /**
     * @param wnd is a handle of window which window procedure can be substituted.
     */
    public WindowProc(Wnd wnd)
    {
        this();
        setWnd(wnd);
    }

    public WindowProc()
    {
        init(new Parameter[]{_wnd, _msg, _wParam, _lParam}, _lResult);
        _defWindowProc = getDefWindowProc();
    }

    public void callback()
    {
        if (isSubstituted())
        {
            notifyListeners(true);
            _lResult.setValue(Wnd.callWindowProc(_wndProc, _wnd, _msg, _wParam, _lParam));
            notifyListeners(false);
        }
        else
        {
            _defWindowProc.invoke(_lResult, _wnd, _msg, _wParam, _lParam);
        }
    }

    private void notifyListeners(boolean beforeWndProc)
    {
        final WindowMessage message = new WindowMessage(this, (int)_msg.getValue(), (int)_wParam.getValue(), (int)_lParam.getValue());
        for (Iterator i = _messageListeners.iterator(); i.hasNext();)
        {
            WindowMessageListener listener = (WindowMessageListener) i.next();
            if (listener.canHandle(message, beforeWndProc))
            {
                _lResult.setValue(listener.handle(message));
                _msg.setValue(message.getMsg());
                _wParam.setValue(message.getWParam());
                _lParam.setValue(message.getLParam());
            }
        }
    }

    private static Function getDefWindowProc()
    {
        final Function function = User32.getInstance().getFunction(FUNCTION_DEF_WINDOW_PROC.toString());
        return function;
    }

    /**
     * Adds specified window message listener.
     * @param listener
     */
    public void addMessageListener(WindowMessageListener listener)
    {
        if (!_messageListeners.contains(listener))
            _messageListeners.add(listener);
    }

    /**
     * Removes specified listener.
     * @param listener
     */
    public void removeMessageListener(WindowMessageListener listener)
    {
        _messageListeners.remove(listener);
    }

    /**
     * Substritutes custom window procedure to enable custom message handling.
     */
    public void substitute()
    {
        _wndProc = new Handle(_wnd.getWindowLong(Wnd.GWL_WNDPROC));
        _wnd.setWindowLong(Wnd.GWL_WNDPROC, this);
        _substituted = true;
    }

    /**
     * Restrores the native window procedure and disables custom message handling.
     */
    public void restoreNative()
    {
        if (_wndProc == null || _wndProc.isNull())
            throw new IllegalStateException();
        _wnd.setWindowLong(Wnd.GWL_WNDPROC, _wndProc);
        _substituted = false;
    }

    /**
     * Check if native window proc. is substituted.
     * @return true if native proc. is substituted; otherwise false.
     */
    public boolean isSubstituted()
    {
        return _substituted;
    }

    public Wnd getWnd()
    {
        return _wnd;
    }

    public void setWnd(Wnd wnd)
    {
        if (wnd == null || wnd.isNull())
            throw new IllegalArgumentException();

        final boolean wasSubstituted = isSubstituted();
        if (wasSubstituted)
        {
            restoreNative();
        }
        _wnd.setValue(wnd.getValue());
        if (wasSubstituted)
        {
            substitute();
        }
    }
}
