
/*
 * Copyright (c) 1998, 1999 Semiotek Inc. All Rights Reserved.
 *
 * This software is the confidential intellectual property of
 * of Semiotek Inc.; it is copyrighted and licensed, not sold.
 * You may use it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation. If you 
 * do not want to use the GPL, you may still use the software after
 * purchasing a proprietary developers license from Semiotek Inc.
 *
 * This software is provided "as is", with NO WARRANTY, not even the 
 * implied warranties of fitness to purpose, or merchantability. You
 * assume all risks and liabilities associated with its use.
 *
 * See the attached License.html file for details, or contact us
 * by e-mail at info@semiotek.com to get a copy.
 */


package org.webmacro.engine;

import org.webmacro.util.java2.*;
import java.util.*;
import java.io.*;
import org.webmacro.util.*;
import java.lang.reflect.*;

// PRIMARY CLASS: Variable

/**
  * A Variable is a reference into a Propertymap.
  * <p>
  * A variable name contains a list of names separated by dots, for
  * example "$User.Identity.email.address" is the list: User, Identity,
  * email, and address. 
  * <p>
  * PLEASE NOTE: Case-sensitivity is enforced. "User" is the the same
  * name as "user".
  * <p>
  * What that means: When a template is interpreted, it is interpreted
  * in terms of data in a hashtable/map called the "context". This is 
  * actually a Map of type Map. The context contains all the 
  * local variables that have been set, as well as other information 
  * that Macros may use to evaluate the request.
  * <p>
  * Variable depends heavily on Property introspection: It is defined
  * as a list of one or more names (separated by dots when written). 
  * <p>
  * Those names are references to sub-objects within the context. The
  * Variable instance, when interpreted, will decend through the context
  * following fields, method references, or hash table look-ups based
  * on its names.
  * <p>
  * For example, the variable "$User.Identity.email.address" implies
  * that there is a "User" object under the Map--either it is
  * a field within the map, or the map has a getUser() method, or 
  * the User can be obtained by calling Map.get("User").
  * <p>
  * The full expansion of $User.Identity.email.address might be:<pre>
  * 
  *    Map.get("User").getIdentity().get("email").address
  *
  * </pre>. Variable (actually the Property class it uses) will figure
  * out how to decend through the object like this until it finds the
  * final reference--which is the "value" of the variable.
  * <p>
  * When searchin for subfields Variable prefers fields over getFoo() 
  * methods, and getFoo() over get("Foo").
  *
  */
public class Variable implements Macro
{

   /**
     * The name of this variable.
     */
   final protected String vname;

   /**
     * The name as an array
     */
   final private Object[] myNames;

   /**
     * Create a variable with the supplied name. The variable references 
     * will look like "$name" where name is replaced by the actual name.
     */
   Variable(Object names[]) {
      vname = makeName(names).intern();
      myNames = names;
   }

   /**
     * Return a string name of this variable
     */
   public final String toString() {
      return "variable:" + vname;
   }

   /**
     * Looks in the hashTable (context) for a value keyed to this variables
     * name and returns the value string. If the resulting value is a Macro,
     * recursively call its evaluate method.
     * @return String 
     */
   final public Object evaluate(Object context)
   {
      try {
         Object val = getValue(context);
         if (val instanceof Macro) {
            return ((Macro) val).evaluate(context); // recurse
         } else {
            return val;
         }
      } catch (NullPointerException e) {
         Engine.log.exception(e);
         Engine.log.warning("Variable: " + vname + " does not exist");
         return "<!--\n unable to access variable " 
            + vname + ": not found in " + context + "\n -->";
      } catch (Exception e) {
         Engine.log.exception(e);
         Engine.log.warning("Variable: " + vname + " does not exist");
         return "<!--\n unable to access variable " 
            + vname + ": " + e + " \n-->";
      }
   }

   /**
     * Look in the hashtable (context) for a value keyed to this variables 
     * name and write its value to the stream.
     * @exception InvalidContextException is required data is missing
     * @exception IOException if could not write to output stream
     */
   final public void write(Writer out, Object context) 
       throws InvalidContextException, IOException
   {
      // force it to throw a class cast exception if things are amiss

      try {
         Object val = getValue(context); 
         if (val instanceof Macro) {
            ((Macro) val).write(out,context);
         } else {
             out.write(val.toString());
         }
      } catch (NullPointerException e) {
         Engine.log.warning("Variable: " + vname + " is undefined");
         out.write("<!--\n unable to access variable " 
            + vname + ": does not exist-->");
      } catch (Exception e) {
         Engine.log.exception(e);
         Engine.log.warning("Variable: " + vname + " is undefined");
         out.write("<!--\n warning: attempt to write undefined variable " 
            + vname + ": " + e + " \n-->");
      } 
   }

   /**
     * Attempt to parse a variable name. The '$' has already been eaten, 
     * expect that the next thing on the parseTool is the variables name,
     * followed by an optional ';'. If the name begins with $$ it will be
     * considered a static parameter and must only contain a single name;
     * the value of the parameter will be returned as a string.
     * @exception ParseException on unrecoverable parse error
     * @exception IOException on failure to read from parseTool
     * @return a Variable object 
     */
   final public static Object parse(ParseTool in) 
      throws ParseException, IOException
   {
      // check that you were called correctly 
      if (in.ttype != '$') {
         return null;   
      }
      in.nextToken();

      boolean isParam = in.parseChar('$'); // param is $$

      Vector names = new Vector();
      Object name;

      while ((name = in.parseString()) != null) {

         if (in.ttype == '(') {
            // it's a method call
            Object args = List.parse(in); 
            if (args instanceof Macro) {
               name = new PropertyMethod((String) name,(Macro) args);
            } else {
               name = new PropertyMethod((String) name,(Object[]) args);
            }
         }

         names.addElement(name);
         if (in.ttype == '.') {
            in.nextToken(); // keep parsing beyond the dot
         } else {
            break; 
         }
      }

      if (names.size() == 0) {
         in.pushBack(); // back to the $ or .
         return null;
      }
      in.parseChar(';'); // eat optional ;

      if (isParam) {
         if (names.size() != 1) {
            throw new ParseException(in,"Parameter names cannot include the dot operator character: .");
         }
         Object value = in.getParamContext().get(names.elementAt(0));
         if (value == null) {
            return Boolean.FALSE;
         }
         return value;
      } else {
         Object[] oname = new Object[names.size()];
         names.copyInto(oname);
         return new Variable(oname);
      }
   }


   /**
     * Test harness
     */
   public static void main(String args[]) {

      Log.setLevel(Log.DEBUG);
      Log.traceExceptions(true);

      try {

         String testCase[] = {
               "$ten",
               "$ten xxxx",
               "$twenty; xxxx $a",
               "$bogus xxx",
               "$test.date.Month",
               "$test.Name",
               "$test.show($ten)",
               "$test.fname"

            };

         // context

         Map context = new HashMap(); 

         System.out.println("Adding to context: ten=10");
         context.put("ten", new Integer(10));  

         System.out.println("Adding to context: twenty=20");
         context.put("twenty", new Integer(20));  

         System.out.println("Adding to context: test=TestObject(\"happy\")");
         context.put("test", new TestObject("happy"));

         System.out.println();
         for (int i = 0; i < testCase.length; i++) {
            try {

               System.out.println("Test " + i + ": " + testCase[i]);
               ParseTool in = new ParseTool("string",new StringReader(testCase[i]));
               in.nextToken();
               Variable v =  (Variable) Variable.parse(in); 
               if (v == null) {
                  System.out.println("Err: did not parse!");
               } else  {
                  System.out.println("Evaluates to: " + v.evaluate(context));
               }
            } catch (Exception e) {
               e.printStackTrace();
            }
            System.out.println();
         }

         try {
            Variable v;
            ParseTool in;

            System.out.println("Instantiating a variable v=$ten");
            in = new ParseTool("string",new StringReader("$ten"));
            in.nextToken();
            v = (Variable) Variable.parse(in);
            System.out.println("Value of v: " + v.evaluate(context));  
            System.out.println("Setting v=TEN");
            v.setValue(context,"TEN");
            System.out.println("Value of v: " + v.evaluate(context));  
            v = null;

            System.out.println();

            System.out.println("Instantiating a variable v=$test.fname");
            in = new ParseTool("string",new StringReader("$test.fname"));
            in.nextToken();
            v = (Variable) Variable.parse(in);
            System.out.println("Value of v: " + v.evaluate(context));  
            System.out.println("Setting v=HI MOM");
            v.setValue(context,"HI MOM");
            System.out.println("Value of v: " + v.evaluate(context));  
            v = null;

            System.out.println();

            System.out.println("Instantiating a variable v=$test.Name");
            in = new ParseTool("string",new StringReader("$test.Name"));
            in.nextToken();
            v = (Variable) Variable.parse(in);
            System.out.println("Value of v: " + v.evaluate(context));  
            System.out.println("Setting v=HI AGAIN");
            v.setValue(context,"HI AGAIN");
            System.out.println("Value of v: " + v.evaluate(context));  
            v = null;

         } catch (Exception e) {
            System.out.println("FAILED");
            e.printStackTrace();
         }

         System.out.println();

         System.out.println("Done.");
      } catch (Exception e) {
         System.out.println("MAIN CAUGHT AN EXCEPTION");
         e.printStackTrace();
      }

   }

   /**
     * Helper method to construct a String name from a Object[] name
     */
   private static String makeName(Object[] names) {
      StringBuffer buf = new StringBuffer();
      for (int i = 0; i < names.length; i++) {
         if (i != 0) {
            buf.append(".");
         }
         buf.append(names[i]);
      }
      return buf.toString();
   }

   /**
     * Look up my value in the corresponding Map, possibly using introspection,
     * and return it
     * @exception InvalidContextException If the property does not exist
     */
   final public Object getValue(Object context) 
      throws InvalidContextException
   {
      try {
         return PropertyOperator.getProperty(context, myNames);
      } catch (Exception e) {
         Engine.log.exception(e);
         String warning = "Variable: unable to access " + this + ";";
         throw new InvalidContextException(warning);
      }
   }

   /**
     * Look up my the value of this variable in the specified Map, possibly
     * using introspection, and set it to the supplied value.
     * @exception InvalidContextException If the property does not exist
     */
   final public void setValue(Object context, Object newValue)
      throws InvalidContextException
   {
      try{
         if (!PropertyOperator.setProperty(context,myNames,newValue)) {
            throw new PropertyException("No method to set \"" + vname + 
               "\" to type " +
               ((newValue == null) ? "null" : newValue.getClass().toString()) 
               + " in supplied context (" + context + ")",null);
         }
      } catch (Exception e) {
         Engine.log.exception(e);
         String warning = "Variable.setValue: unable to access " + this + 
            " (is it a public method/field?)";
         throw new InvalidContextException(warning);
      }
   }
}



