
/*
 * Copyright (c) 1998, 1999 Semiotek Inc. All Rights Reserved.
 *
 * This software is the confidential intellectual property of
 * of Semiotek Inc.; it is copyrighted and licensed, not sold.
 * You may use it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation. If you 
 * do not want to use the GPL, you may still use the software after
 * purchasing a proprietary developers license from Semiotek Inc.
 *
 * This software is provided "as is", with NO WARRANTY, not even the 
 * implied warranties of fitness to purpose, or merchantability. You
 * assume all risks and liabilities associated with its use.
 *
 * See the attached License.html file for details, or contact us
 * by e-mail at info@semiotek.com to get a copy.
 */


package org.webmacro.servlet;

import java.util.*;
import java.io.*;

import javax.servlet.*;
import javax.servlet.http.*;

import org.webmacro.util.*;
import org.webmacro.broker.*;
import org.webmacro.resource.*;
import org.webmacro.engine.Template;


/**
  * This is the abstract base class used by all WebMacro servlets. You
  * can either subclass from it directly, or make use of one of the 
  * generic subclasses provided. 
  * <p>
  * It's primary function is to create a WebContext and manage a 
  * ResourceBroker. It also provides a couple of convenience functions 
  * that access the ResourceBroker and/or WebContext to make some commonly
  * accessed services more readily available. 
  * <p>
  * @see org.webmacro.Handler
  * @see org.webmacro.ResourceBroker
  */
abstract public class WMServlet extends HttpServlet
{

   // INIT METHODS--MANAGE ACCESS TO THE BROKER

   /**
     * Our copy of the ResourceBroker, shared by all instances
     */
   private static ResourceBroker _broker = null;

   /**
     * How many instances are currently using the broker
     */
   private static int _brokerUsers = 0;

   /**
     * Log object used to write out messages
     */
   final static private Log _log = new Log("WMServlet", "WebMacro Abstract Servlet");

   /**
     * null means all OK
     */
   private String _problem = "Not yet initialized: Your servlet API tried to access WebMacro without first calling init()!!!";

   /**
     * This is the old-style init method, it just calls init(), after
     * handing the ServletConfig object to the superclass
     * @exception ServletException if it failed to initialize
     */
   final public synchronized void init(ServletConfig sc) 
      throws ServletException 
   {
      super.init(sc);
      init();
   }

   /**
     * This method is called by the servlet runner--do not call it. It 
     * must not be overidden because it manages a shared instance
     * of the broker--you can overide the start() method instead, which
     * is called just after the broker is initialized.
     * @exception ServletException if it failed to initialize
     */
   final public synchronized void init()
   {
      // this method must be synch. to ensure that the count always agrees
      // with whether the broker exists or not
      _brokerUsers++;
      if (_broker == null) {
         try {
            _broker = new ResourceBroker();
            _broker.join(new Config());  
            String providers = 
               (String) _broker.getValue(Config.TYPE, Config.PROVIDERS);
            Enumeration provEnum = new StringTokenizer(providers);
            String className;
            Class resClass;
            ResourceProvider instance;
            while (provEnum.hasMoreElements()) {
               className = (String) provEnum.nextElement(); 
               try {
                  resClass = Class.forName(className);
                  instance = (ResourceProvider) resClass.newInstance();
                  _broker.join(instance);
                  _log.info("Loaded provider: " + className);
               } catch (Exception e) {
                  _log.exception(e);
                  _log.error("Could not load resource provider \"" +
                        className + "\":" + e);
               }
            }
         } catch (Exception e) {
            _broker = null;
            _brokerUsers = 0; 

            _log.exception(e);
            _log.error("Could not initialize the broker!\n"
                  + "*** Check that webmacro.properties was in your servlet\n"
                  + "*** classpath, in a similar place to webmacro.jar \n"
                  + "*** and that all values were set correctly.\n");

            _problem =
                "Unable to initialize WebMacro:\n\n" + e + "\n\n"
              + "This means critical providers could not be loaded. The usual\n"
              + "cause of this is that the configuration file could not\n"
              + "be loaded from the servlet classpath. Check in particular\n"
              + "that webmacro.properties, WebMacro's configuration file, is\n"
              + "in your classpath (the same place as webmacro.jar).\n"
              + "The fact that you are seeing this error means your servlet\n"
              + "runner did find webmacro.jar, and launched your WebMacro\n"
              + "servlet--however WebMacro was then unable to initialize.\n\n"
              + "WebMacro may have logged additional details to its log, or\n"
              + "to its standard output.\n";
            return;
         }
      }
      try {
         start();
         _problem = null; 
      } catch (ServletException e) {
         _log.exception(e);
         _problem = "WebMacro application code failed to initialize: \n"
            + e + "\n" + "This error is the result of a failure in the\n"
            + e + "code supplied by the application programmer.\n";
      }
   }

   /**
     * This method is called by the servlet runner--do not call it. It 
     * must not be overidden because it manages a shared instance of 
     * the broker--you can overide the stop() method instead, which 
     * will be called just before the broker is shut down.
     */
   final public synchronized void destroy() {
      // this method must be synch. to ensure that the count always agrees
      // with whether the broker exists or not
      stop();
      _brokerUsers--;
      if ((_brokerUsers == 0) && (_broker != null)) {
         _broker.shutdown(); 
         _broker = null;
      }
   }


   // SERVLET API METHODS

   /**
     * Process an incoming GET request: Builds a WebContext up and then 
     * passes it to the handle() method. You can overide this if you want,
     * though for most purposes you are expected to overide handle() 
     * instead.
     * <p>
     * @param req the request we got
     * @param resp the response we are generating
     * @exception ServletException if we can't get our configuration
     * @exception IOException if we can't write to the output stream 
     */
   final protected void doGet(HttpServletRequest req, HttpServletResponse resp)
      throws ServletException, IOException
   { 
      doRequest(new WebContext(_broker,req,resp));
   }

   /**
     * Behaves exactly like doGet() except that it reads data from POST
     * before doing exactly the same thing. This means that you can use 
     * GET and POST interchangeably with WebMacro. You can overide this if
     * you want, though for most purposes you are expected to overide 
     * handle() instead.
     * <p>
     * @param req the request we got
     * @param resp the response we are generating
     * @exception ServletException if we can't get our configuration
     * @exception IOException if we can't read/write to the streams we got
     */
   final protected void doPost(HttpServletRequest req, HttpServletResponse resp)
      throws ServletException, IOException
   {
       doRequest(new WebContext(_broker,req,resp));
   }

   final private void doRequest(WebContext context)
   {
      if (_problem != null) {
         init();
         if (_problem != null) {
            try { 
               Writer out = context.getResponse().getWriter();
               out.write(_problem); 
               out.flush();
               out.close();
            } catch (Exception e) {
               _log.error(_problem); 
            }
            return;
         }
      }
      Template tmpl = null;
      try {
         tmpl = handle(context);
      } catch (HandlerException e) {
         _log.exception(e);
         tmpl = error(context, 
            "Your handler was unable to process the request successfully " +
            "for some reason. Here are the details:<p>" + e);
      } catch (Exception e) {
         _log.exception(e);
         tmpl = error(context, 
            "The handler WebMacro used to handle this request failed for " +
            "some reason. This is likely a bug in the handler written " +
            "for this application. Here are the details:<p>" + e);
      }
      execute(tmpl,context);  
   }

   
   // CONVENIENCE METHODS & ACCESS TO THE BROKER

   /**
     * Create an error template using the built in error handler.
     * This is useful for returning error messages on failure;
     * it is used by WMServlet to display errors resulting from
     * any exception that you may throw from the handle() method.
     * @param context will add error variable to context (see Config)
     * @param error a string explaining what went wrong
     */
   final protected Template error(WebContext context, String error)
   {
      Template tmpl = null;
      _log.warning(error);
      Handler hand = new ErrorHandler();
      try {
         context.put(getConfig(Config.ERROR_VARIABLE), error);
         tmpl = hand.accept(context);
      } catch(ResourceException e2) {
         _log.error("Could not find error variable in Config: " + e2);
      } catch(Exception e2) {
         _log.error("Unable to use ErrorHandler: " + e2);
      }
      return tmpl;
   }

   /**
     * This object is used to access components that have been plugged
     * into WebMacro; it is shared between all instances of this class and
     * its subclasses. It is created when the first instance is initialized,
     * and deleted when the last instance is shut down. If you attempt to 
     * access it after the last servlet has been shutdown, it will either 
     * be in a shutdown state or else null.
     */
   final protected ResourceBroker getBroker() {
      // this method can be unsynch. because the broker manages its own
      // state, plus the only time the _broker will be shutdown or null 
      // is after the last servlet has shutdown--so why would anyone be 
      // accessing us then? if they do the _broker will throw exceptions
      // complaining that it has been shut down, or they'll get a null here.
      return _broker;
   }

   /**
     * Retrieve a template from the "template" provider. Equivalent to 
     * getBroker().getValue(TemplateProvider.TYPE,key)
     * @exception ResourceUnavailableException if the template was not found
     */
   final protected Template getTemplate(String key) 
      throws ResourceUnavailableException
   {
      return (Template) _broker.getValue(TemplateProvider.TYPE, key);
   }

   /**
     * Retrieve a handler from the "handler" provider. Equivalent to 
     * getBroker().getValue(HandlerProvider.TYPE,key)
     * @exception ResourceUnavailableException if the handler was not found
     */
   final protected Handler getHandler(String key)
      throws ResourceUnavailableException
   {
      return (Handler) _broker.getValue(HandlerProvider.TYPE, key);
   }

   /**
     * Retrieve configuration information from the "config" provider.
     * Equivalent to getBrker().getValue(Config.TYPE,key)
     * @exception ResourceUnavailableException could not locate requested information
     */
   final protected String getConfig(String key) 
      throws ResourceUnavailableException
   {
      return (String) _broker.getValue(Config.TYPE, key);
   }


   // DELEGATE-TO METHODS -- COMMON THINGS MADE EASIER

   /**
     * This method takes a populated context and a template and 
     * writes out the interpreted template to the context's output
     * stream. 
     */
   final protected void execute(Template tmpl, WebContext c)
   {
      Writer out = null;
      try {
         out = c.getResponse().getWriter();
         BufferedWriter bw = new BufferedWriter(out);
         tmpl.write(bw, c);
         bw.flush();
      } catch (IOException e) {
         // ignore disconnect
      } catch (Exception e) {
         _log.exception(e);
         String error =
            "WebMacro encountered an error while executing a  template:\n"
            + ((tmpl != null) ?  (tmpl  + ": " + e + "\n") :
                ("The template failed to load; double check the "
                 + "TemplatePath in your webmacro.properties file."));
         _log.warning(error);
         try { out.write(error); } catch (Exception ignore) { }
      } finally {
         try {
            if (out != null) {
               out.flush();
               out.close();
            }
         } catch (Exception e3) {
            // ignore disconnect
         }
      }
   }


   // FRAMEWORK TEMPLATE METHODS--PLUG YOUR CODE IN HERE

   /**
     * Override this method and put your controller code here.
     * This is the primary method that you will write.
     * Use the many other methods on this object to assist you.
     * Your job is to process the input, put whatever will be 
     * needed into the WebContext hash, locate the appropriate
     * template, and return it.
     * @see getTemplate
     * @see getConfig
     * @see getBroker
     * @return the template to be rendered by the WebMacro engine
     * @exception HandlerException throw this to produce vanilla error messages
     * @param context contains all relevant data structures, incl builtins.
     */
   protected abstract Template handle(WebContext context)
         throws HandlerException;

   /**
     * Override this method to implement any startup/init code 
     * you require. The broker will have been created before this 
     * method is called; the default implementation does nothing.
     * This is called when the servlet environment initializes 
     * the servlet for use via the init() method.
     * @exception ServletException to indicate initialization failed
     */
   protected void start() throws ServletException { }
      

   /**
     * Override this method to implement any shutdown code you require.
     * The broker may be destroyed just after this method exits. This 
     * is called when the servlet environment shuts down the servlet 
     * via the shutdown() method. The default implementation does nothing.
     */
   protected void stop() { }


}

