/* ----------------------------------------------------------------------- *
 *                                                                         *
 *    (C) Copyright 1996 by:  aiNet                                        *
 *										Trubarjeva 42                                *
 *                            SI-3000 Celje                                *
 *										Europe, Slovenia                             *
 *     All Rights Reserved                                                 *
 *                                                                         *
 *     Subject: C code for single vector prediction.                       *
 *     File:    T3RUNTIM - The XOR problem with low level model creation   *
 *                                                                         *
 * ----------------------------------------------------------------------- */

/*--------------------------------------------------------------------------
	Here it will be shown how we can colve the XOR problem using
	aiNet C functions

	The XOR problem:
	================
		Number of model vectors: 4
			 Number of variables: 3
	 Number of input variables: 3
		 Any discrete variables: NONE

		Model vectors:  Inp,Inp,Out
				  row 1:  1,  1,  0
				  row 2:  1,  0,  1
				  row 3:  0,  1,  1
				  row 4:  0,  0,  0

	Test vectors (vectors which will be used in prediction) together with
	penalty coefficient and penalty method.

		 Prediction vectors:  Inp  Inp  Out
						  prd 1:  0.9  0.1  ??
						  prd 2:  0.1  0.9  ??
						  prd 3:  0.2  0.2  ??
						  prd 4:  0.7  0.7  ??

		 Penalty coeffcient: 0.5
		 Penalty methods: DYNAMIC

	NOTE: Selected penalty coefficients are in no case optimal.
			They were selected randomly, to perform a few tests.
			The test results were compared with the results calculated by
			the main aiNet 1.14 application.

	--------------------------------------------------------------------------
	Results (rounded at fourth decimal):
	--------------------------------------------------------------------------

		 Penalty cefficient: 0.5
		 Penalty method:     DYNAMIC
												  (RESULT)
		 Prediction vectors:  Inp  Inp  (  Out )
						  prd 1:  0.9  0.1  (0.6948)
						  prd 2:  0.1  0.9  (0.6948)
						  prd 3:  0.2  0.2  (0.3321)
						  prd 4:  0.7  0.7  (0.3869)

 ---------------------------------------------------------------------------*/

/*
 * This file assumes that ainetxx.dll will be loaded at run time,
 * which is default option and no flags need to be defined.
 * ainetxx.lib must NOT be included in the linking process.
 */

#include "ainetdll.h"
#include <stdio.h>
#include <stdlib.h>

/*
 * Path and the filename of dll which will be loaded.
 */

#if defined(__WIN32__)
const char ainetDll[] = "ainet32.dll";
#else
const char ainetDll[] = "ainet16.dll";
#endif
/*
 * Pointers to ainet dll functions. They are made global - all functions
 * can use them.
 */

t_aiRegistration              aiRegistration;
t_aiGetVersion                aiGetVersion;
t_aiCreateModel               aiCreateModel;
t_aiCreateModelFromCSVFile    aiCreateModelFromCSVFile;
t_aiDeleteModel               aiDeleteModel;
t_aiNormalize                 aiNormalize;
t_aiDenormalize               aiDenormalize;
t_aiPrediction                aiPrediction;
t_aiGetNumberOfVariables      aiGetNumberOfVariables;
t_aiGetNumberOfModelVectors   aiGetNumberOfModelVectors;
t_aiGetNumberOfInputVariables aiGetNumberOfInputVariables;
t_aiSetDiscreteFlag           aiSetDiscreteFlag;
t_aiGetDiscreteFlag           aiGetDiscreteFlag;
t_aiSetVariable               aiSetVariable;
t_aiGetVariable               aiGetVariable;
t_aiGetVariableVB             aiGetVariableVB;
t_aiGetCSVFileModelSize       aiGetCSVFileModelSize;
// New in version 1.24
t_aiSetCapacity					aiSetCapacity;
t_aiGetCapacity					aiGetCapacity;
t_aiGetFreeEntries				aiGetFreeEntries;
t_aiInsertModelVector			aiInsertModelVector;
t_aiOverwriteModelVector		aiOverwriteModelVector;
t_aiAppendModelVector			aiAppendModelVector;
t_aiDeleteModelVector			aiDeleteModelVector;
t_aiPredictionEx					aiPredictionEx;
t_aiExcludeModelVector			aiExcludeModelVector;
t_aiExcludeModelVectorRange	aiExcludeModelVectorRange;
t_aiIsModelVectorExcluded		aiIsModelVectorExcluded;
t_aiSaveCSVFile					aiSaveCSVFile;


/*
 *  ainet32.dll module variable.
 */

HINSTANCE hLib;

/*
 *  The load_aiNetLibrary() function is implemented below.
 *  This function will load ainet32.dll and define pointers to
 *  ainet functions.
 */

int load_aiNetLibrary(void);

/*
 *
 */

void main()
{
	/*
	 * Here we present how to create a model in a "low level" way.
	 * We recommend you to avoid this model creation type. Use rather
	 * aiCreateModel functions instead. Please also note that model must
	 * not be deleted using aiDeleteModel function.
	 */

	int i;
	int version;

	float* data[4];				/* Model data    */
	float row1[3] = { 1,1,0 }; /* model vectors */
	float row2[3] = { 1,0,1 };
	float row3[3] = { 0,1,1 };
	float row4[3] = { 0,0,0 };
	int   disc[3] = {0,0,0};	/* Discrete flags */
	float n1[3];					/* Normalization  */
	float n2[3];					/* buffers        */
   unsigned char flag[4];     /* NEW 1.24: additional information for prediction */

	aiModel model = { 0, /* data     */
							4, /* nMV      */
							3, /* nVar     */
							2, /* ni       */
							0, /* discrete */
							0, /* n1       */
							0  /* n2       */
	};

	float predict[4][3] = { { 0.9,0.1, 999 },   /* vectors to be predicted */
									{ 0.1,0.9, 999 },
									{ 0.2,0.2, 999 },
									{ 0.7,0.7, 999 } };

	/*
	 * Setup the model
	 */

	data[0] = row1;
	data[1] = row2;
	data[2] = row3;
	data[3] = row4;
	model.data = data;
	model.discrete = disc;
	model.n1 = n1;
	model.n2 = n2;
   model.flag = flag;

   /*
    * Load DLL
    */
   if( !load_aiNetLibrary() ) {
		exit(EXIT_FAILURE);
	}

	/*
	 * Title
	 */

	version = aiGetVersion();
	printf( "\naiNetDLL version %i.%i! (C) Copyright by aiNet, 1996",
			  version/100, version%100 );
	printf( "\n---------------------------------------------------\n" );

	/*
	 * Register DLL
	 */

	aiRegistration( "Your registration name", "Your code" );

	/*
	 * Output the model
	 */

	printf( "\n             Model name: aiNet DLL test 3 (Low level creation)");
	printf( "\nNumber of model vectors: %i", aiGetNumberOfModelVectors(&model));
	printf( "\n    Number of variables: %i", aiGetNumberOfVariables(&model));
	printf( "\n         Variable names: A,   B,   A xor B" );
	printf( "\n          Discrete flag: %i,   %i,   %i",
			  aiGetDiscreteFlag(&model,1),
			  aiGetDiscreteFlag(&model,2),
			  aiGetDiscreteFlag(&model,3) );
	for( i=1; i<=aiGetNumberOfModelVectors(&model); i++ ) {
		printf( "\n\t\t\t %3.1lf, %3.1lf, %3.1lf",
				  aiGetVariable(&model, i,1),
				  aiGetVariable(&model, i,2),
				  aiGetVariable(&model, i,3) );
	 }

	/*
	 * Normalize the model
	 */

	aiNormalize( &model, NORMALIZE_STATISTICAL );

	/*
	 * Prediction: Pen. coefficient = 0.50, Pen. method = STATIC
	 * This test has dynamic penalty coefficient 0.50
	 */

	printf( "\n\n  Penalty coefficient: 0.50" );
	printf(   "\n       Penalty method: DYNAMIC" );
	printf(   "\n\t A(inp), B(inp), A xor B(out)" );
	for ( i=0; i<4; i++ ) {
		aiPrediction( &model, predict[i], 0.50, PENALTY_DYNAMIC );
		printf( "\n\t%7.4f, %7.4f, %7.4f",
				  predict[i][0],predict[i][1],predict[i][2] );
	}

	/*
	 * Denormalize the model (in this case it is not necessary)
	 */

	aiDenormalize(&model);

	/*
	 * We must not call the aiDeleteModel function here since the model
	 * was not allocated dynamicaly using the aiCreateModel function.
	 */

   FreeLibrary(hLib);

	printf( "\n\nEnd." );
	exit(EXIT_SUCCESS);
}

int load_aiNetLibrary()
{
   /*
    * Load the Dynamic Link Library AINET32.DLL
    */

   hLib = LoadLibrary(ainetDll);
   if((unsigned)hLib<=HINSTANCE_ERROR){
      char bfr[40];
      wsprintf(bfr, "Failure loading library: %s", ainetDll);
      MessageBox(NULL, bfr, "Error", MB_OK|MB_APPLMODAL);
      return 0;
   }

   /*
    * Get all the entry points for the functions in ainet32.dll
    */

	aiRegistration              = (t_aiRegistration)              GetProcAddress(hLib, "aiRegistration");
   aiGetVersion                = (t_aiGetVersion)                GetProcAddress(hLib, "aiGetVersion");
   aiCreateModel               = (t_aiCreateModel)               GetProcAddress(hLib, "aiCreateModel");
   aiCreateModelFromCSVFile    = (t_aiCreateModelFromCSVFile)    GetProcAddress(hLib, "aiCreateModelFromCSVFile");
   aiDeleteModel               = (t_aiDeleteModel)               GetProcAddress(hLib, "aiDeleteModel");
   aiNormalize                 = (t_aiNormalize)                 GetProcAddress(hLib, "aiNormalize");
   aiDenormalize               = (t_aiDenormalize)               GetProcAddress(hLib, "aiDenormalize");
   aiPrediction                = (t_aiPrediction)                GetProcAddress(hLib, "aiPrediction");
   aiGetNumberOfVariables      = (t_aiGetNumberOfVariables)      GetProcAddress(hLib, "aiGetNumberOfVariables");
   aiGetNumberOfModelVectors   = (t_aiGetNumberOfModelVectors)   GetProcAddress(hLib, "aiGetNumberOfModelVectors");
   aiGetNumberOfInputVariables = (t_aiGetNumberOfInputVariables) GetProcAddress(hLib, "aiGetNumberOfInputVariables");
   aiSetDiscreteFlag           = (t_aiSetDiscreteFlag)           GetProcAddress(hLib, "aiSetDiscreteFlag");
   aiGetDiscreteFlag           = (t_aiGetDiscreteFlag)           GetProcAddress(hLib, "aiGetDiscreteFlag");
   aiSetVariable               = (t_aiSetVariable)               GetProcAddress(hLib, "aiSetVariable");
   aiGetVariable               = (t_aiGetVariable)               GetProcAddress(hLib, "aiGetVariable");
   aiGetVariableVB             = (t_aiGetVariableVB)             GetProcAddress(hLib, "aiGetVariableVB");
   aiGetCSVFileModelSize       = (t_aiGetCSVFileModelSize)       GetProcAddress(hLib, "aiGetCSVFileModelSize");
   aiSetCapacity               = (t_aiSetCapacity)               GetProcAddress(hLib, "aiSetCapacity");
   aiGetCapacity               = (t_aiGetCapacity)               GetProcAddress(hLib, "aiGetCapacity");
   aiGetFreeEntries            = (t_aiGetFreeEntries)            GetProcAddress(hLib, "aiGetFreeEntries");
   aiInsertModelVector         = (t_aiInsertModelVector)         GetProcAddress(hLib, "aiInsertModelVector");
   aiOverwriteModelVector      = (t_aiOverwriteModelVector)      GetProcAddress(hLib, "aiOverwriteModelVector");
   aiAppendModelVector         = (t_aiAppendModelVector)         GetProcAddress(hLib, "aiAppendModelVector");
   aiDeleteModelVector         = (t_aiDeleteModelVector)         GetProcAddress(hLib, "aiDeleteModelVector");
   aiPredictionEx              = (t_aiPredictionEx)              GetProcAddress(hLib, "aiPredictionEx");
   aiExcludeModelVector        = (t_aiExcludeModelVector)        GetProcAddress(hLib, "aiExcludeModelVector");
   aiExcludeModelVectorRange   = (t_aiExcludeModelVectorRange)   GetProcAddress(hLib, "aiExcludeModelVectorRange");
   aiIsModelVectorExcluded     = (t_aiIsModelVectorExcluded)     GetProcAddress(hLib, "aiIsModelVectorExcluded");
   aiSaveCSVFile               = (t_aiSaveCSVFile)               GetProcAddress(hLib, "aiSaveCSVFile");

   /*
    * GetProcAddress returns null on failure
    */
   if( aiRegistration == NULL
       || aiGetVersion == NULL
       || aiCreateModel == NULL
       || aiCreateModelFromCSVFile == NULL
       || aiDeleteModel == NULL
       || aiNormalize == NULL
       || aiDenormalize == NULL
       || aiPrediction == NULL
       || aiGetNumberOfVariables == NULL
       || aiGetNumberOfModelVectors == NULL
       || aiGetNumberOfInputVariables == NULL
       || aiSetDiscreteFlag == NULL
       || aiGetDiscreteFlag == NULL
       || aiSetVariable == NULL
       || aiGetVariable == NULL
       || aiGetVariableVB == NULL
       || aiGetCSVFileModelSize == NULL
		 || aiSetCapacity == NULL
       || aiGetCapacity == NULL
       || aiGetFreeEntries == NULL
       || aiInsertModelVector == NULL
       || aiOverwriteModelVector == NULL
       || aiAppendModelVector == NULL
       || aiDeleteModelVector == NULL
       || aiPredictionEx == NULL
       || aiExcludeModelVector == NULL
       || aiExcludeModelVectorRange == NULL
       || aiIsModelVectorExcluded == NULL
       || aiSaveCSVFile == NULL ) {
		MessageBox(NULL, "Failure locating procedures.", "Error",
            MB_OK|MB_APPLMODAL);
      return 0;
	}
   return 1;
}

/* THE END */
