// ForkServerTask.cpp: implementation of the ForkServerTask class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "ForkServerTask.h"
#include <atask\Exception.h>

//////////////////////////////////////////////////////////////////////
// Static Utilities
//////////////////////////////////////////////////////////////////////

static ForkId left_fork_of(PhilosopherId philo_id) 
{
	return (ForkId)philo_id;
}

static ForkId right_fork_of(PhilosopherId philo_id) 
{
	return (left_fork_of(philo_id) + 1) % N_FORKS;
}

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

ForkServerTask::ForkServerTask()
{
	this->activate();
}

ForkServerTask::~ForkServerTask()
{
	this->terminate();
}

//////////////////////////////////////////////////////////////////////
// Member Functions
//////////////////////////////////////////////////////////////////////

void ForkServerTask::body()
{
	bool done = false;
	
	// Initially all forks are available
	for (ForkId id = 0; id < N_FORKS; id++) {
		this->available[id] = true;
	}
	
	while (!done) {
		try {
			Any_Entry* pe_table[N_PHILOSOPHERS + 1 + 1];
			
			for (int ix = 0; ix < N_PHILOSOPHERS; ix++) {
				pe_table[ix] = 
					guarded(&request_and_wait[ix], 
							can_grab_both((PhilosopherId)ix));
			}
			pe_table[N_PHILOSOPHERS + 0] = &release;
			pe_table[N_PHILOSOPHERS + 1] = NULL;
			
			// Start the rendezvous
			Rendezvous accept(pe_table, OR_TERMINATE);
			switch (accept.entry_selector()) {
			case N_PHILOSOPHERS + 0: {
				// accepted release
				PhilosopherId philo_id = release.actual_parameter1(accept);
				
				this->available[left_fork_of(philo_id)] = true;
				this->available[right_fork_of(philo_id)] = true;
				break;
									 }
			default: {
				// accepted one of the request and wait
				PhilosopherId philo_id = (PhilosopherId)accept.entry_selector();
				
				this->available[left_fork_of(philo_id)] = false;
				this->available[right_fork_of(philo_id)] = false;
				break;
					 }
			}
		}
		CATCH_ANY {
			// An exception, try again
		}
	}
}

bool ForkServerTask::can_grab_both(PhilosopherId philo_id)
{
	return this->available[left_fork_of(philo_id)]
			&& this->available[right_fork_of(philo_id)];
}


Priority ForkServerTask::priority()
{
	return high_Priority;
}

size_t ForkServerTask::virtual_sizeof()
{
	return sizeof(ForkServerTask);
}
